/*
 * Decompiled with CFR 0.152.
 */
package org.jlab.coda.hipo;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.jlab.coda.hipo.Compressor;
import org.jlab.coda.hipo.HipoException;
import org.jlab.coda.hipo.RecordHeader;

public class RecordInputStream {
    private static final int DEFAULT_BUF_SIZE = 0x800000;
    private RecordHeader header;
    private ByteBuffer dataBuffer;
    private ByteBuffer recordBuffer;
    private ByteBuffer headerBuffer;
    private Compressor compressor = Compressor.getInstance();
    private int nEntries;
    private int userHeaderOffset;
    private int eventsOffset;
    private ByteOrder byteOrder;

    public RecordInputStream() {
        this.byteOrder = ByteOrder.LITTLE_ENDIAN;
        this.allocate(0x800000);
        this.header = new RecordHeader();
    }

    public RecordInputStream(ByteOrder order) {
        this.byteOrder = order;
        this.allocate(0x800000);
        this.header = new RecordHeader();
    }

    private void allocate(int size) {
        byte[] arrayData = new byte[size];
        this.dataBuffer = ByteBuffer.wrap(arrayData);
        this.dataBuffer.order(this.byteOrder);
        byte[] arrayRecord = new byte[size];
        this.recordBuffer = ByteBuffer.wrap(arrayRecord);
        this.recordBuffer.order(this.byteOrder);
        byte[] arrayHeader = new byte[56];
        this.headerBuffer = ByteBuffer.wrap(arrayHeader);
        this.headerBuffer.order(this.byteOrder);
    }

    public RecordHeader getHeader() {
        return this.header;
    }

    public ByteOrder getByteOrder() {
        return this.byteOrder;
    }

    private void setByteOrder(ByteOrder order) {
        this.byteOrder = order;
        this.dataBuffer.order(order);
        this.recordBuffer.order(order);
        this.headerBuffer.order(order);
    }

    public boolean hasIndex() {
        return this.header.getIndexLength() > 0;
    }

    public boolean hasUserHeader() {
        return this.header.getUserHeaderLength() > 0;
    }

    public byte[] getEvent(int index) {
        int firstPosition = 0;
        if (index > 0) {
            if (index >= this.header.getEntries()) {
                index = this.header.getEntries() - 1;
            }
            firstPosition = this.dataBuffer.getInt((index - 1) * 4);
        } else {
            index = 0;
        }
        int lastPosition = this.dataBuffer.getInt(index * 4);
        int length = lastPosition - firstPosition;
        byte[] event = new byte[length];
        int offset = this.eventsOffset + firstPosition;
        if (this.dataBuffer.hasArray()) {
            System.arraycopy(this.dataBuffer.array(), offset, event, 0, length);
        } else {
            this.dataBuffer.limit(length + offset).position(offset);
            this.dataBuffer.get(event, 0, length);
        }
        return event;
    }

    public ByteBuffer getEvent(ByteBuffer buffer, int index) throws HipoException {
        return this.getEvent(buffer, 0, index);
    }

    public ByteBuffer getEvent(ByteBuffer buffer, int bufOffset, int index) throws HipoException {
        int firstPosition = 0;
        if (index > 0) {
            if (index >= this.header.getEntries()) {
                throw new HipoException("index too large");
            }
            firstPosition = this.dataBuffer.getInt((index - 1) * 4);
        }
        int lastPosition = this.dataBuffer.getInt(index * 4);
        int length = lastPosition - firstPosition;
        int offset = this.eventsOffset + firstPosition;
        if (buffer == null) {
            buffer = ByteBuffer.wrap(new byte[length]);
            bufOffset = 0;
        } else if (bufOffset < 0 || bufOffset + length > buffer.capacity()) {
            if (bufOffset < 0) {
                throw new HipoException("negative offset arg");
            }
            throw new HipoException("buffer with offset " + bufOffset + " is smaller than the event.");
        }
        buffer.order(this.byteOrder);
        if (buffer.hasArray() && this.dataBuffer.hasArray()) {
            System.arraycopy(this.dataBuffer.array(), offset, buffer.array(), buffer.arrayOffset() + bufOffset, length);
        } else {
            this.dataBuffer.limit(offset + length).position(offset);
            buffer.clear();
            buffer.position(bufOffset);
            buffer.put(this.dataBuffer);
        }
        buffer.limit(bufOffset + length).position(bufOffset);
        return buffer;
    }

    public byte[] getUserHeader() {
        int length = this.header.getUserHeaderLength();
        if (length < 1) {
            return null;
        }
        byte[] userHeader = new byte[length];
        if (this.dataBuffer.hasArray()) {
            System.arraycopy(this.dataBuffer.array(), this.userHeaderOffset, userHeader, 0, length);
        } else {
            this.dataBuffer.limit(length + this.userHeaderOffset).position(this.userHeaderOffset);
            this.dataBuffer.get(userHeader, 0, length);
        }
        return userHeader;
    }

    public ByteBuffer getUserHeader(ByteBuffer buffer, int bufOffset) throws HipoException {
        int length = this.header.getUserHeaderLength();
        if (length < 1) {
            return null;
        }
        if (buffer == null) {
            buffer = ByteBuffer.wrap(new byte[length]);
            bufOffset = 0;
        } else if (bufOffset < 0 || bufOffset + length > buffer.capacity()) {
            if (bufOffset < 0) {
                throw new HipoException("negative offset arg");
            }
            throw new HipoException("buffer with offset " + bufOffset + " is smaller than the user header.");
        }
        buffer.order(this.byteOrder);
        if (buffer.hasArray() && this.dataBuffer.hasArray()) {
            System.arraycopy(this.dataBuffer.array(), this.userHeaderOffset, buffer.array(), buffer.arrayOffset() + bufOffset, length);
        } else {
            this.dataBuffer.limit(this.userHeaderOffset + length).position(this.userHeaderOffset);
            buffer.clear();
            buffer.position(bufOffset);
            buffer.put(this.dataBuffer);
        }
        buffer.limit(bufOffset + length).position(bufOffset);
        return buffer;
    }

    public RecordInputStream getUserHeaderAsRecord(ByteBuffer buffer, int bufOffset) throws HipoException {
        ByteBuffer buf = this.getUserHeader(buffer, bufOffset);
        if (buf == null) {
            return null;
        }
        if (buffer == null) {
            bufOffset = 0;
        }
        RecordInputStream record = new RecordInputStream(this.byteOrder);
        record.readRecord(buf, bufOffset);
        return record;
    }

    public void readRecord(RandomAccessFile file, long position) throws HipoException {
        if (file == null || position < 0L) {
            throw new HipoException("bad argument(s)");
        }
        try {
            FileChannel channel = file.getChannel();
            channel.position(position);
            file.read(this.headerBuffer.array());
            this.header.readHeader(this.headerBuffer);
            this.setByteOrder(this.headerBuffer.order());
            int recordLengthWords = this.header.getLength();
            int headerLength = this.header.getHeaderLength();
            int cLength = this.header.getCompressedDataLength();
            int neededSpace = this.header.getIndexLength() + 4 * this.header.getUserHeaderLengthWords() + 4 * this.header.getDataLengthWords();
            int n = neededSpace = neededSpace < cLength ? cLength : neededSpace;
            if (this.dataBuffer.capacity() < neededSpace) {
                this.allocate(neededSpace);
            }
            channel.position(position + (long)headerLength);
            switch (this.header.getCompressionType()) {
                case 1: 
                case 2: {
                    file.read(this.recordBuffer.array(), 0, cLength);
                    Compressor.uncompressLZ4(this.recordBuffer, cLength, this.dataBuffer);
                    break;
                }
                case 3: {
                    file.read(this.recordBuffer.array(), 0, cLength);
                    byte[] unzipped = Compressor.uncompressGZIP(this.recordBuffer.array(), 0, cLength);
                    this.dataBuffer.put(unzipped);
                    break;
                }
                default: {
                    file.read(this.dataBuffer.array(), 0, recordLengthWords - headerLength);
                }
            }
            this.nEntries = this.header.getEntries();
            this.userHeaderOffset = this.nEntries * 4;
            this.eventsOffset = this.userHeaderOffset + this.header.getUserHeaderLengthWords() * 4;
            int event_pos = 0;
            for (int i = 0; i < this.nEntries; ++i) {
                int size = this.dataBuffer.getInt(i * 4);
                this.dataBuffer.putInt(i * 4, event_pos += size);
            }
        }
        catch (IOException ex) {
            Logger.getLogger(RecordInputStream.class.getName()).log(Level.SEVERE, null, ex);
        }
        catch (HipoException ex) {
            Logger.getLogger(RecordInputStream.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    public void readRecord(ByteBuffer buffer, int offset) throws HipoException {
        if (buffer == null || offset < 0) {
            throw new HipoException("bad argument(s)");
        }
        try {
            this.header.readHeader(buffer, offset);
            this.setByteOrder(buffer.order());
            int recordLengthWords = this.header.getLength();
            int headerLength = this.header.getHeaderLength();
            int cLength = this.header.getCompressedDataLength();
            int compDataOffset = offset + headerLength;
            int neededSpace = this.header.getIndexLength() + 4 * this.header.getUserHeaderLengthWords() + 4 * this.header.getDataLengthWords();
            int n = neededSpace = neededSpace < cLength ? cLength : neededSpace;
            if (this.dataBuffer.capacity() < neededSpace) {
                this.allocate(neededSpace);
            }
            switch (this.header.getCompressionType()) {
                case 1: 
                case 2: {
                    if (buffer.hasArray() && this.recordBuffer.hasArray()) {
                        System.arraycopy(buffer.array(), buffer.arrayOffset() + compDataOffset, this.recordBuffer.array(), 0, cLength);
                    } else {
                        buffer.limit(compDataOffset + cLength).position(compDataOffset);
                        this.recordBuffer.put(buffer);
                    }
                    Compressor.uncompressLZ4(this.recordBuffer, cLength, this.dataBuffer);
                    break;
                }
                case 3: {
                    if (buffer.hasArray() && this.recordBuffer.hasArray()) {
                        System.arraycopy(buffer.array(), buffer.arrayOffset() + compDataOffset, this.recordBuffer.array(), 0, cLength);
                    } else {
                        buffer.limit(compDataOffset + cLength).position(compDataOffset);
                        this.recordBuffer.put(buffer);
                    }
                    byte[] unzipped = Compressor.uncompressGZIP(this.recordBuffer.array(), 0, cLength);
                    this.dataBuffer.put(unzipped);
                    break;
                }
                default: {
                    int len = recordLengthWords - headerLength;
                    if (buffer.hasArray() && this.dataBuffer.hasArray()) {
                        System.arraycopy(buffer.array(), buffer.arrayOffset() + compDataOffset, this.dataBuffer.array(), 0, len);
                        break;
                    }
                    buffer.limit(compDataOffset + len).position(compDataOffset);
                    this.dataBuffer.put(buffer);
                }
            }
            this.nEntries = this.header.getEntries();
            this.userHeaderOffset = this.nEntries * 4;
            this.eventsOffset = this.userHeaderOffset + this.header.getUserHeaderLengthWords() * 4;
            int event_pos = 0;
            for (int i = 0; i < this.nEntries; ++i) {
                int size = this.dataBuffer.getInt(i * 4);
                this.dataBuffer.putInt(i * 4, event_pos += size);
            }
        }
        catch (HipoException ex) {
            Logger.getLogger(RecordInputStream.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    public int getEntries() {
        return this.nEntries;
    }

    private void showIndex() {
        for (int i = 0; i < this.nEntries; ++i) {
            System.out.printf("%3d  ", this.dataBuffer.getInt(i * 4));
        }
        System.out.println();
    }

    public static void main(String[] args) {
        try {
            RandomAccessFile outStreamRandom = new RandomAccessFile("example_file.evio", "r");
            long position = 104L;
            RecordInputStream istream = new RecordInputStream();
            istream.readRecord(outStreamRandom, position);
            int nevents = istream.getEntries();
            for (int i = 0; i < nevents; ++i) {
                byte[] event = istream.getEvent(i);
                System.out.printf("%4d : size = %8d\n", i, event.length);
            }
        }
        catch (FileNotFoundException ex) {
            Logger.getLogger(RecordInputStream.class.getName()).log(Level.SEVERE, null, ex);
        }
        catch (HipoException ex) {
            Logger.getLogger(RecordInputStream.class.getName()).log(Level.SEVERE, null, ex);
        }
    }
}

