#ifndef _CDEV_SERVICE_DEFINITION_H_
#define _CDEV_SERVICE_DEFINITION_H_

#ifndef _CDEV_DIRECTORY_TABLE_H_
	#error "You must include cdevDirectoryTable.h to load cdevServiceDefinition.h"
#endif

#include <stdio.h>

// *****************************************************************************
// * class cdevServiceDefinition :
// *	This class stores a single CDEV service definition as defined in the
// *	DDL file.  This data is managed by the cdevServiceList.
// *
// *	This class maintains a variety of internal data items for the following
// *	purposes...
// * 
// *	name      - This is the name of the service as specified in the 
// *                CDEV DDL file.
// *    tags      - These are the tags that will be used as service data when 
// *                loading elements that use this service.
// *	nTags     - This is the number of tags in the tags array.
// *	next      - This is the next service that was loaded from the CDEV DDL 
// *                file.  This list is maintained in the cdevDirectoryTable
// *                object.
// *****************************************************************************
class cdevServiceDefinition
{
private:
	cdevServiceDefinition * next;
	char                  * name;
	char                 ** tags;
	int                     nTags;
public:
	inline  cdevServiceDefinition  ( char *Name=NULL, char ** Tags=NULL, int NTags=0);
	inline ~cdevServiceDefinition  ( void );
	inline void asciiDump          ( FILE * fp = stdout );

	// *********************************************************************
	// * Member access methods.
	// *********************************************************************
	inline char                  * getName   ( void );
	inline char                 ** getTags   ( void );
	inline int                     getTagCnt ( void );     
	inline cdevServiceDefinition * getNext   ( void );
	inline void                    setNext   ( cdevServiceDefinition * Next );
};


// *****************************************************************************
// * cdevServiceDefinition::cdevServiceDefinition :
// *	This is the constructor for the cdevService defintion.  It uses the
// *	following parameters:
// *
// *	Name  - the name of the service being added.
// *	Tags  - the service data tags that will be used by each device.
// *	NTags - the number of tags in the list.
// *****************************************************************************
inline cdevServiceDefinition::cdevServiceDefinition  ( char *Name, char ** Tags, int NTags)
	: next(NULL), name(Name), tags(Tags), nTags(NTags)
	{
	}

// *****************************************************************************
// * cdevServiceDefinition::cdevServiceDefinition :
// *	This is the destructor for the cdevServiceDefinition class.  It will
// *	delete all data items that were provided to the constructor.
// *****************************************************************************
inline cdevServiceDefinition::~cdevServiceDefinition ( void )
	{
	if(name) free(name);
	while(--nTags>=0) free(tags[nTags]);
	if(tags)          delete[] tags;
	}
		
// *****************************************************************************
// * cdevServiceDefinition::asciiDump :
// *	This method is used to write the contents of the service definition 
// *	object to an ouput file.
// *****************************************************************************
inline void cdevServiceDefinition::asciiDump ( FILE * fp )
	{
	fprintf(fp, "\nservice %s\n\t{\n\t", name);
	if(tags && nTags)
		{
		int idx;
		fprintf(fp, "tags {");
		for(idx=0; idx<nTags-1; idx++)
			{
			fprintf(fp, "%s, ", tags[idx]);
			}
		fprintf(fp, "%s}\n\t", tags[idx]);
		}
	fprintf(fp, "}\n");
	}

// *****************************************************************************
// * cdevServiceDefinition::getName :
// *	This method allows the caller to retrieve the name of the class.
// *****************************************************************************
inline char * cdevServiceDefinition::getName ( void )
	{
	return name;
	}

// *****************************************************************************
// * cdevServiceDefinition::getTags :
// *	This method allows the caller to retrieve the tag names.
// *****************************************************************************
inline char ** cdevServiceDefinition::getTags ( void )
	{
	return tags;
	}

// *****************************************************************************
// * cdevServiceDefinition::getTagCnt :
// *	This method allows the caller to retrieve the number of tag names.
// *****************************************************************************
inline int cdevServiceDefinition::getTagCnt ( void )
	{
	return nTags;
	}

// *****************************************************************************
// * cdevServiceDefinition::getNext :
// *	This method allows the caller to retrieve the next service definition
// *	in the list.
// *****************************************************************************
inline cdevServiceDefinition * cdevServiceDefinition::getNext ( void )
	{
	return next;
	}

// *****************************************************************************
// * cdevServiceDefinition::setNext :
// *	This method allows the caller to set the next service definition in the
// *	list.
// *****************************************************************************
inline void cdevServiceDefinition::setNext ( cdevServiceDefinition * Next )
	{
	next = Next;
	}

#endif /* _CDEV_SERVICE_DEFINITION_H_ */
