//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
//-----------------------------------------------------------------------------
//
// Description:
//       cdevAccount classes:  These classes provide the basic functionality
//       for CDEV accounting. 
//
// Author:  Walt Akers
//-----------------------------------------------------------------------------

#ifndef _CDEV_ACCOUNT_H_
#define _CDEV_ACCOUNT_H_ 1

#include <cdevSystem.h>

// *****************************************************************************
// * class cdevAccountRecord :
// *	The cdevAccountRecord class is used to store information about a single
// *	CDEV application. This class contains the mechanisms necessary to write
// *	this information to a disk file and then read entries back in sequentially
// *****************************************************************************
class cdevAccountRecord 
{
public:
	enum {MAX_TEMP_BUF=4096};

	cdevAccountRecord        ( void );
	~cdevAccountRecord       ( void );
	
	void    reset            ( void );
	int     streamIn         ( FILE * fp  );
	int     streamIn         ( int    fd  );
	char *  streamIn         ( char * ptr );
	int     streamOut        ( FILE * fp  );
	int     streamOut        ( int    fd  );
	
	void    asciiDump        ( FILE * fp = stdout );
	
	cdevAccountRecord * getNext ( void );
	void                setNext ( cdevAccountRecord * Next );
	
	pid_t   getProcessID     ( void );
	void    setProcessID     ( pid_t ProcessID );
	char *  getStartTime     ( void );
	void    setStartTime     ( time_t StartTime );
	char *  getAppName       ( void );
	void    setAppName       ( char * AppName );
	char ** getArguments     ( void );
	void    setArguments     ( char ** args, int nArgs);
	char *  getUserName      ( void );
	void    setUserName      ( char * UserName );
	char *  getHostName      ( void );
	void    setHostName      ( char * HostName );
	char *  getSystemName    ( void );
	void    setSystemName    ( char * OSName );
	char *  getSystemRelease ( void );
	void    setSystemRelease ( char * OSRelease );
	char *  getCDEVVersion   ( void );
	void    setCDEVVersion   ( char * CDEVVersion );
	char ** getServiceList   ( void );
	void    setServiceList   ( char ** ServiceList, int nServices );
	void    addService       ( char * ServiceName );
		
protected:
	// *********************************************************************
	// * This is the pointer to the next cdevAccountRecord object that may
	// * be stored in a cdevAccountTable
	// *********************************************************************
	cdevAccountRecord * next;

	// *********************************************************************
	// * This is a static buffer that will be used to perform reads
	// * whenever the size of the data to be read does not exceed the
	// * basic size.
	// *********************************************************************
	static char tempBuf[MAX_TEMP_BUF];
	
	// *********************************************************************
	// * This is the process ID of the running CDEV application.
	// *********************************************************************
	pid_t   processID;
	
	// *********************************************************************
	// * This is the time that the application was started.
	// *********************************************************************
	char    startTime  [20];
	
	// *********************************************************************
	// * This is the name of the application and the arguments that were
	// * provided when it was started.
	// *********************************************************************
	char    appName    [256];
	int     argCnt;
	char ** argList;
	
	// *********************************************************************
	// * This is the name of the user that started the application.
	// *********************************************************************
	char    userName   [128];

	// *********************************************************************
	// * This is the name of the host where the application was started
	// * and pertinent information about the operating system.
	// *********************************************************************
	char    hostName   [128];
	char    osName     [32];
	char    osRelease  [32];

	// *********************************************************************
	// * This is the version of CDEV that the application is using. 
	// * Includes the suffix (SHARED) or (ARCHIVE).
	// *********************************************************************
	char    cdevVersion[64];

	// *********************************************************************
	// * These are the services that the application has loaded.
	// *********************************************************************
	int     serviceCnt;
	int     serviceMax;
	char ** serviceList;
};


// *****************************************************************************
// * class cdevAccountEntry:
// *	This class is used to create a single instance of a cdevAccountRecord
// *	that will be associated with a CDEV application.  This class instance
// *	will be called to write accounting information to the CDEV Accounting
// *	file whenever a significant event (start-up, service load, shutdown)
// *	occurs.
// *****************************************************************************
class cdevAccountEntry : public cdevAccountRecord
{
public:
	cdevAccountEntry  ( char * OutputFile = NULL );
	~cdevAccountEntry ( void );
	int  updateRecord ( void );
	int  removeRecord ( void );
	void initialize   ( void );
	void addService   ( char * service );   

private:
	char outputFile[256];
};

extern cdevAccountEntry * cdevGlobalAccountEntry;

#endif /* _CDEV_ACCOUNT_H_ */
