//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
//-----------------------------------------------------------------------------
//
// Description:
//      RSVC database interface 
//
// Author:  Jie Chen
//
//
//
//
#ifndef _RSVC_VIRTUAL_DATABASE_H
#define _RSVC_VIRTUAL_DATABASE_H

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <rsvcErr.h>
#include <rsvcData.h>
#include <rsvcServerConfig.h>

class rsvcVirtualDbase
{
public:
  // constructor
  // default constructor, users have to call open explicitly
  rsvcVirtualDbase (void);

  // constructor: name is a file name, flags are the same flags as
  // standard open routine.
  // data_size is estimate of each data stored in the database
  rsvcVirtualDbase (char* name, int flags, int keyType = RSVC_STRING,
		    size_t data_size = 496,
		    size_t cache_size = 0, size_t page_size = 0, 
		    int mode = 0666);
  
  // destructor
  virtual ~rsvcVirtualDbase (void);

  // operation

  // open a database
  virtual int open      (char* name, int flags, int keyType = RSVC_STRING,
			 size_t cache_size = 0,
			 int mode = 0666);
  // create a database
  virtual int create    (char* name, int flags, int keyType = RSVC_STRING,
			 size_t data_size = 496,
			 size_t cache_size = 0, size_t page_size = 0, 
			 int mode = 0666);

  // close connection to the database
  virtual int close     (void) {return RSVC_SUCCESS;}

  // get a rsvcData. 
  virtual int get       (rsvcData& /* data */, rsvcData& /* key */) {return RSVC_SUCCESS;}
  // insert a rsvcData which must have a key inside
  virtual int put       (rsvcData& /* data */, rsvcData& /* key */) {return RSVC_SUCCESS;}
  // delete a data object pointed by key value
  virtual int del       (rsvcData& /* key */) {return RSVC_SUCCESS;}
  // flush all internal buffer to disk
  virtual int flush     (void) {return RSVC_SUCCESS;}

  // return database name
  char* database (void) const {return name_;}
  // return file descriptor for the opened database
  virtual int   fd       (void) {return 0;}
  // check to see the database is open or not
  virtual int   opened   (void) const {return opened_;}

  // iterator for sequential access to database
  virtual int cursorInit (void) {return RSVC_SUCCESS;}

  // set cursor to the position pointed by key 'key',
  // and return data 'data' if successful
  virtual int cursorSet (rsvcData& /* data */, rsvcData& /* key */) {return RSVC_SUCCESS;}

  // move cursor to the next position. 
  // return CDEV_SUCCESS on success, CDEV_ERROR: check errno
  // return CDEV_INVALIDOBJ: no more
  virtual int cursorNext (rsvcData& /* data */, rsvcData& /* key */) {return RSVC_SUCCESS;}

  // move cursor to the previous position
  // return CDEV_SUCCESS on success, CDEV_ERROR: check errno
  // return CDEV_INVALIDOBJ: no more
  virtual int cursorPrev (rsvcData& /* data */, rsvcData& /* key */) {return RSVC_SUCCESS;}

  // merge current value pointed by cursor:
  // this routine can only used to update the data with the same
  // or smaller size than the exisiting data
  // If you like to update arbitrary size of data, use del and put
  virtual int cursorUpdate (rsvcData& /* data */) {return RSVC_SUCCESS;}

  // move cursor to the beginning of the database
  // no need to provide key
  virtual int cursorFirst  (rsvcData& /* data */, rsvcData& /* key */) {return RSVC_SUCCESS;}

  // move cursor to the end of the database
  // no need to provide key
  virtual int cursorLast   (rsvcData& /* data */, rsvcData& /* key */) {return RSVC_SUCCESS;}

  // close cursor operation
  virtual int cursorFinish (void) {return RSVC_SUCCESS;}

  virtual const char* className (void) const {return "rsvcVirtualDbase";}

protected:
  // open flag and mode
  int       flag_;
  int       mode_;
  // page size
  size_t    page_size_;
  // database name
  char*     name_;

  // flag to tell whether this database is opened or not
  int       opened_;

  // key type of this database
  int       keyType_;

  // error status of open or create of this database
  int       err_;

  // deny access to copy and assignment operator
  rsvcVirtualDbase (const rsvcVirtualDbase& dbase);
  rsvcVirtualDbase& operator = (const rsvcVirtualDbase& dbase);
};
#endif  
  
  
  
