//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
//-----------------------------------------------------------------------------
// 
// Description:
//	 Single Linked List for pointers rsvcHashable pointer
//       Primary used inside rsvcHash
//
//       Note: remove and clean operations on the list
//             will only remove link nodes without removal of
//             the content inside the nodes. It is callers' 
//             responsiblity to clean up those contents
//
//	
// Author:  Jie Chen
//       CEBAF Data Acquisition Group
//
//
//
//
#ifndef _RSVC_HSLIST_H
#define _RSVC_HSLIST_H

#include <stdio.h>
#include <string.h>

#if defined (RSVC_USE_THREAD) && defined (_REENTRANT)
#include <cpSynch.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>


//======================================================================
//	class rsvcSlist
//        Single Linked List for void* pointer
//======================================================================
class rsvcHSlistLink;
class rsvcHSlistIterator;
class rsvcHSlistCursor;
class rsvcHash;
class rsvcHashable;

class rsvcHSlist
{
public:
  // constructors
  rsvcHSlist          (void);
  rsvcHSlist          (const rsvcHSlist & source);
  virtual ~rsvcHSlist (void);

  // operations

  // add list item to the beginning of the list
  virtual void        add               (rsvcHashable* value);
  // add item to the end of list
  virtual void        addToEnd          (rsvcHashable* value);

  // remove a list item from the list
  // return 0: nothing to remove
  // return 1: remove success
  virtual int         remove            (rsvcHashable* value);

  // clean up the list. 
  virtual void        deleteAllValues   (void);

  // return first element of the list
  virtual rsvcHashable*  firstElement   (void);

  // return last element of the list
  virtual rsvcHashable*  lastElement    (void);

  // duplicate ths whole list
  virtual rsvcHSlist*       duplicate   (void);
  
  // check whether this list contains a particular item
  // return 1: yes. return 0: no
  virtual int              includes     (rsvcHashable* value);

  // Is list empty
  // return 1: yes. return 0: no
  virtual int              isEmpty      (void);

  // remove first element of the list
  virtual void             removeFirst  (void);

  // return number of elements inside the list
  virtual int              count        (void);

protected:
  // data field
  rsvcHSlistLink*           ptrToFirstLink;

  // number elements in the list
  int                       count_;

#if defined (RSVC_USE_THREAD) && defined (_REENTRANT)
  cpMutex                  lock_;
#endif

  // some internal functions which must be called with lock held
  virtual void           add_i              (rsvcHashable* value); 
  virtual int            isEmpty_i          (void) const;
  virtual void           deleteAllValues_i  (void);
  virtual void           removeFirst_i      (void);
  virtual rsvcHashable*  firstElement_i     (void);
  virtual rsvcHashable*  lastElement_i      (void);

  // friends
  friend class  rsvcHSlistIterator;
  // cannot modify list in anyways
  friend class  rsvcHSlistCursor;
  friend class  rsvcHash;

private:
  // deny access to assignment operation
  rsvcHSlist&   operator = (const rsvcHSlist& list);
};

//======================================================================
//	class rsvcHSlistLink
//        Single linked list link node
//======================================================================
class rsvcHSlistLink
{
public:
  // constructor
  rsvcHSlistLink         (rsvcHashable* linkValue, rsvcHSlistLink * nextPtr);

  // insert a new element following the current value
  rsvcHSlistLink* insert (rsvcHashable* val);
  // return data value
  rsvcHashable*  data   (void);

  // return next data item
  rsvcHSlistLink* next   (void);


private:

  // duplicate
  rsvcHSlistLink*    duplicate (void);

  // data areas
  rsvcHashable*      value;
  rsvcHSlistLink*     ptrToNextLink;

  // friends
  friend class       rsvcHSlist;
  friend class       rsvcHSlistIterator;
  friend class       rsvcHSlistCursor;
};

//===================================================================
//	class rsvcHSlistIterator
//		implements iterator protocol for linked lists
//		also permits removal and addition of elements
//===================================================================

class rsvcHashIterator;

class rsvcHSlistIterator
{
public:
  // constructor
  rsvcHSlistIterator        (rsvcHSlist& aList);

  // iterator protocol
  virtual int              init (void);
  virtual rsvcHashable*    operator () (void);
  virtual int              operator !  (void);
  virtual int              operator ++ (void);
  virtual void             operator =  (rsvcHashable* value);

  // new methods specific to list iterators

  // remove current item pointed by the iterator from the list
  void                     removeCurrent(void);
  
  // add an item to the list before the position pointed by the iterator
  void                     addBefore(rsvcHashable*  newValue);

  // add an item to the list after the position pointed by the iterator
  void                     addAfter(rsvcHashable*   newValue);

  // search an item and move the iterator to that position
  int                      searchSame(rsvcHashable* &value);

protected:
  // some internal functions which must be called with lock held
  virtual int              init_i (void);
  // move cursor forward
  virtual int              forward_i (void);
  // internal remove current
  void                     removeCurrent_i (void);
  // data areas
  rsvcHSlistLink *    currentLink;
  rsvcHSlistLink *    previousLink;
  rsvcHSlist&         theList;
#if defined (RSVC_USE_THREAD) && defined (_REENTRANT)
  cpMutex             lock_;
#endif

private:
  // friend class
  friend class rsvcHashIterator;

  // deny access of copy and assignment
  rsvcHSlistIterator (const rsvcHSlistIterator& ir);
  rsvcHSlistIterator& operator = (const rsvcHSlistIterator& ir);
};

//===================================================================
//	class rsvcHSlistCursor
//		implements cursor protocol for linked lists
//===================================================================
class rsvcHSlistCursor
{
public:
  // constructor
  rsvcHSlistCursor           (const rsvcHSlist& aList);

  // iterator protocol
  virtual int               init        (void);
  virtual rsvcHashable*     operator () (void);
  virtual int               operator !  (void);
  virtual int               operator ++ (void);
  virtual void              operator =  (rsvcHashable* value);

  int                       searchSame  (rsvcHashable* &value);

protected:
  // internal functions which must be called with lock 
  virtual int               init_i      (void);  
  // data areas
  rsvcHSlistLink *          currentLink;
  rsvcHSlistLink *          previousLink;
  const rsvcHSlist&         theList;
#if defined (RSVC_USE_THREAD) && defined (_REENTRANT)
  cpMutex             lock_;
#endif

private:
  // deny access to copy and assignment operations
  rsvcHSlistCursor (const rsvcHSlistCursor &);
  rsvcHSlistCursor& operator = (const rsvcHSlistCursor &);
};
#endif
