//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
//-----------------------------------------------------------------------------
// 
// Description:
//	 cdevError class (abstract base class)
//	
// Author:  Jie Chen & Chip Watson
//
// Revision History:
//   cdevError.h,v
// Revision 1.5  1997/08/27  18:23:31  chen
// Change error reporting to site specific scheme
//
// Revision 1.4  1995/12/14  19:11:14  chen
// add support to C interface
//
// Revision 1.3  1995/10/17  17:44:13  akers
// Added setThreshold functionality and made data items protected
//
// Revision 1.2  1995/10/17  15:29:45  chen
// change reportError interface
//
// Revision 1.1.1.1  1995/06/16  17:14:08  epics
// initial import of cdev
//
//
#ifndef _CDEV_ERROR_H
#define _CDEV_ERROR_H

#include <stdarg.h>
#include <cdevSpec.h>

#define _CDEV_MAX_ERR_HANDLERS   5
#define CDEV_SEVERITY_VOID_DATA  0xcde5da2a

class cdevRequestObject;

typedef void (*cdevErrorHandler)(int severity, 
				 char *text, 
				 cdevRequestObject* obj);

class CDEV_CLASS_SPEC cdevError
{
 public:
  // destructor
  virtual ~cdevError (void);

  virtual int autoErrorOn  (void);
  // PURPOSE: turn on flag which enables auto error on
  // REQUIRE: derived class can implement diffrently
  // PROMISE: return 0

  virtual int autoErrorOff (void);
  // PURPOSE: turn on flag which disables auto error on
  // REQUIRE: derived class can implement diffrently
  // PROMISE: return 0

  virtual int vreportError (int severity,
                            const char * name,
			    cdevRequestObject * obj,
			    const char * formatString,
			    va_list argp);

  // PURPOSE: report error to somewhere
  // REQUIRE: formatString can have anything that works in printf
  // PROMISE: same as printf
						   	
  virtual int reportError (int severity, 
			   const char *name, 
			   cdevRequestObject* obj, 
			   const char *formatString,...);
  // PURPOSE: report error to somewhere
  // REQUIRE: formatString can have anything that works in printf
  // PROMISE: same as printf

  virtual void reportError (int severity, 
			    char *message,
			    cdevRequestObject* obj);
  // PURPOSE: report error to somewhere with a whole string
  // REQUIRE: user whole string
  // PROMISE: 


  virtual void setErrorHandler (cdevErrorHandler handler = 0);
  // PURPOSE: allow caller to set up his/her own error Handler
  // REQUIRE: handle > 0 stands for a real function pointer
  // PROMISE: next time error will be handled by new error handler

  virtual void setThreshold ( int errorThreshold );
  // PURPOSE: allow the caller to specify the minimum severity to report
  // REQUIRE: errorThreshold >=0
  // PROMISE: errors with severity < errorThreshold wil not be reported

  virtual int  attachErrorHandler (cdevErrorHandler handler);
  // PURPOSE: allow caller to attach error handler to trap error messages
  // REQUIRE: handler > 0
  // PROMISE: return CDEV_SUCCESS if this handler is attached to trap messgaes

  virtual int  detachErrorHandler (cdevErrorHandler handler);
  // PURPOSE: allow caller to dettach error handler
  // REQUIRE: handler > 0
  // PROMISE: return CDEV_SUCCESS if this handler is removed.

  virtual void reportError (void *data,
			    cdevRequestObject* obj);
  // PURPOSE: report error to somewhere with user defined data object.
  //          This routine is intended for site developer use only to
  //          support site wide error logging system.
  // REQUIRE: user whole string
  // PROMISE: 

  virtual const char *className(void) const {return "cdevError";}

 protected:
  // constructers, avoiding direct instantation
  cdevError (void);
  cdevError (cdevErrorHandler handler);
  
  // data areas
  cdevErrorHandler        errorHandler_;
  static cdevErrorHandler defaultErrorHandler_;
  int                     autoErrorOn_;
  int                     threshold_;

  // list of error handlers
  int                     numhandlers_;
  cdevErrorHandler        handlers_[_CDEV_MAX_ERR_HANDLERS];
};
#endif  
