#include <cmlogBrowser.h>
#include <cmlogClient.h>

#include <tcl.h>
#include <tk.h>

#include <sys/types.h>
#include <ctype.h>
#include <string.h>
#include <iostream.h>

// cdev includes
#include <cdevData.h>

// Forward declarations
extern "C" {void Cmlog_Init(Tcl_Interp *);};
int  Cmlog_Conn_Cmd           (ClientData, Tcl_Interp *, int, char **);
int  Cmlog_Cmd                (ClientData, Tcl_Interp *, int, char **);
int  Cmlog_Cmd_Connect        (ClientData, Tcl_Interp *, int, char **);
int  Cmlog_Cmd_Disconnect     (ClientData, Tcl_Interp *, int, char **);
int  Cmlog_Cmd_Connected      (ClientData, Tcl_Interp *, int, char **);
int  Cmlog_Cmd_Monitor        (ClientData, Tcl_Interp *, int, char **);
int  Cmlog_Cmd_Query          (ClientData, Tcl_Interp *, int, char **);
static void Cmlog_queryCbk    (int, void*, cmlogPacket*);
static void Cmlog_monCbk      (int, void*, cmlogPacket*);
static void Cmlog_Cmd_Delete  (ClientData);
static void Cmlog_getData     (ClientData, int);
static void Cmlog_clearResult (void);
static int  Cmlog_cdevToTcl   (cdevData &, Tcl_DString &);

// Globals
#define CMLOG_WAITING 99

// Structure containing data for query/monitor callbacks
typedef struct cmlog_tclStuff {
    Tcl_Interp   *interp;	// tcl interpretor
    cmlogBrowser browser;	// each cmlog server connection has its own browser
    char	 tclcmd[80];	// tcl command to invoke for get, query, etc.
    char	 tclarray[80];	// tcl array name in which to put returned data
    double	 timeStart;     // query starts this time
    double	 timeFinal;     // query ends this time
    int		 monitorStatus; // status of monitor callback
    int		 queryStatus;   // status of query callback
    int		 monitorOn;     // 1= mon on, 0= mon off (prevent additional ons)
    int		 append;        // 1= append data, 0= replace data
    int		 error;		// 1= error occurred; 0= OK
} cmlog_ts;


//...............................

void
Cmlog_Init(Tcl_Interp *interp)
{
  Tcl_CreateCommand(interp, "cmlog", Cmlog_Cmd, 0, 0);
}

//...............................

int
Cmlog_Cmd(ClientData clientData, Tcl_Interp *interp, int argc,
	    char **argv)
{
  if ((argc > 1) && (strcmp(argv[1], "connect") == 0)) {
    return Cmlog_Cmd_Connect(clientData, interp, argc, argv);
  }

  Tcl_AppendResult(interp, "usage: cmlog connect <cmdName> <dataArrayName> [<port>]", 0);
  return TCL_ERROR;
}

//...............................

int
Cmlog_Cmd_Connect(ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
    unsigned short port=0;
    int   fd;
    cmlog_ts *pcts = NULL;

    if ((argc > 5 ) || (argc < 4) || (*argv[2] == '\0') || (*argv[3] == '\0')) {
	Tcl_AppendResult(interp, "usage: cmlog connect <cmdName> <dataArrayName> [<port>]", 0);
	return TCL_ERROR;
    }

    // set port number if given
    if (argc > 4 && *argv[4] != '\0') {
	port = atoi(argv[4]);
    }
    
    // store relevant data in struct
    if ((pcts = new cmlog_ts) == NULL)
    {
	Tcl_AppendResult(interp, "cmlog connect: Cannot allocate needed memory", 0);
	return TCL_ERROR;
    }
    // store tcl command & tcl data array names
    strncpy(pcts->tclcmd, argv[2], 78);
    *((pcts->tclcmd)+79) = '\0';
    strncpy(pcts->tclarray, argv[3], 78);
    *((pcts->tclarray)+79) = '\0';
    pcts->interp = interp;
    pcts->monitorStatus = CMLOG_WAITING;
    pcts->queryStatus   = CMLOG_INCOMPLETE;
    pcts->timeStart     = 0.0;
    pcts->timeFinal     = 0.0;
    pcts->monitorOn     = 0;
    pcts->append        = 1;
    pcts->error         = 0;

    // to add a new tag:
    // cdevData::insertTag (1301, "junk");

    // connect browser to cmlogServer
    if (port > 0) {
	if (pcts->browser.connect(port,3) == CMLOG_SUCCESS) {
	    printf ("Connect to the cmlog server on port %s\n", argv[4]);
	} else {
	    Tcl_AppendResult(interp, "cmlog connect: Cannot connect to the cmlog server on port ", argv[4], 0);
	    printf ("Cannot connect to the cmlog server on port %s\n", argv[4]);
	    return TCL_ERROR;
	}
    }
    else 
    {
	if (pcts->browser.connect() == CMLOG_SUCCESS) {
	    printf ("Connect to the cmlog server\n");
	} else {
	    Tcl_AppendResult(interp, "cmlog connect: Cannot connect to the cmlog server", 0);
	    printf ("Cannot connect to the cmlog server\n");
	    return TCL_ERROR;
	}
    }

    // another command by this name will be automatically deleted
    Tcl_CreateCommand(interp, argv[2], Cmlog_Conn_Cmd, (ClientData *)pcts, Cmlog_Cmd_Delete);
    // create an array variable argv[3], and set argv3(length) = 0
    if (Tcl_SetVar2(interp, argv[3], "length", "0", TCL_LEAVE_ERR_MSG|TCL_GLOBAL_ONLY) == NULL)
	return TCL_ERROR;

    fd =  pcts->browser.getFd();
    Tk_CreateFileHandler(fd, TK_READABLE, Cmlog_getData, (ClientData *) pcts);

    return TCL_OK;
}


//...............................

void
Cmlog_Cmd_Delete(ClientData clientData)
{    
    cmlog_ts *pcts = (cmlog_ts *) clientData;
    delete pcts;
}

//...............................

int
Cmlog_Conn_Cmd(ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{  
  if (argc < 2) {
    Tcl_AppendResult(interp, argv[0], ": missing command name", 0);
    return TCL_ERROR;
  }

  if (strcmp(argv[1], "query") == 0)
    return Cmlog_Cmd_Query(clientData, interp, argc, argv); 
  else if (strcmp(argv[1], "monitor") == 0)
    return Cmlog_Cmd_Monitor(clientData, interp, argc, argv); 
  else if (strcmp(argv[1], "connected") == 0)
    return Cmlog_Cmd_Connected(clientData, interp, argc, argv);
  else if (strcmp(argv[1], "disconnect") == 0) 
    return Cmlog_Cmd_Disconnect(clientData, interp, argc, argv);

  Tcl_AppendResult(interp, "For ", argv[0], ", ", argv[1], " is an unsupported command", 0);
  return TCL_ERROR;
}

//...............................

int
Cmlog_Cmd_Disconnect(ClientData clientData, Tcl_Interp *interp,
		     int argc, char **argv)
{
    cmlog_ts *pcts = (cmlog_ts *) clientData;
    int fd;
    
    if (argc != 2) {
	Tcl_AppendResult(interp, "usage: ", argv[0], " disconnect", 0);
	return TCL_ERROR;
    }
  
    fd =  pcts->browser.getFd();
    Tk_DeleteFileHandler(fd);
    pcts->browser.disconnect();
    return Tcl_DeleteCommand(interp, argv[0]);
}

//...............................


int
Cmlog_Cmd_Connected(ClientData clientData, Tcl_Interp *interp,
		    int argc, char **argv)
{
    cmlog_ts *pcts = (cmlog_ts *) clientData;

    if (argc !=2) {
	Tcl_AppendResult(interp, "usage: ", argv[0], " connected", 0);
	return TCL_ERROR;
    }

    sprintf(interp->result, "%d", pcts->browser.connected());
    return TCL_OK;
}

//...............................

int
Cmlog_Cmd_Query(ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
    cmlog_ts *pcts = (cmlog_ts *) clientData;
    int res, status, maxNum, append;
    double start, end;
    cdevData data;

    if ((argc < 5) || (argc > 7)) {
	Tcl_AppendResult(interp, "Usage: ", argv[0], " query <append> <starttime> <endtime> [<maxNumOfItems>] [<filter>]", 0);
	return TCL_ERROR;
    }

    start = atof(argv[3]);
    end   = atof(argv[4]);
    data.insert ("start", start);
    data.insert ("end", end);
//printf ("Cmlog_Cmd_Query: start,end = %lf, %lf\n", start, end);

    if (argc > 6) {
	if (argv[6] != NULL) {
	    data.insert ("queryMsg", argv[6]);
//printf ("Cmlog_Cmd_Query: filter is %s\n",argv[6]);
	}
	else {
	    ;
//printf ("Cmlog_Cmd_Query: filter is NULL\n");
	}
    }
    if (argc > 5) {
	maxNum = atoi(argv[5]);
//printf ("Cmlog_Cmd_Query: numItems = %d\n", maxNum);
	if (maxNum > 0)
	    data.insert ("numberItems", maxNum);
    }
    pcts->append    = atoi(argv[2]);
    pcts->timeStart = start;
    pcts->timeFinal = end;
    
    status = pcts->browser.queryCallback ("query", data, Cmlog_queryCbk, (void *)pcts);
    if (status != CMLOG_SUCCESS) {
	Tcl_AppendResult(interp, "Query command to cmlog server failed", 0);
	return TCL_ERROR;
    }

    return TCL_OK;
}


//...............................

int
Cmlog_Cmd_Monitor(ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
    cmlog_ts *pcts = (cmlog_ts *) clientData;
    char  flag[15];
    char  command[1024];
    cdevData data;

    if (((argc < 4) || (argc > 5))||
	((strcmp(argv[2],"on")!=0) && (strcmp(argv[2],"off")!=0)) ||
	((strcmp(argv[3],"tags")!=0) && (strcmp(argv[3],"data")!=0)))
    {
	Tcl_AppendResult(interp, "Usage: ", argv[0], " monitor on/off  tags/data [<filter>]", 0);
	return TCL_ERROR;
    }
    
    if (strcmp(argv[3],"data")==0)
	strcpy(flag,"loggingData");
    else
	strcpy(flag,"allTags");
    
    if (strcmp(argv[2],"on") == 0) {
//printf("Cmlog_Cmd_Monitor: ON\n");
	sprintf(command, "%s %s", "monitorOn", flag);
	if (pcts->monitorOn == 1)
	    return TCL_OK;
	else
	    pcts->monitorOn = 1;
    }
    else {
//printf("Cmlog_Cmd_Monitor: OFF\n");
	sprintf(command, "%s %s", "monitorOff", flag);
	if (pcts->monitorOn == 0)
	    return TCL_OK;
	else
	    pcts->monitorOn = 0;
    }
    
    if ((argc == 5) && (argv[4] != NULL)) {
	data.insert ("queryMsg", argv[4]);
    }

    pcts->browser.queryCallback (command, data, Cmlog_monCbk, (void *)clientData);

    return TCL_OK;
}

//...............................

void
Cmlog_getData(ClientData clientData, int mask)
{
    cmlog_ts *pcts = (cmlog_ts *) clientData;
//printf("Cmlog_getData:  pendIO\n");
    pcts->browser.pendIO();
    return;
}


//...............................

static void
Cmlog_queryCbk (int status, void* arg, cmlogPacket* data)
{
    Tcl_DString	result;
    cdevData	*res = 0;
    static int	previousState = CMLOG_SUCCESS;
    static int	count = 0;
    int		i, flags = TCL_LEAVE_ERR_MSG|TCL_GLOBAL_ONLY;
    cmlog_ts	*pcts = (cmlog_ts *) arg;
    char	num[25];
    char	*pchar;
    char	*lengthStr;
   
    pcts->queryStatus = status;
    Tcl_DStringInit(&result);

    // If error occurred in previous call to this func in which data couldn't
    // be written to tcl or couldn't handle cdev result, then ignore all
    // subsequent data.
    if (pcts->error == 1)
    {
printf("QueryCbk: error...");
	Tcl_DStringFree(&result);
	previousState = status;
	if (status == CMLOG_INCOMPLETE)
	{
	    return;
	}
	else
	{
printf("QueryCbk: query data error\n");
	    pcts->error = 0;
	    Tcl_SetVar2 (pcts->interp, pcts->tclarray, "qstate", "error", flags);
	    Tcl_SetVar2 (pcts->interp, pcts->tclarray, "qdataIn", "2", flags);
	    return;
	}
    }
    // If this is the signal for an end to a data transmission,
    // end things and reset static variables for next round.
    else if ((status == CMLOG_SUCCESS) && (previousState == CMLOG_INCOMPLETE))
    {
//printf("QueryCbk: query success\n");
	Tcl_DStringFree(&result);
	previousState = status;
	Tcl_SetVar2 (pcts->interp, pcts->tclarray, "qstate", "success", flags);
	pchar = Tcl_SetVar2 (pcts->interp, pcts->tclarray, "qdataIn", "2", flags);
	if (pchar == NULL) {
	    printf("QueryCbk: query error in setting \"qdataIn\" array element\n");
	}
	return;
    }
    // If this is the signal for an end to a data transmission,
    // end things and reset static variables for next round.
    else if ((status == CMLOG_PAUSED) && (previousState == CMLOG_INCOMPLETE))
    {
//printf("QueryCbk: query paused\n");
	Tcl_DStringFree(&result);
	previousState = status;
	Tcl_SetVar2 (pcts->interp, pcts->tclarray, "qstate", "paused", flags);
	pchar = Tcl_SetVar2 (pcts->interp, pcts->tclarray, "qdataIn", "2", flags);
	if (pchar == NULL) {
	    printf("QueryCbk: query error in setting \"qdataIn\" array element\n");
	}
	return;
    }
    // If this is a data transmission with nothing in it,
    // clear the tcl data array & set length element to zero
    // to indicate the fact.
    else if (status == CMLOG_NOTFOUND)
    {
//printf("QueryCbk: query data notfound\n");
	Tcl_DStringFree(&result);
	previousState = status;
	Tcl_UnsetVar(pcts->interp, pcts->tclarray, flags);
	if (pcts->append == 0)
	    Tcl_SetVar2 (pcts->interp, pcts->tclarray, "append", "0", flags);
	else
	    Tcl_SetVar2 (pcts->interp, pcts->tclarray, "append", "1", flags);
	Tcl_SetVar2 (pcts->interp, pcts->tclarray, "length", "0", flags);
	Tcl_SetVar2 (pcts->interp, pcts->tclarray, "qstate", "notfound", flags);
	pchar = Tcl_SetVar2 (pcts->interp, pcts->tclarray, "qdataIn", "2", flags);
	if (pchar == NULL) {
	    printf("Query_Cbk: query error in setting \"qdataIn\" array element\n");
	}
	return;
    }
    // If there is some other condition, print error message.
    else if (status != CMLOG_INCOMPLETE)
    {
	Tcl_DStringFree(&result);
	previousState = status;
	Tcl_SetVar2 (pcts->interp, pcts->tclarray, "qstate", "error", flags);
	if (status == CMLOG_QUERYMSG_ERR) {
	    printf("QueryCbk: query message syntax error\n");
	    Tcl_SetVar2 (pcts->interp, pcts->tclarray, "error",
		"query message syntax error", flags);
	}
	else {
	    printf("QueryCbk: query error condition, status = %d, prevstatus = %d\n",status,previousState);
	    Tcl_SetVar2 (pcts->interp, pcts->tclarray, "error",
		"query unknown error", flags);
	}
	
	Tcl_SetVar2 (pcts->interp, pcts->tclarray, "qdataIn", "2", flags);
	return;
    }
    
    // If start of new transfer
    if (previousState != CMLOG_INCOMPLETE)
    {
	Tcl_UnsetVar(pcts->interp, pcts->tclarray, flags);
	Tcl_SetVar2 (pcts->interp, pcts->tclarray, "start",  "1", flags);
	if (pcts->append == 0)
	    Tcl_SetVar2 (pcts->interp, pcts->tclarray, "append", "0", flags);
	else 
	    Tcl_SetVar2 (pcts->interp, pcts->tclarray, "append", "1", flags);
	count = 0;
	sprintf(num,"%lf",pcts->timeStart);
	Tcl_SetVar2 (pcts->interp, pcts->tclarray, "timestart", num, flags);
	sprintf(num,"%lf",pcts->timeFinal);
	Tcl_SetVar2 (pcts->interp, pcts->tclarray, "timefinal", num, flags);
    }

//printf ("QueryCbk: query status = %d\n", status);

    if (data)
    {
	cmlogMsg** msgs = data->messages ();
	for (i = 0; i < data->numberOfData (); i++)
	{
	    cmlog_cdevMessage& idata = (cmlog_cdevMessage&)(*msgs[i]);
	    res = idata.getData ();
//res->asciiDump();
	    if (Cmlog_cdevToTcl(*res, result) != TCL_OK)
	    {
		printf("QueryCbk: query cannot handle cdev result\n");
		Tcl_SetVar2 (pcts->interp, pcts->tclarray, "error",
		    "cannot handle cdev result", flags);
		pcts->error = 1;
		goto error;
	    }
	    count++;
	    sprintf(num,"%d",count);
//printf("num = %s\n",num);
	    pchar = Tcl_SetVar2(pcts->interp, pcts->tclarray, num,
			Tcl_DStringValue(&result), flags);
	    if (pchar == NULL) {
		printf("QueryCbk: query error in setting array element\n");
		Tcl_SetVar2 (pcts->interp, pcts->tclarray, "error",
		    "error in setting array element", flags);
		pcts->error = 1;
		goto error;
	    }
//printf("Q_Cbk: %s(%d)=%s\n", pcts->tclarray,count,Tcl_DStringValue(&result));
	}
	
	pchar = Tcl_SetVar2(pcts->interp, pcts->tclarray, "length", num, flags);
	if (pchar == NULL) {
	    printf("QueryCbk: query error in setting \"length\" array element\n");
	    Tcl_SetVar2 (pcts->interp, pcts->tclarray, "error",
		"error in setting \"length\" array element", flags);
	    pcts->error = 1;
	}
	
     error:	
	for (i = 0; i < data->numberOfData (); i++) 
	    delete msgs[i];
	delete []msgs;
    }
    
    previousState = status;
    Tcl_DStringFree(&result);
    return;
}

//...............................

static void
Cmlog_monCbk (int status, void* arg, cmlogPacket* data)
{
    Tcl_DString	result;
    cdevData	*res = 0;
    static int	previousState = CMLOG_WAITING;
    int		i, count = 0;
    int		flags = TCL_LEAVE_ERR_MSG|TCL_GLOBAL_ONLY;
    cmlog_ts	*pcts = (cmlog_ts *) arg;
    char	num[25];
    char	*pchar;
    char	*lengthStr;
    
    pcts->monitorStatus = status;
    pcts->error = 0;
    Tcl_DStringInit(&result);
    
//printf ("Cmlog_monCbk: status of monitorOn is %d\n", status);
    
    // If this is the signal for an end to a data transmission,
    // end things and reset static variables for next round.
    if (status == CMLOG_CBK_FINISHED)
    {
//printf("Mon_Cbk: monitor turned off\n");
	Tcl_DStringFree(&result);
	previousState = status;
	Tcl_SetVar2 (pcts->interp, pcts->tclarray, "mstate", "cbkfinished", flags);
	Tcl_SetVar2 (pcts->interp, pcts->tclarray, "mdataIn", "2", flags);
	return;
    }
    // Error Condition
    else if (status != CMLOG_SUCCESS)
    {
printf("Mon_Cbk: monitor error\n");
	Tcl_DStringFree(&result);
	previousState = status;
	printf("Mon_Cbk: Error condition %d\n",status);
	Tcl_SetVar2 (pcts->interp, pcts->tclarray, "mstate", "error", flags);
	Tcl_SetVar2 (pcts->interp, pcts->tclarray, "error",
		    "unknown monitor error", flags);
	Tcl_SetVar2 (pcts->interp, pcts->tclarray, "mdataIn", "2", flags);
	return;
    }

    // Append data to existing array
    Tcl_SetVar2 (pcts->interp, pcts->tclarray, "append", "1", flags);

    if (data)
    {
	cmlogMsg** msgs = data->messages ();
	for (i = 0; i < data->numberOfData (); i++)
	{
	    cmlog_cdevMessage& idata = (cmlog_cdevMessage&)(*msgs[i]);
	    res = idata.getData ();
//res->asciiDump();
	    if (Cmlog_cdevToTcl(*res, result) != TCL_OK)
	    {
		printf("MonCbk: monitor cannot handle cdev result\n");
		Tcl_SetVar2 (pcts->interp, pcts->tclarray, "error",
		    "cannot handle cdev result", flags);
		Tcl_SetVar2 (pcts->interp, pcts->tclarray, "mdataIn", "2", flags);
		goto error;
	    }
	    count++;
	    sprintf(num,"%d",count);
//printf("Mon_Cbk: %s(%d)=%s\n", pcts->tclarray,count,Tcl_DStringValue(&result));
	    pchar = Tcl_SetVar2(pcts->interp, pcts->tclarray, num,
			Tcl_DStringValue(&result), flags);
	    if (pchar == NULL) {
		printf("Mon_Cbk: monitor error in setting array element\n");
		Tcl_SetVar2 (pcts->interp, pcts->tclarray, "error",
		    "error in setting array element", flags);
		Tcl_SetVar2 (pcts->interp, pcts->tclarray, "mdataIn", "2", flags);
		goto error;
	    }
	}
	
	pchar = Tcl_SetVar2(pcts->interp, pcts->tclarray, "length", num, flags);
	if (pchar == NULL) {
	    printf("Mon_Cbk: monitor error in setting \"length\" array element\n");
	    Tcl_SetVar2 (pcts->interp, pcts->tclarray, "error",
		"error in setting \"length\" array element", flags);
	    Tcl_SetVar2 (pcts->interp, pcts->tclarray, "mdataIn", "2", flags);
	    goto error;
	}
	
	Tcl_SetVar2 (pcts->interp, pcts->tclarray, "mstate", "success", flags);
	pchar = Tcl_SetVar2 (pcts->interp, pcts->tclarray, "mdataIn", "2", flags);
	if (pchar == NULL) {
	    printf("Mon_Cbk: monitor error in setting \"mdataIn\" array element\n");
	}
	
     error:  
	for (i = 0; i < data->numberOfData (); i++) 
	    delete msgs[i];
	delete []msgs;
    }

//printf("Mon_Cbk: monitor success\n");
    Tcl_DStringFree(&result);
    previousState = status;
    return;
}

//...............................
// Format a cdevData into a tcl result

static int
Cmlog_cdevToTcl(cdevData & result, Tcl_DString & ds) {
  int    tag;
  size_t dim;
  char   *pchar; 
  char   cs[256];

  Tcl_DStringInit(&ds);
  cdevDataIterator iter(&result);
 
  // Retrieve result as char*, cdevData does conversion
  for (iter.init(); !iter ; ++iter) {
    tag = iter.tag();
    result.tagI2C(tag, pchar);
    result.getDim(tag, &dim);
    if (dim != 0) continue;

    result.get(tag,cs,255);
    Tcl_DStringStartSublist(&ds);
    Tcl_DStringAppendElement(&ds,pchar);
    Tcl_DStringAppendElement(&ds,cs);
    Tcl_DStringEndSublist(&ds);
  }  
  return TCL_OK;
}

