/*
 *	tclStruct package
 *  Support 'C' structures in Tcl
 *
 *  Written by Matthew Costello
 *  (c) 1995 AT&T Global Information Solutions, Dayton Ohio USA
 *
 *  See the file "license.terms" for information on usage and
 *  redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */
#include "stInternal.h"
STRUCT_SCCSID("@(#)tclStruct:stTypeDef.c	1.2	95/09/12")

/* vxWorks has no strdup */
#ifdef VXWORKS
extern char * strdup _ANSI_ARGS_((char *s));
#endif

/*******************************************************************/
/*
 * Struct_Def, easily callable from C version of Struct_TypeDefCmd
 *   'def' is one of:
 *	typename[*count]
 *	{typename [options]}
 *	{"struct" [options]
 *		{type1 name1 [options]}
 *		{type2 name2 [options]} }
 */

Struct_TypeDef *
Struct_ParseDefOptions(cdata,interp,type,pelem,argc,argv)
  ClientData cdata;
  Tcl_Interp *interp;                 /* Current interpreter. */
  Struct_TypeDef *type;
  Struct_StructElem *pelem;
  int argc;
  char **argv;
{
    int i;

    /*  If there are no type options, then we've nothing to do.
     */
    if (argc <= 0)
	return type;
    Struct_CheckType(type,"ParseDefOptions");

    /*  If there are type options, and the type we have is used
     *  elsewhere, then make a clone of the type rather than
     *  having the unintended side effect of modifying somebody
     *  elses type definition.
     */
    if (type->refcount > 1) {
	type = Struct_CloneType( cdata, interp, (char *)NULL, type );
	if (type == NULL)
	    return NULL;
    }

    /*  Now process any options.  Options ALWAYS occur in pairs
     *  with the option name followed by the value.
     */
    if (argc & 01) {
	Tcl_AppendResult(interp,"type options occur in pairs", NULL );
	return NULL;
    }
    for ( i = 0; i < argc; i++ ) {
	if (strcmp(argv[i],"align") == 0) {
	    int value;
	    if (Tcl_GetInt(interp,argv[++i],&value) == TCL_ERROR)
		return NULL;
	    if (value <= 0) {
		Tcl_AppendResult(interp,"\"align\" must be positive", NULL );
		return NULL;
	    }
	    if ( (value != type->align) &&
		 (type->refcount > 1) &&
		 ((type = Struct_CloneType( cdata, interp, (char *)NULL, type )) == NULL) )
			return NULL;
	    type->align = value;
	} else if (strcmp(argv[i],"basic") == 0) {
	    int flags = type->flags & ~(STRUCT_FLAG_TRACE_BASIC);
	    int v;
	    if (Tcl_GetBoolean(interp,argv[++i],&v) == TCL_ERROR)
		return NULL;
	    if (v)
		flags |= STRUCT_FLAG_TRACE_BASIC;
	    if (flags != type->flags) {
		if ((type->flags & STRUCT_FLAG_IS_ARRAY) &&
		    (type->u.a.array_elem->flags & STRUCT_FLAG_TRACE_ARRAY)) {
		    if ( (type->refcount > 1) &&
			 ((type = Struct_CloneType( cdata, interp, (char *)NULL, type )) == NULL) )
				return NULL;
		    if (v)
			type->TraceProc = type->u.a.array_elem->TraceProc;
		    else
			type->TraceProc = Struct_TraceArray;
		    type->flags = flags;
		} else {
		    Tcl_AppendResult(interp,"\"basic\" cannot be changed", NULL );
		    return NULL;
		}
	    }
	} else if (strcmp(argv[i],"endian") == 0) {
	    int flags = type->flags & ~(STRUCT_FLAG_BIG_ENDIAN);
	    if (strcmp( argv[++i], "big" ) == 0)
		flags |= STRUCT_FLAG_BIG_ENDIAN;
	    else if (strcmp( argv[i], "little" ) == 0)
		flags &= ~STRUCT_FLAG_BIG_ENDIAN;
	    else {
		Tcl_AppendResult(interp,"unrecognized endian \"",
			argv[i], "\"", NULL );
		return NULL;
	    }
	    if (type->flags & STRUCT_FLAG_USE_ENDIAN) {
		if ( (flags != type->flags) &&
		     (type->refcount > 1) &&
		     ((type = Struct_CloneType( cdata, interp, (char *)NULL, type )) == NULL) )
			    return NULL;
		type->flags = flags;
	    } else {
		Tcl_AppendResult(interp,"\"endian\" not used by this type", NULL );
		return NULL;
	    }
	} else if (strcmp(argv[i],"fill") == 0) {
	    if (!(type->flags & STRUCT_FLAG_USE_FILL)) {
		Tcl_AppendResult(interp,"\"fill\" not used by this type", NULL );
		return NULL;
	    } else if ( (type->fill == NULL) || (strcmp(type->fill,argv[i+1]) != 0) ) {
		if ( (type->refcount > 1) &&
		     ((type = Struct_CloneType( cdata, interp, (char *)NULL, type )) == NULL) )
			return NULL;
		if (type->fill != NULL)
		    ckfree( type->fill );
		type->fill = strdup( argv[++i] );
	    } else
		i++;
#ifdef STRUCT_FLAG_USE_JUST
	} else if (strcmp(argv[i],"justify") == 0) {
	    int flags = type->flags & ~STRUCT_FLAG_JUST_MASK;
	    int n = strlen( argv[++i] );
	    if (strncmp(argv[i],"left",n) == 0) {
		flags |= STRUCT_FLAG_JUST_LEFT;
	    } else if (strncmp(argv[i],"right",n) == 0) {
		flags |= STRUCT_FLAG_JUST_RIGHT;
	    } else if (strncmp(argv[i],"center",n) == 0) {
		flags |= STRUCT_FLAG_JUST_CENTER;
	    } else if (strncmp(argv[i],"none",n) == 0) {
		flags |= STRUCT_FLAG_JUST_NONE;
	    } else {
		Tcl_AppendResult(interp,"unrecognized justification \"",
			argv[i], "\"", NULL );
		return NULL;
	    }
	    if (!(type->flags & STRUCT_FLAG_USE_JUST)) {
		Tcl_AppendResult(interp,"\"justify\" not used by this type", NULL );
		return NULL;
	    }
	    if ( (flags != type->flags) &&
		 (type->refcount > 1) &&
		 ((type = Struct_CloneType( cdata, interp, (char *)NULL, type )) == NULL) )
			return NULL;
	    type->flags = flags;
#endif	/*STRUCT_FLAG_USE_JUST*/
	} else if (strcmp(argv[i],"nullok") == 0) {
	    int flags = type->flags & ~(STRUCT_FLAG_NULL_OK);
	    int v;
	    if (Tcl_GetBoolean(interp,argv[++i],&v) == TCL_ERROR)
		return NULL;
	    if (v)
		flags |= STRUCT_FLAG_NULL_OK;
	    if (!(type->flags & STRUCT_FLAG_USE_NULLOK)) {
		Tcl_AppendResult(interp,"\"nullok\" not used by this type", NULL );
		return NULL;
	    }
	    if ( (flags != type->flags) &&
		 (type->refcount > 1) &&
		 ((type = Struct_CloneType( cdata, interp, (char *)NULL, type )) == NULL) )
			return NULL;
	    type->flags = flags;
	} else if (strcmp(argv[i],"offset") == 0) {
	    if (pelem == NULL) {
		Tcl_AppendResult(interp,"\"offset\" only valid in structure", NULL );
		return NULL;
	    }
	    if (Tcl_GetInt(interp,argv[++i],&pelem->offset) == TCL_ERROR)
		return NULL;
	    if (pelem->offset < 0) {
		Tcl_AppendResult(interp,"\"offset\" cannot be negative", NULL );
		return NULL;
	    }
	} else if (strcmp(argv[i],"strict") == 0) {
	    int flags = type->flags & ~(STRUCT_FLAG_STRICT);
	    int v;
	    if (Tcl_GetBoolean(interp,argv[++i],&v) == TCL_ERROR)
		return NULL;
	    if (v)
		flags |= STRUCT_FLAG_STRICT;
	    if ( (flags != type->flags) &&
		 (type->refcount > 1) &&
		 ((type = Struct_CloneType( cdata, interp, (char *)NULL, type )) == NULL) )
			return NULL;
	    type->flags = flags;
	} else if (strcmp(argv[i],"unsigned") == 0) {
	    int flags = type->flags & ~STRUCT_FLAG_UNSIGNED;
	    int v;
	    if (Tcl_GetBoolean(interp,argv[++i],&v) == TCL_ERROR)
		return NULL;
	    if (v)
		flags |= STRUCT_FLAG_UNSIGNED;
	    if (!(type->flags & STRUCT_FLAG_USE_SIGN)) {
		Tcl_AppendResult(interp,"\"unsigned\" not usable on this type", NULL );
		return NULL;
	    }
	    
/* Some crazy bug in the GNU 68k compiler chokes on the code below */	    
/*	    if ( (flags != type->flags) &&
		 (type->refcount > 1) &&
		 ((type = Struct_CloneType( cdata, interp, (char *)NULL, type )) == NULL) )
			return NULL;
*/
/* Replace it with equivalent code that compiles */
	    if ((flags != type->flags) && (type->refcount > 1)) {
		Struct_TypeDef *type_temp;
		type_temp = Struct_CloneType( cdata, interp, (char *)NULL, type );
		if (type_temp == NULL) {
		    return NULL;
		} else {
		    type = type_temp;
		}
	    }
/* end */
	    type->flags = flags;
	} else {
	    Tcl_AppendResult(interp,"unrecognized type option \"",
		    argv[i], "\"", NULL );
	    return NULL;
	}
    }
    return type;
}




Struct_TypeDef *
Struct_ParseType(cdata,interp,def)
  ClientData cdata;
  Tcl_Interp *interp;                 /* Current interpreter. */
  char *def;
{
    Struct_TypeDef *type;
    int argc;
    char **argv;
    int i;

#ifdef DEBUG
    if (struct_debug & (DBG_PARSETYPE))
    printf("Struct_ParseType( definition = \"%s\"\n", def );
#endif

    /*  Split up the definition into a list of components.
     */
    if (Tcl_SplitList(interp,def,&argc,&argv) == TCL_ERROR)
	return NULL;
    if (argc < 1) {
	Tcl_AppendResult(interp, "empty type definition",NULL);
	ckfree((char *)argv);
	return NULL;
    }

    /* Structures are handled specially.
     */
    if (strcmp(argv[0],"struct") == 0) {
	Struct_StructElem *pelem;
	int cur_offset = 0;
	int cur_align = 1;
	int align;
	if ((pelem = (Struct_StructElem *)ckalloc( argc * sizeof(Struct_StructElem) )) == NULL) {
	    Tcl_SetResult(interp,"Can't allocate structure member definition!",TCL_STATIC);
	    return NULL;
	}
	memset( (char *)pelem, 0x00, argc * sizeof(Struct_StructElem) );
	if ((type = Struct_NewType(cdata,interp,(char *)NULL,0,
				STRUCT_FLAG_IS_STRUCT,Struct_TraceStruct)) == NULL) {
		ckfree( (char *)pelem );
		return NULL;
	}
	type->align = 0;
	type->u.s.struct_def = pelem;
	type->u.s.num_elements = 0;
	for ( i = 1; i < argc; ) {
	    int margc; char **margv;
	    /*  Figure out how many pieces are in the current element.
	     *  If it is >1 then we have a structure element.
	     */
	    if (Tcl_SplitList(interp,argv[i],&margc,&margv) == TCL_ERROR)
		return NULL;
	    /* Do we have a member of the structure? */
	    if (margc > 1) {
#ifdef DEBUG
		if (struct_debug & (DBG_PARSETYPE))
	        printf("Struct_ParseType: struct member name = %s, type = %s\n",
		    margv[1] ? margv[1] : "<none>", margv[0] );
#endif

		if (type->flags & STRUCT_FLAG_VARLEN) {
		    Tcl_AppendResult(interp,"variable length element must be last", NULL );
		    ckfree( (char *)margv);
		    return NULL;
		}

		pelem->offset = type->size;
		pelem->type = Struct_ParseType(cdata,interp,margv[0]);
		if (pelem->type == NULL) {
		    ckfree( (char *)margv);
		    return NULL;
		}
		if (*margv[1] != '\0')
		    pelem->name = strdup( margv[1] );
		else
		    pelem->name = strdup( "" );

		/* Now process any options.  */
		pelem->offset = -1;
		pelem->type = Struct_ParseDefOptions(cdata,interp,pelem->type,pelem,margc-2, margv+2 );
		if (pelem->type == NULL)
			return NULL;

		/* Calculate alignment (and offset) for this element */
		align = pelem->type->align;
		if (type->align > 0 && type->align < align)
		    align = type->align;
		if (pelem->offset < 0) {
		    /* Calculate offset */
		    if (align > 1) {
			    cur_offset = (cur_offset + align - 1) / align;
			    cur_offset *= align;
		    }
		    pelem->offset = cur_offset;
		} else if ( (align > 1) &&
			    ((pelem->offset % align) != 0) ) {
		    Tcl_AppendResult(interp,"offset incompatible with alignment", NULL );
		    ckfree( (char *)margv);
		    return NULL;
		}
		cur_offset = pelem->offset + pelem->type->size;
		if (align > cur_align) {
			cur_align = align;
		}
		if (cur_offset > type->size)
			type->size = cur_offset;	/* worry about alignment later */
		if (pelem->type->flags & STRUCT_FLAG_VARLEN)
			type->flags |= STRUCT_FLAG_VARLEN;
#ifdef DEBUG
		if (struct_debug & (DBG_PARSETYPE))
		printf("Struct_ParseType: struct member name = %s, type = %s, offset = %d, co=%d, ca=%d\n",
			pelem->name, Struct_TypeName(pelem->type), pelem->offset,
			cur_offset, cur_align );
#endif
		i++;
		if (pelem->name[0] == '\0') {
		    /* Don't keep place-holder structure elements */
		    free( pelem->name );
		    memset( (char *)pelem, 0x00, sizeof(Struct_StructElem) );
		} else {
		    type->u.s.num_elements++;
		    pelem++;
		}
	    }
	    
/* This following section was written to bypass a GNU compiler error */	    
	    if (margc <= 1) {
		int compare1, compare2;
		compare1 = strcmp(argv[i],"size");
		compare2 = strcmp(argv[i],"align");
		if (compare1 == 0) {
		    if (++i >= argc) {
			Tcl_AppendResult(interp,"missing structure size", NULL );
			ckfree( (char *)margv);
			return NULL;
		    }
		    if (Tcl_GetInt(interp,argv[i++],&type->size) == TCL_ERROR) {
			ckfree( (char *)margv);
			return NULL;
		    }
		}
		if ((compare1 != 0) && (compare2 == 0)) {
		    if (++i >= argc) {
			Tcl_AppendResult(interp,"missing structure align", NULL );
			ckfree( (char *)margv);
			return NULL;
		    }
		    if (Tcl_GetInt(interp,argv[i++],&type->align) == TCL_ERROR) {
			ckfree( (char *)margv);
			return NULL;
		    }
		    if (type->align <= 0) {
			Tcl_AppendResult(interp,"\"align\" must be positive", NULL );
			ckfree( (char *)margv);
			return NULL;
		    }
		}
		if ((compare1 != 0) && (compare2 != 0)) {
		    Tcl_AppendResult(interp,"unrecognized structure option \"",
			argv[i], "\"", NULL );
		    ckfree( (char *)margv);
		    return NULL;
		}
	    }
/* This above section was written to bypass a GNU compiler error */	    
	    ckfree( (char *)margv);
	}

	/*  Now check the current parameters */
	if (cur_align > type->align)
		type->align = cur_align;
	if (!(type->flags & STRUCT_FLAG_VARLEN)) {
		type->size = (type->size + type->align - 1) / type->align;
		type->size *= type->align;
	}
	
#ifdef DEBUG
	if (struct_debug & (DBG_PARSETYPE))
        printf("Struct_ParseType() = %s (struct with %d members)\n",
		Struct_TypeName(type), type->u.s.num_elements );
#endif
	return type;
    }

    /*  If we have a straight typename, then just return a pointer
     *  to the existing type.  If this type includes an array
     *  designator then that will be returned as well.
     */
    if ((type = Struct_LookupType(cdata,interp,argv[0])) == NULL) {
	return NULL;
    }

    /*  Now process any options.  Options ALWAYS occur in pairs
     *  with the option name followed by the value.
     */
    type = Struct_ParseDefOptions(cdata,interp,type,NULL,argc-1, argv+1 );
    if (type == NULL)
	return NULL;

#ifdef DEBUG
    if (struct_debug & (DBG_PARSETYPE))
    printf("Struct_ParseType() = %s\n", Struct_TypeName(type) );
#endif
    return type;
}



/*  Called ONLY by Struct_DefType() to fix up pointers to ourselves.
 */
static void Struct_FixSelfReferences( type, newtype, oldtype )
  Struct_TypeDef *type, *newtype, *oldtype;
{
    Struct_CheckType(oldtype,"FixSelfReferences");
#ifdef DEBUG
    if (struct_debug & (DBG_PARSETYPE))
    printf("Struct_FixSelfRef: checking %s\n", Struct_TypeName(type) );
#endif
    if (type->flags & (STRUCT_FLAG_IS_POINTER|STRUCT_FLAG_IS_ARRAY)) {
	if (type->u.a.array_elem == oldtype) {
#ifdef DEBUG
	    if (struct_debug & (DBG_NEWTYPE|DBG_PARSETYPE)) {
	      printf("Struct_FixSelfRef: replacing %s", Struct_TypeName(oldtype) );
	      printf(" in %s", Struct_TypeName(type) );
	      printf(" with %s\n", Struct_TypeName(newtype) );
	    }
#endif
	    type->u.a.array_elem = newtype;
	    type->flags |= STRUCT_FLAG_RECURSIVE;
	} else {
	    Struct_FixSelfReferences( type->u.a.array_elem, newtype, oldtype );
	}
    } else if (type->flags & STRUCT_FLAG_IS_STRUCT) {
	Struct_StructElem *pelem;
	for ( pelem = type->u.s.struct_def; pelem->type != NULL; pelem++ ) {
	    Struct_FixSelfReferences( pelem->type, newtype, oldtype );
	}
    }
}



int
Struct_DefType(cdata,interp,typename,def) 
  ClientData cdata;
  Tcl_Interp *interp;                 /* Current interpreter. */
  CONST char *typename;
  char *def;
{
    Struct_TypeDef   *type;
    Struct_TypeDef *faketype;
    Tcl_HashEntry *entryPtr;

#ifdef DEBUG
    if (struct_debug & (DBG_PARSETYPE))
    printf("Struct_DefType( typename = \"%s\", definition = \"%s\"\n",
	typename, def );
#endif

    /* We want to register the type first, before it has even
     * been defined, so that a structure can contain a pointer
     * to itself.
     */
    if ( ((faketype = Struct_NewType(cdata,interp, typename,
				0, STRUCT_FLAG_NONE,NULL)) == NULL) ||
	 (Struct_RegisterType(cdata, interp, typename, faketype) == TCL_ERROR) ) {
	return TCL_ERROR;
    }

    /* Recursively descend through the type 'def', building
     * the completed type definition one piece at a time.
     */
    if ((type = Struct_ParseType( cdata, interp, def )) == NULL)
	return TCL_ERROR;

    /* If this is an already existing type, then clone a copy
     * of it to name.
     */
    if ( (type->refcount > 1) &&
	 ((type = Struct_CloneType( cdata, interp, typename, type )) == NULL) )
	    return TCL_ERROR;

    /* Fix up any self-referential pointers in our type definition.
     * Then remove the fake type.
     */
    Struct_FixSelfReferences( type, type, faketype );
    if ((entryPtr = Tcl_FindHashEntry( Struct_TypeHash(cdata), (char *)typename )) == NULL) {
	Tcl_AppendResult(interp, "cannot find the type I just added", NULL);
	return TCL_ERROR;
    }
    Tcl_DeleteHashEntry(entryPtr);
    Struct_ReleaseType(faketype);	/* once for creation */
    Struct_ReleaseType(faketype);	/* and once for registration */

    /* Now really register the type.
     */
    if (Struct_RegisterType(cdata, interp, typename, type) == TCL_ERROR) {
	Struct_ReleaseType(type);
	return TCL_ERROR;
    }

    /* Now that it is registered, we can release it. */
    Struct_ReleaseType(type);

    return TCL_OK;
}
 
 
 
/*
 * Associate a field with a memory location
 *
 * usage : struct_def object field type size offset flags
 * usage : struct_def typename def
 *	def = {type name {addlinfo}}
 */
int
Struct_TypeDefCmd(cdata, interp, argc, argv)
  ClientData cdata;                   /* Client Data */
  Tcl_Interp *interp;                 /* Current interpreter. */
  int argc;                           /* Number of arguments. */
  char **argv;                        /* Argument strings. */
{
    CONST char *name;

    Struct_PkgInfo(cdata,si_cmdCount) += 1;
    if (argc!=3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
                       " typename typedef\"", (char *) NULL);
	return TCL_ERROR;
    }
#ifdef DEBUG
    if (struct_debug & (DBG_COMMAND)) Struct_PrintCommand(argc,argv);
#endif
    if (cdata==NULL) {
	Tcl_AppendResult(interp,"Called ",argv[0]," with NULL client data",(char *)NULL);
	return TCL_ERROR;
    }
    name = (strcmp(argv[1],"#auto") == 0) ?
	Struct_GenerateName("type") : argv[1];

    if (Struct_DefType(cdata,interp,name,argv[2])==TCL_ERROR)
	return TCL_ERROR;
    Tcl_AppendResult(interp,name,NULL);
    return TCL_OK;
}



int
Struct_UnTypeDefCmd(cdata, interp, argc, argv)
  ClientData cdata;                   /* Client Data */
  Tcl_Interp *interp;                 /* Current interpreter. */
  int argc;                           /* Number of arguments. */
  char **argv;                        /* Argument strings. */
{
    Struct_TypeDef *type;
    Tcl_HashEntry *entryPtr;

    Struct_PkgInfo(cdata,si_cmdCount) += 1;
    if (argc!=2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
                       " typeName\"", (char *) NULL);
	return TCL_ERROR;
    }
#ifdef DEBUG
    if (struct_debug & (DBG_COMMAND)) Struct_PrintCommand(argc,argv);
#endif
    if (cdata==NULL) {
	Tcl_AppendResult(interp,"Called ",argv[0]," with NULL client data",(char *)NULL);
	return TCL_ERROR;
    }

    if ((type = Struct_LookupType(cdata,interp,argv[1])) == NULL)
	return TCL_ERROR;

    if (type->name == NULL) {
	Struct_ReleaseType(type);
	Tcl_AppendResult(interp,"\"",argv[1],"\" is an anonymous type",(char *)NULL);
	return TCL_ERROR;
    }


    if (type->refcount != 2) {
	Struct_ReleaseType(type);
	Tcl_AppendResult(interp,"type \"",argv[1],"\" is still in use",(char *)NULL);
	return TCL_ERROR;
    }

    /*  Find the type name in the hash table.  */
    if ((entryPtr = Tcl_FindHashEntry(Struct_TypeHash(cdata),argv[1])) == NULL) {
	Struct_ReleaseType(type);
	Tcl_AppendResult(interp,"cannot find registered type in Struct_UnTypeDefCmd",
		(char *)NULL);
	return TCL_ERROR;
    }

    /* Remove it from the hash table */
    Tcl_DeleteHashEntry(entryPtr);

    /*  Now free the underlying type */
    Struct_ReleaseType(type);	/* once for creation */
    Struct_ReleaseType(type);	/* and once for registration */
    return TCL_OK;
}
