//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
//-----------------------------------------------------------------------------
//
// Description:
//      CODA run control requestObject Class
//
// Author:  Jie Chen
//
// Revision History:
//   $Log: codaRequestObject.cc,v $
//   Revision 1.6  2000/08/21 18:28:41  abbottd
//   Sun 5.0 C++ compiler fix
//
//   Revision 1.5  1999/07/28 20:10:08  rwm
//   Jie Chen-s changes for latest rcServer
//
//   Revision 1.4  1998/05/28 17:47:12  heyes
//   new GUI look
//
//   Revision 1.3  1997/02/18 17:06:29  chen
//   minor changes
//
//   Revision 1.2  1997/02/03 13:47:27  heyes
//   add ask command
//
//   Revision 1.1.1.1  1996/10/11 13:39:32  chen
//   run control source
//
//
#include <cdevTranObj.h>
#include <cdevGroup.h>
#include <cdevErrCode.h>
#include <cdevCommon.h>
#include <cdevDirectory.h>
#include <rcClient.h>

#include "codaRequestObject.h"
#include "codaService.h"
#include "codaMonObj.h"
#include "dplite.h"
int codaRequestObject::numAttributes = 13;

codaAttr codaRequestObject::codaAttributes[]=
{
  {"value",       0x00000001},
  {"status",      0x00000002},
  {"severity",    0x00000004},
  {"time",        0x00000008},
  {"units",       0x00000010},
  {"displayHigh", 0x00000020},
  {"displayLow",  0x00000040},
  {"alarmHigh",   0x00000080},
  {"alarmLow",    0x00000100},
  {"warningHigh", 0x00000200},
  {"warningLow",  0x00000400},
  {"controlHigh", 0x00000800},
  {"controlLow",  0x00001000},
};

codaStateString codaRequestObject::states[] =
{
  {"dormant", DA_DORMANT},
  {"booting", DA_BOOTING},
  {"booted", DA_BOOTED},
  {"configuring", DA_CONFIGURING},
  {"configured", DA_CONFIGURED},
  {"downloading", DA_DOWNLOADING},
  {"downloaded", DA_DOWNLOADED},
  {"prestarting", DA_PRESTARTING},
  {"paused", DA_PAUSED},
  {"pausing", DA_PAUSING},
  {"activating", DA_ACTIVATING},
  {"active", DA_ACTIVE},
  {"ending", DA_ENDING},
  {"verifying", DA_VERIFYING},
  {"verified", DA_VERIFIED},
 {"prestarted", DA_PRESTARTED},
  {"resetting", DA_TERMINATING},
};

int codaRequestObject::numStates = 17;



codaRequestObject::codaRequestObject (char* device, char* msg,
				      codaService* svc,
				      cdevSystem& system)
:cdevRequestObject (device, msg, system), device_ (0), attr_ (0), readonly_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("             Create codaRequestObject Class Object\n");
#endif
  // get service data
  int status = findSvcData (device, msg, system);
  // get attribute and action
  status = codaRequestObject::findAction (msg, action_, attr_);
  // cache device name
  device_ = new char[::strlen (device) + 1];
  ::strcpy (device_, device);
  // get pointer of real client handler
  codaClient_ = &(svc->codaClient_);
}

codaRequestObject::~codaRequestObject (void)
{
#ifdef _TRACE_OBJECTS
  printf ("             Delete codaRequestObject Class Object\n");
#endif
  // delete attr and device
  delete []device_;
  delete []attr_;
}

int
codaRequestObject::getState (void)
{
  if (codaClient_->connected ())
    return CDEV_STATE_CONNECTED;
  else
    return CDEV_STATE_NOTCONNECTED;
}

int
codaRequestObject::setContext (cdevData& cxt)
{
  // clean the mask
  reqMask_ = 0x0;
  reqType_ = codaRequestObject::VALUE;

  // get bitmask from cdev requestobject layer
  (void)cdevRequestObject::setContext (cxt);

  int temp;
  for (int i = 0; i < codaRequestObject::numAttributes; i++) {
    if (cxt.get ((codaRequestObject::codaAttributes[i]).attrName, &temp) 
	== CDEV_SUCCESS) 
      reqMask_ |= (codaRequestObject::codaAttributes[i]).attrValue;
  }
  
  if (reqMask_ == 0x0)
    reqType_ = codaRequestObject::VALUE;
  else if (reqMask_ >= codaRequestObject::VALUE &&
      reqMask_ < codaRequestObject::STATUS)
    reqType_ = codaRequestObject::VALUE;
  else if (reqMask_ >= codaRequestObject::STATUS &&
	   reqMask_ < codaRequestObject::TIMESTMP)
    reqType_ = codaRequestObject::VALUE;
  else if (reqMask_ >= codaRequestObject::TIMESTMP &&
	   reqMask_ <  codaRequestObject::GRINFO)
    reqType_ = codaRequestObject::VALUE;
  else if (reqMask_ >= codaRequestObject::GRINFO &&
	   reqMask_ < codaRequestObject::CTRLINFO)
    reqType_ = codaRequestObject::GRINFO;
  else
    reqType_ = codaRequestObject::GRINFO;

#ifdef _CDEV_DEBUG
  printf("reqType_ is %d mask is 0x%x \n", reqType_, reqMask_);
#endif
  return CDEV_SUCCESS;
}
  
int
codaRequestObject::send (cdevData& out, cdevData& result)
{
  return send (&out, &result);
}

int
codaRequestObject::send (cdevData* out, cdevData& result)
{
  return send (out, &result);
}

int
codaRequestObject::send (cdevData& out, cdevData* result)
{
  return send (&out, result);
}

int
codaRequestObject::send (cdevData* out, cdevData* result)
{
  int status = CDEV_ERROR;

  if (action_ == codaRequestObject::MSG) {
    char *reslt;
    char name[256];
    char cmd[1024];

    // command argument is associated with tag message
    out->get ("message",cmd, sizeof (cmd));

    if (DP_cmd(device_,cmd,&reslt) == 0 ) {
      result->insert ("value",reslt);
    } else {
      result->insert ("status",CDEV_ERROR);
      result->insert ("severity",reslt);
    }
    free(reslt);
    return CDEV_SUCCESS;
  }

  if (codaClient_->connected () ) {
    switch (action_) {
    case codaRequestObject::GET:
      if (result == 0)
	return CDEV_INVALIDARG;
      {
	cdevGroup getGrp (2, system_);
	getGrp.start ();
	sendNoBlock (out, result);
	getGrp.end ();
	getGrp.pend (CODA_DEFAULT_TIMEOUT);
	if (getGrp.allFinished ())
	  status = CDEV_SUCCESS;
	else
	  status = CDEV_TIMEOUT;
      }
      break;
    case codaRequestObject::DISCONNECT:
      codaClient_->disconnect ();
      status = CDEV_SUCCESS;
      break;
    case codaRequestObject::CONNECTED:
      if (result == 0)
	return CDEV_INVALIDARG;
      result->insert ("value", "yes");
      status = CDEV_SUCCESS;
      break;
    case codaRequestObject::LOAD:
    case codaRequestObject::CONFIGURE:
    case codaRequestObject::DOWNLOAD:
    case codaRequestObject::PRESTART:
    case codaRequestObject::PAUSE:
    case codaRequestObject::RESUME:
    case codaRequestObject::END:
    case codaRequestObject::RESET:
    case codaRequestObject::GO:
    case codaRequestObject::ABORT:
    case codaRequestObject::STATE:
    case codaRequestObject::NEW_STATE:
    case codaRequestObject::ASK:
      {
	cdevGroup getGrp (2, system_);
	getGrp.start ();
	sendNoBlock (out, result);
	getGrp.end ();
	getGrp.pend (CODA_DEFAULT_TIMEOUT);
	if (getGrp.allFinished ())
	  status = CDEV_SUCCESS;
	else
	  status = CDEV_TIMEOUT;
      }
      break;
    case codaRequestObject::SET:
      if (readonly_)
	return CDEV_NOACCESS;
      if (out == 0)
	return CDEV_INVALIDARG;
      {
	int sendErr = 0;
	cdevGroup setGrp (2, system_);
	setGrp.start ();
	if (sendNoBlock (out, result) != CDEV_SUCCESS)
	  sendErr = 1;
	setGrp.end ();
	if (!sendErr) {
	  setGrp.pend (CODA_DEFAULT_TIMEOUT);
	  if (setGrp.allFinished ())
	    status = CDEV_SUCCESS;
	  else
	    status = CDEV_TIMEOUT;
	}
	else
	  status = CDEV_INVALIDARG;
      }
      break;
    case codaRequestObject::MONITOR_ON:
      status = CDEV_INVALIDARG;
      break;
    case codaRequestObject::MONITOR_OFF:
      status = CDEV_INVALIDARG;
      break;
    default:
      status = CDEV_INVALIDARG;
      break;
    }
  }
  else {
    if (action_ == codaRequestObject::CONNECTED) {
      result->insert ("value", "no");
      status = CDEV_SUCCESS;
    }
    else if (action_ == codaRequestObject::CONNECT) {
      codaService *svc = (codaService *)service_;
      status = svc->connect (deviceName_);
    }
    else
      status = CDEV_NOTCONNECTED;
  }
  
  return status;
}

int
codaRequestObject::getValueCbk (cdevTranObj* obj)
{
  int cbkst;
  cbkst = codaClient_->getValueCallback (device_, attr_,
		 &(codaRequestObject::defaultGetCallback),
		 (void *)obj);
  if (cbkst == CODA_SUCCESS)
    return CDEV_SUCCESS;
  else {
    // no callback will be called, remove transaction obejct
    // otherwise groups will block forever
    cdevCallback *cbkobj = obj->userCallback_;
    cdevRequestObject *reqobj = obj->reqObj_;
    cdevData result;

    if (cbkobj)
      (*cbkobj->callbackFunction())(CDEV_IOFAILED,
				    cbkobj->userarg(),
				    *reqobj,
				    result);
    delete obj;
    return CDEV_IOFAILED;
  }
}

int
codaRequestObject::setValueCbk (cdevTranObj* obj)
{
  int cbkst;
  daqData* data = 0;
  if (codaRequestObject::convertData (*(obj->resultData_),
				      device_, attr_,
				      data) == CDEV_SUCCESS) {
    cbkst = codaClient_->setValueCallback (*data,
			   &(codaRequestObject::defaultSetCallback),
			   (void *)obj);
    if (cbkst == CODA_SUCCESS) {
      delete data;
      return CDEV_SUCCESS;
    }
    else {
      // no callback will be called, remove transaction obejct
      // otherwise groups will block forever
      cdevCallback *cbkobj = obj->userCallback_;
      cdevRequestObject *reqobj = obj->reqObj_;
      cdevData result;

      if (cbkobj)
	(*cbkobj->callbackFunction())(CDEV_IOFAILED,
				      cbkobj->userarg(),
				      *reqobj,
				      result);
      delete obj;
      delete data;
      return CDEV_IOFAILED;
    }
  }
  else
    return CDEV_INVALIDARG;
}

int
codaRequestObject::stateValueCbk (cdevTranObj* obj)
{
  int cbkst;
  cbkst = codaClient_->getValueCallback (device_, "status",
		 &(codaRequestObject::defaultStateCallback),
		 (void *)obj);
  if (cbkst == CODA_SUCCESS)
    return CDEV_SUCCESS;
  else {
    // no callback will be called, remove transaction obejct
    // otherwise groups will block forever
    cdevCallback *cbkobj = obj->userCallback_;
    cdevRequestObject *reqobj = obj->reqObj_;
    cdevData result;

    if (cbkobj)
      (*cbkobj->callbackFunction())(CDEV_IOFAILED,
				    cbkobj->userarg(),
				    *reqobj,
				    result);
    delete obj;
    return CDEV_IOFAILED;
  }
}

int
codaRequestObject::sendCommandCbk (int cmd, cdevTranObj* obj)
{
  int cbkst;
  daqData* data = 0;
  if (codaRequestObject::convertData (*(obj->resultData_),
				      device_, "command",
				      data) == CDEV_SUCCESS) {
    cbkst = codaClient_->sendCmdCallback (cmd, *data,
			   &(codaRequestObject::defaultCmdCallback),
			   (void *)obj);
    if (cbkst == CODA_SUCCESS) {
      delete data;
      return CDEV_SUCCESS;
    }
    else {
      // no callback will be called, remove transaction obejct
      // otherwise groups will block forever
      cdevCallback *cbkobj = obj->userCallback_;
      cdevRequestObject *reqobj = obj->reqObj_;
      cdevData result;

      if (cbkobj)
	(*cbkobj->callbackFunction())(CDEV_IOFAILED,
				      cbkobj->userarg(),
				      *reqobj,
				      result);
      delete obj;
      delete data;
      return CDEV_IOFAILED;
    }
  }
  else
    return CDEV_INVALIDARG;
}

int
codaRequestObject::changeStateCbk (int cmd, cdevTranObj* obj)
{
  // no extra data needed for state transition
  daqData data (device_, "command", cmd);
  if (codaClient_->sendCmdCallback (cmd, data,
		    &(codaRequestObject::defaultCmdCallback),
		    (void *)obj) == CODA_SUCCESS)
    return CDEV_SUCCESS;
  else {
    // no callback will be called, remove transaction obejct
    // otherwise groups will block forever
    cdevCallback *cbkobj = obj->userCallback_;
    cdevRequestObject *reqobj = obj->reqObj_;
    cdevData result;

    if (cbkobj)
      (*cbkobj->callbackFunction())(CDEV_IOFAILED,
				    cbkobj->userarg(),
				    *reqobj,
				    result);
    delete obj;
    return CDEV_IOFAILED;
  }
}

int
codaRequestObject::newStateCbk (int cmd, cdevTranObj* obj)
{
  // get two states information
  int dtype, i, j, k;
  daqData* qdata = 0;
  int dim = 0, count = 0;
  int st = CDEV_SUCCESS;
  int status = CDEV_SUCCESS;

  // get pointer to outbound data
  cdevData* data = obj->resultData_;

  if ((status = data->getDim (codaService::CODA_TAG_VALUE, 
			      (size_t *)&dim)) == CDEV_SUCCESS) {
    if (dim > 1)
      return CDEV_ERROR;
    else if (dim == 1) {
      if ((st = data->getElems (codaService::CODA_TAG_VALUE, 
				(size_t *)&count)) != CDEV_SUCCESS)
	return st;
    }
    else
      count = 1;
  }
  else
    return status;

  if (count != 2) // must be two states denoting initial/final state
    return CDEV_ERROR;
  
  dtype = data->getType (codaService::CODA_TAG_VALUE);
  
  if (dtype != CDEV_STRING)
    return CDEV_ERROR;

  char **temp;
  temp = new char* [count];
  int    sts[2];

  if (data->get (codaService::CODA_TAG_VALUE, temp) == CDEV_SUCCESS) {
    k = 0;
    for (i = 0; i < count; i++)
      for (j = 0; j < codaRequestObject::numStates; j++) {
	if (::strcmp (temp[i], codaRequestObject::states[j].string) == 0) 
	  sts[k++] = codaRequestObject::states[j].state;
      }
    if (k != 2)
      return CDEV_ERROR;
    
    qdata = new daqData (device_, "command", sts, count);

    for (i = 0; i < count; i++)
      delete []temp[i];
    delete []temp;
  }
  else
    return CDEV_ERROR;
  
  if (codaClient_->sendCmdCallback (cmd, *qdata,
		    &(codaRequestObject::defaultCmdCallback),
		    (void *)obj) == CODA_SUCCESS) {
    delete qdata;
    return CDEV_SUCCESS;
  }
  else {
    // no callback will be called, remove transaction obejct
    // otherwise groups will block forever
    cdevCallback *cbkobj = obj->userCallback_;
    cdevRequestObject *reqobj = obj->reqObj_;
    cdevData result;

    if (cbkobj)
      (*cbkobj->callbackFunction())(CDEV_IOFAILED,
				    cbkobj->userarg(),
				    *reqobj,
				    result);
    delete qdata;
    delete obj;
    return CDEV_IOFAILED;
  }
}

int
codaRequestObject::monitorValue (codaMonObj* obj)
{
  int cbkst;

  cbkst = codaClient_->monitorOnCallback (device_, attr_,
		 &(codaRequestObject::defaultMonCallback),
		 (void *)obj);
  if (cbkst == CODA_SUCCESS)
    return CDEV_SUCCESS;
  else {
    // no callback will be called, remove transaction obejct
    // otherwise groups will block forever
    cdevCallback *cbkobj = obj->rep->userCallback_;
    cdevRequestObject *reqobj = obj->rep->reqObj_;
    cdevData result;

    if (cbkobj)
      (*cbkobj->callbackFunction())(CDEV_IOFAILED,
				    cbkobj->userarg(),
				    *reqobj,
				    result);
    if (obj->tobj) { // remove transaction object
      delete obj->tobj;
      obj->tobj = 0;
    }
    delete obj;
    return CDEV_IOFAILED;
  } 
}

int
codaRequestObject::monitorOffValue (cdevCallback* callback)
{
  daqData *qdata = codaClient_->data (device_, attr_);
  if (qdata) {
    codaSlist& list = qdata->monitorCbkList ();
    codaSlistIterator ite (list);
    codaRcCallback *cbk = 0;  // codaRcCallback is what stored
    codaMonObj *obj = 0;
    cdevTranObj* xobj = 0;
    int found = 0;

    if (callback->callbackFunction () == 0) { // remove all monitor callbacks
      // create monitor off group
      cdevGroup offGrp;
      offGrp.start ();

      for (ite.init(); !ite; ++ite) {
	cbk = (codaRcCallback *)ite ();
	// user argument is codaMonObj
	obj = (codaMonObj *)cbk->userarg ();
	// remove all monitored callbacks
	found = 1;
	// create a transaction object to force synchronous behaviour
	xobj = new cdevTranObj (&system_, this, 0, 0); 
	// attach obj to xobj so that obj can be delete inside monitor off
	// callback
	cdevCallback *junkCbk = new cdevCallback (0, (void *)obj);
	xobj->userCallback_ = junkCbk;
	// note: to delete callback and obj --
	codaClient_->monitorOffCallback (device_, attr_,
			    &(codaRequestObject::defaultMonCallback),
			    (void *)obj,
			    &(codaRequestObject::defaultMonOffCallback),
			    (void *)xobj);
      }
      offGrp.end ();
      if (found)
	return offGrp.pend (CODA_DEFAULT_TIMEOUT);
    }
    else {
      // create monitor off group
      cdevGroup offGrp (2);
      offGrp.start ();
      
      for (ite.init(); !ite; ++ite) {
	cbk = (codaRcCallback *)ite ();
	// user argument is codaMonObj
	obj = (codaMonObj *)cbk->userarg ();
	if (*callback == *(obj->rep->userCallback_)) {
	  found = 1;
	  // create a transaction object to force synchronous behaviour
	  xobj = new cdevTranObj (&system_, this, 0, 0); 
	  // attach obj to xobj so that obj can be delete inside monitor off
	  // callback
	  cdevCallback *junkCbk = new cdevCallback (0, (void *)obj);
	  xobj->userCallback_ = junkCbk;
	  // note: to delete callback and obj --
	  codaClient_->monitorOffCallback (device_, attr_,
				   &(codaRequestObject::defaultMonCallback),
				   (void *)obj,
				   &(codaRequestObject::defaultMonOffCallback),
				   (void *)xobj);
	  break;
	}
      }
      offGrp.end ();
      if (found)
	return offGrp.pend (CODA_DEFAULT_TIMEOUT);
    }
    return CDEV_SUCCESS;
  }
  else
    return CDEV_ERROR;
}

int
codaRequestObject::sendNoBlock (cdevData& out, cdevData& result)
{
  return sendNoBlock (&out, &result);
}

int
codaRequestObject::sendNoBlock (cdevData* out, cdevData& result)
{
  return sendNoBlock (out, &result);
}

int
codaRequestObject::sendNoBlock (cdevData& out, cdevData* result)
{
  return sendNoBlock (&out, result);
}

int
codaRequestObject::sendNoBlock (cdevData* out, cdevData* result)
{
  if (deferExecution (out, result))
    return CDEV_SUCCESS;
  
  int status = CDEV_ERROR;
  cdevTranObj* xobj = 0;

  if (codaClient_->connected ()) {
    switch (action_) {
    case codaRequestObject::GET:
      if (result == 0)
	return CDEV_INVALIDARG;
      xobj = new cdevTranObj (&system_, this, result, 0);
      status = getValueCbk (xobj);
      break;
    case codaRequestObject::SET:
      if (out == 0)
	return CDEV_INVALIDARG;
      xobj = new cdevTranObj (&system_, this, out, 0);
      status = setValueCbk (xobj);
      break;
    case codaRequestObject::STATE:
      if (result == 0)
	return CDEV_INVALIDARG;
      xobj = new cdevTranObj (&system_, this, result, 0);
      status = stateValueCbk (xobj);
      break;
    case codaRequestObject::LOAD:
      {
	cdevData* data;
	int newdata = 1;
	void* vptr = 0;
	if (out && 
	    out->find (codaService::CODA_TAG_VALUE, vptr) == CDEV_SUCCESS ) 
	  newdata = 0;


	if (newdata) {
	  data = new cdevData();
	  
	  char* extra[2];
	  extra[0] = ::getenv ("EXPID");
	  if (!extra[0]) {
	    fprintf (stderr, "Environment variable EXPID is not set\n");
	    exit (1);
	  }
	  extra[1] = deviceName_;

	  if (data->insert (codaService::CODA_TAG_VALUE, extra, 2) 
	      != CDEV_SUCCESS)
	    return CDEV_INVALIDARG;     
	}
	else
	  data = out;
	  
	xobj = new cdevTranObj (&system_, this, data, 0);
	status = sendCommandCbk (DALOADDBASE, xobj);
      }
      break;
    case codaRequestObject::ASK:
      {
	if (out == 0) 
	  return CDEV_INVALIDARG;

	// create cdev callback which has no function 
	// but a user argument holding result data pointer
	cdevCallback* tmpcbk = new cdevCallback(0, (void *)result);
	xobj = new cdevTranObj (&system_, this, out, tmpcbk);
	status = sendCommandCbk (DAASK, xobj);
      }
      break;
    case codaRequestObject::CONFIGURE:
      if (out == 0)
	return CDEV_INVALIDARG;
      xobj = new cdevTranObj (&system_, this, out, 0);
      status = sendCommandCbk (DACONFIGURE, xobj);
      break;
    case codaRequestObject::DOWNLOAD:
      xobj = new cdevTranObj (&system_, this, out, 0);
      status = changeStateCbk (DADOWNLOAD, xobj);
      break;
    case codaRequestObject::PRESTART:
      xobj = new cdevTranObj (&system_, this, out, 0);
      status = changeStateCbk (DAPRESTART, xobj);
      break;
    case codaRequestObject::PAUSE:
      xobj = new cdevTranObj (&system_, this, out, 0);
      status = changeStateCbk (DAPAUSE, xobj);
      break;
    case codaRequestObject::RESUME:
      xobj = new cdevTranObj (&system_, this, out, 0);
      status = changeStateCbk (DARESUME, xobj);
      break;
    case codaRequestObject::END:
      xobj = new cdevTranObj (&system_, this, out, 0);
      status = changeStateCbk (DAEND, xobj);
      break;
    case codaRequestObject::GO:
      xobj = new cdevTranObj (&system_, this, out, 0);
      status = changeStateCbk (DAGO, xobj);
      break;
    case codaRequestObject::RESET:
      xobj = new cdevTranObj (&system_, this, out, 0);
      status = changeStateCbk (DARESET, xobj);
      break;
    case codaRequestObject::ABORT:
      xobj = new cdevTranObj (&system_, this, out, 0);
      status = changeStateCbk (DAABORT, xobj);
      break;
    case codaRequestObject::NEW_STATE:
      xobj = new cdevTranObj (&system_, this, out, 0);
      status = newStateCbk (DACHANGE_STATE, xobj);
      break;
    case codaRequestObject::MONITOR_ON:
    case codaRequestObject::MONITOR_OFF:
      status = CDEV_INVALIDARG;
      break;
    default:
#ifdef _CDEV_DEBUG
      printf ("Unsupported action message in codaService %d\n",action_);
#endif
      status = CDEV_INVALIDARG;
      break;
    }
  }
  else
    status = CDEV_NOTCONNECTED;
  return status;
}

int
codaRequestObject::sendCallback (cdevData& data, cdevCallback& callback)
{
  return sendCallback (&data, callback);
}

int
codaRequestObject::sendCallback (cdevData* data, cdevCallback& userCallback)
{
  if (deferExecution (data, &userCallback))
    return CDEV_SUCCESS;
  
  int status = CDEV_ERROR;
  cdevTranObj *xobj = 0;
  codaMonObj*  obj = 0;
  
  // copy userCallback into the following callback, which will free 
  // the user from worrying about cdevCallback object
  // note: cdevCallback's new operator is an overloaded operator
  cdevCallback* callback = 0;

  if (codaClient_->connected ()) {
    switch (action_) {
    case codaRequestObject::GET:
      callback = new cdevCallback (userCallback);
      xobj = new cdevTranObj (&system_, this, 0, callback);
      status = getValueCbk (xobj);
      break;
    case codaRequestObject::ASK:
      callback = new cdevCallback (userCallback);
      xobj = new cdevTranObj (&system_, this, data, callback);
      status = sendCommandCbk (DAASK, xobj);
      status = getValueCbk (xobj);
      break;
    case codaRequestObject::SET:
      if (data == 0)
	return CDEV_INVALIDARG;
      callback = new cdevCallback (userCallback);
      xobj = new cdevTranObj (&system_, this, data, callback);
      status = setValueCbk (xobj);
      break;
    case codaRequestObject::LOAD:
      {
	cdevData* out;
	int newdata = 1;
	void* vptr = 0;
	if (data && 
	    data->find (codaService::CODA_TAG_VALUE, vptr) == CDEV_SUCCESS ) 
	  newdata = 0;


	if (newdata) {
	  out = new cdevData();
	  
	  char* extra[2];
	  extra[0] = ::getenv ("EXPID");
	  if (!extra[0]) {
	    fprintf (stderr, "Environment variable EXPID is not set\n");
	    exit (1);
	  }
	  extra[1] = deviceName_;

	  if (out->insert (codaService::CODA_TAG_VALUE, extra, 2) 
	      != CDEV_SUCCESS)
	    return CDEV_INVALIDARG;     
	}
	else
	  out = data;

	callback = new cdevCallback (userCallback);
	xobj = new cdevTranObj (&system_, this, out, callback);
	status = sendCommandCbk (DALOADDBASE, xobj);
      }
      break;
    case codaRequestObject::CONFIGURE:
      if (data == 0)
	return CDEV_INVALIDARG;
      callback = new cdevCallback (userCallback);
      xobj = new cdevTranObj (&system_, this, data, callback);
      status = sendCommandCbk (DACONFIGURE, xobj);
      break;
    case codaRequestObject::DOWNLOAD:
      callback = new cdevCallback (userCallback);
      xobj = new cdevTranObj (&system_, this, data, callback);
      status = changeStateCbk (DADOWNLOAD, xobj);
      break;
    case codaRequestObject::PRESTART:
      callback = new cdevCallback (userCallback);
      xobj = new cdevTranObj (&system_, this, data, callback);
      status = changeStateCbk (DAPRESTART, xobj);
      break;
    case codaRequestObject::PAUSE:
      callback = new cdevCallback (userCallback);
      xobj = new cdevTranObj (&system_, this, data, callback);
      status = changeStateCbk (DAPAUSE, xobj);
      break;
    case codaRequestObject::RESUME:
      callback = new cdevCallback (userCallback);
      xobj = new cdevTranObj (&system_, this, data, callback);
      status = changeStateCbk (DARESUME, xobj);
      break;
    case codaRequestObject::END:
      callback = new cdevCallback (userCallback);
      xobj = new cdevTranObj (&system_, this, data, callback);
      status = changeStateCbk (DAEND, xobj);
      break;
    case codaRequestObject::RESET:
      callback = new cdevCallback (userCallback);
      xobj = new cdevTranObj (&system_, this, data, callback);
      status = changeStateCbk (DARESET, xobj);
      break;
    case codaRequestObject::GO:
      callback = new cdevCallback (userCallback);
      xobj = new cdevTranObj (&system_, this, data, callback);
      status = changeStateCbk (DAGO, xobj);
      break;
    case codaRequestObject::ABORT:
      callback = new cdevCallback (userCallback);
      xobj = new cdevTranObj (&system_, this, data, callback);
      status = changeStateCbk (DAABORT, xobj);
      break;
    case codaRequestObject::NEW_STATE:
      callback = new cdevCallback (userCallback);
      xobj = new cdevTranObj (&system_, this, data, callback);
      status = newStateCbk (DACHANGE_STATE, xobj);
      break;
    case codaRequestObject::MONITOR_ON:
      callback = new cdevCallback (userCallback);
      xobj = new cdevTranObj (&system_, this, 0, callback);
      xobj->disableDeleteCbk ();
      obj = new codaMonObj (*xobj);
      // Note: codaMonObj will be removed by monitor off, otherwise it will
      //       stays forever until the program exits 1-11-96.
      status = monitorValue (obj);
      break;
    case codaRequestObject::MONITOR_OFF:
      // this is actually a synchronous call, 
      // so userCallback with local pointer is safe
      status = monitorOffValue (&userCallback);
      break;
    default:
#ifdef _CDEV_DEBUG
      printf ("Unsupported action in codaService %d\n", action_);
#endif
      status = CDEV_INVALIDARG;
      break;
    }
  }
  else
    status = CDEV_NOTCONNECTED;
  return status;
}

void
codaRequestObject::defaultGetCallback (int status,
				       void* arg,
				       daqNetData* data)
{
  cdevTranObj* xobj = (cdevTranObj *)arg;
  codaRequestObject* reqObj = (codaRequestObject *)xobj->reqObj_;
  cdevData *resData = xobj->resultData_;
  cdevCallback *cbkobj = xobj->userCallback_;
  cdevData result;

  if (status == CODA_SUCCESS) {
   if (resData) // sendNoBlock
      resData->remove ();
    else
      resData = &result;

    // convert data from coda data type to cdevData type
    codaRequestObject::convertData (*data, 
				    codaService::CODA_TAG_VALUE,
				    reqObj->reqMask_,
				    reqObj->reqType_,
				    *resData);

    if (cbkobj) { // called from sendCallback
      (*cbkobj->callbackFunction ()) (CDEV_SUCCESS, cbkobj->userarg (),
				      *reqObj, result);
    }
  }
  else {
    if (cbkobj) { // called from sendCallback
      (*cbkobj->callbackFunction ()) (CDEV_ERROR, cbkobj->userarg (),
				      *reqObj, result);
    }
  }
  delete xobj;
}


void
codaRequestObject::defaultSetCallback (int status,
				       void* arg,
				       daqNetData*)
{
  cdevTranObj* xobj = (cdevTranObj *)arg;
  codaRequestObject* reqObj = (codaRequestObject *)xobj->reqObj_;
  cdevData *resData = xobj->resultData_;
  cdevCallback *cbkobj = xobj->userCallback_;

  if (status == CODA_SUCCESS) {
    if (cbkobj)  // called from sendCallback
      (*cbkobj->callbackFunction ()) (CDEV_SUCCESS, cbkobj->userarg (),
				      *reqObj, *resData);
  }
  else {
    if (cbkobj)  // called from sendCallback
      (*cbkobj->callbackFunction ()) (CDEV_ERROR, cbkobj->userarg (),
				      *reqObj, *resData);
  }
  delete xobj;
}

void
codaRequestObject::defaultCmdCallback (int status,
				       void* arg,
				       daqNetData* data)
{
  cdevTranObj* xobj = (cdevTranObj *)arg;
  codaRequestObject* reqObj = (codaRequestObject *)xobj->reqObj_;
  cdevData *resData = 0;
  cdevCallback *cbkobj = xobj->userCallback_;
  cdevData result;

  if (cbkobj && (cbkobj->callbackFunction () == 0)) { 
    // using a dumb callback as a way to hold incoming
    resData = (cdevData *)cbkobj->userarg ();
    // convert data from coda data type to cdevData type
    codaRequestObject::convertData (*data, 
				    codaService::CODA_TAG_VALUE,
				    reqObj->reqMask_,
				    reqObj->reqType_,
				    *resData);
  }

  if (status == CODA_SUCCESS) {
    if (cbkobj && cbkobj->callbackFunction ())  // called from sendCallback
      (*cbkobj->callbackFunction ()) (CDEV_SUCCESS, cbkobj->userarg (),
				      *reqObj, result);
  }
  else {
    if (cbkobj && cbkobj->callbackFunction ())  // called from sendCallback
      (*cbkobj->callbackFunction ()) (CDEV_ERROR, cbkobj->userarg (),
				      *reqObj, result);
  }
  delete xobj;
}

void
codaRequestObject::defaultStateCallback (int status,
					 void* arg,
					 daqNetData* data)
{
  int i;
  cdevTranObj* xobj = (cdevTranObj *)arg;
  codaRequestObject* reqObj = (codaRequestObject *)xobj->reqObj_;
  cdevData *resData = xobj->resultData_;
  cdevCallback *cbkobj = xobj->userCallback_;
  cdevData result;

  if (status == CODA_SUCCESS) {
   if (resData) // sendNoBlock
      resData->remove ();
    else
      resData = &result;

   // find string representation of state
   int state = (int)(*data);
   int found = 0;
   for (i = 0; i < codaRequestObject::numStates; i++) {
     if (state == codaRequestObject::states[i].state) {
       found = 1;
       break;
     }
   }
   daqNetData newdata (reqObj->device_, "status", "unknown");
   if (found) 
     newdata = codaRequestObject::states[i].string;
   
   // convert data from coda data type to cdevData type
   codaRequestObject::convertData (newdata, 
				   codaService::CODA_TAG_VALUE,
				   reqObj->reqMask_,
				   reqObj->reqType_,
				   *resData);
    if (cbkobj) { // called from sendCallback
      (*cbkobj->callbackFunction ()) (CDEV_SUCCESS, cbkobj->userarg (),
				      *reqObj, result);
    }
  }
  else {
    if (cbkobj) { // called from sendCallback
      (*cbkobj->callbackFunction ()) (CDEV_ERROR, cbkobj->userarg (),
				      *reqObj, result);
    }
  }
  delete xobj;
}

void
codaRequestObject::defaultMonCallback (int status,
				       void* arg,
				       daqNetData* data)
{
  codaMonObj* obj = (codaMonObj *)arg;
  codaRequestObject* reqObj = (codaRequestObject *)obj->rep->reqObj_;
  cdevCallback *cbkobj = obj->rep->userCallback_;
  cdevData result;

  if (obj->tobj) { // remove transaction object
    delete obj->tobj;
    obj->tobj = 0;
  }
  if (status == CODA_SUCCESS) {
    codaRequestObject::convertData (*data,
				    codaService::CODA_TAG_VALUE,
				    reqObj->reqMask_,
				    reqObj->reqType_,
				    result);
    (*cbkobj->callbackFunction ()) (CDEV_SUCCESS, cbkobj->userarg (),
				    *reqObj, result);
  }
  else {
    (*cbkobj->callbackFunction ()) (CDEV_ERROR, cbkobj->userarg (),
				    *reqObj, result);    
  }
}

void
codaRequestObject::defaultMonOffCallback (int status,
					  void* arg,
					  daqNetData*)
{
  cdevTranObj* xobj = (cdevTranObj *)arg;
  // get the fake callback which has monitorOn codaMonObj associated with it
  cdevCallback* junkCbk = xobj->userCallback_;
  codaMonObj* monObj = (codaMonObj *)junkCbk->userarg ();
  if (status == CODA_SUCCESS) // only delete successful removal
    delete monObj;
  delete junkCbk;
  // finally delete this transaction object
  delete xobj;
}

void
codaRequestObject::convertData (daqNetData& data,
				int tag,
				int reqMask,
				int reqType,
				cdevData& resData)
{
  switch (data.type ()) {
  case CODA_STR:
    if (data.count () == 1) {
      resData.insert (tag, (char *)data);
      printf("disp hi/lo\n");
      resData.insert ("displayHigh", 0);
      resData.insert ("displayLow",  0);
      //      codaRequestObject::fillResult (reqMask, reqType, data, resData);
    }
    else {
      char **insertBuf = new char* [data.count()];
      int count = (int)data.count ();
      // get array of data
      data.getData (insertBuf, count);
      resData.insert (tag, insertBuf, count);
//      codaRequestObject::fillResult (reqMask, reqType, data, resData);
      for (int i = 0; i < count; i++)
	delete []insertBuf[i];
      printf("disp hi/lo\n");
      resData.insert ("displayHigh", count-1);
      resData.insert ("displayLow",  0);
      delete []insertBuf;
    }
    break;
  case CODA_INT:
    if (data.count () == 1) {
      resData.insert (tag, (int)data);
//      codaRequestObject::fillResult (reqMask, reqType, data, resData);
    }
    else {
      int* insertBuf = new int[data.count()];
      int count = (int)data.count();
      // get array of data
      data.getData (insertBuf, count);
      resData.insert (tag, insertBuf, count);
//      codaRequestObject::fillResult (reqMask, reqType, data, resData);
      delete []insertBuf;
    }
    break;
  case CODA_FLT:
    if (data.count () == 1) {
      resData.insert (tag, (float)data);
//      codaRequestObject::fillResult (reqMask, reqType, data, resData);
    }
    else {
      float* insertBuf = new float[data.count()];
      int count = (int)data.count();
      // get array of data
      data.getData (insertBuf, count);
      resData.insert (tag, insertBuf, count);
//      codaRequestObject::fillResult (reqMask, reqType, data, resData);
      delete []insertBuf;
    }
    break;
  case CODA_DBL:
    if (data.count () == 1) {
      resData.insert (tag, (double)data);
//      codaRequestObject::fillResult (reqMask, reqType, data, resData);
    }
    else {
      double* insertBuf = new double[data.count()];
      int count = (int)data.count();
      // get array of data
      data.getData (insertBuf, count);
      resData.insert (tag, insertBuf, count);
//      codaRequestObject::fillResult (reqMask, reqType, data, resData);
      delete []insertBuf;
    }
    break;
  default:
    printf ("Fatal Error: unsuppeorted data type in coda service\n");
    break;
  }
}

int
codaRequestObject::convertData (cdevData& data,
				char *compname,
				char *attrname,
				daqData* &qdata)
{
  int dtype;
  int dim = 0, count = 0;
  int st = CDEV_SUCCESS;
  int status = CDEV_SUCCESS;

  if ((status = data.getDim (codaService::CODA_TAG_VALUE, 
			    (size_t *)&dim)) == CDEV_SUCCESS) {
    if (dim > 1)
      return CDEV_ERROR;
    else if (dim == 1) {
      if ((st = data.getElems (codaService::CODA_TAG_VALUE, 
			 (size_t *)&count)) != CDEV_SUCCESS)
	return st;
    }
    else
      count = 1;
  }
  else
    return status;
  
  dtype = data.getType (codaService::CODA_TAG_VALUE);
  switch (dtype){
  case CDEV_STRING:
    if (count == 1) {
      char temp[CODA_MAX_STRLEN];
      if (data.get (codaService::CODA_TAG_VALUE, temp,
		    CODA_MAX_STRLEN) == CDEV_SUCCESS)
	qdata = new daqData (compname, attrname, temp);
      else {
	char **temp;
	temp = new char* [count];
	if (data.get (codaService::CODA_TAG_VALUE, temp) == CDEV_SUCCESS) {
	  qdata = new daqData (compname, attrname, temp, count);
	  for (int i = 0; i < count; i++)
	    delete []temp[i];
	  delete []temp;
	}
	return CDEV_ERROR;
      }
    }
    else {
      char **temp = new char*[count];
      // each item will be allocated by the following
      if (data.get (codaService::CODA_TAG_VALUE, temp) == CDEV_SUCCESS) {
	qdata = new daqData (compname, attrname, temp, count);
	for (int i = 0; i < count; i++) 
	  delete temp[i];
      }
      else {
	qdata = 0;
	return CDEV_ERROR;
      }
      delete []temp;
    }
    break;
  case CDEV_INT16:
  case CDEV_UINT16:
    if (count == 1) {
      short sval;
      int   val;
      if (data.get (codaService::CODA_TAG_VALUE, &sval) == CDEV_SUCCESS) {
	val = (int)sval;
	qdata = new daqData (compname, attrname, val);
      }
      else {
	qdata = 0;
	return CDEV_ERROR;
      }
    }
    else {
      short *sval = new short[count];
      int   *val = new int[count];
      if (data.get (codaService::CODA_TAG_VALUE, sval) == CDEV_SUCCESS) {
	for (int i = 0; i < count; i++)
	  val[i] = (int)sval[i];
	qdata = new daqData (compname, attrname, val, count);
	delete []sval;
      }
      else {
	qdata = 0;
	return CDEV_ERROR;
      }
      delete []sval;
      delete []val;
    }
    break;
  case CDEV_INT32:
  case CDEV_UINT32:
    if (count == 1) {
      int val;
      if (data.get (codaService::CODA_TAG_VALUE, &val) == CDEV_SUCCESS) 
	qdata = new daqData (compname, attrname, val);
      else {
	qdata = 0;
	return CDEV_ERROR;
      }
    }
    else {
      int *val = new int[count];
      if (data.get (codaService::CODA_TAG_VALUE, val) == CDEV_SUCCESS)
	qdata = new daqData (compname, attrname, val, count);
      else {
	qdata = 0;
	return CDEV_ERROR;
      }
      delete []val;
    }
    break;
  case CDEV_FLOAT:
    if (count == 1) {
      float val;
      if (data.get (codaService::CODA_TAG_VALUE, &val) == CDEV_SUCCESS) 
	qdata = new daqData (compname, attrname, val);
      else {
	qdata = 0;
	return CDEV_ERROR;
      }
    }
    else {
      float *val = new float[count];
      if (data.get (codaService::CODA_TAG_VALUE, val) == CDEV_SUCCESS)
	qdata = new daqData (compname, attrname, val, count);
      else {
	qdata = 0;
	return CDEV_ERROR;
      }
      delete []val;
    }
    break;
  case CDEV_DOUBLE:
    if (count == 1) {
      double val;
      if (data.get (codaService::CODA_TAG_VALUE, &val) == CDEV_SUCCESS) 
	qdata = new daqData (compname, attrname, val);
      else {
	qdata = 0;
	return CDEV_ERROR;
      }
    }
    else {
      double *val = new double[count];
      if (data.get (codaService::CODA_TAG_VALUE, val) == CDEV_SUCCESS)
	qdata = new daqData (compname, attrname, val, count);
      else {
	qdata = 0;
	return CDEV_ERROR;
      }
      delete []val;
    }
    break;
  default:
    status = CDEV_ERROR;
    break;
  }
  return CDEV_SUCCESS;
}

int
codaRequestObject::findAction (char* msg,
			       int& action,
			       char* &attr)
{
  char token0[64], token1[512], token2[64];
  int  status;

  status = ::sscanf (msg, "%s %s",token0, token1);

  if (::strcmp (token0, "get") == 0)
    action = codaRequestObject::GET;
  else if (::strcmp (token0, "set") == 0)
    action = codaRequestObject::SET;
  else if (::strcmp (token0, "monitorOn") == 0)
    action = codaRequestObject::MONITOR_ON;
  else if (::strcmp (token0, "monitorOff") == 0)
    action = codaRequestObject::MONITOR_OFF;
  else if (::strcmp (token0, "load") == 0)
    action = codaRequestObject::LOAD;
  else if (::strcmp (token0, "configure") == 0)
    action = codaRequestObject::CONFIGURE;
  else if (::strcmp (token0, "download") == 0)
    action = codaRequestObject::DOWNLOAD;
  else if (::strcmp (token0, "prestart") == 0)
    action = codaRequestObject::PRESTART;
  else if (::strcmp (token0, "pause") == 0)
    action = codaRequestObject::PAUSE;
  else if (::strcmp (token0, "resume") == 0)
    action = codaRequestObject::PAUSE;
  else if (::strcmp (token0, "end") == 0)
    action = codaRequestObject::END;
  else if (::strcmp (token0, "reset") == 0)
    action = codaRequestObject::RESET;
  else if (::strcmp (token0, "go") == 0)
    action = codaRequestObject::GO;
  else if (::strcmp (token0, "abort") == 0)
    action = codaRequestObject::ABORT;
  else if (::strcmp (token0, "disconnect") == 0)
    action = codaRequestObject::DISCONNECT;
  else if (::strcmp (token0, "connected") == 0)
    action = codaRequestObject::CONNECTED;  
  else if (::strcmp (token0, "connect") == 0)
    action = codaRequestObject::CONNECT;
  else if (::strcmp (token0, "state") == 0)
    action = codaRequestObject::STATE;
  else if (::strcmp (token0, "ask") == 0)
    action = codaRequestObject::ASK;
  else if (::strcmp (token0, "msg") == 0)
    action = codaRequestObject::MSG;
  else if (::strcmp (token0, "newState") == 0)
    action = codaRequestObject::NEW_STATE;
  else
    action = codaRequestObject::UNKNOWN;

  if (status > 1) {
    // get attribute name
    attr = new char[::strlen (token1) + 1];
    ::strcpy (attr, token1);
  }
  else
    attr = 0;

  return CDEV_SUCCESS;
}

int
codaRequestObject::findSvcData (char* device, char* msg,
				cdevSystem& system)
{
  char message[256], fullname[128];

  if (system.prefix())
    ::sprintf (fullname, "%s%s",system.prefix(),device);
  else
    ::strcpy (fullname, device);

  ::sprintf (message,"resolveServiceData %s %s", fullname, msg);

  cdevData result;
  int status = (system.nameServer()).send (message, 0, result);
  if (status == CDEV_SUCCESS) {
    int rd = 0;
    if (result.get (codaService::CODA_TAG_RO, &rd) == CDEV_SUCCESS)
      readonly_ = 1;
  }
  else
    return CDEV_ERROR;
}
  
