/*----------------------------------------------------------------------------*
 *  Copyright (c) 2001        Southeastern Universities Research Association, *
 *                            Thomas Jefferson National Accelerator Facility  *
 *                                                                            *
 *    This software was developed under a United States Government license    *
 *    described in the NOTICE file included as part of this distribution.     *
 *                                                                            *
 *    Author:  Carl Timmer                                                    *
 *             timmer@jlab.org                   Jefferson Lab, MS-12H        *
 *             Phone: (757) 269-5130             12000 Jefferson Ave.         *
 *             Fax:   (757) 269-5800             Newport News, VA 23606       *
 *                                                                            *
 *----------------------------------------------------------------------------*/

import java.lang.*;
import java.io.*;
import java.net.*;
import java.util.*;
import java.util.regex.*;
import java.sql.*;

/**
 * This class implements a server which listens for anyone using the modified
 * libmsql library for connecting to an msql database. Instead, this server
 * connects to a JDBC database. It starts one thread for each tcp socket
 * connection and provides a bridge for communication between each user and
 * a JDBC database specified when starting up this server.
 *
 * @author Carl Timmer
 * @version 1
 */

public class CodaDatabaseProxyServer {

  /** Port number to listen on. */
  private int port = 8101;
  
  /** Indicates whether this server should create tables and
    * insert data into the database for testing purposes. */
  private boolean inTestMode;
  
  /** Creates a new CodaDatabaseProxyServer object. */
  CodaDatabaseProxyServer() throws IOException {
    // Find out which database to connect to and how to do it.
    
    // First check to see if properties were set on the command line.
    String url = System.getProperty("url");
    String user = System.getProperty("user");
    String driver = System.getProperty("driver");
    String password = System.getProperty("password");
    
    // If properties were NOT set, look for the relevant environmental variables.
    if (url == null) {url = UnixEnvironment.getenv("CODA_DB_URL");}
    if (user == null) {user = UnixEnvironment.getenv("CODA_DB_USER");}
    if (driver == null) {driver = UnixEnvironment.getenv("CODA_DB_DRIVER");}
    if (password == null) {password = UnixEnvironment.getenv("CODA_DB_PASSWORD");}
    
    // Error if we don't at least have driver & url
    if (url == null || driver == null) {
        throw new IOException("Cannot find environmental variable(s): CODA_DB_URL and/or CODA_DB_DRIVER");
    }
    
    // Set user & password to defaults if unset at this point
    if (user == null) {user = "dummy";}
    if (password == null) {password = "";}
    
    System.setProperty("url", url);
    System.setProperty("user", user);
    System.setProperty("driver", driver);
    System.setProperty("password", password);
    
    // Check to see if port given by environmental variable
    String tcpPort = UnixEnvironment.getenv("MSQL_TCP_PORT");
    if (tcpPort != null) {
        try {
            port = Integer.parseInt(tcpPort);
        }
        catch (NumberFormatException ex) {}
    }
  }

  
  /** Start server on command line. */
  public static void main(String[] args) {
    int arg = 0;
    byte debug = 0;
    CodaDatabaseProxyServer server = null;
    
    // Catch error reading environmental variables
    try {
        server = new CodaDatabaseProxyServer();
    }
    catch (IOException ex) {
        System.out.println(ex.getMessage());
        ex.printStackTrace();
        System.exit(-1);
    }
    
    while (arg < args.length) {
      if (args[arg].equalsIgnoreCase("-debug")) {
	arg++;
	if (arg < args.length) {
	  try {debug = Byte.parseByte(args[arg++]);}
	  catch (NumberFormatException ex) {}
	}
      }
      else if (args[arg].equalsIgnoreCase("-test")) {
	arg++;
        server.inTestMode = true;
      }
      else if (args[arg].equalsIgnoreCase("-port")) {
	arg++;
	if (arg < args.length) {
	  try {server.port = Integer.parseInt(args[arg++]);}
	  catch (NumberFormatException ex) {}
	}
      }
      else {arg++;}
    }
    server.start(debug);
  }
  
  /** Start up a server's threads. */
  public void start(byte debug) {
    System.out.println("Running CODA Database Server");
    
    if (inTestMode) {
        fillDatabase();
    }
    
    // open a listening socket
    try {
      ServerSocket listeningSocket = new ServerSocket(port);  // IOEx
      while (true) {
	// socket to client created
	Socket sock;
	sock = listeningSocket.accept();
        // Set tcpNoDelay so no packets are delayed
        sock.setTcpNoDelay(true);
        // set buffer size
        sock.setReceiveBufferSize(2048);
        sock.setSendBufferSize(2048);
	// create thread to deal with client
	ClientThread connection = new ClientThread(sock, debug);
	connection.start();
      }
    }
    catch (SocketException ex) {
      System.out.println(ex.getMessage());
      ex.printStackTrace();
    }
    catch (IOException ex) {
      System.out.println(ex.getMessage());
      ex.printStackTrace();
    }
    return;
  }
  
  

  /**
   *  Create tables and fills them with data. This method is useful for
   *  testing and debugging CODA applications since the table definitions
   *  are some of those used in CODA.
   *  @param con database connection object.
   *  @param stmt database connection object's statement object.
   */
  public void fillDatabase() {

    String url = System.getProperty("url");
    String user = System.getProperty("user");
    String driver = System.getProperty("driver");
    String password = System.getProperty("password");
    
    // Extract some convenient substrings. The database name
    // (or catalog in jdbc) is the last text (zzz) in a url of the
    // form: jdbc:dbname://host:port/zzz . Or, in the case of postgres,
    // it is the last text (zzz) in the form: jdbc:dbname:zzz
    String  currentDatabase = null;
    boolean gotcurrentDB = false;
    String  regex = "(jdbc:\\w+:(//[a-zA-Z0-9_\\.]+:\\d+)*)/*(\\w+)";
    Pattern pattern = Pattern.compile(regex);
    Matcher matcher = pattern.matcher(url);
    if (matcher.find()) {
	currentDatabase = matcher.group(3);
	gotcurrentDB = true;
    }
        
    // There is no database name/catalog in the url,
    // so return without doing anything.
    if (!gotcurrentDB) {
        return;
    }
 
    Connection con;
    Statement stmt;
    DatabaseMetaData meta;
    
    try {
      // Register driver
      Class.forName(driver).newInstance();
      // Connect to database
      con = DriverManager.getConnection(url, user, password);
      // Create SQL statement
      stmt = con.createStatement();
      // Create database meta data object
      meta = con.getMetaData();
    }
    catch (Exception ex) {
      System.out.println("Cannot load test data into database");
      return;
    }


    try {
        // Remove tables if they already exist.
        try {stmt.executeUpdate("DROP TABLE testing");}        catch (SQLException e) {}
        try {stmt.executeUpdate("DROP TABLE testing_option");} catch (SQLException e) {}
        try {stmt.executeUpdate("DROP TABLE testing_pos");}    catch (SQLException e) {}
        try {stmt.executeUpdate("DROP TABLE testing_script");} catch (SQLException e) {}

        // Create "testing" table
        stmt.executeUpdate("CREATE TABLE testing (name VARCHAR(32), code VARCHAR(255), inputs VARCHAR(255), outputs VARCHAR(255), first VARCHAR(32), next VARCHAR(32), inuse VARCHAR(32))");

        // Insert a couple of rows
        stmt.executeUpdate(
            "INSERT INTO testing (name, CODE, outputs, first, inuse) VALUES ('cu', '{/home/timmer/rol/test_list_big.so junk}', 'ebc:aslan', 'yes', 'no')");
        stmt.executeUpdate(
            "INSERT INTO testing (name, code, inputs, first, inuse)  VALUES ('ebc', '{CODA}{CODA}', 'cu:aslan', 'yes', 'no')");

        // Create "testing_option" table
        stmt.executeUpdate("CREATE TABLE testing_option (name VARCHAR(32), value VARCHAR(80))");

        // Insert some rows.
        stmt.executeUpdate("INSERT INTO testing_option (name, value) VALUES ('dataLimit', '0')");
        stmt.executeUpdate("INSERT INTO testing_option (name, value) VALUES ('eventLimit', '0')");
        stmt.executeUpdate("INSERT INTO testing_option (name, value) VALUES ('tokenInterval', '64')");
        stmt.executeUpdate("INSERT INTO testing_option (name, value) VALUES ('rocMask', '2')");


        // Create "testing_pos" table
        stmt.executeUpdate("CREATE TABLE testing_pos (name VARCHAR(32), row INTEGER, col INTEGER)");
        // Insert some rows.
        stmt.executeUpdate("INSERT INTO testing_pos (name, row, col) VALUES ('cu', 2, 1)");
        stmt.executeUpdate("INSERT INTO testing_pos (name, row, col) VALUES ('ebc', 2, 3)");

        // Create "testing_script" table
        stmt.executeUpdate("CREATE TABLE testing_script (name VARCHAR(32), state VARCHAR(32), script VARCHAR(128))");

        // See if "sessions", "process", & "runTypes" tables exist. If not, create.
        String    name = null;
        ResultSet rs = meta.getTables(currentDatabase, null, "%", null);
        boolean gotSessions = false, gotProcess = false, gotRunTypes = false;

        while(rs.next()) {
          // database name
          name = rs.getString(3);

          if (name.equals("sessions")) {
            gotSessions = true;
          }
          else if (name.equals("process")) {
            gotProcess = true;   
          }
          else if (name.equals("runTypes")) {
            gotRunTypes = true;   
          }
        }

        String session = UnixEnvironment.getenv("SESSION");
        if (session == null) session = "test";
        
        if (!gotSessions) {
          // Create "sessions" table
          stmt.executeUpdate("CREATE TABLE sessions (name VARCHAR(64), id INTEGER, owner VARCHAR(32), inuse VARCHAR(32), log_name VARCHAR(32), rc_name VARCHAR(32), runNumber INTEGER, config VARCHAR(32))");
          stmt.executeUpdate("INSERT INTO sessions (name, id, owner, inuse, log_name, rc_name, runNumber, config) VALUES ('" +
                             session + "', 75, 'aslan timmer 2175 102', 'no', 'errors', 'RunControl', 1001, 'config')");
        }
        else {
          // Check to see if "sessions" table has entry under sesssion. If not, add.
          boolean rowExists = false;
	  rs = stmt.executeQuery("SELECT name FROM sessions WHERE name LIKE '" + session + "'");
	  while(rs.next()) {
	    String s = rs.getString(1);
            if (s.equals(session)) {
              // Don't add data as it is already in the database
              rowExists = true;
            }
	  }
          if (!rowExists) {
            stmt.executeUpdate("INSERT INTO sessions (name, id, owner, inuse, log_name, rc_name, runNumber, config) VALUES ('" +
                               session + "', 75, 'aslan timmer 2175 102', 'no', 'errors', 'RunControl', 1001, 'config')");
          }
        }

        if (!gotProcess) {
          // Create "process" table
          stmt.executeUpdate("CREATE TABLE process (name VARCHAR(32), id INTEGER, cmd VARCHAR(128), type VARCHAR(32), host VARCHAR(32), port INTEGER, state VARCHAR(32), pid INTEGER, inuse VARCHAR(32), clone VARCHAR(32))");
          stmt.executeUpdate("INSERT INTO process (name, id, cmd, type, host, port, state, pid, inuse, clone) VALUES (" +
                             "'cu', 27, '/home/timmer/cvs/coda/source/dac/coda_roc', 'ROC', 'aslan', 0, 'down', -1, 'no', 'no')");
          stmt.executeUpdate("INSERT INTO process (name, id, cmd, type, host, port, state, pid, inuse, clone) VALUES (" +
                             "'ebc', 28, '/home/timmer/cvs/coda/source/dac/coda_eb', 'EB', 'aslan', 0, 'down', -1, 'no', 'no')");
        }
        else {
          boolean rowExists = false;
	  rs = stmt.executeQuery("SELECT name FROM process WHERE name LIKE 'cu' OR name LIKE 'ebc'");
	  while (rs.next()) {
	    String s = rs.getString(1);
            if (s.equals("cu") || s.equals("ebc")) {
              // Don't add data as it is already in the database
              rowExists = true;
            }
	  }
          if (!rowExists) {
            stmt.executeUpdate("INSERT INTO process (name, id, cmd, type, host, port, state, pid, inuse, clone) VALUES (" +
                               "'cu', 27, '/home/timmer/cvs/coda/source/dac/coda_roc', 'ROC', 'aslan', 0, 'down', -1, 'no', 'no')");
            stmt.executeUpdate("INSERT INTO process (name, id, cmd, type, host, port, state, pid, inuse, clone) VALUES (" +
                               "'ebc', 28, '/home/timmer/cvs/coda/source/dac/coda_eb', 'EB', 'aslan', 0, 'down', -1, 'no', 'no')");
          }
        }

        if (!gotRunTypes) {
          // Create "runTypes" table
          stmt.executeUpdate("CREATE TABLE runTypes (name VARCHAR(32), id INTEGER, inuse VARCHAR(32), category VARCHAR(32))");
          stmt.executeUpdate("INSERT INTO runTypes (name, id, inuse, category) VALUES (" +
                             "'config', 7, 'no', 'blech')");
        }
        else {
          // Check to see if "sessions" table has entry under sesssion. If not, add.
          boolean rowExists = false;
	  rs = stmt.executeQuery("SELECT name FROM runTypes WHERE name LIKE 'config'");
	  while (rs.next()) {
	    String s = rs.getString(1);
            if (s.equals("config")) {
              // Don't add data as it is already in the database
              rowExists = true;
            }
	  }
          if (!rowExists) {
            stmt.executeUpdate("INSERT INTO runTypes (name, id, inuse, category) VALUES (" +
                               "'config', 7, 'no', 'blech')");
          }
        }
        
        // Close connection
	stmt.close();
	con.close();
    }
    catch (SQLException ex ) {
        ex.printStackTrace();
    }

    return;
  }


}


/**
 * This class handles all communication between the server and a ROC or
 * libmsql user.
 *
 * @author Carl Timmer
 */

class ClientThread extends Thread {
  
  /** Tcp socket. */
  private Socket sock;
  /** Data input stream built on top of the socket's input stream (with an
   *  intervening buffered input stream). */
  private DataInputStream  in;
  /** Data output stream built on top of the socket's output stream (with an
   *  intervening buffered output stream). */
  private DataOutputStream out;
  
  /** Database currently being used. */
  private String currentDatabase;
  /** JDBC url without the ending database (or catalog) name. */
  private String baseUrl;
  /** Complete JDBC database url. */
  private String url;
  /** Database user id. */
  private String user;
  /** Database password for the specified user id. */
  private String password;
  /** Database driver. */
  private String driver;
  /** Level of debug output. A value of 0 means no output and a value
   *  a value of 3 is maximum output. */
  private byte debug = 0;
  /** Connection to the database. */
  private Connection con;
  /** Statement object of the database connection. */
  private Statement stmt;
  /** Database metadata. */
  private DatabaseMetaData meta;

  /** Size (# chars) of the largest error message libmsql can handle. */
  private final int  maxErrorMsgSize    = 159;
  /** Server version number. */
  private final byte version            = 1; 
  
  // Communication & Error codes
  /** Everything OK. */
  private final byte ok                 =  0;
  /** General error. */
  private final byte error              = -1;
  /** Error in sql syntax, query, or update. */
  private final byte sqlError           = -2;
  /** Error in jdbc database access/metadata. */
  private final byte jdbcError          = -3;
  /** Need to connect to database. */
  private final byte cannotConnectError = -4;
  /** No data to send. */
  private final byte noData             = -5;
  /** End of row. */
  private final byte endRow             = -99;
  /** End of data. */
  private final byte endData            = -100;
    
    

  /**
   *  Create a new ClientThread object.
   *  @param socket Tcp socket.
   *  @param dbase index for selecting database to connect to.
   *  @param debugLevel level of debug output.
   */
  ClientThread(Socket socket, byte debugLevel) {
    
    url = System.getProperty("url");
    user = System.getProperty("user");
    driver = System.getProperty("driver");
    password = System.getProperty("password");
    
    // Extract some convenient substrings. The database name
    // (or catalog in jdbc) is the last text (zzz) in a url of the
    // form: jdbc:dbname://host:port/zzz . Or, in the case of postgres,
    // it is the last text (zzz) in the form: jdbc:dbname:zzz
    
    boolean gotcurrentDB = false;
    String  regex = "(jdbc:\\w+:(//[a-zA-Z0-9_\\.]+:\\d+)*)/*(\\w+)";
    Pattern pattern = Pattern.compile(regex);
    Matcher matcher = pattern.matcher(url);
    if (matcher.find()) {
	baseUrl = matcher.group(1);
	currentDatabase = matcher.group(3);
	gotcurrentDB = true;
    }
        
    // There is no database name/catalog in the url.
    if (!gotcurrentDB) {
        baseUrl = url;
        currentDatabase = "default";
    }
    
    sock = socket;
    debug = debugLevel;
  }

  /**
   * Read and ignore all remaining bytes on socket input stream.
   */
  private final void clearSocketInput() throws IOException {
    int bytesLeft = in.available();
    while (bytesLeft > 0) {
      in.skipBytes(bytesLeft);
      bytesLeft = in.available();
    }
    return;
  }  


  /** Start thread to handle communications with user of libmsql. */
  public void run() {

    try {
      byte    reply = version, err;
      boolean isError = false;
      String  errorMsg = null;
      
      // buffered communication streams for efficiency
      in  = new DataInputStream(new BufferedInputStream(sock.getInputStream(), 2048));
      out = new DataOutputStream(new BufferedOutputStream(sock.getOutputStream(), 2048));

      try {
        if (debug > 0) {
          System.out.println("Load driver " + driver + " at url " + url);
          System.out.println("  user =  " + user + ", password = " + password);
        }
	// Register driver
        Class.forName(driver).newInstance();
        // Connect to database
        con = DriverManager.getConnection(url, user, password);
        // Create SQL statement
        stmt = con.createStatement();
        // Create database meta data object
	meta = con.getMetaData();
      }
      catch (InstantiationException ex) {
	errorMsg = ex.getMessage();
        ex.printStackTrace();
	isError = true;
      }
      catch (IllegalAccessException ex) {
	errorMsg = ex.getMessage();
        ex.printStackTrace();
	isError = true;
      }
      catch (ClassNotFoundException ex) {
	errorMsg = ex.getMessage();
        ex.printStackTrace();
	isError = true;
      }
      catch (SQLException ex) {
	errorMsg = ex.getMessage();
        ex.printStackTrace();
	isError = true;
      }
      
      if (isError) {
	// Return error msg. Make sure it isn't too long.
	try {
	  int msgLength = errorMsg.length() > maxErrorMsgSize ? maxErrorMsgSize : errorMsg.length();
	  out.writeByte(jdbcError);
	  out.writeInt(msgLength);
	  out.write(errorMsg.getBytes("ASCII"), 0, msgLength);
	  out.flush();
	}
	catch (UnsupportedEncodingException ex) {
	}
	return;
      }
      else {
	// send version info back to ROC
	out.writeByte(version);
	out.flush();
      }
      
      /* wait for and process requests */
      commandLoop();

      return;
    }
    catch (IOException ex) {
      System.out.println(ex.getMessage());
      ex.printStackTrace();
    }
    finally {
      // we are done with the socket
      try {sock.close();}
      catch (IOException ex) {}
    }
  }


  /**  Wait for and implement commands from the client. */
  private void commandLoop() {
    
    // List of possible client commands
    final byte query      = 1;
    final byte update     = 2;
    final byte getAll     = 3;
    final byte disconnect = 4;
    final byte selectDB   = 5;
    final byte listDBs    = 6;
    final byte listTables = 7;
    final byte listFields = 8;
    
    // Type of data
    final byte jdbcNone = 0;
    final byte jdbcInt  = 1;
    final byte jdbcChar = 2;
    final byte jdbcReal = 3;
    
    // Database query result
    ResultSet result = null;
    // Meta data on a database query result
    ResultSetMetaData metaResult = null;
    // Flag specifying that an error has occurred
    boolean isError = false;
    // Command sent from client
    byte command;
    // Error msg to pass to client
    String errorMsg = null;
    // General variables  
    int size;
    byte err = ok;
    byte[] buffer = new byte[2048];
    
    
    // The Command Loop ...
    try {
      while (true) {
	// Read the remote command.
	command = in.readByte();
	
	switch (command) {

          case  getAll:
          {
            if (debug > 0) {
              System.out.println("get all data");
            }
	    // The following code writes all the return data as well as
	    // some metadata. In order to closely follow the original libmsql
	    // communication protocol, first the data is sent line-by-line.
	    // Then the meta data is sent the same way. Everything is sent
	    // as a string.
	    
	    try {
	      // check to see if there are any results to go through
	      if (result == null) {
                if (debug > 1) {
                  System.out.println("Client must (re)query before issuing \"next\" command");
		}
		err = noData;
		errorMsg = "Must (re)query the database.";
		isError = true;
		break;
	      }
	      
	      ByteArrayOutputStream baos = new ByteArrayOutputStream();
	      DataOutputStream dos = new DataOutputStream(baos);
	      
	      // Send the data first
	      String  s;	      
	      boolean firstTimeThru = true;
	      int     columns = metaResult.getColumnCount();
	      // No error (yet ...)
	      dos.writeByte(ok);
	      
              if (debug > 1) {
                System.out.println("send data");
              }
	      
	      // All data sent as strings
	      while(result.next()) {
		if (!firstTimeThru) {
		  // mark end of row
		  dos.writeByte(endRow);
        	  if (debug > 2) {
        	    System.out.println("");
		  }
		}
		for (int i=1; i <= columns; i++) {
		  // postgres pads char fields with spaces
		  s = result.getString(i).trim();
		  if (s == null) {
		    dos.writeInt(0);
                    if (debug > 2) {
                      System.out.println("  null");
		    }
		  }
		  else {
		    // cedit chokes on zero-length strings
		    if (s.length() < 1) {
		    	s = " ";
		    }
		    dos.writeInt(s.length());
		    dos.write(s.getBytes("ASCII"));
                    if (debug > 2) {
                      System.out.println("  [" + s + "]");
		    }
		  }
		}
		firstTimeThru = false;
	      }
	      // If no data, send 0 for each column
	      if (firstTimeThru) {
		for (int i=1; i <= columns; i++) {
		  dos.writeInt(0);
		}
	      }
	      
	      // Mark end of data
              dos.writeByte(endData);
              dos.flush();
	      out.write(baos.toByteArray());
	      out.flush();
	      
	      
	      // Now send the meta data -
	      // 1 row of 6 fields for each column
	      int       type;
	      ResultSet rs = null;
	      String    table = null, colName = null;
	      baos.reset();
              if (debug > 1) {
                System.out.println("send metadata");
	      }
	      
	      for (int i=1; i <= columns; i++) {
		// table name
		table = metaResult.getTableName(i);
		
		// Postgres chokes and returns a blank, but that makes
		// the msql client lib also choke. Solution is to make
		// up a table name.
		if ((table == null) || (table == "")) {
		  table = "junk";
		}
		
		dos.writeInt(table.length());
		dos.write(table.getBytes("ASCII"));
		
                if (debug > 2) {
                  if (table == null) System.out.println("  table is null");
		  else if (table == "") System.out.println("  table is blank");
		  else System.out.println("  table = " + table);
		}
		
		// column name
		colName = metaResult.getColumnName(i);
		if ((colName == null) || (colName == "")) {
		  dos.writeInt(0);
		}
		else {
		  dos.writeInt(colName.length());
		  dos.write(colName.getBytes("ASCII"));
		}
                if (debug > 2) {
                  System.out.println("  col = " + colName);
		}
		
		// type of data
		type = metaResult.getColumnType(i);
		if (type == java.sql.Types.INTEGER) {
		  s = "" + jdbcInt;
		  dos.writeInt(s.length());
		  dos.write(s.getBytes("ASCII"));
                  if (debug > 2) {
                    System.out.println("  type = int");
		  }
		}
		else if ((type == java.sql.Types.CHAR) ||
		         (type == java.sql.Types.VARCHAR)) {
		  s = "" + jdbcChar;
		  dos.writeInt(s.length());
		  dos.write(s.getBytes("ASCII"));
                  if (debug > 2) {
                    System.out.println("  type = char");
		  }
		}
		else if (type == java.sql.Types.REAL) {
		  s = "" + jdbcReal;
		  dos.writeInt(s.length());
		  dos.write(s.getBytes("ASCII"));
                  if (debug > 2) {
                    System.out.println("  type = real");
		  }
		}
		else {
		  s = "" + jdbcNone;
		  dos.writeInt(s.length());
		  dos.write(s.getBytes("ASCII"));
                  if (debug > 2) {
                    System.out.println("  type = unknown");
		  }
		}
		
		// length of char field
		// (but wrongly set to 0 for char types in MySQL)
		s = "" + metaResult.getPrecision(i);
		// length of char field - another way (wrong in hsqldb)
		rs = meta.getColumns(currentDatabase, null, table, colName);
		if ((rs != null) && (rs.next())) {
		  String s2 = rs.getString(7);
		  if (s2 != null) s = s2;
		}
		dos.writeInt(s.length());
		dos.write(s.getBytes("ASCII"));
		if (debug > 2) {
                  System.out.println("  max size (for CHAR) = " + s);
		}
		
		// is field not null?
		s = "Y";
		if (metaResult.isNullable(i) == ResultSetMetaData.columnNullable) {
		  s = "N";
		}
		dos.writeInt(s.length());
		dos.write(s.getBytes("ASCII"));
		if (debug > 2) {
                  System.out.println("  nullable = " + s);
		}
		
		// Is field primary key? Look for this column's name
		// in primary key list.
		s = "N";
		if ((table != null) && (table != "")) {
		  rs = meta.getPrimaryKeys(currentDatabase, null, table);
		  while (rs.next()) {
		    if (rs.getString("COLUMN_NAME").equals(colName)) {
		      s = "Y";
		      break;
		    }
		  }
		}
		dos.writeInt(s.length());
		dos.write(s.getBytes("ASCII"));
		if (debug > 2) {
                  System.out.println("  primary key = " + s);
		}
		
		// Mark end of row for all except last row
		if (i == columns) break;
		dos.writeByte(endRow);
        	if (debug > 2) {
        	  System.out.println("");
		}
	      }
	      
	      // Mark end of data
              dos.writeByte(endData);
              dos.flush();
	      out.write(baos.toByteArray());
	      out.flush();
	    }
	    catch (UnsupportedEncodingException ex) {
	      // Never happens
	    }
	    catch (SQLException ex) {
	      err = jdbcError;
	      errorMsg = ex.getMessage();
              System.out.println(errorMsg);
              ex.printStackTrace();
	      isError = true;
	    }
	  }
	  break;


          case  listDBs:
          {
	    if (debug > 0) {
              System.out.println("list databases");
	    }
	    // Send list of DB names.
	    try {
	      ByteArrayOutputStream baos = new ByteArrayOutputStream();
	      DataOutputStream dos = new DataOutputStream(baos);
	      
	      String    name = null;
	      boolean   firstTimeThru = true;
	      ResultSet rs = meta.getCatalogs();
	      dos.writeByte(ok);
	      
	      while(rs.next()) {
		if (!firstTimeThru) {
		  // mark end of row
		  dos.writeByte(endRow);
		}
		// database name
		name = rs.getString(1);
		if ((name == null) || (name == "")) {
		  dos.writeInt(0);
		}
		else {
	          if (debug > 2) {
                    System.out.println("  " + name);
		  }
		  dos.writeInt(name.length());
		  dos.write(name.getBytes("ASCII"));
		}
		firstTimeThru = false;
	      }
	      // if no databases listed, send the current one
	      if (firstTimeThru) {
		  dos.writeInt(currentDatabase.length());
		  dos.write(currentDatabase.getBytes("ASCII"));
	      }
	      // mark end of data
              dos.writeByte(endData);
              dos.flush();
	      out.write(baos.toByteArray());
	      out.flush();
	    }
	    catch (UnsupportedEncodingException ex) {
	      // never happens
	    }
	    catch (SQLException ex) {
	      err = jdbcError;
	      errorMsg = ex.getMessage();
              System.out.println(errorMsg);
              ex.printStackTrace();
	      isError = true;
	    }
	  }
	  break;


          case  listTables:
          {
	    if (debug > 0) {
              System.out.println("list tables for database " + currentDatabase);
	    }
	    // Send list of table names.
	    try {
	      ByteArrayOutputStream baos = new ByteArrayOutputStream();
	      DataOutputStream dos = new DataOutputStream(baos);
	      
	      String    name = null;
	      boolean   firstTimeThru = true;
	      ResultSet rs = meta.getTables(currentDatabase, null, "%", null);
	      dos.writeByte(ok);
	      
	      while(rs.next()) {
		if (!firstTimeThru) {
		  // mark end of row
		  dos.writeByte(endRow);
		}
		// database name
		name = rs.getString(3);
		if ((name == null) || (name == "")) {
		  dos.writeInt(0);
		}
		else {
	          if (debug > 2) {
                    System.out.println("  " + name);
		  }
		  dos.writeInt(name.length());
		  dos.write(name.getBytes("ASCII"));
		}
		firstTimeThru = false;
	      }
	      // if no tables listed, send a null
	      if (firstTimeThru) {
		  dos.writeInt(0);
	      }
	      // mark end of data
              dos.writeByte(endData);
              dos.flush();
	      out.write(baos.toByteArray());
	      out.flush();
	    }
	    catch (UnsupportedEncodingException ex) {
	      // never happens
	    }
	    catch (SQLException ex) {
	      err = jdbcError;
	      errorMsg = ex.getMessage();
              System.out.println(errorMsg);
              ex.printStackTrace();
	      isError = true;
	    }
	  }
	  break;


          case  listFields:
          {
	    // Send list of a table's field names.
	    try {
	      // get table name
	      size = in.readInt();
	      if (size > buffer.length) {
		buffer = new byte[size];
	      }
	      in.readFully(buffer, 0, size);
	      String table = new String(buffer, 0, size, "ASCII");
	      if (debug > 0) {
                System.out.println("list fields for table " + table);
              }
	    
	      ByteArrayOutputStream baos = new ByteArrayOutputStream();
	      DataOutputStream dos = new DataOutputStream(baos);
	      
	      // 1 row of 6 fields for each column of "table"
	      ResultSet rsPkey, rs = meta.getColumns(currentDatabase, null, table, "%");
	      boolean   firstTimeThru = true;
	      String    s, colName = null;
	      int       type, nullable;
	      
	      dos.writeByte(ok);
	      
	      while (rs.next()) {
		if (!firstTimeThru) {
		  // mark end of row
		  dos.writeByte(endRow);
        	  if (debug > 2) {
        	    System.out.println("");
		  }
		}

		// table name we already have
		dos.writeInt(table.length());
		dos.write(table.getBytes("ASCII"));
		
		// column name
		colName = rs.getString(4);
		if ((colName == null) || (colName == "")) {
		  dos.writeInt(0);
		}
		else {
		  dos.writeInt(colName.length());
		  dos.write(colName.getBytes("ASCII"));
		}
	        if (debug > 2) {
                  System.out.println("  col = " + colName);
		}
		
		// type of data
		type = rs.getInt(5);
		if (type == java.sql.Types.INTEGER) {
		  s = "" + jdbcInt;
		  dos.writeInt(s.length());
		  dos.write(s.getBytes("ASCII"));
	          if (debug > 2) {
                    System.out.println("  type = int");
		  }
		}
		else if ((type == java.sql.Types.CHAR) ||
		         (type == java.sql.Types.VARCHAR)) {
		  s = "" + jdbcChar;
		  dos.writeInt(s.length());
		  dos.write(s.getBytes("ASCII"));
	          if (debug > 2) {
                    System.out.println("  type = char");
		  }
		}
		else if (type == java.sql.Types.REAL) {
		  s = "" + jdbcReal;
		  dos.writeInt(s.length());
		  dos.write(s.getBytes("ASCII"));
	          if (debug > 2) {
                    System.out.println("  type = real");
		  }
		}
		else {
		  s = "" + jdbcNone;
		  dos.writeInt(s.length());
		  dos.write(s.getBytes("ASCII"));
	          if (debug > 2) {
                    System.out.println("  type = unknown");
		  }
		}
		
		// max length of char field - precision of numeric field
		s = "" + rs.getInt(7);
		dos.writeInt(s.length());
		dos.write(s.getBytes("ASCII"));
	        if (debug > 2) {
                  System.out.println("  max size (for CHAR) = " + s);
		}
		
		// is field not null?
		s = "Y";
		if (rs.getInt(11) == DatabaseMetaData.columnNullable) {
		  s = "N";
		}
		dos.writeInt(s.length());
		dos.write(s.getBytes("ASCII"));
	        if (debug > 2) {
                  System.out.println("  nullable? = " + s);
		}
		
		// Is field primary key? Look for this column's name
		// in primary key list.
		s = "N";
		rsPkey = meta.getPrimaryKeys(currentDatabase, null, table);
		while (rsPkey.next()) {
		  if (rsPkey.getString("COLUMN_NAME").equals(colName)) {
		    s = "Y";
		    break;
		  }
		}
		dos.writeInt(s.length());
		dos.write(s.getBytes("ASCII"));
	        if (debug > 2) {
                  System.out.println("  primary key? = " + s);
		}
		
		firstTimeThru = false;
	      }
	      
	      // mark end of data
              dos.writeByte(endData);
              dos.flush();
	      out.write(baos.toByteArray());
	      out.flush();
	    }
	    catch (UnsupportedEncodingException ex) {
	      // never happens
	    }
	    catch (SQLException ex) {
	      err = jdbcError;
	      errorMsg = ex.getMessage();
              System.out.println(errorMsg);
              ex.printStackTrace();
	      isError = true;
	    }
	  }
	  break;


          case  query:
          {
	    // read query
	    size = in.readInt();
	    if (size > buffer.length) {
	      buffer = new byte[size];
	    }
	    in.readFully(buffer, 0, size);
	    String sqlQuery = new String(buffer, 0, size, "ASCII");
	    if (debug > 0) {
              System.out.println("query = " + sqlQuery);
	    }
	    
	    try {
	      result = stmt.executeQuery(sqlQuery);
	    }
	    catch (SQLException ex) {
	      err = sqlError;
	      errorMsg = ex.getMessage();
              System.out.println(errorMsg);
              ex.printStackTrace();
	      isError = true;
	      break;
	    }
	    
	    try {
	      metaResult = result.getMetaData();
	      int count = metaResult.getColumnCount();
	      out.writeByte(ok);
	      out.writeInt(count);
	      out.flush();
	    }
	    catch (SQLException ex) {
	      err = jdbcError;
	      errorMsg = ex.getMessage();
              System.out.println(errorMsg);
              ex.printStackTrace();
	      isError = true;
	    }
	  }
	  break;


          case  update:
          {
	    // read update
	    size = in.readInt();
	    if (size > buffer.length) {
	      buffer = new byte[size];
	    }
	    in.readFully(buffer, 0, size);
	    String sqlUpdate = new String(buffer, 0, size, "ASCII");
	    result = null;
	    if (debug > 0) {
              System.out.println("update = " + sqlUpdate);
	    }
	    
	    try {
	      stmt.executeUpdate(sqlUpdate);
	      out.writeByte(ok);
	      out.flush();
	    }
	    catch (SQLException ex) {
	      err = sqlError;
	      errorMsg = ex.getMessage();
              System.out.println(errorMsg);
              ex.printStackTrace();
	      isError = true;
	    }
	  }
	  break;

	  
          case  disconnect:
          {
	    if (debug > 0) {
              System.out.println("disconnect");
	    }
	    try {
	      stmt.close();
	      con.close();
	    }
	    catch (SQLException ex) {
	    }
	  }
	  return;
	  
	  
          case  selectDB:
          {
	    // read database name
	    size = in.readInt();
	    if (size > buffer.length) {
	      buffer = new byte[size];
	    }
	    in.readFully(buffer, 0, size);
	    String db = new String(buffer, 0, size, "ASCII");
	    err = ok;
	    if (debug > 0) {
              System.out.println("select database " + db);
	    }
	    
	    if (currentDatabase.equals(db)) {
	      // Nothing needs to be done since we're already
	      // connected to that database.
	      if (debug > 1) {
                System.out.println("  already in selected database so do nothing");
	      }
	    }
	    else {
	      // List all databases (or catalogs in jdbc language).
	      // If none or only 1 is listed, do nothing. Otherwise,
	      // close the connection to the current db and open up
	      // a new connection.
	      int numDbs = 0;
	      boolean match = false;
	      try {
		result = meta.getCatalogs();
		while (result.next()) {
		  numDbs++;
		  // does the db we're looking for exist?
		  if (db.equals(result.getString(1))) {
		    match = true;
		  }
		}
	        result = null;
	      }
	      catch (SQLException ex) {
	        result = null;
		err = jdbcError;
		errorMsg = ex.getMessage();
        	System.out.println(errorMsg);
        	ex.printStackTrace();
		isError = true;
		break;
	      }
	      
	      if (match == false) {
		if (debug > 1) {
                  System.out.println("  database does not exist so ignore command");
		}
	      }
              else if ((numDbs > 1) && (match)) {
	        // Connect to the new db and disconnect the old db .
		// First try connecting with the new db.
		// If that works, dump the old connection.
		Connection conNew;
		Statement  stmtNew;
		DatabaseMetaData metaNew;

		isError = false;
		url  = baseUrl + "/" + db;
	        if (debug > 1) {
                  System.out.println("  try to connect to " + url);
		}
		  
		try {
		  conNew = DriverManager.getConnection(url, user, password);
		  stmtNew = conNew.createStatement();
		  metaNew = conNew.getMetaData();
		}
		catch (SQLException ex) {
		  err = cannotConnectError;
		  errorMsg = ex.getMessage();
        	  System.out.println(errorMsg);
        	  ex.printStackTrace();
		  isError = true;
		  break;
		}
                
		try {
		  stmt.close();
		  con.close();
		}  
 		catch (SQLException ex) {
		  // ignore
		}
		
       	        con  = conNew;
        	stmt = stmtNew;
		meta = metaNew;
		currentDatabase = db;
	      }
	    }
	    
	    out.writeByte(ok);
	    out.flush();
	  }
	  break;


          default:
	  {
	    err = error;
	    isError = true;
	    errorMsg = "Unknown command";
	  }	
	} // switch(command)
	
	if (isError) {
	  // read everything on the socket and throw it away
	  clearSocketInput();

	  // Return error msg. Make sure it isn't too long.
	  try {
	    int msgLength = errorMsg.length() > maxErrorMsgSize ? maxErrorMsgSize : errorMsg.length();
	    out.writeByte(err);
	    out.writeInt(msgLength);
	    out.write(errorMsg.getBytes("ASCII"), 0, msgLength);
	    out.flush();
	  }
	  catch (UnsupportedEncodingException ex) {
	  }

	  // reset flag
	  isError = false;
	}
      }  // while(true)
    }   // try

    catch (IOException ex) {
    }

    // We only end up down here if there's a communication error.
    // Perhaps the client has crashed.
    if (debug > 0) {
      System.out.println("Client being disconnected");
    }
    
    try {
      stmt.close();
      con.close();
    }
    catch (SQLException ex) {
    }

    return;
  }
}
