/*
 **    libmsqlJDBC.c     -
 **
 **
 ** Copyright (c) 1993-95  David J. Hughes
 ** Copyright (c) 1995  Hughes Technologies Pty Ltd
 ** Copyright (c) 2001  Southeastern Universities Research Association,
 **                     Thomas Jefferson National Accelerator Facility,
 **                     Carl Timmer, author
 **			This file was modified to connect to a java server
 **			which translates the msql server instructions into
 **			jdbc so users of msql can switch to using any jdbc
 **			database by using this library.
 **
 ** Permission to use, copy, and distribute for non-commercial purposes,
 ** is hereby granted without fee, providing that the above copyright
 ** notice appear in all copies and that both the copyright notice and this
 ** permission notice appear in supporting documentation.
 **
 ** This software is provided "as is" without any expressed or implied warranty.
 **
 ** ID = "$Id:"
 **
 */
#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <errno.h>
#include <ctype.h>
#include <arpa/inet.h>

#include <signal.h>
#include <netdb.h>
#ifdef VXWORKS
#include <sysLib.h>
#else
#include <pwd.h>
#include <varargs.h>
#endif
#include <stdlib.h>
#include <string.h>

#include "common/portability.h"

#undef HAVE_SYS_UN_H

#ifdef HAVE_SYS_UN_H
#  include <sys/un.h>
#endif

#include "common/site.h"

#include "msql_priv.h"
#include "errmsg.h"

#define _LIB_SOURCE
#include "msql.h"

#ifndef	INADDR_NONE
#define INADDR_NONE	-1
#endif

#define BAD_PARAM_ERROR "Bad parameter"
static	int	curServerSock,
		numFields,
		queryTableSize,
		fieldTableSize,
		protoInfo;
static	ms_data	*tmpDataStore = NULL,
		*queryData = NULL,
		*fieldData = NULL;

int		msqlTcpPort = 0;
char		msqlErrMsg[160];

RETSIGTYPE	(*oldHandler)();

static void	msqlDebug();

#define	resetError()	(void)bzero(msqlErrMsg,sizeof(msqlErrMsg))
#define chopError()	{ char *cp; cp = msqlErrMsg+strlen(msqlErrMsg) -1; \
			    if (*cp == '\n') *cp = 0;}
#define safeFree(x)     {if(x) { (void)free(x); x = NULL; } }
#define MAX_IX 1024

extern char	*packet;
static char	*hosts[MAX_IX];
static int	ports[MAX_IX];
static int	socks[MAX_IX];
static int	currIx=0;

static	int	readTimeout;

/* Constants used for communication with jdbc server */
#define JDBCSERVERVERSION 1
/* Commands from client */
static char jdbcQuery      = 1;
static char jdbcUpdate     = 2;
static char jdbcGetAll     = 3;
static char jdbcDisconnect = 4;
static char jdbcSelectDB   = 5;
static char jdbcListDBs    = 6;
static char jdbcListTables = 7;
static char jdbcListFields = 8;

/* Type of query data returned */
static char jdbcNone = 0;
static char jdbcInt  = 1;
static char jdbcChar = 2;
static char jdbcReal = 3;

/* Error codes */
static char ok                 =  0;   /* everything OK */
static char error              = -1;   /* general error */
static char sqlError           = -2;   /* error in sql syntax, query, or update */
static char jdbcError          = -3;   /* error in jdbc database access */
static char cannotConnectError = -4;   /* cannot to connect to db */
static char noData             = -5;   /* no data sent */
static char endRow             = -99;  /* end of row */
static char endData            = -100; /* end of data */

/* Client version number */
static char version = 1;

/* prototype */
static int readQueryData(int sock);

#define	MOD_QUERY	1
#define MOD_API		2
#define MOD_MALLOC	4
#define MOD_ERROR	8

static int 	debugLevel=0,
                debugInit = 0;

/**************************************************************************
 **	msqlInitDebug
 **
 **	Purpose	: 
 **	Args	: 
 **	Returns	: 
 **	Notes	: 
 */

static void msqlInitDebug()
{
  char	*env,
  *tmp,
  *tok;

  env = getenv("MINERVA_DEBUG");
  if(env)
    {
      tmp = (char *)strdup(env);
    }
  else
    return;
  printf("\n-------------------------------------------------------\n");
  printf("MINERVA_DEBUG found. libmsql started with the following:-\n\n");
  tok = (char *)strtok(tmp,":");
  while(tok)
    {
      if (strcmp(tok,"query") == 0)
	{
	  debugLevel |= MOD_QUERY;
	  printf("Debug level : query\n");
	}
      if (strcmp(tok,"api") == 0)
	{
	  debugLevel |= MOD_API;
	  printf("Debug level : api\n");
	}
      if (strcmp(tok,"malloc") == 0)
	{
	  debugLevel |= MOD_MALLOC;
	  printf("Debug level : malloc\n");
	}
      tok = (char *)strtok(NULL,":");
    }
  safeFree(tmp);
  printf("\n-------------------------------------------------------\n\n");
}


/**************************************************************************
 **	setServerSock 	
 **
 **	Purpose	: Store the server socket currently in use
 **	Args	: Server socket
 **	Returns	: Nothing
 **	Notes	: The current socket is stored so that the signal
 **		  handlers know which one to shut down.
 */

static void setServerSock(sock)
     int	sock;
{
  curServerSock = sock;
}




/**************************************************************************
 **	closeServer 	
 **
 **	Purpose	: Shut down the server connection
 **	Args	: Server socket
 **	Returns	: Nothing
 **	Notes	: This is used by msqlClose and the signal handlers
 */

static void closeServer(sock)
     int	sock;
{
  char cmd;

  msqlDebug(MOD_API,"Server socket (%d) closed\n", sock);
  
  /* command */
  cmd = jdbcDisconnect;
  tcp_write(sock, (void *) &cmd, sizeof(cmd));

  shutdown(sock,2);
  /* signal(SIGPIPE,oldHandler); */
  close(sock);
}





/**************************************************************************
 **	msqlClose
 **
 **	Purpose	: Send a QUIT to the server and close the connection
 **	Args	: Server socket
 **	Returns	: Nothing
 **	Notes	: 
 */
void msqlClose(sock)
     int	sock;
{
  if (hosts[sock])
    free(hosts[sock]);
  
}

void msqlClose_real(sock)
     int	sock;
{
  setServerSock(sock);
  closeServer(sock);
}





/**************************************************************************
 **	pipeHandler
 **
 **	Purpose	: Close the server connection if we get a SIGPIPE
 **	Args	: sig
 **	Returns	: Nothing
 **	Notes	: 
 */

RETSIGTYPE pipeHandler(sig)
     int	sig;
{
  msqlDebug(MOD_API,"Hit by pipe signal\n");
  closeServer(curServerSock);
  signal(SIGPIPE,pipeHandler);
  return;
}




static freeQueryData(cur)
     ms_data	*cur;
{
  ms_data	*prev;
  int	offset;

  while(cur)
    {
      offset = 0;
      while(offset < cur->width)
	{
	  safeFree(cur->data[offset]);
	  offset++;
	}
      safeFree(cur->data);
      prev = cur;
      cur = cur->next;
      safeFree(prev);
      msqlDebug(MOD_MALLOC, "Query data row - free @ %X\n", prev);
    }
  safeFree(cur);
}





static freeFieldList(fieldData)
     m_fdata	*fieldData;
{
  m_fdata	*cur,
  *prev;

  cur = fieldData;
  while(cur)
    {
      prev = cur;
      cur = cur->next;
      safeFree(prev->field.table);
      safeFree(prev->field.name);
      safeFree(prev);
      msqlDebug(MOD_MALLOC, "Field List Entry- free @ %X\n", prev);
    }
}



/**************************************************************************
 **	msqlFreeResult
 **
 **	Purpose	: Free the memory allocated to a table returned by a select
 **	Args	: Pointer to head of table
 **	Returns	: Nothing
 **	Notes	: 
 */

void msqlFreeResult(result)
     m_result  *result;
{
  freeQueryData(result->queryData);
  freeFieldList(result->fieldData);
  safeFree(result);
  msqlDebug(MOD_MALLOC,"Result Handle - Free @ %X\n",result);
}



/**************************************************************************
 **	tableToFieldList
 **
 */
static m_fdata *tableToFieldList(data)
     ms_data	*data;
{
  ms_data	*curRow;
  m_fdata	*curField,
                *prevField,
                *head = NULL;

  curRow = data;
  while(curRow)
    {
      curField = (m_fdata *)malloc(sizeof(m_fdata));
      msqlDebug(MOD_MALLOC,"Field List Entry - malloc @ %X of %d\n",
		curField, sizeof(m_fdata));
      (void)bzero(curField, sizeof(m_fdata));
      if (head)
	{
	  prevField->next = curField;
	  prevField = curField;
	}
      else
	{
	  head = prevField = curField;
	}

      curField->field.table = (char *)strdup((char *)curRow->data[0]);
      curField->field.name = (char *)strdup((char *)curRow->data[1]);
      curField->field.type = atoi((char*)curRow->data[2]);
      curField->field.length = atoi((char*)curRow->data[3]); /* max length of field */
      curField->field.flags = 0;
      if (*curRow->data[4] == 'Y')
	curField->field.flags |= NOT_NULL_FLAG;
      if (*curRow->data[5] == 'Y')
	curField->field.flags |= PRI_KEY_FLAG;
      curRow = curRow->next;
    }
  return(head);
}

/**************************************************************************
 **	msqlConnectPort
 **
 */
int msqlConnectPort(host, port) 
char *host;
int   port;
{
  int val, sock;
  
  if (port < 1)
    return -1;
    
  if (host != NULL) {
    hosts[currIx] = malloc(strlen(host) + 1);
    strcpy(hosts[currIx],host);
  } else {
    hosts[currIx] = NULL;
  }
  ports[currIx] = port;
  val = currIx;
 
  sock = msqlConnect_real(val);
  if (sock == -1)
    return -1;
    
  socks[currIx] = sock;

  currIx = (currIx +1) & 1023;
  return val;
}


/**************************************************************************
 **	msqlConnect
 **
 */
int msqlConnect(host) 
char *host;
{
  int val, sock;
  
  if (host != NULL) {
    hosts[currIx] = malloc(strlen(host) + 1);
    strcpy(hosts[currIx],host);
  } else {
    hosts[currIx] = NULL;
  }
  ports[currIx] = 0;
  val = currIx;
  
  sock = msqlConnect_real(val);
  if (sock == -1)
    return -1;  
    
  socks[currIx] = sock;
  
  currIx = (currIx +1) & 1023;
  return val;
}

/**************************************************************************
 **	msqlConnect_real
 **
 **	Purpose	: Form a connection to a mSQL server
 **	Args	: hostname of server
 **	Returns	: socket for further use.  -1 on error
 */

int msqlConnect_real(index)
  int		index;
{
  char		*cp,
		*host,
		*envVar,
		*unixPort,
		version = JDBCSERVERVERSION,
		serverVersion = JDBCSERVERVERSION;
  struct	sockaddr_in IPaddr;
  struct servent *serv_ptr;
  struct	hostent *hp;
  u_long	IPAddr;
  int		opt,
		sock,
		size;
  
  resetError();
  initNet();
  if (!debugInit)
    {
      debugInit++;
      msqlInitDebug();
    }

  /*
   ** Grab a socket and connect it to the server
   */
  host = hosts[index];
  if ((!host) && (envVar = getenv("MSQL_TCP_HOST")))
      host = envVar;

  if (!host)
      host = "localhost";

  if (ports[index])
      msqlTcpPort = ports[index];

  if ((msqlTcpPort < 1) && (envVar = getenv("MSQL_TCP_PORT")))
      msqlTcpPort = atoi(envVar);

  if (msqlTcpPort < 1)
      msqlTcpPort = MSQL_PORT;

#ifndef VXWORKS
  if ((msqlTcpPort < 1) && (serv_ptr = getservbyname("msql", "tcp")))
      msqlTcpPort = serv_ptr->s_port;
#endif

  msqlDebug(MOD_API,"Server name = %s.  Using TCP sock (%d)\n",
	    host, msqlTcpPort);

  sock = socket(AF_INET,SOCK_STREAM,0);
  if (sock < 0)
    {
      sprintf(msqlErrMsg,IPSOCK_ERROR);
      return(-1);
    }
  setServerSock(sock);
  opt = 1;
  setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, (char *)&opt, sizeof(opt));
  /* add this, CTCTCT */
  setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, (char *)&opt, sizeof(opt));

  bzero(&IPaddr,sizeof(IPaddr));
  IPaddr.sin_family = AF_INET;

  /*
   ** The server name may be a host name or IP address
   */

  if ((IPAddr = inet_addr(host)) != INADDR_NONE)
    {
      bcopy(&IPAddr,&IPaddr.sin_addr,sizeof(IPAddr));
    }
  else
    {
      hp = gethostbyname(host);
      if (!hp)
	{
	  sprintf(msqlErrMsg,
		  UNKNOWN_HOST,
		  host);
	  close(sock);
	  return(-1);
	}
      bcopy(hp->h_addr,&IPaddr.sin_addr, hp->h_length);
    }
  IPaddr.sin_port = htons(msqlTcpPort);
  if (connect(sock,(struct sockaddr *) &IPaddr,sizeof(IPaddr))<0)
    {
      sprintf(msqlErrMsg,CONN_HOST_ERROR, host);
      perror("Connect");
      close(sock);
      return(-1);
    }
  
  oldHandler = signal(SIGPIPE,pipeHandler);
  msqlDebug(MOD_API,"Connection socket = %d\n",sock);

  /* read server's version number */
  if (tcp_read(sock, &serverVersion, sizeof(serverVersion)) != sizeof(serverVersion)) {
      closeServer(sock);
      strcpy(msqlErrMsg,SERVER_GONE_ERROR);
      return(-1);
  }
  /* check for error */
  if (serverVersion  < ok) {
      if (tcp_read(sock, &size, sizeof(size)) != sizeof(size)) {
	  closeServer(sock);
	  strcpy(msqlErrMsg,SERVER_GONE_ERROR);
	  return(-1);
      }
      size = ntohl(size);
      if (tcp_read(sock, msqlErrMsg, (size_t) size) != (size_t) size) {
	  closeServer(sock);
	  strcpy(msqlErrMsg,SERVER_GONE_ERROR);
	  return(-1);
      }
      return(-1);
  }
  else if (serverVersion != version) {
      closeServer(sock);
      strcpy(msqlErrMsg, "msqlConnect_real: The jdbc server is not the expected version.");
      return(-1);
  }

  return(sock);
}






/**************************************************************************
 **	msqlSelectDB
 **
 **	OK
 **
 **	Purpose	: Tell the server which database we want to use
 **	Args	: Server sock and DB name
 **	Returns	: -1 on error
 **	Notes	: 
 */
int msqlSelectDB(index,db)
     int	index;
     char	*db;
{
  int    size, sock;
  char	 err, *buf;
  size_t bufsize, dbsize;
  

  if ((db == NULL) || (*db == 0)) {
    strcpy(msqlErrMsg,BAD_PARAM_ERROR);
    return (-1);
  }
  
  sock = socks[index];
  msqlDebug(MOD_API,"Select Database = \"%s\" socket = %d\n",db,sock);

  resetError();
  setServerSock(sock);
  
  dbsize  = strlen(db);
  size    = htonl((int) dbsize);
  bufsize = dbsize + sizeof(int) + sizeof(char);
  buf     = (char *) malloc(bufsize);
  if (buf == NULL) {
      closeServer(sock);
      strcpy(msqlErrMsg, "Cannot allocate memory");
      return(-1);
  }
  msqlDebug(MOD_MALLOC,"TCP buffer - malloc @ %X of %d\n", buf, bufsize);

  /* select DB command */
  buf[0] = jdbcSelectDB;
  /* length of DB name */
  memcpy((void *) &buf[1], (const void *) &size, sizeof(int));
  /* DB name */
  memcpy((void *) &buf[5], (const void *) db, dbsize);
  
  if (tcp_write(sock, (void *) buf, bufsize) != bufsize) {
      closeServer(sock);
      strcpy(msqlErrMsg, "Cannot communicate with JDBC Server");
      free((void *) buf);
      return(-1);
  }
  free((void *) buf);
  
  /* Error code returned */
  if (tcp_read(sock, &err, sizeof(err)) != sizeof(err)) {
      closeServer(sock);
      strcpy(msqlErrMsg,SERVER_GONE_ERROR);
      return(-1);
  }
  
  /* Check the result  */
  if (err != ok) {
      if (tcp_read(sock, &size, sizeof(size)) != sizeof(size)) {
	  closeServer(sock);
	  strcpy(msqlErrMsg,SERVER_GONE_ERROR);
	  return(-1);
      }
      size = ntohl(size);
      if (tcp_read(sock, msqlErrMsg, (size_t) size) != (size_t) size) {
	  closeServer(sock);
	  strcpy(msqlErrMsg,SERVER_GONE_ERROR);
	  return(-1);
      }
      
      return(-1);
  }

  msqlDebug(MOD_API,"Select Database returning\n");

  return(0);
}

/**************************************************************************
 **	msqlStoreResult
 **
 **	Purpose	: Store the data returned from a query
 **	Args	: None
 **	Returns	: Result handle or NULL if no data
 **	Notes	: 
 */

m_result *msqlStoreResult()
{
  m_result *tmp;

  if (!queryData && !fieldData)
    {
      return(NULL);
    }
  tmp = (m_result *)malloc(sizeof(m_result));

  msqlDebug(MOD_MALLOC,"Result Handle - malloc @ %X of %d\n",
	    tmp, sizeof(m_result));

  if (!tmp)
    {
      return(NULL);
    }
  (void)bzero(tmp, sizeof(m_result));
  tmp->queryData = queryData;
  tmp->numRows = queryTableSize;
  tmp->fieldData = tableToFieldList(fieldData);
  tmp->numFields = fieldTableSize;
  tmp->cursor = tmp->queryData;
  tmp->fieldCursor = tmp->fieldData;
  freeQueryData(fieldData);
  queryData = NULL;
  fieldData = NULL;
  return(tmp);
}





/**************************************************************************
 **	msqlFetchField
 **
 **	Purpose	: Return a row of the query results
 **	Args	: result handle
 **	Returns	: pointer to row data
 **	Notes	: 
 */

m_field	*msqlFetchField(handle)
     m_result *handle;
{
  m_field	*tmp;

  if (!handle->fieldCursor)
    {
      return(NULL);
    }
  tmp = &(handle->fieldCursor->field);
  handle->fieldCursor = handle->fieldCursor->next;
  return(tmp);
}



/**************************************************************************
 **	msqlFetchRow
 **
 **	Purpose	: Return a row of the query results
 **	Args	: result handle
 **	Returns	: pointer to row data
 **	Notes	: 
 */

m_row	msqlFetchRow(handle)
     m_result *handle;
{
  m_row	tmp;

  if (!handle->cursor)
    {
      return(NULL);
    }
  tmp = handle->cursor->data;
  handle->cursor = handle->cursor->next;
  return(tmp);
}




/**************************************************************************
 **	msqlFieldSeek
 **
 **	Purpose	: Move the result cursor
 **	Args	: result handle, offset
 **	Returns	: Nothing.  Just sets the cursor
 **	Notes	: The data is a single linked list so we can go backwards
 */

void msqlFieldSeek(handle, offset)
     m_result *handle;
     int	offset;
{
  m_fdata	*tmp;

  msqlDebug(MOD_API,"msqlFieldSeek() pos = \n",offset);

  tmp = handle->fieldData;
  while(offset)
    {
      if (!tmp)
	break;
      tmp = tmp->next;
      offset--;
    }
  handle->fieldCursor = tmp;
}



/**************************************************************************
 **	msqlDataSeek
 **
 **	Purpose	: Move the result cursor
 **	Args	: result handle, offset
 **	Returns	: Nothing.  Just sets the cursor
 **	Notes	: The data is a single linked list so we can go backwards
 */

void msqlDataSeek(handle, offset)
     m_result *handle;
     int	offset;
{
  ms_data	*tmp;


  msqlDebug(MOD_API,"msqlDataSeek() pos = \n",offset);

  tmp = handle->queryData;
  while(offset)
    {
      if (!tmp)
	break;
      tmp = tmp->next;
      offset--;
    }
  handle->cursor = tmp;
}



/**************************************************************************
 **	msqlQuery
 **
 */

int msqlQuery(index,q)
     int	index;
     char	*q;
{
  int	 i, len, res, size, sock;
  char	 query[6], cmd, err, *buf;
  size_t bufsize, qsize;
  

  if ((q == NULL) || (*q == 0)) {
      strcpy(msqlErrMsg,BAD_PARAM_ERROR);
      return (-1);
  }
  sock = socks[index];
  msqlDebug(MOD_QUERY,"Query = \"%s\"\n",q);
  resetError();
  setServerSock(sock);
  
  /* now see if the query is a SELECT or not */
  strncpy(query, (const char *) q, (size_t) 6);
  /* change to upper case */
  for (i=0; i < 6; i++) {
      query[i] = (char) toupper((int) query[i]);
  }
  /* if SELECT statement, use query, else use update */
  if (strncmp((const char *) query, "SELECT", (size_t) 6) == 0) {
      cmd = jdbcQuery;
  }
  else {
      cmd = jdbcUpdate;
  }

  /* Issue the query/update */
  qsize   = strlen(q);
  size    = htonl((int) qsize);
  bufsize = qsize + sizeof(int) + sizeof(char);
  buf     = (char *) malloc(bufsize);
  if (buf == NULL) {
      closeServer(sock);
      strcpy(msqlErrMsg, "Cannot allocate memory");
      return(-1);
  }

  /* command */
  buf[0] = cmd;
  /* length of query */
  memcpy((void *) &buf[1], (const void *) &size, sizeof(int));
  /* query string */
  memcpy((void *) &buf[5], (const void *) q, qsize);
  
  if (tcp_write(sock, (void *) buf, bufsize) != bufsize) {
      closeServer(sock);
      strcpy(msqlErrMsg, "Cannot communicate with JDBC Server");
      free((void *) buf);
      return(-1);
  }
  free((void *) buf);
  
  /* Error code returned */
  if (tcp_read(sock, &err, sizeof(err)) != sizeof(err)) {
      closeServer(sock);
      strcpy(msqlErrMsg,SERVER_GONE_ERROR);
      return(-1);
  }
  
  if (err != ok) {
      if (tcp_read(sock, &size, sizeof(size)) != sizeof(size)) {
	  closeServer(sock);
	  strcpy(msqlErrMsg,SERVER_GONE_ERROR);
	  return(-1);
      }
      size = ntohl(size);
      if (tcp_read(sock, msqlErrMsg, (size_t) size) != (size_t) size) {
	  closeServer(sock);
	  strcpy(msqlErrMsg,SERVER_GONE_ERROR);
	  return(-1);
      }
      
      return(-1);
  }
  
  if (cmd == jdbcUpdate) {
      return(0);
  }
  
  numFields = 0;
  if (tcp_read(sock, &numFields, sizeof(int)) != sizeof(int)) {
      closeServer(sock);
      strcpy(msqlErrMsg,SERVER_GONE_ERROR);
      return(-1);
  }
  numFields = ntohl(numFields);
  /* if no results, return */
  if (numFields == 0) {
      return(0);
  }
  else if (numFields < 0) {
      closeServer(sock);
      strcpy(msqlErrMsg,UNKNOWN_ERROR);
      return(-1);
  }

  /*
   ** numFields > 0 therefore we have data waiting on the socket.
   ** Grab it and dump it into a table structure.  If there's
   ** uncollected data free it - it's no longer available.
   */
  if (queryData) {
      freeQueryData(queryData);
      freeQueryData(fieldData);
      queryData = NULL;
      fieldData = NULL;
  }
  
  /* send command to get all query data */
  cmd = jdbcGetAll;  
  if (tcp_write(sock, (void *) &cmd, sizeof(char)) != sizeof(char)) {
      closeServer(sock);
      strcpy(msqlErrMsg, "Cannot communicate with JDBC Server");
      return(-1);
  }

  /* error code returned */
  if (tcp_read(sock, &err, sizeof(err)) != sizeof(err)) {
      closeServer(sock);
      strcpy(msqlErrMsg,SERVER_GONE_ERROR);
      return(-1);
  }

  if (err != ok) {
      if (tcp_read(sock, &size, sizeof(size)) != sizeof(size)) {
	  closeServer(sock);
	  strcpy(msqlErrMsg,SERVER_GONE_ERROR);
	  return(-1);
      }
      size = ntohl(size);
      if (tcp_read(sock, msqlErrMsg, (size_t) size) != (size_t) size) {
	  closeServer(sock);
	  strcpy(msqlErrMsg,SERVER_GONE_ERROR);
	  return(-1);
      }

      return(-1);
  }
  
  queryTableSize = readQueryData(sock);
  if (queryTableSize < 0) {
      closeServer(sock);
      return(-1);
  }
  queryData = tmpDataStore;
  tmpDataStore = NULL;
  
  
  /* get metadata - 6 fields in each row describing a data column */
  numFields = 6;
  fieldTableSize = readQueryData(sock);
  if (fieldTableSize < 0) {
      closeServer(sock);
      return(-1);
  }
  fieldData = tmpDataStore;
  tmpDataStore = NULL;

  return(0);
}




/**************************************************************************
 **	readQueryData
 **
 */

static int readQueryData(sock)
     int  sock;
{
  int	  off, len, numRows, size;
  char	  end;
  ms_data  *cur;
	
  /* each dowhile loop is one row */
  numRows = 0;
  do {
      numRows++;
      if(!tmpDataStore) {
	  tmpDataStore = cur = (ms_data *)malloc(sizeof(ms_data));
      }
      else {
	  cur->next = (ms_data *)malloc(sizeof(ms_data));
	  cur = cur->next;
      }
      msqlDebug(MOD_MALLOC,"Query data row - malloc @ %X of %d\n", cur, sizeof(ms_data));
      (void)bzero(cur,sizeof(ms_data));
      cur->data = (char **)malloc(numFields * sizeof(char *));
      (void)bzero(cur->data,numFields * sizeof(char *));
      cur->width = numFields;
      off = 0;
      
      /* read in all fields of one row */
      while(off < numFields) {
	  /* read length of following string */
	  if (tcp_read(sock, &len, sizeof(int)) != sizeof(int)) {
	      closeServer(sock);
	      strcpy(msqlErrMsg,SERVER_GONE_ERROR);
	      return(-1);
	  }
	  len = ntohl(len);
          /* printf("readQueryData: len = %d\n", len); */
	  if (len < 0) {
	      closeServer(sock);
	      strcpy(msqlErrMsg,SERVER_GONE_ERROR);
	      return(-1);
	  }
	  else if (len == 0) {
	      cur->data[off] = (char *)NULL;
	  }
	  else {
	      cur->data[off] = (char *)malloc(len+1);
	      (void)bzero(cur->data[off],len+1);
	      
	      if (tcp_read(sock, cur->data[off], (size_t) len) != (size_t) len) {
		  closeServer(sock);
		  strcpy(msqlErrMsg,SERVER_GONE_ERROR);
		  return(-1);
	      }
              /* printf("readQueryData: data = %s\n", cur->data[off]); */
	  }
	  off++;
      }
      
      /* read end code */
      if (tcp_read(sock, &end, sizeof(end)) != sizeof(end)) {
	  closeServer(sock);
	  strcpy(msqlErrMsg,SERVER_GONE_ERROR);
	  return(-1);
      }
            
  } while (end == endRow);
  
  /* The proxy server, unfortunately, does not know in advance how many rows there are in a
   * table. As a result of that, the server assumes at least one row of data. But in reality,
   * if there is only 1 row of all NULL items, there is no row at all and "numRows" should
   * be zero. So find and fix that problem here.
   */
  if (numRows == 1) {
      int allIsNull = 1;
      off = 0;
      while (off < numFields) {
	  if (cur->data[off++] != NULL) {
	      allIsNull = 0;
	      break;
	  }
      }
      if (allIsNull) {
	  free(tmpDataStore);
	  tmpDataStore = NULL;
	  numRows = 0;
      }
  }
  
  return(numRows);
}




/**************************************************************************
 **	msqlListDBs
 **
 */

m_result *msqlListDBs(index)
     int	index;
{
  int      size, sock;
  char     cmd, err;
  m_result *tmp;
  
  sock = socks[index];
  
  msqlDebug(MOD_API,"msqlListDBs: sock = %d\n",sock);
  tmp = (m_result *)malloc(sizeof(m_result));
  if (!tmp)
    {
      return(NULL);
    }
  (void)bzero(tmp, sizeof(m_result));
  msqlDebug(MOD_MALLOC,"Result Handle - malloc @ %X of %d\n",
	    tmp, sizeof(m_result));

  numFields = 1;
  
  /* send command to get all database names */
  cmd = jdbcListDBs;  
  if (tcp_write(sock, (void *) &cmd, sizeof(char)) != sizeof(char)) {
      closeServer(sock);
      strcpy(msqlErrMsg, "Cannot communicate with JDBC Server");
      free(tmp);
      return(NULL);
  }

  /* error code returned */
  if (tcp_read(sock, &err, sizeof(err)) != sizeof(err)) {
      closeServer(sock);
      strcpy(msqlErrMsg,SERVER_GONE_ERROR);
      free(tmp);
      return(NULL);
  }

  if (err != ok) {
      if (tcp_read(sock, &size, sizeof(size)) != sizeof(size)) {
	  closeServer(sock);
	  strcpy(msqlErrMsg,SERVER_GONE_ERROR);
          free(tmp);
	  return(NULL);
      }
      size = ntohl(size);
      if (tcp_read(sock, msqlErrMsg, (size_t) size) != (size_t) size) {
	  closeServer(sock);
	  strcpy(msqlErrMsg,SERVER_GONE_ERROR);
          free(tmp);
	  return(NULL);
      }

      free(tmp);
      return(NULL);
  }
  
  tmp->numRows = readQueryData(sock);
  if (tmp->numRows < 0)
    {
      (void)free(tmp);
      closeServer(sock);
      return(NULL);
    }
  tmp->queryData = tmpDataStore;
  tmp->cursor = tmp->queryData;
  tmp->numFields = 1;
  tmp->fieldData = (m_fdata *)malloc(sizeof(m_fdata));
  msqlDebug(MOD_MALLOC,"Field List Entry - malloc @ %X of %d\n",
	    tmp->fieldData, sizeof(m_fdata));
  (void)bzero(tmp->fieldData, sizeof(m_fdata));
  tmp->fieldData->field.table = (char *)strdup("mSQL Catalog");
  tmp->fieldData->field.name = (char *)strdup("Database");
  tmp->fieldData->field.type = CHAR_TYPE;
  tmp->fieldData->field.length = NAME_LEN;
  tmp->fieldData->field.flags = 0;
  tmp->fieldCursor = tmp->fieldData;
  tmpDataStore = NULL;
  return(tmp);
}





/**************************************************************************
 **	msqlListTables
 **
 */

m_result *msqlListTables(index)
     int	index;
{
  int      size, sock;
  char     cmd, err;
  m_result *tmp;
  
  sock = socks[index];
  
  msqlDebug(MOD_API,"msqlListTables: sock = %d\n", sock);
  tmp = (m_result *)malloc(sizeof(m_result));
  if (!tmp)
    {
      return(NULL);
    }
  msqlDebug(MOD_MALLOC,"Result Handle - malloc @ %X of %d\n",
	    tmp, sizeof(m_result));
  (void)bzero(tmp, sizeof(m_result));
  numFields = 1;
  
  /* send command to get all table names */
  cmd = jdbcListTables;  
  if (tcp_write(sock, (void *) &cmd, sizeof(char)) != sizeof(char)) {
      closeServer(sock);
      strcpy(msqlErrMsg, "Cannot communicate with JDBC Server");
      free(tmp);
      return(NULL);
  }

  /* error code returned */
  if (tcp_read(sock, &err, sizeof(err)) != sizeof(err)) {
      closeServer(sock);
      strcpy(msqlErrMsg,SERVER_GONE_ERROR);
      free(tmp);
      return(NULL);
  }

  if (err != ok) {
      if (tcp_read(sock, &size, sizeof(size)) != sizeof(size)) {
	  closeServer(sock);
	  strcpy(msqlErrMsg,SERVER_GONE_ERROR);
          free(tmp);
	  return(NULL);
      }
      size = ntohl(size);
      if (tcp_read(sock, msqlErrMsg, (size_t) size) != (size_t) size) {
	  closeServer(sock);
	  strcpy(msqlErrMsg,SERVER_GONE_ERROR);
          free(tmp);
	  return(NULL);
      }

      free(tmp);
      return(NULL);
  }
  
  tmp->numRows = readQueryData(sock);
  if (tmp->numRows < 0)
    {
      (void)free(tmp);
      closeServer(sock);
      return(NULL);
    }
  tmp->queryData = tmpDataStore;
  tmp->numFields = 0;
  tmp->cursor = tmp->queryData;
  tmp->fieldCursor = NULL;
  tmpDataStore = NULL;
  tmp->numFields = 1;
  tmp->fieldData = (m_fdata *)malloc(sizeof(m_fdata));
  msqlDebug(MOD_MALLOC,"Field List Entry - malloc @ %X of %d\n",
	    tmp->fieldData, sizeof(m_fdata));
  (void)bzero(tmp->fieldData, sizeof(m_fdata));
  tmp->fieldData->field.table = (char *)strdup("mSQL Catalog");
  tmp->fieldData->field.name = (char *)strdup("Table");
  tmp->fieldData->field.type = CHAR_TYPE;
  tmp->fieldData->field.length = NAME_LEN;
  tmp->fieldData->field.flags = 0;
  tmp->fieldCursor = tmp->fieldData;
  return(tmp);
}


/**************************************************************************
 **	msqlListFields
 **
 */

m_result *msqlListFields(index,table)
     int	index;
     char	*table;
{
  int      len, size, sock;
  char     cmd, err;
  m_result *tmp;

  if ((table == NULL) || (*table == 0)) {
    strcpy(msqlErrMsg,BAD_PARAM_ERROR);
    return (NULL);
  }
  
  sock = socks[index];
  
  msqlDebug(MOD_API,"msqlListFields: sock = %d, table = %s\n", sock, table);
  tmp = (m_result *)malloc(sizeof(m_result));
  if (!tmp)
    {
      return(NULL);
    }
  msqlDebug(MOD_MALLOC,"Result Handle - malloc @ %X of %d\n",
	    tmp, sizeof(m_result));
  (void)bzero(tmp, sizeof(m_result));
  numFields = 6;
  
  /* send command to get all table's field names */
  cmd = jdbcListFields;  
  if (tcp_write(sock, (void *) &cmd, sizeof(char)) != sizeof(char)) {
      closeServer(sock);
      strcpy(msqlErrMsg, "Cannot communicate with JDBC Server");
      free(tmp);
      return(NULL);
  }
  /* send table name */
  size = len = strlen(table);
  size = htonl(size);
  if (tcp_write(sock, (void *) &size, sizeof(int)) != sizeof(int)) {
      closeServer(sock);
      strcpy(msqlErrMsg,SERVER_GONE_ERROR);
      free(tmp);
      return(NULL);
  }
  if (tcp_write(sock, (void *) table, (size_t) len) != (size_t) len) {
      closeServer(sock);
      strcpy(msqlErrMsg,SERVER_GONE_ERROR);
      free(tmp);
      return(NULL);
  }

  /* error code returned */
  if (tcp_read(sock, &err, sizeof(err)) != sizeof(err)) {
      closeServer(sock);
      strcpy(msqlErrMsg,SERVER_GONE_ERROR);
      free(tmp);
      return(NULL);
  }

  if (err != ok) {
      if (tcp_read(sock, &size, sizeof(size)) != sizeof(size)) {
	  closeServer(sock);
	  strcpy(msqlErrMsg,SERVER_GONE_ERROR);
          free(tmp);
	  return(NULL);
      }
      size = ntohl(size);
      if (tcp_read(sock, msqlErrMsg, (size_t) size) != (size_t) size) {
	  closeServer(sock);
	  strcpy(msqlErrMsg,SERVER_GONE_ERROR);
          free(tmp);
	  return(NULL);
      }

      free(tmp);
      return(NULL);
  }
  
  tmp->numFields = readQueryData(sock);
  if(tmp->numFields < 0)
    {
      closeServer(sock);
      (void)free(tmp);
      return(NULL);
    }
  tmp->fieldData = tableToFieldList(tmpDataStore);
  tmp->fieldCursor = tmp->fieldData;
  tmp->queryData = NULL;
  tmp->cursor = NULL;
  tmp->numRows = 0;
  freeQueryData(tmpDataStore);
  tmpDataStore = NULL;
  return(tmp);
}




#ifndef VXWORKS
/**************************************************************************
 **	msqlDebug
 **
 */

static void msqlDebug(va_alist)
     va_dcl
{
  va_list args;
  char	msg[1024],
  *fmt;
  int	module,
  out = 0;

  va_start(args);
  module = (int) va_arg(args, int *);
  if (! (module & debugLevel))
    {
      va_end(args);
      return;
    }

  fmt = (char *)va_arg(args, char *);
  if (!fmt)
    return;
  (void)vsprintf(msg,fmt,args);
  va_end(args);
  printf("[libmsql] %s",msg);
  fflush(stdout);
}

#else
static void msqlDebug(module,fmt,a,b,c,d,e,f,g,h)
     int   module;
     char *fmt;
     char *a;
     char *b;
     char *c;
     char *d;
     char *e;
     char *f;
     char *g;
     char *h;
{
  char	msg[1024],
  out = 0;

  if (! (module & debugLevel))
    {
      return;
    }

  if (!fmt)
    return;
  (void)sprintf(msg,fmt,a,b,c,d,e,f,g,h);

  printf("[libmsql] %s",msg);
  fflush(stdout);
}
#endif


/*****************************************************/
static int tcp_read(int fd, void *vptr, int n)
{
  int	nleft;
  int	nread;
  char	*ptr;

  ptr = (char *) vptr;
  nleft = n;
  
  while (nleft > 0) {
    if ( (nread = read(fd, ptr, nleft)) < 0) {
      if (errno == EINTR) {
        nread = 0;		/* and call read() again */
      }
      else {
        return(-1);
      }
    } else if (nread == 0) {
      break;			/* EOF */
    }
    
    nleft -= nread;
    ptr   += nread;
  }
  return(n - nleft);		/* return >= 0 */
}


/*****************************************************/
static int tcp_write(int fd, const void *vptr, int n)
{
  int		nleft;
  int		nwritten;
  const char	*ptr;

  ptr = (char *) vptr;
  nleft = n;
  
  while (nleft > 0) {
    if ( (nwritten = write(fd, ptr, nleft)) <= 0) {
      if (errno == EINTR) {
        nwritten = 0;		/* and call write() again */
      }
      else {
        return(-1);		/* error */
      }
    }

    nleft -= nwritten;
    ptr   += nwritten;
  }
  return(n);
}




/*  Dummy functions for cedit for the moment...ejw,ct */


int msqlCreateDB(sock,DB)
     int	sock;
     char	*DB;
{
  return(0);
}

int msqlDropDB(sock,DB)
     int	sock;
     char	*DB;
{
  return(0);
}


