/*
 * ------------------------------------------------------------------------
 *      PACKAGE:  [incr Tk]
 *  DESCRIPTION:  Building mega-widgets with [incr Tcl]
 *
 *  [incr Tk] provides a framework for building composite "mega-widgets"
 *  using [incr Tcl] classes.  It defines a set of base classes that are
 *  specialized to create all other widgets.
 *
 *  This file defines procedures used to manage mega-widget options
 *  specified within class definitions.
 *
 * ========================================================================
 *  AUTHOR:  Michael J. McLennan       Phone: (610)712-2842
 *           AT&T Bell Laboratories   E-mail: michael.mclennan@att.com
 *     RCS:  $Id: itk_option.c,v 1.1.1.1 1996/08/21 19:25:44 heyes Exp $
 * ========================================================================
 *             Copyright (c) 1993-1995  AT&T Bell Laboratories
 * ------------------------------------------------------------------------
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */
#include <assert.h>
#include "itkInt.h"

#ifndef lint
static char rcsid[] = "$Id: itk_option.c,v 1.1.1.1 1996/08/21 19:25:44 heyes Exp $";
#endif

/*
 *  List of all classes containing itk options:
 */
static Tcl_HashTable *ItkClasses = NULL;


/*
 * ------------------------------------------------------------------------
 *  Itk_ClassOptionDefineCmd()
 *
 *  Invoked when a class definition is being parse to handle an
 *  itk_option declaration.  Adds a new option to a mega-widget
 *  declaration, with some code that will be executed whenever the
 *  option is changed via "configure".  If there is already an existing
 *  option by that name, then this new option is folded into the
 *  existing option, but the <init> value is ignored.  The X11 resource
 *  database names must be consistent with the existing option.
 *
 *  Handles the following syntax:
 *
 *      itk_option define <switch> <resName> <resClass> <init> ?<config>?
 *
 *  Returns TCL_OK/TCL_ERROR to indicate success/failure.
 * ------------------------------------------------------------------------
 */
/* ARGSUSED */
int
Itk_ClassOptionDefineCmd(clientData, interp, argc, argv)
    ClientData clientData;   /* class parser info */
    Tcl_Interp *interp;      /* current interpreter */
    int argc;                /* number of arguments */
    char **argv;             /* argument strings */
{
    ItclObjectInfo *info = (ItclObjectInfo*)clientData;
    Itcl_Class *cdefn = (Itcl_Class*)Itcl_PeekStack(&info->cdefnStack);

    int newEntry;
    ItkClassOptTable *optTable;
    Tcl_HashEntry *entry;
    ItkClassOption *opt;
    char *config;

    /*
     *  Make sure that the arguments look right.  The option switch
     *  name must start with a '-'.
     */
    if (*argv[1] != '-') {
        Tcl_AppendResult(interp, "bad option name \"", argv[1],
            "\": should be -", argv[1],
            (char*)NULL);
        return TCL_ERROR;
    }
    if (strstr(argv[1],".")) {
        Tcl_AppendResult(interp, "bad option name \"", argv[1],
            "\": illegal character \".\"",
            (char*)NULL);
        return TCL_ERROR;
    }
    if (!islower(*argv[2])) {
        Tcl_AppendResult(interp, "bad resource name \"", argv[2],
            "\": should start with a lower case letter",
            (char*)NULL);
        return TCL_ERROR;
    }
    if (!isupper(*argv[3])) {
        Tcl_AppendResult(interp, "bad resource class \"", argv[3],
            "\": should start with an upper case letter",
            (char*)NULL);
        return TCL_ERROR;
    }

    /*
     *  Make sure that this option has not already been defined in
     *  the context of this class.  Options can be redefined in
     *  other classes, but can only be defined once in a given
     *  class.  This ensures that there will be no confusion about
     *  which option is being referenced if the configuration code
     *  is redefined by a subsequent "body" command.
     */
    optTable = Itk_CreateClassOptTable(interp, cdefn);
    entry = Tcl_CreateHashEntry(&optTable->options, argv[1], &newEntry);

    if (!newEntry) {
        Tcl_AppendResult(interp, "option \"", argv[1],
            "\" already defined in class \"",
            Itcl_GetNamespPath(cdefn->namesp), "\"",
            (char*)NULL);
        return TCL_ERROR;
    }

    /*
     *  Create a new option record and add it to the table for this
     *  class.
     */
    config = (argc >= 6) ? argv[5] : NULL;
    if (Itk_CreateClassOption(interp, cdefn, argv[1], argv[2], argv[3],
        argv[4], config, &opt) != TCL_OK) {
        return TCL_ERROR;
    }

    Tcl_SetHashValue(entry, (ClientData)opt);
    Itk_OptListAdd(&optTable->order, entry);
    return TCL_OK;
}


/*
 * ------------------------------------------------------------------------
 *  Itk_ClassOptionIllegalCmd()
 *
 *  Invoked when a class definition is being parse to handle an
 *  itk_option declaration.  Handles an "illegal" declaration like
 *  "add" or "remove", which can only be used after a widget has
 *  been created.  Returns TCL_ERROR along with an error message.
 * ------------------------------------------------------------------------
 */
/* ARGSUSED */
int
Itk_ClassOptionIllegalCmd(clientData, interp, argc, argv)
    ClientData clientData;   /* class parser info */
    Tcl_Interp *interp;      /* current interpreter */
    int argc;                /* number of arguments */
    char **argv;             /* argument strings */
{
    Tcl_AppendResult(interp, "can only ", argv[0],
        " options for a specific widget\n",
        "(move this command into the constructor)",
        (char*)NULL);

    return TCL_ERROR;
}


/*
 * ------------------------------------------------------------------------
 *  Itk_ConfigClassOption()
 *
 *  Invoked whenever a class-based configuration option has been
 *  configured with a new value.  If the option has any extra code
 *  associated with it, the code is invoked at this point to bring
 *  the widget up-to-date.
 *
 *  Returns TCL_OK on success, or TCL_ERROR (along with an error
 *  message in the interpreter) if anything goes wrong.
 * ------------------------------------------------------------------------
 */
/* ARGSUSED */
int
Itk_ConfigClassOption(interp,obj,cdata,newval)
    Tcl_Interp *interp;        /* interpreter managing the class */
    Itcl_Object *obj;          /* object being configured */
    ClientData cdata;          /* class option */
    char *newval;              /* new value for this option */
{
    ItkClassOption *opt = (ItkClassOption*)cdata;

    int status;
    Itcl_ActiveNamespace nsToken;

    /*
     *  If the option has any config code, execute it now.
     *  Make sure that the namespace context is set up correctly.
     */
    if (opt->config) {
        nsToken = Itcl_ActivateNamesp2(interp, opt->cdefn->namesp,
            (ClientData)obj);

        if (nsToken) {
            status = Itcl_EvalCmdImplement(interp, (ItclCmdMember*)NULL,
                opt->config, 0, (char**)NULL);
            Itcl_DeactivateNamesp(interp, nsToken);
        }
        else {
            status = TCL_ERROR;
        }
        return status;
    }
    return TCL_OK;
}


/*
 * ------------------------------------------------------------------------
 *  Itk_CreateClassOptTable()
 *
 *  Finds or creates an option table which will contain all of the
 *  class-based configuration options for a mega-widget.  These are
 *  the options included in the class definition which add new behavior
 *  to the mega-widget.
 *
 *  This table is automatically deleted by Itk_TraceClassDestroy
 *  whenever the class namespace is destroyed.  The "unset" operation
 *  of a private class variable is used to detect the destruction of
 *  the namespace.
 *
 *  Returns a pointer to an option table which will contain pointers to
 *  ItkClassOption records.
 * ------------------------------------------------------------------------
 */
ItkClassOptTable*
Itk_CreateClassOptTable(interp,cdefn)
    Tcl_Interp *interp;        /* interpreter managing the class */
    Itcl_Class *cdefn;         /* class definition */
{
    int newEntry;
    Tcl_HashEntry *entry;
    ItkClassOptTable *optTable;
    Itcl_ActiveNamespace nsToken;
    int pLevel;

    /*
     *  If the lookup table for class definition has not yet been
     *  created, do it now.
     */
    if (ItkClasses == NULL) {
        ItkClasses = (Tcl_HashTable*)ckalloc(sizeof(Tcl_HashTable));
        Tcl_InitHashTable(ItkClasses, TCL_ONE_WORD_KEYS);
    }

    /*
     *  Look for the specified class definition in the table.
     *  If it does not yet exist, then create a new slot for it.
     *  When a table is created for the first time, add a
     *  special sentinel variable "_itk_option_data" to the
     *  class namespace, and put a trace on this variable.
     *  Whenever it is destroyed, have it delete the option table
     *  for this class.
     */
    entry = Tcl_CreateHashEntry(ItkClasses, (char*)cdefn, &newEntry);
    if (newEntry) {
        optTable = (ItkClassOptTable*)ckalloc(sizeof(ItkClassOptTable));
        Tcl_InitHashTable(&optTable->options, TCL_STRING_KEYS);
        Itk_OptListInit(&optTable->order, &optTable->options);

        Tcl_SetHashValue(entry, (ClientData)optTable);

        nsToken = Itcl_ActivateNamesp(interp, cdefn->namesp);
        if (nsToken) {
            pLevel = Itcl_VarProtection(interp, ITCL_PRIVATE);

            Tcl_TraceVar(interp, "_itk_option_data",
                (TCL_TRACE_UNSETS | ITCL_GLOBAL_VAR),
                Itk_TraceClassDestroy, (ClientData)cdefn);

            Itcl_VarProtection(interp, pLevel);
            Itcl_DeactivateNamesp(interp, nsToken);
        }
    }
    else {
        optTable = (ItkClassOptTable*)Tcl_GetHashValue(entry);
    }
    return optTable;
}


/*
 * ------------------------------------------------------------------------
 *  Itk_FindClassOptTable()
 *
 *  Looks for an option table containing all of the class-based
 *  configuration options for a mega-widget.  These are the options
 *  included in a class definition which add new behavior to the
 *  mega-widget.
 *
 *  Returns a pointer to an option table which will contain pointers to
 *  Itk_ClassOption records.  If a table does not exist for this class,
 *  this returns NULL.
 * ------------------------------------------------------------------------
 */
ItkClassOptTable*
Itk_FindClassOptTable(cdefn)
    Itcl_Class *cdefn;         /* class definition */
{
    Tcl_HashEntry *entry;

    /*
     *  If the lookup table for class definition has not yet been
     *  created, then there is no record for this class.
     */
    if (ItkClasses == NULL) {
        return NULL;
    }

    /*
     *  Look for the specified class definition in the table.
     */
    entry = Tcl_FindHashEntry(ItkClasses, (char*)cdefn);
    if (entry) {
        return (ItkClassOptTable*)Tcl_GetHashValue(entry);
    }
    return NULL;
}


/*
 * ------------------------------------------------------------------------
 *  Itk_DeleteClassOptTable()
 *
 *  Deletes a hash table which contains all of the class-based
 *  configuration options for a mega-widget.  These are the options
 *  included in the class definition which add new behavior to the
 *  mega-widget.  If no data exists for the class, this procedure
 *  does nothing.
 *
 *  This procedure is usually invoked by Itk_TraceClassDestroy
 *  whenever a class namespace is destroyed, to automatically
 *  discard option data for that class.
 * ------------------------------------------------------------------------
 */
void
Itk_DeleteClassOptTable(interp,cdefn)
    Tcl_Interp *interp;        /* interpreter managing the class */
    Itcl_Class *cdefn;         /* class definition */
{
    Tcl_HashEntry *entry;
    ItkClassOptTable *optTable;
    Tcl_HashSearch place;
    ItkClassOption *opt;

    /*
     *  Look for the specified class definition in the table.
     *  If it is found, delete all the option records and tear
     *  down the table.
     */
    if (ItkClasses) {
        entry = Tcl_FindHashEntry(ItkClasses, (char*)cdefn);
        if (entry) {
            optTable = (ItkClassOptTable*)Tcl_GetHashValue(entry);
            Tcl_DeleteHashEntry(entry);

            entry = Tcl_FirstHashEntry(&optTable->options, &place);
            while (entry) {
                opt = (ItkClassOption*)Tcl_GetHashValue(entry);
                Itk_DelClassOption(opt);
                entry = Tcl_NextHashEntry(&place);
            }
            Tcl_DeleteHashTable(&optTable->options);
            Itk_OptListFree(&optTable->order);
            ckfree((char*)optTable);
        }
    }
}


/*
 * ------------------------------------------------------------------------
 *  Itk_TraceClassDestroy()
 *
 *  Invoked automatically whenever the "_itk_option_data" variable
 *  is destroyed within a class namespace.  This should be a signal
 *  that the namespace is being destroyed.
 *
 *  Calls Itk_DeleteClassOptTable() to release any option data that
 *  exists for the class.
 *
 *  Returns NULL on success, or a pointer to a string describing any
 *  error that is encountered.
 * ------------------------------------------------------------------------
 */
/* ARGSUSED */
char*
Itk_TraceClassDestroy(cdata,interp,name1,name2,flags)
    ClientData cdata;          /* class definition data */
    Tcl_Interp *interp;        /* interpreter managing the class */
    char *name1;               /* name of variable involved in trace */
    char *name2;               /* name of array element within variable */
    int flags;                 /* flags describing trace */
{
    Itcl_Class *cdefn = (Itcl_Class*)cdata;

    Itk_DeleteClassOptTable(interp, cdefn);
    return NULL;
}


/*
 * ------------------------------------------------------------------------
 *  Itk_CreateClassOption()
 *
 *  Creates the data representing a configuration option for an
 *  Archetype mega-widget.  This record represents an option included
 *  in the class definition.  It adds new behavior to the mega-widget
 *  class.
 *
 *  If successful, returns TCL_OK along with a pointer to the option
 *  record.  Returns TCL_ERROR (along with an error message in the
 *  interpreter) if anything goes wrong.
 * ------------------------------------------------------------------------
 */
int
Itk_CreateClassOption(interp,cdefn,switchName,resName,resClass,defVal,
    config,optPtr)

    Tcl_Interp *interp;            /* interpreter managing the class */
    Itcl_Class *cdefn;             /* class containing this option */
    char *switchName;              /* name of command-line switch */
    char *resName;                 /* resource name in X11 database */
    char *resClass;                /* resource class name in X11 database */
    char *defVal;                  /* last-resort default value */
    char *config;                  /* configuration code */
    ItkClassOption **optPtr;       /* returns: option record */
{
    ItkClassOption *opt;
    ItclCmdImplement *mimpl;

    /*
     *  If this option has any "config" code, then try to create
     *  an implementation for it.
     */
    if (config) {
        if (Itcl_CreateCmdImplement(interp, (char*)NULL, config, &mimpl)
            != TCL_OK) {
            return TCL_ERROR;
        }
    }
    else {
        mimpl = NULL;
    }

    /*
     *  Create the record to represent this option.
     */
    opt = (ItkClassOption*)ckalloc(sizeof(ItkClassOption));
    opt->cdefn = cdefn;

    opt->switchName = (char*)ckalloc((unsigned)(strlen(switchName)+1));
    strcpy(opt->switchName, switchName);

    opt->resName = (char*)ckalloc((unsigned)(strlen(resName)+1));
    strcpy(opt->resName, resName);

    opt->resClass = (char*)ckalloc((unsigned)(strlen(resClass)+1));
    strcpy(opt->resClass, resClass);

    opt->init = (char*)ckalloc((unsigned)(strlen(defVal)+1));
    strcpy(opt->init, defVal);

    opt->config = mimpl;

    *optPtr = opt;
    return TCL_OK;
}

/*
 * ------------------------------------------------------------------------
 *  Itk_FindClassOption()
 *
 *  Searches for a class-based configuration option for an Archetype
 *  mega-widget.   The specified name is treated as the "switch" name
 *  (e.g., "-option"), but this procedure will recognize it even without
 *  the leading "-".
 *
 *  If an option is found that was defined in the specified class,
 *  then this procedure returns a pointer to the option definition.
 *  Otherwise, it returns NULL.
 * ------------------------------------------------------------------------
 */
ItkClassOption*
Itk_FindClassOption(cdefn,switchName)
    Itcl_Class *cdefn;             /* class containing this option */
    char *switchName;              /* name of command-line switch */
{
    ItkClassOption *opt = NULL;

    Tcl_DString buffer;
    ItkClassOptTable *optTable;
    Tcl_HashEntry *entry;

    /*
     *  If the switch does not have a leading "-", add it on.
     */
    Tcl_DStringInit(&buffer);
    if (*switchName != '-') {
        Tcl_DStringAppend(&buffer, "-", -1);
        Tcl_DStringAppend(&buffer, switchName, -1);
        switchName = Tcl_DStringValue(&buffer);
    }

    /*
     *  Look for the option table for the specified class, and check
     *  for the requested switch.
     */
    optTable = Itk_FindClassOptTable(cdefn);
    if (optTable) {
        entry = Tcl_FindHashEntry(&optTable->options, switchName);
        if (entry) {
            opt = (ItkClassOption*)Tcl_GetHashValue(entry);
        }
    }
    Tcl_DStringFree(&buffer);
    return opt;
}

/*
 * ------------------------------------------------------------------------
 *  Itk_DelClassOption()
 *
 *  Destroys a configuration option previously created by
 *  Itk_CreateClassOption().
 * ------------------------------------------------------------------------
 */
void
Itk_DelClassOption(opt)
    ItkClassOption *opt;  /* pointer to option data */
{
    ckfree(opt->switchName);
    ckfree(opt->resName);
    ckfree(opt->resClass);
    ckfree(opt->init);

    if (opt->config) {
        Itcl_DeleteCmdImplement((ClientData)opt->config);
    }

    ckfree((char*)opt);
}
