/*
 * dd_dump_callbacks.c 
 *         :  Callbacks for the dd_dump facility.   
 * Author  :  C.Witzig
 * Date    :  Mai 31, 1992 
 * Mods    :
 *  Dec 5, 1993: environment variable DD_DUMP_DIR
 *
 */

/*
 * dd_dump allows to get events from the DD system and dump
 * its contents.
 * It creates its own fifo DUMP.
 * The following options are available:
 * Event_selection = ALL: Every event ends up in the DUMP fifo.
 * Event_selection = ONREQ: Only events are put into the DUMP
 *                   fifo, when one asks specifically for an event.
 *                   (This mode should be used during data taking).
 * Auto_dump ON:  If an event is found, it is automatically dumped 
 *                on the screen.
 * Auto_dump OFF: One has to press the dump button.
 *
 * 
 * This file has four parts:
 * 1. Code for screen handling.
 * 2. Code for setting the options.
 * 3. Code for start/stop/hook up to dd
 * 4. Event retrieval and dump.
 *
 */



#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <sys/signal.h>
#include <stdio.h>
#include <X11/Intrinsic.h>
#include <Xm/Xm.h>
#include <Xm/Text.h>
#include <Xm/TextF.h>
#include <e787util.h>
#include <Xm/Scale.h>

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <errno.h>

#include <dd_user.h>
#include <ybos/bank_def.hinc>
#include <ybos/lrid_bank.hinc>

#include <lrid_bank.hinc>
#include <scal_mon.h>


#define TRUE 1
#define FALSE 0

/*
 * MAX_TF_LEN: Max length to read from TextField input
 * YBOS_BANK_NAME_LEN: is always four
 * MAX_NBANKS : Max number of banks to dump.
 */
#define MAX_TF_LEN 1000
#define YBOS_BANK_NAME_LEN 4
#define MAX_NBANKS 100


extern Display *display;

extern Widget DDD_TEXT;
extern Widget DDD_TF;
extern Widget DDD_SAVE_TEXT;
extern Widget EVS_ALL_B;
extern Widget EVSEL_SUB_B;
extern Widget AUTO_ON_B;
extern Widget AUTO_OFF_B;
extern Widget NEXT_EV_B;
extern Widget HELP_B;
extern Widget STOP_B;
extern Widget EXIT_B;
extern Widget DUMP_B;
extern Widget START_B;

Arg al[64];
static int DDD_TextLen;

int got_event = FALSE;
int auto_dump = TRUE;

int attached = FALSE;
int started  = FALSE;
int all_events = TRUE;

int status;

/*
 * List of banks which are to be dumped.
 * nbanks2dump = -1 --> dump all banks.
 */
int  nbanks2dump = -1;
char banks2dump[MAX_NBANKS][YBOS_BANK_NAME_LEN+1];


/*
 * Print format is set by default to hex, but can be 
 * changed interactively.
 */
#define IFORMAT_HEX 0
#define IFORMAT_HEXASCII 1
#define IFORMAT_OCT 2
#define IFORMAT_DEC 3
char *formats[4] = {
  "%9x",
  "%9x",
  "%18o",
  "%12d"};
int iformat = IFORMAT_HEX;
char *format = "%9x";


/*
 * If the banks are dumped to disk or printed they need 
 * a file pointer ...
 */
FILE *fp;


/*
 * The event will go into fev, set default values for fifo.
 */
fifo_entry fev;
char fifo_name[FIFO_NAME_LEN] = "DUMP";
int fifo_wait = FWAIT_ASYNC;
int fifo_mode = FMODE_ONREQ;
int fifo_prescale = FMODE_NOPRESCALE;
int fifo_ctl[FIFO_HDR_CTL_LEN] = { 0, 0, 0, 0};

#define DDD_HELP_LINES 10
static char *help_text[10] = {
"\n\n \
============================= DD DUMP HELP ============================= \n\
1. In order to dump events, you have to start up (press start).\n\
   By default, the process will create the DUMP fifo and receive all the\n\
   events.\n",
"2. DD Mode: ALL means that you get every event. This might block\n\
   the trigger during data taking. SUB will get you a subsample of the \n\
   events and the data taking should not be blocked.\n",
"3. Auto dump ON dumps automatically every acuired event to screen.  \n\
   If auto dump is OFF, then only the bank headers will be printed. \n",
"4. Dump Format: The following options are available: Hex, Hex and ASCII,\n\
   OCTal and DECimal.\n",
"5. Start/Stop: To start and stop (enable/disable) event dumping.\n",
"6. Exit: Guess what!!! \n",
"7. Dump the current aquired event. \n",
"8. Next Event: Will read the next event and dump the YBOS bank headers \n\
   and print the banks if auto dump is on. \n",
"\nBank List:\n----------\n\
By default every bank is dumped. You can enter a list of YBOS banks into \n\
the text field to the right of the label 'bank list' which selects the\n\
desired banks. Hit RETURN to validate your choice. (you can cut and \n\
paste into the text field). \n",
"Note:\n\
The default fifo for the dump fifo can be changed by entering the \n\
following command into the textfield: \n\
init FIFONAME 1 4000 w1 b1 w2 b2 \n\
where FIFONAME is the fifoname (max 6 characters), w1,b1,w2,b2 the DD fifo \n\
control words. (If you are not familiar with the dd system you\n\
better do not try this command out during data taking!) \n\
\n\n\n"};


char text[10000], text1[1000];

void do_init();
void DDD_TF_prbanks();
void release_event();
void dump_bank_list();
void dump_event();

/*
 * Callbacks for Start/Stop
 *-------------------------------------------------------------------
 */

void DDD_Start()
{
  if ( !attached )
	 do_init();

  ddu_start();
  started = TRUE;

  XtSetArg(al[0], XmNsensitive, FALSE);
  XtSetValues( START_B, al, 1);
  XtSetArg(al[0], XmNsensitive, TRUE);
  XtSetValues( STOP_B, al, 1);

  XFlush(display);
}

void DDD_Stop()
{
  release_event();
  ddu_stop();
  started = FALSE;

  XtSetArg(al[0], XmNsensitive, TRUE);
  XtSetValues( START_B, al, 1);
  XtSetArg(al[0], XmNsensitive, FALSE);
  XtSetValues( STOP_B, al, 1);

  XFlush(display);
}

void DDD_Exit()
{

  if ( attached ){
	 release_event();
	 if ( (status = ddu_close() ) < 0){
		fprintf(stderr,"ERROR: ddu_close failed - status %i \n",status);
		exit(1);
	 }
  }
  exit(0);
}

void do_init()
{
  struct fifo_mode fmode;

  if ( attached ){
	 DDD_Stop();
	 ddu_close();
  }
  
  fmode.mode = fifo_mode;
  fmode.prescale = fifo_prescale;
  fmode.wait = fifo_wait;
  fmode.user = FMODE_SINGLE_USER;
  fmode.p2ct = fifo_ctl;
  if ( (status = ddu_init(fifo_name, fmode) ) < 0){
    fprintf(stderr,"ERROR: ddu_init failed - status %i \n",status);
    exit(1);
  }
  attached = TRUE;

  DDD_TF_prbanks();
}



/*
 * Screen handling
 *--------------------------------------------------------------------
 */

void update_screen()
{
  XmTextPosition lpos;

  lpos = XmTextGetLastPosition(DDD_TEXT);
  XmTextSetInsertionPosition(DDD_TEXT,lpos);
  XFlush(display);

  if (DDD_TextLen == 0)
	 DDD_TextLen = XmTextGetMaxLength(DDD_TEXT);

  if (lpos > ( (int) (0.8*(float)DDD_TextLen) ) ){
	 printf("bigger now - update !!\n");
	 printf("dd text: %i %i %i \n",lpos,DDD_TextLen, (int)(0.7*DDD_TextLen) );
	 XmTextSetString(DDD_TEXT,"");
  }
}


void pr2screen(char *text)
{
  XmTextPosition lpos;

  if ( strlen(text) > 0){
	 lpos = XmTextGetLastPosition(DDD_TEXT);
	 XmTextInsert(DDD_TEXT, lpos, text);
  }
}



/*
 * Callbacks for Options
 *---------------------------------------------------------------------
 */

void DDD_Auto_off()
{
  auto_dump = FALSE;
}

void DDD_Auto_on()
{
  auto_dump = TRUE;
}

void DDD_Ev_sub()
{
  all_events = FALSE;

  if ( ( attached ) && (fifo_mode != FMODE_ONREQ) ){
	 fifo_mode = FMODE_ONREQ;
	 do_init();
  }
}

void DDD_Ev_all()
{
  all_events = TRUE;

  if ( ( attached ) && (fifo_mode != FMODE_ALL) ){
	 fifo_mode = FMODE_ALL;
	 do_init();
  }
}

void DDD_FHEX()
{
  iformat = IFORMAT_HEX;
  format = formats[IFORMAT_HEX];
}

void DDD_FHEXASCII()
{
  iformat = IFORMAT_HEXASCII;
  format = formats[IFORMAT_HEXASCII];
}

void DDD_FOCT()
{
  iformat = IFORMAT_OCT;
  format = formats[IFORMAT_OCT];
}

void DDD_FDEC()
{
  iformat = IFORMAT_DEC;
  format = formats[IFORMAT_DEC];
}


void DDD_Help()
{
  int i;
  for (i=0;i<DDD_HELP_LINES;i++)
	 pr2screen(help_text[i]);

  update_screen();
}



/*
 * Callbacks for TextField input/output
 *-------------------------------------------------------------------
 * The following input commands are expected:
 * 1. Init for specifying which fifo to attach to,
 * 2. all for dumping all the banks,
 * 3. a list of banks to dump.
 */

void dump_file_header(FILE *fp)
{
  time_t t;
  char *p2time;
  struct lrid_bank_str *lrid_hdr;

  fprintf(fp,"\n\n******************** E787 online dump facility ********************\n");
  t = time(NULL);
  p2time = ctime(&t);
  fprintf(fp," ---> Current time is %s ",p2time);
  lrid_hdr = (struct lrid_bank_str*) (fev.p2da+1+sizeof(struct bank_header)/4);
  fprintf(fp,"---> Run number %d run type %d record number %d \n",
			 lrid_hdr->run_number, lrid_hdr->run_type, lrid_hdr->record_number);
}




void DDD_Print()
{
  char *p2file;

  p2file = getenv("DD_DUMP_FILE");
  
  fp = fopen(p2file,"w");
  if (fp == NULL)
	 sprintf(text,"\n\ncannot open file %s !!! - contents NOT saved \n \
 ===> check environment variable $DD_DUMP_FILE\n\n ",p2file);
  else{
	 dump_file_header(fp);
	 dump_bank_list(fp);
	 dump_event(fp);
	 if ( fclose(fp) )
		sprintf(text,"\n\nerror while closing file\n");
	 else
		sprintf(text,"\n\nfile %s sent to line printer \n",p2file);
	 system("lpr -Pe787lp0 $DD_DUMP_FILE");
  }
  pr2screen(text);
  update_screen();
}


void DDD_Save()
{
  char *p2file;

  if ( !got_event ){
	 pr2screen("You must first get an event!\n");
	 update_screen();
  }

  p2file = XmTextGetString(DDD_SAVE_TEXT);
  
  fp = fopen(p2file,"w");
  if (fp == NULL)
	 sprintf(text,"\ncannot open file %s !!! - contents NOT saved \n",p2file);
  else{
	 dump_file_header(fp);
	 dump_bank_list(fp);
	 dump_event(fp);
	 if ( fclose(fp) )
		sprintf(text,"\nerror while closing file\n");
	 else
		sprintf(text,"\n\nevents saved in %s\n",p2file);
}
  pr2screen(text);
  update_screen();
}

void DDD_TF_prbanks()
{
  int i;
  char text[MAX_TF_LEN] = "";

  if (nbanks2dump == -1)
	 strcpy(text,"ALL");
  else{
	 for (i=0;i<nbanks2dump;i++){
		strcat(text,&banks2dump[i][0]);
		strcat(text," ");
	 }
  }
  XmTextFieldSetString(DDD_TF,text);
  XFlush(display);
}

void DDD_TF_inp()
{
  int i;
  char *p2text, *p2c, text1[100],text2[100];

  p2text = XmTextFieldGetString(DDD_TF);

  if ( ( strstr(p2text,"init") != NULL ) || ( strstr(p2text,"INIT") ) ){
	 sscanf(p2text,"%s %s %i %i %i %i %i %i",text1,text2,&fifo_mode,&fifo_wait,
			  &fifo_ctl[0],&fifo_ctl[1],&fifo_ctl[2],&fifo_ctl[3]);
	 strncpy(fifo_name,text2,FIFO_NAME_LEN-1);
	 do_init();
	 pr2screen("\nNEW FIFO INITIALISED .....\nPress START to begin\n\n");
	 update_screen();
	 return;
  }
		
  if ( ( strstr(p2text,"all") != NULL ) || ( strstr(p2text,"ALL") ) ){
	 nbanks2dump = -1;
	 DDD_TF_prbanks();
	 return;
  }

  nbanks2dump = 0;
  p2c = strtok(p2text," ");

  while ( (p2c != NULL) && (nbanks2dump < MAX_NBANKS) ){
	 strncpy(&banks2dump[nbanks2dump++][0],p2c,YBOS_BANK_NAME_LEN);
	 p2c = strtok(NULL," ");
  }
  DDD_TF_prbanks();
}



/*
 * Actual Event dumping 
 *-----------------------------------------------------------------------
 */


int *get_next_bank(int *ptr0, int *ptr)
{
  struct bank_header *bhdr;

  bhdr = (struct bank_header*) ptr;
  if ( (ptr+=(bhdr->bank_length+sizeof(struct bank_header)/4-1)) >= ptr0+(*ptr0) )
	 return NULL;

  return ptr;
}


int to_dump(int *ptr)
{
  int i;
  struct bank_header *bhdr;

  if (nbanks2dump < 0)
	 return TRUE;

  bhdr = (struct bank_header*) ptr;
  for (i=0; i<nbanks2dump; i++){
	 if (  !strcasecmp( bhdr->bank_name, &banks2dump[i][0]) )
		return TRUE;
  }
  return FALSE;
}


void release_event()
{
  if ( got_event ){
	 if ( ddu_put_fev(fev) != 0 ){
		fprintf(stderr,"ERROR in ddu_put_fev \n");
		exit(1);
	 }
  }
  got_event = FALSE;
}

void DDD_Dump_Blist()
{
  if ( !got_event){
	 pr2screen("I ain't got no event, babe \n");
	 update_screen();
	 return;
  }
  dump_bank_list(NULL);
}


void DDD_Dump()
{
  if ( !got_event){
	 pr2screen("I ain't got no event, babe \n");
	 update_screen();
	 return;
  }
  dump_bank_list(NULL);
  dump_event(NULL);
}


void dump_bank_list(FILE *fp)
{
  struct bank_header *bhdr;
  int *ptr0, *ptr;
  int nbank;

  nbank = 0;
  ptr0 = fev.p2da;
  ptr  = fev.p2da+1;
  strcpy(text,"\nBank list :");

  while ( ptr != NULL ){
	 bhdr = (struct bank_header*) ptr;
	 sprintf(text1,"%6s",bhdr->bank_name);
	 strcat(text,text1);
	 if ( (++nbank)%10 == 0 )
		strcat(text,"\n           ");
	 
	 ptr = get_next_bank(ptr0, ptr);
  }

  if (fp == NULL){
	 pr2screen(text);
	 update_screen();
  }
  else
	 fprintf(fp,"%s",text);
}


void dump_bank(int *ptr, FILE *fp)
{
  int i, j, *p;
  char *p2c, textascii[100], textascii1[100];
  struct bank_header *bhdr;

  bhdr = (struct bank_header*) ptr;
  sprintf(text,"\n\nBank %4s : Number of data words %i",
                         bhdr->bank_name,bhdr->bank_length-1);

  strcpy(text1,"");
  strcpy(textascii,"   ");
  p = ptr + (sizeof(struct bank_header)/4);

  for (i=0; i<bhdr->bank_length-1; i++){
	 if (i%5 == 0){
		strcat(text,textascii);
		if (fp == NULL) pr2screen(text);
		else fprintf(fp,"%s",text);
		sprintf(text,"\n%4i: ",i);
		strcpy(textascii,"   ");
	 }
	 if ( iformat == IFORMAT_HEXASCII ){
		strcpy(textascii1,".... ");
		p2c = (char*)p;
		for (j=0;j < sizeof( int ); j++){
		  if ( isalnum(*p2c) )
			 textascii1[j] = *p2c;
		  p2c++;
		}
		strcat(textascii,textascii1);
	 }
	 sprintf(text1,format,*p++);
	 strcat(text,text1);
  }
  
  if ( (j = (--i)%5) != 4){
	 if ( iformat == IFORMAT_HEXASCII ){
		for (i=0; i<4-j; i++)
		  strcat(text,"         ");
		strcat(text,textascii);
	 }
	 strcat(text,"\n");
	 if (fp == NULL) pr2screen(text);
	 else fprintf(fp,"%s",text);
  }
  else{
	 if (fp == NULL) pr2screen("\n");
	 else fprintf(fp,"%s",text);
  }

  strcpy(text,"");
}


void dump_event(FILE *fp)
{
  int *ptr0, *ptr;

  ptr0 = fev.p2da;
  ptr  = fev.p2da+1;

  while ( ptr != NULL ){
	 if ( to_dump(ptr) )
		dump_bank(ptr,fp);
	 ptr = get_next_bank(ptr0, ptr);
  }
  
  update_screen();
}


void DDD_Next_Event()
{
  int status;

  if ( !started ){
	 pr2screen("You have to start first\n");
	 update_screen();
	 return;
  }

  release_event();
  
  status = ddu_get_fev(&fev);

  if ( status < 0 ){
	 fprintf(stderr,"ERROR in ddu_get_fev \n");
	 exit(1);
  }
  else if (status > 0){
	 pr2screen("got no event\n");
	 update_screen();
  }
  else{
	 got_event = TRUE;
	 
	 sprintf(text,"\n\n\n\n \
******************************** new event ********************************\n\n \
FIFO event: address %p length %6i ctl %6i %6x %6i %6x \n",
				fev.p2da, fev.len, fev.ctlw1,fev.ctlb1,fev.ctlw2,fev.ctlb2);
	 pr2screen(text);
	 
	 dump_bank_list(NULL);
	 
	 if ( auto_dump )
		dump_event(NULL);
  }

}
