/*
 * dd_dcom.c   :  Routines that deal with the dcom information 
 *             :  each process attached to the dd.
 *
 * Author   :  C.Witzig
 * Date     :  Apr 11, 1992 
 * Mods     :  Apr 26, 1995: add semaphore flags for active
 *             masters and dd proc in order
 *             to check whether they are still around.
 *
 */

#include <stdio.h>
#include <sys/types.h>
#include <string.h>
#include <time.h>

#include <errno.h>
#include <signal.h>

#include <dd_dcom.h>
#include <fifo.h>
#include <dd_log.h>
#include <dd_signal_def.h>
#include <dd_network.h>

#define TRUE 1
#define FALSE 0

int my_own_ddpid = -1;

struct dd_dcom_str *dd_dcom;

void show_dd_dcom()
{
  int i, semcnt[MAX_DD_SEM];

  dd_dcom_get_semcnt(semcnt);

  printf("\n\n");
  printf("              DD DCOM AREA             \n");
  printf(" ---------------------------------------------\n");
  printf(" dd state ........................ %14s \n",dd_states[dd_dcom->dd_status]);
  printf(" number of dd masters attached ... %9d \n",dd_dcom->dd_nmstr);
  printf(" number of processes attached .... %9d \n",dd_dcom->dd_nproc_att);
  printf(" number of events requested ...... %9u \n",dd_dcom->nfev_req);
  printf(" number of volatile buffers made . %9u \n",dd_dcom->nbuf_vol_made);
  printf(" number of volatile buffers ...... %9u \n",dd_dcom->db_nbuf_vol);
  printf(" brc_lock flag ................... %9d \n",dd_dcom->brc_lock);
  printf(" dd dcom lock semaphore .......... %9d \n",semcnt[0]);
  printf(" dd process semaphores: ");
  for (i = 0; i<MAX_DD_SEM; i++) {
    (i==0) ? printf("sem %2d val %3d",i,semcnt[i]) : 
     printf("\n                        sem %3d: val %3d ",i,semcnt[i]) ;
  }
  printf("\n\n");
}

void dd_dcom_detmaster()
/*
 * called from atexit
 * is used only if the process for whatever reason hasn't
 * detached from the DD system
 */
{
  int i;

  if ( dd_dcom ) {
    for ( i=0; i<DD_MAX_NMSTR; i++) {
      if ( dd_dcom->dd_master[i].pid == getpid() ) {
	sprintf(dd_log_msg,"dd_dcom_detmaster: removing master from dcom\n");
	dd_log(DD_INFO, dd_log_msg);
	dd_dcom->dd_master[i].pid = -1;
	if ( dd_dcom->dd_master[i].dd_sem >= 0 ) {
	  /*
	   * I assume that dd_semval is negative already, so that the 2nd arg becomes pos.
	   * This would correspond to the release of resources. Why not just put 1? 
	   * Because if dd_semval is pos., then you end up waiting to get a sem. CTCTCT
	   */
	  if ( dd_dcom_set_procsem(dd_dcom->dd_master[i].dd_sem, -dd_dcom->dd_master[i].dd_semval)) {
	    sprintf(dd_log_msg,"dd_dcom_detmaster: error in dd_dcom_set_procsem\n");
	    dd_log(DD_ERROR, dd_log_msg);
	    return;
	  }
	  /* corresponds to sembuf.sem_num CTCTCT */
	  dd_dcom->dd_master[i].dd_sem = -1;
	  /* corresponds to sembuf.sem_op CTCTCT */
	  dd_dcom->dd_master[i].dd_semval = -1;
	}
	dd_dcom->dd_nmstr--;
	if ( dd_dcom->dd_nmstr <= 0 ){
	  sprintf(dd_log_msg,"dd_cleanup: last master - call dds_close\n");
	  dd_log(DD_INFO, dd_log_msg);
 	  dds_close(0);
	  return;
	}
      }
    }
  }
}

int dd_dcom_addmaster()
{
  int i, ival, nsem;
  
  dd_dcom->dd_nmstr++;

  for (i=0; i<DD_MAX_NMSTR; i++){
    if (dd_dcom->dd_master[i].pid <= 0)
      break;
  }
  if (i == DD_MAX_NMSTR ){
    sprintf(dd_log_msg," dd_dcom_init: no empty dd master slot \n");
    dd_log(DD_ERROR, dd_log_msg);
    /* The next line added by CTCTCT */
    dd_dcom->dd_nmstr--;
    return -1;
  }

 /* 
  * sembuf.sem_num = nsem
  * Since DD_MAX_NMSTR=15 & DD_PROC_PER_SEM=10 , nsem = 1 or 2   CTCTCT
  */
  nsem = i / DD_PROC_PER_SEM + 1;
  
  /*
   * sembuf.sem_op = ival
   * i,ival = 0,1  1,2  2,4  3,8  ... 9,512
   * I don't get this at all.  CTCTCT
   */
  ival = 1 << (i % DD_PROC_PER_SEM);
  
  if ( dd_dcom_set_procsem(nsem, ival ) ){
    sprintf(dd_log_msg," dd_dcom_addmaster: error in dd_dcom_set_procsem\n");
    dd_log(DD_ERROR, dd_log_msg);
    /* The next line added by CTCTCT */
    dd_dcom->dd_nmstr--;
    return -1;
  }

  dd_dcom->dd_master[i].pid = getpid();
  dd_dcom->dd_master[i].dd_sem = nsem;
  dd_dcom->dd_master[i].dd_semval = ival;

  atexit(dd_dcom_detmaster);

  return i;
}


int dd_dcom_init()
{
  int i,ii;

/*
  dd_dcom->fifo_semid = fifo_semid;
  dd_dcom->fifo_lock_semid = fifo_lock_semid;
  dd_dcom->fifo_shmid = fifo_shmid;
  dd_dcom->fifo_max_in_use = 0;
  dd_dcom->db_shmid = db_shmid;
*/

  dd_dcom->nfev_req = 0;
  dd_dcom->nbuf_vol_made = 0;
  dd_dcom->db_nbuf_vol = 0;
  
  dd_dcom->dd_nproc_att = 0;
  dd_dcom->dd_garbage_d_pid = 0;
  dd_dcom->dd_garbage_d_uid = 0;
  
  for (i=0;i<DD_MAX_NPROC_ATT;i++){
    dd_dcom->dd_proc[i].pid = -1;
    dd_dcom->dd_proc[i].fifo = -1;
    dd_dcom->dd_proc[i].nputfev = 0;
    dd_dcom->dd_proc[i].ngetfev = 0;
    dd_dcom->dd_proc[i].status = DD_PROC_STATE_UNUSED;
    dd_dcom->dd_proc[i].brc_flag = DD_BRC_OFF;
    strcpy(dd_dcom->dd_proc[i].pname," ");
    for (ii = 0; ii<DD_BRCCMD_MAXCMD; ii++)
      dd_dcom->dd_proc[i].brc_cmd[ii] = NULL;
  }

  return 0;
}


int dd_dcom_addproc(char *fname, int dd_wait)
/*
 * Adds another process in the dcom area:
 * 1. Checks that there is room for another process (slot my_own_ddpid).
 * 2. Sets status
 * 3. Builds the process name, i.e. fname + "_x", where 
 *    x is the number of process with this name modulo DD_MAX_PROC_ATT
 * 4. Writes the start time into dcom.
 */
{
  int i, len, nproc, iproc, nsem, ival;
  char *p2c;
  char pname[100];
  time_t tt;
  struct tm *ts;

  if ( my_own_ddpid < 0 ) {
    if (dd_dcom->dd_nproc_att >= DD_MAX_NPROC_ATT){
      sprintf(dd_log_msg," dd_dcom_addproc: no more processes allowed \n");
      dd_log(DD_ERROR, dd_log_msg);
       return -1;
    }

    for ( i = 0; i<DD_MAX_NPROC_ATT; i++) {
      if ( dd_dcom->dd_proc[i].pid == -1)
	break;
    }

    nsem = 1 + (DD_MAX_NMSTR/DD_PROC_PER_SEM+1) + (i/DD_PROC_PER_SEM+1);
    ival = 1 << (i % DD_PROC_PER_SEM);
    if ( dd_dcom_set_procsem(nsem, ival) ) {
      sprintf(dd_log_msg," dd_dcom_addproc: error in dd_dcom_set_procsem\n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
    dd_dcom->dd_proc[i].dd_sem = nsem;
    dd_dcom->dd_proc[i].dd_semval = ival;

    my_own_ddpid = i;
    dd_dcom->dd_nproc_att++;
  }

  ( dd_wait != DD_WAIT_ASYNC ) ? ( dd_wait = DD_WAIT_SLEEP ) : ( dd_wait = DD_WAIT_ASYNC);

  dd_dcom->dd_proc[my_own_ddpid].pid  = getpid();
  dd_dcom->dd_proc[my_own_ddpid].fifo = my_own_fifo;
  dd_dcom->dd_proc[my_own_ddpid].dd_wait = dd_wait;
  dd_dcom->dd_proc[my_own_ddpid].nputfev = 0;
  dd_dcom->dd_proc[my_own_ddpid].ngetfev = 0;
  dd_dcom->dd_proc[my_own_ddpid].brc_flag = DD_BRC_OFF;
  for (i = 0; i < DD_BRCCMD_MAXCMD; i++ )
    dd_dcom->dd_proc[my_own_ddpid].brc_cmd[i] = NULL;

  if ( my_own_fifo >= 0 ) {
    nproc = 0;
    i = 0;
    len = strlen(fname);
    for (i=0;i<DD_MAX_NPROC_ATT;i++){
      if ( dd_dcom->dd_proc[i].status != DD_PROC_STATE_UNUSED ) {
	if ( !(strncmp(dd_dcom->dd_proc[i].pname, fname,len) )) {
	  if ( ( p2c = strstr(dd_dcom->dd_proc[i].pname,"_") ) != NULL){
	    iproc = atoi(++p2c);
	    if (iproc > nproc)
	      nproc = iproc;
/* Why isn't there a break here? Is a pname not unique? CTCTCT */
	  }
	}
      }
    }
/* Why the modulo DD_MAX_NPROC_ATT? CTCTCT */
    nproc = (++nproc)%DD_MAX_NPROC_ATT;
    
    if (strcmp(dd_peername,"") == 0)
      sprintf(pname,"%s_%i",fname,nproc);
    else
      sprintf(pname,"serv2%s4%s_%i",dd_peername,fname,nproc);
    
    strncpy(dd_dcom->dd_proc[my_own_ddpid].pname,pname,
	    sizeof((dd_dcom->dd_proc[my_own_ddpid].pname))-1);
  }
  else
    strcpy(dd_dcom->dd_proc[my_own_ddpid].pname, "a monitor process");

  tt = time(NULL);
  ts = localtime(&tt);
  dd_dcom->dd_proc[my_own_ddpid].tstart = *ts;

  dd_dcom->dd_proc[my_own_ddpid].master_id = -1;
  for (i=0; i<DD_MAX_NMSTR; i++) {
    if ( dd_dcom->dd_master[i].pid == dd_dcom->dd_proc[my_own_ddpid].pid) {
      dd_dcom->dd_proc[my_own_ddpid].master_id = i;
      break;
    }
  }

  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_IDLE;

  return 0;
}

int dd_dcom_start()
{
  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
  return 0;
}

int dd_dcom_stop()
{
  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_IDLE;
  return 0;
}


int dd_dcom_remproc(int iproc)
{
  dd_dcom->dd_proc[my_own_ddpid].pid  = -1;
  dd_dcom->dd_proc[my_own_ddpid].fifo = -1;
  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_UNUSED;
  dd_dcom->dd_proc[my_own_ddpid].nputfev = 0;
  dd_dcom->dd_proc[my_own_ddpid].ngetfev = 0;
  strcpy(dd_dcom->dd_proc[my_own_ddpid].pname,"");

  if ( dd_dcom_set_procsem(dd_dcom->dd_proc[my_own_ddpid].dd_sem, -dd_dcom->dd_proc[my_own_ddpid].dd_semval) ) {
    sprintf(dd_log_msg," dd_dcom_remproc: error in dd_dcom_set_procsem\n");
    dd_log(DD_ERROR, dd_log_msg); 
  }
  dd_dcom->dd_proc[my_own_ddpid].dd_sem = -1;
  dd_dcom->dd_proc[my_own_ddpid].dd_semval = -1;

  if ( dd_dcom->dd_proc[my_own_ddpid].master_id >= 0 ) {
    dd_dcom_remmaster();
    dd_dcom->dd_proc[my_own_ddpid].master_id = -1;
  }

  dd_dcom->dd_nproc_att--;

  return 0;

}

int dd_dcom_remmaster()
{
/*
 * removes a master id, no not use my_own_ddpid as this might
 * have been overwritten
 */
  int i;

  if ( dd_dcom ) {
    for (i=0; i<DD_MAX_NMSTR; i++){
      if ( dd_dcom->dd_master[i].pid == getpid() ) {
	dd_dcom->dd_master[i].pid = -1;
	dd_dcom->dd_nmstr--;
      }
    }
  }
  return 0;
}

void dd_dcom_clear()
{
  int i, *p;

  p = (int*) dd_dcom;
/* can use bzero ? CTCTCT */
  for (i=0;i<sizeof(struct dd_dcom_str)/4;i++)
    *p++ = 0;
}
