/*
 * dbfi.c   :  Routines that deal with the db and the fifos
 *          :  in a rather general way. Note that the fifo routines in
 *          :  this file work on all fifos (fifoS_xxx)
 *          :  Furthermore it contains ALL the code for broadcasting
 *          :  fifo entries thru the dd system.
 *          :  Content: dbfi_reset()
 *          :           dbfi_connect()
 *          :           dbfi_empty_fifo()
 *          :           fifos_attach()
 *          :           fifos_init()
 * Author   :  C.Witzig
 * Date     :  Apr 1, 1992 (no joke!)
 * Mods     :  Jan 2, 1996: add dbfi_get_free_buffers to return the
 *                    number of free buffers in the system
 *             Feb 15, 1996: add dbfi_fifo_exist(char *fifoname)
 *          :  Feb 22, 1996: added call to dd_dcom_lock/unlock in 
 *                         : dd_dcom_attach
 *             Feb 28, 1996: added call to brc_cmd in dbfi_get_fev and
 *                           extend the broadcast mechanism to support 
 *                           the user commands
 *             Apr 16, 1996: add dbfi_fifo_active
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <signal.h>

#include <dd_dcom.h>
#include <fifo.h>
#include <db.h>
#include <dbfi.h>
#include <dd_signal_def.h>
#include <dd_user_entries.h>
#include <dd_usr_brccmd.h>
#include <dd_log.h>

#define TRUE  1 
#define FALSE 0

/* when flushing over the net wait at least BRC_FLUSH_WAIT ms for the
 * broadcast (different for the wait on the flush ! )
 */
#define BRC_FLUSH_WAIT 500

static int broadcast_rec       = FALSE;
static int ndbfi_broadcast_rec = 0;
static int user_int_rec = FALSE;

extern void dd_shutdown_alert();

void dbfi_user_int(int sig)
{
  sprintf(dd_log_msg, "user interrupt attached to signal %d seen\n",DD_USER_INTERRUPT);
  dd_log(DD_INFO, dd_log_msg);

  if ( DD_USER_INTERRUPT > 0 ) {
    signal(DD_USER_INTERRUPT, dbfi_user_int);
  }

  user_int_rec = TRUE;
  return;
}



int dbfi_free_fev(struct fifo_entry fev)
/*
 * Free shared mem belonging to extra large (volatile) events. CTCTCT
 */
{
  int shmid;

  shmid = fev.shmid;
  if ( shmid == -1){
    sprintf(dd_log_msg," dbfi_free_fev: bad shmid !!! \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  /* deletes shared mem from sys CTCTCT */
  if ( db_freemem(shmid) ){
    sprintf(dd_log_msg," dbfi_free_fev: error in db_freemem \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  dd_dcom_lock();
  dd_dcom->db_nbuf_vol--;
  dd_dcom_unlock();

  return 0;
}

int dbfi_get_free_buffers()
/* returns sem value of fifo_semid for INPUT_FIFO, CTCTCT */
{
  int cnt;
  
  return fifo_get_asemcnt(INPUT_FIFO, &cnt) ? -1 : cnt;
}

int dbfi_fifo_exist(char *fifoname)
/* returns fifo number (position in shm, 0-N) or -1 if non-existant CTCTCT */
{
  return fifo_exist(fifoname);
}

int dbfi_fifo_active(char *fifoname)
{
  int ififo;
  struct fifo_header fhdr;

  if ( (ififo = fifo_exist(fifoname)) < 0 )
    return -1;

  if ( fifo_get_header(ififo, &fhdr) )
    return -1;

  return (fhdr.fstatus == FSTAT_ACTIVE) ? 0 : 1;
}

int dbfi_fork_garbage_d()
/* Fork the garbage daemon process CTCTCT */
{
  pid_t pid;
  char pname[100], *p2c;
  int status;

  /* if child process ... CTCTCT */
  if ( (pid = fork()) == 0 ){
    /* immunize child from parent death CTCTCT */
    signal(SIGHUP,  SIG_IGN);
    signal(SIGINT,  SIG_IGN);
    signal(SIGQUIT, SIG_IGN);
    signal(SIGTERM, SIG_IGN);
    
    p2c = getenv("DD_BIN");
    strcpy(pname,p2c);
    strncat(pname,"/dd_garbage_d", sizeof(pname) - strlen(p2c) - 1);
    if ( status = execlp(pname, pname, (char*)NULL) ){
      sprintf(dd_log_msg,"error in exec-ing dd_garbage_d %s\n",strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
      dd_error_inv();
      return -1;
    }
  }
  if ( pid < 0 ){
    sprintf(dd_log_msg,"error in forking dd_garbage_d %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    dd_error_inv();
    return -1;
  }  
  dd_dcom->dd_garbage_d_pid = pid;
  sprintf(dd_log_msg,"forked off garbage_d %d\n",pid);
  dd_log(DD_INFO, dd_log_msg);

  return 0;
}

int dbfi_make_fev(int size,struct fifo_entry *p2fev)
/* Make a volatile buffer for the extra large event CTCTCT */
{
  static int imax0 = 20;
  int i,imax,shmid;
  int *ptr;
  struct fifo_entry fev1;

  imax = 0;

  /*
   * Loop to wait if too many volatile buffers already exist. CTCTCT
   */
  while ( dd_dcom->db_nbuf_vol > dd_dcom->dd_config.db_max_buf_vol){
    dd_nap(0.05);
    imax++;
  }
  
  if (imax == imax0){
    sprintf(dd_log_msg," dbfi_make_fev: too many volatile buffers \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  /*
   * Gets a piece of shared memory "on the fly" for event
   * bigger than MAX_EVSIZE (in multiples of 512 bytes). CTCTCT
   */
   if (db_getmem(&size,&shmid) ){
    sprintf(dd_log_msg," dbfi_make_fev: error in db_getmem \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  /* attach to shared mem "shmid" CTCTCT */
  if (db_attmem(shmid,&ptr) ){
    sprintf(dd_log_msg," dbfi_make_fev: error in db_attmem \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  /* set new user id for shared mem CTCTCT */
  if (db_set_shmuid(shmid,dd_dcom->dd_garbage_d_uid) ){
    sprintf(dd_log_msg," dbfi_make_fev: error in db_set_uid \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  fev1.shmid = shmid;
  fev1.dboff = 0;
  fev1.p2da = ptr;
  fev1.len = size;
  fev1.ctlw1 = -1;
  fev1.ctlb1 = -1;
  fev1.ctlw2 = -1;
  fev1.ctlb2 = -1;

  /* keep track of the number of volatile buffers CTCTCT */
  dd_dcom_lock();
  dd_dcom->db_nbuf_vol++;
  dd_dcom_unlock();

  *p2fev = fev1;
  return 0;
}

int dbfi_req_fev(int size, struct fifo_entry *p2fev)
/* process requesting event of certain size CTCTCT */
/* Who's doing the requesting? Seems to be broadcast connected CTCTCT */
{
  int status, mode;
  struct fifo_entry fev1;

  /* The following lines change dd_dcom data, but this function is
   * not always called with semaphore locks around it  CTCTCT
   */
  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_REQFEV;
  dd_dcom->nfev_req++;
  if (size > db_maxevsize){
    /* Make a volatile buffer for the extra large event CTCTCT */
    if ( dbfi_make_fev(size,&fev1) ) {
      dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
      return -1;
    }
    dd_dcom->nbuf_vol_made++;
    status = 0;
  }
  else{
    mode = dd_dcom->dd_proc[my_own_ddpid].dd_wait;
    /* puts next INPUT_FIFO event into fev1 CTCTCT */
    if ( (status = fifo_read(INPUT_FIFO, &fev1, mode)) ){
      if (status < 0) {
	sprintf(dd_log_msg," ddu_req_fev: error in fifo_read  \n");
	dd_log(DD_ERROR, dd_log_msg);
      }
      if (  dd_dcom->dd_proc[my_own_ddpid].status == DD_PROC_STATE_EXIT ){
	printf("dbfi_req_fev: EXIT signal seen\n");
	return status;
      }
      dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
      return status;
    }
    fev1.p2da = db_ptr_shm + fev1.dboff;
    fev1.len = db_maxevsize;
  }
  *p2fev = fev1;
  dd_dcom->dd_proc[my_own_ddpid].ngetfev++;
  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;

  return status;
}

int dbfi_get_fev(struct fifo_entry *fev)
/* get event from own or BROADCAST_FIFO CTCTCT */
{
  int ififo, status, mode;
  struct fifo_entry fev1;

  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_GETFEV;
  mode = dd_dcom->dd_proc[my_own_ddpid].dd_wait;

 retry:
  ififo = my_own_fifo;
  if ( ndbfi_broadcast_rec > 0 ){
	 ififo = BROADCAST_FIFO;
	 ndbfi_broadcast_rec--;
  }

  broadcast_rec = FALSE;
  if ( ( status = fifo_read(ififo, &fev1, mode)) ){
    if ( status < 0 ) {
      sprintf(dd_log_msg," dbfi_get_fev: error in fifo_read  \n");
      dd_log(DD_ERROR, dd_log_msg);
    }
    if ( ( status > 0 ) && ( broadcast_rec ) )
      goto retry;
    if ( dd_dcom->dd_proc[my_own_ddpid].status == DD_PROC_STATE_EXIT ) {
      sprintf(dd_log_msg, "dbfi_get_fev: EXIT signal seen\n");
      dd_log(DD_INFO, dd_log_msg);
      return status;
    }
    if ( dd_dcom->dd_proc[my_own_ddpid].status == DD_PROC_STATE_DEBLOCKED) {
      sprintf(dd_log_msg," fifo_read: process %i got deblocked \n",my_own_ddpid);
      dd_log(DD_INFO, dd_log_msg);
    }
    if ( user_int_rec ) {
      sprintf(dd_log_msg, "user_interrupted semaphore operation\n");
      dd_log(DD_INFO, dd_log_msg); 
      user_int_rec = FALSE;
    }
    dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
    return status;
  }

  dd_dcom->dd_proc[my_own_ddpid].ngetfev++;

  /* Is this in case we need to attach to a volatile buffer CTCTCT */
  if (fev1.shmid != -1){
    if (db_attmem(fev1.shmid,&fev1.p2da) ){
      sprintf(dd_log_msg," dbfi_get_fev: error in db_attmem \n");
      dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
      return -1;
    }
  }
  else
    fev1.p2da = db_ptr_shm + fev1.dboff;

  /* What is this stuff? CTCTCT */
  if ( (fev1.ctlw2 == BROADCAST_MARKER) && ( (fev1.ctlb2 & DD_BRCCMD_FIELD) == DD_BRCCMD_MARKER) ) {
    int cmd = fev1.ctlb2 & ~DD_BRCCMD_FIELD ;
    if ( cmd > 0 && cmd < DD_BRCCMD_MAXCMD ) {
      if ( dd_dcom->dd_proc[my_own_ddpid].brc_cmd[cmd] != NULL )
	(*dd_dcom->dd_proc[my_own_ddpid].brc_cmd[cmd])(fev1.ctlw1, fev1.ctlb2);
    }
  }

  *fev = fev1;
  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;

  return 0;
}

int dbfi_put_fev(struct fifo_entry fev)
/*
 * Called when a process got an event from its fifo and wants to put it
 * back into the dd system.
 */
{
  int next_fifo;

  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_PUTFEV;

  if ( fev.ctlw2 == BROADCAST_MARKER ){
    /* if not volatile, return to INPUT, else dump CTCTCT */
    if (fev.shmid == -1)
      next_fifo = INPUT_FIFO;
    else
      next_fifo = GARBAGE_FIFO;
  }
  else{
    /* get_next_fifo (fifo.c):
     * Starting from fifo number my_own_fifo, it loops through the remaining
     * fifos and checks whether the event fits into another fifo. If not, then
     * it returns the INPUT_FIFO. CTCTCT
     */
    if ( (next_fifo = get_next_fifo(my_own_fifo,fev) ) < 0){
      sprintf(dd_log_msg," dbfi_put_fev: error in get_next_fifo  \n");
      dd_log(DD_ERROR, dd_log_msg);
      dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
      return -1;
    }
  }
  
  /* if volatile, detach memory CTCTCT */
  if (fev.shmid != -1){
    if ( db_detmem(fev.p2da) ){
      sprintf(dd_log_msg," dbfi_put_fev: error in db_detmem \n");
      dd_log(DD_ERROR, dd_log_msg);
      dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
      return -1;
    }
  }
  
  /* if all fifos done with event, put back in INPUT_FIFO CTCTCT */
  if ( (next_fifo == GARBAGE_FIFO) && (fev.shmid == -1) )
    next_fifo = INPUT_FIFO;
  
  /*
   * if next fifo = INPUT_FIFO, and it's a voltile buffer,
   * return an error - otherwise big problems CTCTCT
   */
  if ( (next_fifo == INPUT_FIFO) && (fev.shmid != -1) ) {
    printf(" dbfi_put_fev: no putting volatile buffer in INPUT_FIFO, check garbage daemon\n");
    sprintf(dd_log_msg," dbfi_put_fev: no putting volatile buffer in INPUT_FIFO, check garbage daemon\n");
    dd_log(DD_ERROR, dd_log_msg);
    dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
    return -1;
  }
  
  /* put event into next fifo CTCTCT */
  if ( fifo_write(next_fifo,fev) ){
    sprintf(dd_log_msg," dbfi_put_fev: error in fifo_write  \n");
    dd_log(DD_ERROR, dd_log_msg);
    dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
    return -1;
  }

  dd_dcom->dd_proc[my_own_ddpid].nputfev++;
  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
  return 0;
}

int dbfi_ins_fev(struct fifo_entry fev)
/*
 * identical to dbfi_put_fev except that it starts from the
 * first (input) fifo instead of my_own_fifo when checking to
 * see whether the event goes to another fifo next. CTCTCT
 */
{
  int next_fifo;

  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_PUTFEV;

  if ( fev.ctlw2 == BROADCAST_MARKER ){
    if (fev.shmid == -1)
      next_fifo = INPUT_FIFO;
    else
      next_fifo = GARBAGE_FIFO;
  }
  else{
    if ( (next_fifo = get_next_fifo(INPUT_FIFO,fev) ) < 0){
      sprintf(dd_log_msg," dbfi_ins_fev: error in get_next_fifo  \n");
      dd_log(DD_ERROR, dd_log_msg);
      dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
      return -1;
    }
  }
  
  if (fev.shmid != -1){
    if ( db_detmem(fev.p2da) ){
      sprintf(dd_log_msg," dbfi_ins_fev: error in db_detmem \n");
      dd_log(DD_ERROR, dd_log_msg);
      dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
      return -1;
    }
  }
  
  if ( (next_fifo == GARBAGE_FIFO) && (fev.shmid == -1) )
    next_fifo = INPUT_FIFO;
  
  /*
   * if next fifo = INPUT_FIFO, and it's a voltile buffer,
   * return an error - otherwise big problems CTCTCT
   */
  if ( (next_fifo == INPUT_FIFO) && (fev.shmid != -1) ) {
    sprintf(dd_log_msg," dbfi_put_fev: no putting volatile buffer in INPUT_FIFO, check garbage daemon\n");
    dd_log(DD_ERROR, dd_log_msg);
    dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
    return -1;
  }
  
  if ( fifo_write(next_fifo,fev) ){
    sprintf(dd_log_msg," dbfi_ins_fev: error in fifo_write  \n");
    dd_log(DD_ERROR, dd_log_msg);
    dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
    return -1;
  }

  dd_dcom->dd_proc[my_own_ddpid].nputfev++;
  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;

  return 0;
}



static int dbfi_brcsignal_fev(struct fifo_entry fev, int wait_flag)
/*
 * 1. Make a list of all interested processes
 * 2. Fill the broadcast fifo with copies of the fev 
 * 3. Signal all the processes in your list
 * 4. If wait_flag is set, wait until the broadcast fifo is empty or 
 *    the maximum number of milliseconds.
 * 5. Check if there are processes that did not acknowledge the signal.
 *
 * NOTE: If there is a timeout, the broadcast fifo is only emptied if 
 *       processes did not acknowledge the signal. If the process did
 *       acknowledge but not yet read the broadcast fifo, then we return
 *       anyway, assuming that the process will read it later. 
 *
 * Mod: Feb 29, 1996: add brc_cmd mechanism 
 */
{
  int i, npid, ififo, cop_shot, status, nsem, nmsec;
  int proc_dd_id[DD_MAX_NPROC_ATT];
  pid_t my_pid, pid2sig[DD_MAX_NPROC_ATT];
  struct fifo_entry fev2rel, fevar[DD_MAX_NPROC_ATT];

  dd_dcom_lock();
  npid = 0;
  my_pid = getpid();
  for (i=0;i<DD_MAX_NPROC_ATT;i++){
    if ( ( dd_dcom->dd_proc[i].pid > 0) && 
	( dd_dcom->dd_proc[i].pid  != my_pid ) && 
	( dd_dcom->dd_proc[i].fifo != INPUT_FIFO ) && 
   	( dd_dcom->dd_proc[i].brc_flag == DD_BRC_SIGNAL ) && 
	( dd_dcom->dd_proc[i].status != DD_PROC_STATE_IDLE) ){
      if ( (fev.ctlb2 & DD_BRCCMD_FIELD) == DD_BRCCMD_MARKER ) {
	int cmd = fev.ctlb2 & ~DD_BRCCMD_FIELD;
	if ( cmd > 0 && cmd < DD_BRCCMD_MAXCMD ) {
	  if ( dd_dcom->dd_proc[i].brc_cmd[cmd] == NULL )
	    continue;
	}
      }	
      proc_dd_id[npid] = i;
      pid2sig[npid++] = dd_dcom->dd_proc[i].pid;
    }
  }

  for (i=0; i<npid; i++){
    dbfi_req_fev(fev.len, &fevar[i]);
    memcpy(fevar[i].p2da, fev.p2da, 4*fev.len);
    fevar[i].len   = fev.len;
    fevar[i].ctlw1 = fev.ctlw1;
    fevar[i].ctlb1 = fev.ctlb1;
    fevar[i].ctlw2 = fev.ctlw2;
    fevar[i].ctlb2 = fev.ctlb2;
  }

  for (i = 0; i < npid; i++){
    if (fevar[i].shmid != -1){
      if ( db_detmem(fevar[i].p2da) ){
	sprintf(dd_log_msg," dbfi_brc_fev: error in db_detmem \n");
	dd_log(DD_ERROR, dd_log_msg);
	dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
	dd_dcom_unlock();
	return -1;
      }
    }
    
    fevar[i].ctlw2 = BROADCAST_MARKER;
    if ( fifo_write(BROADCAST_FIFO, fevar[i]) ){
      sprintf(dd_log_msg," dbfi_brc_fev: error in fifo_write  \n");
      dd_log(DD_ERROR, dd_log_msg);
      dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
      dd_dcom_unlock();
      return -1;
    }
    dd_dcom->dd_proc[my_own_ddpid].nputfev++;
    dd_dcom->dd_proc[proc_dd_id[i]].brc_ack = FALSE;
    if ( (cop_shot = kill(pid2sig[i], BROADCAST_SIGNAL)) != 0 ) {
      sprintf(dd_log_msg,"dbfi_brc_fev: kill to proc %i %s \n",
	      pid2sig[i],strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
    }
  }
  dd_dcom_unlock();
  
  status = 0;
  fifo_get_asemcnt(BROADCAST_FIFO, &nsem);
  if ( wait_flag ){
    nmsec = wait_flag/20+1;
    for (i=0; ( (i<nmsec) && (nsem != 0) && (!status) ); i++){
      dd_nap(0.05); 
      if ( ( status = fifo_get_asemcnt(BROADCAST_FIFO, &nsem) ) < 0){
	sprintf(dd_log_msg, "dbfi_brc_fev: error in fifo_get_asemcnt\n");
	dd_log(DD_ERROR, dd_log_msg);
     }
    }
    if ( i == nmsec ){
      sprintf(dd_log_msg,"proc %i dbfi_brcsignal_fev: %i not responding %i %i\n",
	      getpid(),nsem,i,nmsec);
      dd_log(DD_ERROR, dd_log_msg);
      status = nsem;
    }
  }
  else
    dd_nap(0.1);

/*  sprintf(dd_log_msg,"proc: %i dbfi_brcsignal_fev: after signal %i %i %i %i \n",
	  getpid(),nsem,i,nmsec,status); */
  for (i=0;i<npid;i++){
    if (dd_dcom->dd_proc[proc_dd_id[i]].brc_ack != TRUE){
      sprintf(dd_log_msg,"brc: process %i %i did NOT ack !!!!\n",i,pid2sig[i]);
      dd_log(DD_ERROR, dd_log_msg);
      fifo_read(BROADCAST_FIFO,&fev2rel, DD_WAIT_ASYNC);
      dbfi_rel_fev(fev2rel);
    }
    else
      dd_dcom->dd_proc[my_own_ddpid].nputfev++;
  }

  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;

  return status;
}


static int dbfi_brcusmail_fev(struct fifo_entry fev)
/*
 * Just write the broadcast fev into the fifo. If there are more than
 * one process attached to the fifo, then only one of them will get the
 * broadcast fev!
 */
{
  int i, ififo, status;
  pid_t my_pid, pid2sig[DD_MAX_NPROC_ATT];
  struct fifo_entry fev2send;

  dd_dcom_lock();

  status = 0;
  my_pid = getpid();

  for (i=0;i<DD_MAX_NPROC_ATT;i++){
    if ( ( dd_dcom->dd_proc[i].pid > 0) && 
	( dd_dcom->dd_proc[i].pid  != my_pid ) && 
	( dd_dcom->dd_proc[i].fifo != INPUT_FIFO ) && 
	( dd_dcom->dd_proc[i].brc_flag == DD_BRC_USMAIL ) && 
	( dd_dcom->dd_proc[i].status != DD_PROC_STATE_IDLE) ){
      
      if ( fev.ctlb2 & DD_BRCCMD_FIELD == DD_BRCCMD_MARKER ) {
	int cmd = fev.ctlb2 & ~DD_BRCCMD_FIELD;
	if ( cmd > 0 && cmd < DD_BRCCMD_MAXCMD ) {
	  if ( dd_dcom->dd_proc[i].brc_cmd[cmd] == NULL )
	    continue;
	}
      }

      dbfi_req_fev(fev.len, &fev2send);
      memcpy(fev2send.p2da, fev.p2da, 4*fev.len);
      fev2send.len   = fev.len;
      fev2send.ctlw1 = fev.ctlw1;
      fev2send.ctlb1 = fev.ctlb1;
      fev2send.ctlw2 = BROADCAST_MARKER;
      fev2send.ctlb2 = fev.ctlb2;
      
      if (fev2send.shmid != -1){
	if ( ( status = db_detmem(fev2send.p2da)) ){
	  sprintf(dd_log_msg," dbfi_brcusmail_fev: error in db_detmem \n");
	  dd_log(DD_ERROR, dd_log_msg);
 	  break;
	}
      }
      
      if ( (status = fifo_write(dd_dcom->dd_proc[i].fifo, fev2send)) ){
	sprintf(dd_log_msg," dbfi_brcusmail_fev: error in fifo_write  \n");
	dd_log(DD_ERROR, dd_log_msg);
 	break;
      }
    }
  }
      
  dd_dcom_unlock();

  return status;
}

static void brc_lock()
{
  int ok = FALSE;

  while ( !ok ){
    dd_dcom_lock();
    if ( dd_dcom->brc_lock == FALSE ){
      dd_dcom->brc_lock = TRUE;
      ok = TRUE;
      dd_dcom_unlock();
    }else{
      dd_dcom_unlock();
      dd_nap(0.05);
    }
  }
}

static void brc_unlock()
{
  dd_dcom_lock();
  dd_dcom->brc_lock = FALSE;
  dd_dcom_unlock();
}


static int dbfi_brc_flush()
/*
 * makes a broadcast after the flush by sending the
 * command DD_BRC_FLUSH
 */
{
  struct fifo_entry fev;
  int status;
  int len = dd_dcom->dd_config.db_maxevsize - 1;

  if ( dbfi_req_fev(len, &fev) == 0 ) {
    fev.len = 10;
    fev.ctlw1 = -1;
    fev.ctlb1 = -1; 
    fev.ctlw2 = BROADCAST_MARKER;
    fev.ctlb2 = DD_BRCCMD_MARKER | DD_BRCCMD_FLUSH;
    if ( (status = dbfi_brc_fev(fev, BRC_FLUSH_WAIT)) != 0 ) {
      sprintf(dd_log_msg, "dbfi_flush: error in brc_flush - status %d\n",status);
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }

    if ( (status = dbfi_rel_fev(fev)) != 0 ) {
      sprintf(dd_log_msg, "dbfi_flush: error in dbfi_rel_fev - status %d\n",status);
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }

  } else {
    sprintf(dd_log_msg, "dbfi_flush: error in dbfi_req_fev for brc_flush - status %d\n",status);
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  return 0;
}

  


int dbfi_brc_fev(struct fifo_entry fev, int wait_flag)
/*
 * Broadcasts an fev over the whole dd system, i.e. every consumer process 
 * will get a copy. If wait_flag != 0, then waits until every 
 * consumer process has actually received the fev. The units of
 * wait_flags are msec (i.e. wait_flag = 2 --> maximal wait of 2 ms).
 *
 * There are two ways to broadcast the event: 
 * If brc_flag == DD_BRC_SIGNAL then the event is broadcasted over
 * the broadcast fifo. In this case it is garanteed that every process
 * gets the fev, if several are attached to the same fifo.
 * [Internally we duplicate the fev to the desired number and then every
 * consumer process afterwards has his private copy, which he will
 * eventually put back into the INPUT fifo.]
 * If brc_flag == DD_BRC_USMAIL then the fev is simply written into the
 * fifo of that process, regardless of the number of processes on that fifo.
 * Obviously the second way has a much lower priority than the first one
 * (and is therefore named after its famous brother!).
 *
 * Only one process can broadcast at a time (simple locking mechanism brc_lock).
 *
 */
{
  int status1, status2;

  brc_lock();

  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_BRCFEV;

/*  sprintf(dd_log_msg,"\n\n pid %i start broadcasting\n",getpid()); */

  if ( (status1 = dbfi_brcsignal_fev(fev, wait_flag)) ) {
    sprintf(dd_log_msg,"dbfi_brc_fev: error in dbfi_brcsignal_fev %i\n",status1);
    dd_log(DD_ERROR, dd_log_msg);
  }

  if ( (status2 = dbfi_brcusmail_fev(fev)) ) {
    sprintf(dd_log_msg,"dbfi_brc_fev: error in dbfi_brcnormal_fev %i\n",status2);
    dd_log(DD_ERROR, dd_log_msg);
  }

  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;

/*  
  sprintf(dd_log_msg,"pid %i stop broadcasting\n",getpid());
  dd_log(DD_INFO, dd_log_msg);
*/

  brc_unlock();

  return (status1|status2);
}


int dbfi_rel_fev(struct fifo_entry fev)
{
  int next_fifo;

  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_RELFEV;
  next_fifo = INPUT_FIFO;
  
  if (fev.shmid != -1){
    if ( db_detmem(fev.p2da) ){
      sprintf(dd_log_msg," dbfi_rel_fev: error in db_detmem \n");
      dd_log(DD_ERROR, dd_log_msg);
      dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
      return -1;
    }
  }
  
  if ( (next_fifo == INPUT_FIFO) && (fev.shmid != -1) ){
    if ( dbfi_free_fev(fev) ){
      sprintf(dd_log_msg," dbfi_rel_fev: error in dbfi_free_fev \n");
      dd_log(DD_ERROR, dd_log_msg);
       dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
      return -1;
    }
  }
  else{
    if ( fifo_write(next_fifo,fev) ){
      sprintf(dd_log_msg," dbfi_rel_fev: error in fifo_write  \n");
      dd_log(DD_ERROR, dd_log_msg);
       dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;
      return -1;
    }
  }
  dd_dcom->dd_proc[my_own_ddpid].nputfev++;
  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_ACTIVE;

  return 0;
}

void dbfi_rec_brc(int sig)
{
  signal(BROADCAST_SIGNAL,dbfi_rec_brc);
  dd_dcom->dd_proc[my_own_ddpid].brc_ack = TRUE;
  ndbfi_broadcast_rec++;
  broadcast_rec = TRUE;
}

int dbfi_link_signal(int sig)
{
  signal(BROADCAST_SIGNAL,dbfi_rec_brc);
  ndbfi_broadcast_rec = 0;

  if ( DD_USER_INTERRUPT > 0 ) {
    signal(DD_USER_INTERRUPT, dbfi_user_int);
  }
  signal(DD_SHUTDOWN_SIGNAL, dd_shutdown_alert);
  

  return 0;
}


int dbfi_fill_input_fifo()
/*
 * Fills the INPUT FIFO with the pointers to the slots in the db.
 * Called either from dbfi_connect (called from dds_init or dds_create)
 * or as part of the dd flush command at each start/end of run.
 * In addition it saves the offset within db for later crosscheck.
 */
{
  int iev;
  struct fifo_entry fev;

  fev.shmid = -1;
  fev.len   = -1;
  fev.ctlw1 = -1;
  fev.ctlb1 = -1;
  fev.ctlw2 = -1;
  fev.ctlb2 = -1;
  for (iev = 0;iev<db_maxnbev;iev++){
    fev.dboff = iev*db_maxevsize;
    fev.p2da  = db_ptr_shm + iev*db_maxevsize;
    if (fifo_write(INPUT_FIFO,fev)){
      sprintf(dd_log_msg," dbfi_fill_input_fifo: error in writing into input fifo \n");
      dd_log(DD_ERROR, dd_log_msg);
       return -1;
    }
    dd_dcom->db_ptr_offset[iev] = fev.dboff;
  }
  dd_dcom->db_nbuf_vol = 0;

  return 0;
}


int dbfi_empty_fifo(int which_fifo)
/*
 * Empties an individual fifo but does not throw the events that
 * it finds away, but transfers them into the INPUT_FIFO. This
 * routine is used when an individual process wants to detach from
 * the dd pool.
 */
{
  int i;
  int fstatus[MAX_FIFO_NB_MAX],fcount[MAX_FIFO_NB_MAX];
  int fev1[MAX_FIFO_NB_MAX],fev2[MAX_FIFO_NB_MAX];
  int icnt, mode;
  struct fifo_entry fev;

  dd_nap(0.1);
  if ( fifo_rate(fstatus,fcount,fev1,fev2) ){
    sprintf(dd_log_msg,"dbfi_empty_fifo: cannot get fifo counts \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  mode = dd_dcom->dd_proc[my_own_ddpid].dd_wait;
  for (icnt = 0;icnt<fcount[which_fifo];icnt++){
    fifo_read(which_fifo,&fev, mode);
    if (fev.shmid == -1){
      if (fifo_write(INPUT_FIFO,fev)){
	sprintf(dd_log_msg," dbfi_empty_fifo: error in fifo_write \n");
	dd_log(DD_ERROR, dd_log_msg);
 	return -1;
      }
    }
    else{
      if (dbfi_free_fev(fev) ){
	sprintf(dd_log_msg," dbfi_empty_fifo: error in dbfi_free_fev \n");
	dd_log(DD_ERROR, dd_log_msg);
 	return -1;
      }
    }
  }
  if (fifo_rate(fstatus,fcount,fev1,fev2)){
    sprintf(dd_log_msg," dbfi_empty_fifo: cannot get fifo counts \n");
    dd_log(DD_ERROR, dd_log_msg);
     return -1;
  }
  dd_nap(0.1);
  if (fcount[which_fifo] != 0) {
    sprintf(dd_log_msg," dbfi_empty_fifo : fifo is not emtpy  \n");
    dd_log(DD_ERROR, dd_log_msg);
     return -1;
  }
  return 0;
}

int dbfi_reset()
/*
 * Resets the db and fifos, that is:
 * empties the fifos if it is not yet empty and check with the saved
 * set in dcom to make sure that no pointer got lost (this is only
 * a last crosscheck that the dd system was working ok in the last run).
 * NOTE: It does not touch the fifo headers ..... just makes sure that ALL
 *       the fifos are empty (including INPUT fifo). 
 */
{
  int i,ififo, mode;
  int fstatus[MAX_FIFO_NB_MAX],fcount[MAX_FIFO_NB_MAX];
  int fev1[MAX_FIFO_NB_MAX],fev2[MAX_FIFO_NB_MAX];
  int ievfound, icnt, found, fatal;
  struct fifo_entry fev;
  
 /*
  * returns arrays with the status, fifo count and difference in
  * no. of events that came in/went out of the fifo since the last
  * time that it was called
  */
  if ( fifo_rate(fstatus,fcount,fev1,fev2) ){
    sprintf(dd_log_msg,"dbfi_reset: cannot get fifo counts \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fatal = FALSE;
  ievfound = 0;
  mode = dd_dcom->dd_proc[my_own_ddpid].dd_wait;
  for (ififo=0;ififo<max_fifo_nb;ififo++){
    /* reading thru each entry in fifo CTCTCT */
    for (icnt = 0;icnt<fcount[ififo];icnt++){
    /*
     * Reads the next fifo_entry from the fifo ififo.
     * Return code -1: some very serious error
     *             >0: no available pointer in the fifo or interrupted system
     *                 call while waiting for the next event. (the variable
     *                 errno is returned:
     *                 errno = EAGAIN: IPC_NOWAIT flag is set 
     *                                 (i.e. fifo in async mode)
     *                 errno = EINTR : interrupted system call 
     *                                 (i.e. signal arrived while waiting).
     */
      fifo_read(ififo,&fev, mode);
      if (fev.shmid == -1){
	found = FALSE;
	/* check all event locations to see if = this event's */
	for (i=0;(i<db_maxnbev)&&(!found);i++){
	  if ( dd_dcom->db_ptr_offset[i] == fev.dboff ){
	    found = TRUE;
	    ievfound++;
	  }
	}
	if ( !found ){
	  sprintf(dd_log_msg," dbfi_reset: offset %d not found in save list \n",
		  fev.dboff);
	  dd_log(DD_ERROR, dd_log_msg);
 	  fatal = TRUE;
	}
      }
      else{
	/* deletes shared mem from sys?! CTCTCT */
	if (dbfi_free_fev(fev) ){
	  sprintf(dd_log_msg," dbfi_reset: error in dbfi_free_fev \n");
	  dd_log(DD_ERROR, dd_log_msg);
 	  return -1;
	}
      }
    }
  }
  if (dd_dcom->db_nbuf_vol != 0){
    sprintf(dd_log_msg," dbfi_reset: not all volatile buffers deleted \n");
    dd_log(DD_ERROR, dd_log_msg);
     return -1;
  }
  
  if ( fatal )
    return -1;

  return 0;
} 


int dbfi_flush(int brc_flush)
/*
 * Flushes the DD system, that means:
 * 1. lock the INPUT_FIFO such that producers are no longer
 *    able to inject events
 * 2. loop over all other fifos and take the events out.
 *    Store them temporarily in the broadcast_fifo (this is 
 *    safe as dd_dcom is locked, therefore no process can
 *    make a broadcast during that time.
 * 3. unlock INPUT_FIFO
 * 4. Move the events from BROADCAST_FIFO to INPUT_FIFO.
 * 5. unlock dd_dcom.
 *
 * Returns -1 in case of error, 0 ok, >0 equal number of 
 * processes still having events.
 */
{
  int i,ififo, nwait;
  int nproc;
  struct fifo_entry fev;

  dd_dcom_lock();
  lock_input_fifo();
  dd_dcom->dd_status = DD_STATUS_FLUSH;

  nproc = 0;
  nwait = 0;
  for (ififo=0;ififo<max_fifo_nb;ififo++){
    if ( (ififo != INPUT_FIFO) && (ififo != BROADCAST_FIFO)  && (ififo != GARBAGE_FIFO) ) {
      while ( fifo_read(ififo, &fev, DD_WAIT_ASYNC) == 0 ) {
	if (fev.shmid == -1){
	  fifo_write(BROADCAST_FIFO,fev);
	}
	else{
	  if (dbfi_free_fev(fev) ){
	    sprintf(dd_log_msg," dbfi_flush: error in dbfi_free_fev \n");
	    dd_log(DD_ERROR, dd_log_msg);
 	    return -1;
	  }
	}
      }

      for ( i = 0; i<DD_MAX_NPROC_ATT; i++) {
	if ( dd_dcom->dd_proc[i].pid > 0 ){
	  if ( dd_dcom->dd_proc[i].fifo == ififo ) {
	    while ( (dd_dcom->dd_proc[i].status != DD_PROC_STATE_PUTFEV) && 
		    (dd_dcom->dd_proc[i].status != DD_PROC_STATE_GETFEV) && (nproc == 0) ) {
	      dd_nap(DD_FLUSH_WAIT_TICK);
	      if ( ++nwait >= DD_FLUSH_MAXWAIT ) {
		int delta_ev = dd_dcom->dd_proc[i].ngetfev-dd_dcom->dd_proc[i].nputfev;
		if ( delta_ev == 0 ) {
		  if ( dd_dcom->dd_proc[i].status == DD_PROC_STATE_IDLE )
		    sprintf(dd_log_msg, "dbfi_flush: process %d on fifo %d stayed in  idle state during flush\n",
			    dd_dcom->dd_proc[i].pid, ififo);
		  else
		    sprintf(dd_log_msg,"dbfi_flush: process pid %d on fifo %d did not read/write its fifo \n",
			    dd_dcom->dd_proc[i].pid, ififo);
		  dd_log(DD_INFO, dd_log_msg);
		}
		else {
		  sprintf(dd_log_msg,"dbfi_flush: process pid %d on fifo %d still holds %d fev \n",
			  dd_dcom->dd_proc[i].pid, ififo,delta_ev);
		  dd_log(DD_ERROR, dd_log_msg);
		}
		nproc++;
		break;
	      }
	    }
	  }
	}
      }
    }
  }

  unlock_input_fifo();
  while ( fifo_read(BROADCAST_FIFO, &fev, DD_WAIT_ASYNC) == 0 ) 
    fifo_write(INPUT_FIFO,fev);

  dd_dcom->dd_status = DD_STATUS_OK;
  dd_dcom_unlock();

  if ( brc_flush ) 
    dbfi_brc_flush();

  return nproc ;
}

int dbfi_connect()
/* 
 * Connect the db and the fifo together, that is:
 * 1. get the db and fill the pointers to the slots into the INPUT fifo.
 * 2. initialise INPUT fifo
 * 3. set INPUT fifo active
 */
{
  int ctl[FIFO_HDR_CTL_LEN];
  int i, itest;
  char ftype[5];
  struct fifo_mode fmode;

  if (db_maxnbev > fifo_depth){
    sprintf(dd_log_msg," dbfi_connect: cannot put so many pointers into fifo \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  for (i=0;i<sizeof(ctl)/4;i++)
    ctl[i]=0;

  sprintf(ftype,"STA%2i",INPUT_FIFO);
  fmode.mode = FMODE_ALL;
  fmode.prescale = FMODE_NOPRESCALE;
  fmode.wait = FWAIT_SLEEP;
  fmode.suser = FMODE_MULTI_USER;
  fmode.p2ctl = ctl;
  
 /*
  * Make a fifo for a process. The return value is either the fifo number,
  * to which the process must refer to in subsequent calls (my_own_fifo) 
  * or - when < 0 - signals an error code.
  * The character ftype is either "STA" or "DYN". If it is "STA" and ends with a 
  * number, then this number is interpreted as the requested fifo number.
  * If the requested fifo is already in use  then an error code is returned.
  * If successful, the fifo is initialised and set into IDLE mode.
  * (Note: We must loop over all fifos.)
  */
  if (fifo_make("INPUT",ftype, fmode) != INPUT_FIFO ){
    sprintf(dd_log_msg," dbfi_connect: cannot make INPUT fifo  \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  /* protect it so garbage daemon can't kill it */
  if (fifo_set_protection(INPUT_FIFO,FIFO_PROTECTED)){
    sprintf(dd_log_msg,"dbfi_connect: cannot set BROADCAST fifo to protected \n");
    dd_log(DD_ERROR, dd_log_msg);
     return -1;
  }
  if (fifo_set_status(INPUT_FIFO,FSTAT_IDLE)){
    sprintf(dd_log_msg,"dbfi_connect: cannot activate INPUT fifo \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  /* define static fifo "TAPE" and set to IDLE CTCTCT */
  ddu_def_static_fifos();

  sprintf(ftype,"STA%2i",BROADCAST_FIFO);
  fmode.mode = FMODE_ALL;
  fmode.prescale = FMODE_NOPRESCALE;
  fmode.wait = FWAIT_SLEEP;
  fmode.suser = FMODE_MULTI_USER;
  fmode.p2ctl = ctl;
  if (fifo_make("BCAST", ftype, fmode) != BROADCAST_FIFO ){
    sprintf(dd_log_msg," dbfi_connect: cannot make BROADCAST fifo  \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  if (fifo_set_protection(BROADCAST_FIFO,FIFO_PROTECTED)){
    sprintf(dd_log_msg,"dbfi_connect: cannot set BROADCAST fifo to protected \n");
    dd_log(DD_ERROR, dd_log_msg);
     return -1;
  }
  if (fifo_set_status(BROADCAST_FIFO,FSTAT_IDLE)){
    sprintf(dd_log_msg,"dbfi_connect: cannot set BROADCAST fifo to idle \n");
    dd_log(DD_ERROR, dd_log_msg);
     return -1;
  }

  /* define Garbage daemon/fifo CTCTCT */
  fmode.mode = FMODE_ALL;
  fmode.prescale = FMODE_NOPRESCALE;
  fmode.wait = FWAIT_SLEEP;
  fmode.suser = FMODE_MULTI_USER;
  fmode.p2ctl = ctl;
  sprintf(ftype,"STA%2i",GARBAGE_FIFO);
  if (fifo_make("GARB", ftype, fmode) != GARBAGE_FIFO ){
    sprintf(dd_log_msg," dbfi_connect: cannot make GARBAGE fifo  \n");
    dd_log(DD_ERROR, dd_log_msg);
     return -1;
  }
  if (fifo_set_protection(GARBAGE_FIFO,FIFO_PROTECTED)){
    sprintf(dd_log_msg,"dbfi_connect: cannot set GARBAGE fifo to protected \n");
    dd_log(DD_ERROR, dd_log_msg);
     return -1;
  }
  if (fifo_set_status(GARBAGE_FIFO,FSTAT_IDLE)){
    sprintf(dd_log_msg,"dbfi_connect: cannot activate GARBAGE fifo \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
/*
  if ( dbfi_fork_garbage_d() ){
    sprintf(dd_log_msg,"dbfi_connect: error in dbfi_fork_garbage \n");
    dd_log(DD_ERROR, dd_log_msg);
     return -1;
  }
*/

 /*
  * Fills the INPUT FIFO with the pointers to the slots in the db.
  * Called either from dbfi_connect (called from dds_init or dds_create)
  * or as part of the dd flush command at each start/end of run.
  * In addition it saves the offset within db for later crosscheck.
  */
  if ( dbfi_fill_input_fifo() ){
    sprintf(dd_log_msg,"dbfi_connect: error in dbfi_fill_input_fifo \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if (fifo_set_status(INPUT_FIFO,FSTAT_ACTIVE)){
    sprintf(dd_log_msg,"dbfi_connect: cannot activate INPUT fifo \n");
    dd_log(DD_ERROR, dd_log_msg);
     return -1;
  }
  return 0;
}

int fifos_unblock()
{
  int i;

  for (i=0;i<DD_MAX_NPROC_ATT;i++){
    if ( dd_dcom->dd_proc[i].pid > 0 ){
      if ( (dd_dcom->dd_proc[i].fifo   !=  INPUT_FIFO) && 
	  ( dd_dcom->dd_proc[i].fifo   !=  GARBAGE_FIFO) && 
	  ( dd_dcom->dd_proc[i].fifo   !=  BROADCAST_FIFO) && 
	  ( dd_dcom->dd_proc[i].status !=  DD_PROC_STATE_IDLE) ){
	  dd_dcom->dd_proc[i].status = DD_PROC_STATE_DEBLOCKED;
	}
    }
  }
  
  dd_nap(0.2);

  for (i=0;i<max_fifo_nb;i++){
    if ( (i != INPUT_FIFO) && (i != GARBAGE_FIFO) ){
      if ( fifo_unblock_proc(i) ){
	sprintf(dd_log_msg," fifos_unblock: error in fifo_unblock_proc \n");
	dd_log(DD_ERROR, dd_log_msg);
 	return -1;
      }
    }
  }
  return 0;
}

int fifos_attach(char *filename)
/*
 * Attaches the semaphores and shared memory (if icreate then creates them)
 */
{
  if ( fifo_attach(filename, dd_dcom->dd_config.max_fifo_nb, dd_dcom->dd_config.fifo_depth) ){
    sprintf(dd_log_msg," dds_create: fifo_attach failed  \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  return 0;
}

int fifos_init()
/*
 * Reinitialises the fifos completely (all the connections to 
 * processes are lost) and connects db and fifo.
 */
{
  int ififo;

  for (ififo=0;ififo<max_fifo_nb;ififo++){
    if (fifo_init(ififo) ){
      sprintf(dd_log_msg," fifos_init: error in initialising fifo \n");
      dd_log(DD_ERROR, dd_log_msg);
       return -1;
    }
  }
  return 0;
}
