/*
 * db.c     :  General routines to allocate and attach the global data 
 *          :  buffer as define in db.h
 * Author   :  C.Witzig
 * Date     :  Apr 2, 1992 
 * Mods     :
 *          :  Feb 21, 95 add argument *buf in shmctl call for HP
 *          :  Feb 21, 95: comment munpin out (temporarily)
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <errno.h>

#include <db.h>
#include <dd_log.h>

#define TRUE 1
#define FALSE 0

int db_shmid = 0;
int *db_ptr_shm;
int *db_shm_end;

int db_maxnbev, db_maxevsize;

int db_init(char *filename, int nbuffers, int maxsize)
{
  int istat;
  int ishmsize, ishmflg;
  int idummy;
  int isizpg;
  key_t key;
  static int first = TRUE;
  struct shmid_ds *buf;

  if ( ! first ) 
    return 0;

 /* db_maxnbev   = max number of events
  * db_maxevsize = max event size in longs
  * CTCTCT
  */
  db_maxnbev = nbuffers;
  db_maxevsize = maxsize;

 /* PAGE_SIZE = 128 longwords (= 512 bytes),
  * allocate mem for an event in page-sized chunks,
  * isizpg    = max size of an event in pages
  * ishmsize  = total size of shared mem in bytes
  * ishmflg   = permission to read & write by owner, group, & world
  * CTCTCT
  */
  isizpg =  db_maxevsize/PAGE_SIZE + 1;
  ishmsize = 4*db_maxnbev* PAGE_SIZE * isizpg;
  ishmflg = 0666 ;

/*
  if (db_shmid != 0) {
    if (shmdt((void *)db_ptr_shm) < 0)
      printf("db_init: can't detach from shared mem\n");
    else
      printf("db_init: detached from shared mem\n");
    if ( ( shmctl(db_shmid,IPC_RMID, buf) ) < 0) { 
      printf("db_init: can't destroy shm\n");
    }
  }
*/

  key = ftok(filename,DB_SHM_KEY);
  if ( key == -1 ) {
    perror(filename);
    return -1;
  }
  printf("key = %d\n", (int) key);
 
  if((db_shmid = shmget(key,ishmsize,ishmflg)) < 0) {
    ishmflg = 0666 | IPC_CREAT;
    printf("shm_get failed, so will create\n");
    if((db_shmid = shmget(key,ishmsize,ishmflg)) < 0) {
      printf("shm_get failed to create\n");
      sprintf(dd_log_msg," db_init: shmget creating %s \n",strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
  }
  
printf("db_shmid = %d\n",db_shmid);

  if ((idummy = (int) shmat(db_shmid, NULL, 0)) == -1) {
    sprintf(dd_log_msg,"db_init: shmat %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  db_ptr_shm = (int*)idummy;   
  db_shm_end = db_ptr_shm + ishmsize/4;
printf("db_ptr_shm = %u, db_shm_end = %u\n", db_ptr_shm, db_shm_end);

  return 0;
}


int db_close(int idestroy)
/* if idestroy, remove shared mem,
 * else, detach from shared mem.
 * CTCTCT
 */
{
  int i1,i2,isizpg,ishmsize;
  struct shmid_ds *buf;

  if (idestroy){
    ishmsize = 4*db_maxnbev* PAGE_SIZE * isizpg;
/*
  if (munpin(db_ptr_shm,ishmsize)){
    sprintf(dd_log_msg,"db_close: error in munpin %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
*/
    if ( ( shmctl(db_shmid,IPC_RMID, buf) ) < 0){ 
      sprintf(dd_log_msg,"db_close: shmctl %s \n",strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
  }
  else{
    if ( (shmdt((void *)db_ptr_shm)) < 0){
      sprintf(dd_log_msg," db_close: shmdt  %s \n",strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
  }
  return 0;
}

       
int db_getmem(int *size, int *shmid)
/*
 * Gets a piece of shared memory "on the fly" for 
 * events bigger than MAX_EVSIZE (in multiples of 512 bytes).
 * Input: size in longwords requested.
 * Output:size in longwords obtained and shmid.
 */
{
  int isizpg, ishmsize, ishmflg, id;
  int idummy;

 /* PAGE_SIZE = 128 longwords = 512 bytes,
  * allocate mem for an event in page-sized chunks,
  * isizpg    = size of event in pages
  * ishmsize  = total size of shared mem in bytes
  * ishmflg   = permission to read & write by owner, group, & world
  * IPC_CREAT | IPC_EXCL = creates a new entry for the key (IPC_PRIVATE)
  * only if the entry doesn't already exist. If an existing entry
  * is found, an error occurs, since the IPC channel already exists.
  * Specifying IPC_PRIVATE guarantees that a unique IPC channel is
  * created.
  * CTCTCT
  */
  isizpg =  *size/PAGE_SIZE + 1;
  *size = isizpg*PAGE_SIZE;
  ishmsize = 4*isizpg*PAGE_SIZE;
  ishmflg = 0666 | IPC_CREAT | IPC_EXCL;

  if( (id = shmget(IPC_PRIVATE,ishmsize,ishmflg)) < 0) {
    sprintf(dd_log_msg," db_getmem: shmget  %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  *shmid = id;

  return 0;
}

int db_attmem(int shmid, int **ptr2ptr)
/* attach to shared mem "shmid" CTCTCT */
{

  int idummy;
  if ((idummy = (int) shmat(shmid, NULL, 0)) == -1) {
    sprintf(dd_log_msg," db_attmem: shmat %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  *ptr2ptr = (int*)idummy; 
  return 0;
}

int db_set_shmuid(int shmid, uid_t new_uid)
/* set new user id for shared mem CTCTCT */
{
  struct shmid_ds shm_ds;
 
  if ( shmctl(shmid, IPC_STAT, &shm_ds) < 0 ){
	 sprintf(dd_log_msg," db_set_shmuid: shmctl1 %s \n",strerror(errno));
	 dd_log(DD_ERROR, dd_log_msg);
	 return -1;
  }
  shm_ds.shm_perm.uid = new_uid;
  if ( shmctl(shmid, IPC_SET, &shm_ds) < 0 ){
	 sprintf(dd_log_msg," db_set_shmuid: shmctl %s \n",strerror(errno));
	 dd_log(DD_ERROR, dd_log_msg);
	 return -1;
  }
  return 0;
}

int db_detmem(int *shmptr)
/* detach from shared mem CTCTCT */
{
  if ( (shmdt((void *)shmptr)) < 0){
    sprintf(dd_log_msg," db_detmem: error in shmdt  %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  return 0;
}


int db_freemem(int shmid)
/* remove or free shared mem CTCTCT */
{
  int status, uid;
  struct shmid_ds shm_ds, *shmbuf;

  shmbuf = &shm_ds;
  status = shmctl(shmid, IPC_STAT, shmbuf);
  if ( ( shmctl(shmid,IPC_RMID, shmbuf) ) < 0){ 
	 sprintf(dd_log_msg," db_freemem: shmctl %s \n",strerror(errno));
	 dd_log(DD_ERROR, dd_log_msg);
	 return -1;
  }
  return 0;
}
