#ifndef FIFO_H
#define FIFO_H


#ifdef _LANGUAGE_C
/*
 * fifo.h   :  Only header file for the fifos.
 * Author   :  C.Witzig
 * Date     :  Apr 2, 1992 
 * Mods     :
 *          :  Apr 3, 1995: added fifo system header and 
 *                    change the role of MAX_FIFO_NB from 
 *                    a define'ed queantity to a variable.
 *                    Keep a varialbe MAX_FIFO_NB_MAX for
 *                    array dimension etc...
 *
 */

#include <sys/ipc.h>

/* 
 * Fifo system header:
 */
#define FIFO_SYS_UNUSED 0
#define FIFO_SYS_INITIALISED 1
#define FIFO_SYS_ERROR 2



/*
 * size of the structures fifo_header and a_fifo (see below)
 * ----------------------------------------------------------
 * MAX_FIFO_NB   : maximum number of allowed fifos
 * FIFO_HDR_LEN  : length of the struct fifo_header
 * FIFO_HDR_CTL_LEN : number of control words per fifo event 
 *                    (length of the array ctl)
 * FIFO_CNTRL    : number of control words in fifo header 
 * FIFO_DEPTH    : number of entries per fifo
 * FIFO_ENLEN    : length of a fifo event
 *
 * ATTENTION     : These numbers are correlated with the size of the
 *                 global data buffer db (as defined in db.h) 
 */
#endif

#define MAX_FIFO_NB_MAX 10
#define FIFO_DEPTH_MAX 256

#define FIFO_SYS_NAME_LEN 100
/* change from 15 to 17  Feb 15, 1996 */
#define FIFO_HDR_LEN 40
#define FIFO_HDR_CTL_LEN 4
#define FIFO_CNTRL 3
#define FIFO_ENLEN 8
/* #define FIFO_NAME_LEN 7 */
#define FIFO_NAME_LEN 30
#define FIFO_TYPE_LEN 4


#ifdef _LANGUAGE_C
/*
 * Values for the fifo headers
 * ---------------------------
 */
#endif
#define FSTAT_UNUSED 0
#define FSTAT_IDLE   1
#define FSTAT_ACTIVE 2

#define FMODE_ALL   0
#define FMODE_COND  1
#define FMODE_ONREQ 2
#define FMODE_NULL  3
#define FMODE_NOPRESCALE 1
/*
 * another mode added for hallB
 * to enable wierd selection
 * criteria, CTCTCT
 */
#define FMODE_USER  4

/* whether the fifo can be used by one or several processes only*/
#define FMODE_SINGLE_USER 1
#define FMODE_MULTI_USER 0

#define FWAIT_SLEEP 0
#define FWAIT_ASYNC IPC_NOWAIT

#define FIFO_UNPROTECTED 0
#define FIFO_PROTECTED   1

#ifdef _LANGUAGE_C
/*
 * semaphore and shared memory keys (to be transferred later into dcom).
 */
#endif
#define FIFO_SEM_KEY 10
#define FIFO_SEM_KEY_LOCK 11
#define FIFO_SHM_KEY 5

#ifdef _LANGUAGE_C
/*
 * INPUT, TAPE, BROADCAST and GARBAGE fifo are the only STATIC fifos, i.e.
 * they do not disappear when dd is reinitialised or reset. Their
 * absolute position in the queue of the fifos is important and
 * should be garanteed (INPUT is the first fifo, as it is here that
 * all the processes find empty slots and TAPE should be closest
 * to it as it is the highest prioriy process attached to the dd.)
 */
#endif
#define INPUT_FIFO     0
#define GARBAGE_FIFO   fifo_sys_start->max_fifo_nb -2
#define BROADCAST_FIFO  fifo_sys_start->max_fifo_nb -1

#ifdef _LANGUAGE_C
/*
 * TYPEDEFs for the FIFOs:
 *------------------------
 * Define the structures for the fifos:
 * a_fifo has a header ( = fifo_header) and a body, which is an
 * array of fifo entries (fifo events), which is turn is naturally
 * another structure - the fifo_entry!
 */

/*
 * TYPEDEF for the FIFO HEADER:
 *-----------------------------
 * fhdr.fnumber   : number of the fifo (1 ... MAX_FIFO_NB).
 * fhdr.fname     : name of the fifo as chosen by the process.
 * fhdr.ftype     : STAtic or DYNamic, depending whether fifo dies with
 *                  its process.
 * fhdr.fstatus   : Status of the fifo (active/idle....).
 * fhdr.fmode     : fifo mode for readout.
 * fhdr.fctl..    : Control words.
 * fhdr.fnproc_att: number of processes attached to the fifo.
 * fhdr.wait_mode : sleeping or async.
 * fhdr.protection: protection
 * fhdr.fpid2sig  : for async wait mode.
 * fhdr.fpidcreat : pid of the process that created the fifo.
 */

typedef struct fifo_sys_header{
  char sys_name[FIFO_SYS_NAME_LEN];
  pid_t pid_creat;
  time_t time_creat;
  int   status;
  int max_fifo_nb;
  int fifo_depth;
}fifo_sys_header;

typedef struct fifo_header{
  int fnumber;
  char fname[FIFO_NAME_LEN];
  char ftype[FIFO_TYPE_LEN];
  int fstatus;
  int fmode;
  int fctlw1;
  int fctlb1;
  int fctlw2;
  int fctlb2;
  int fnproc_att;
  int fwait_mode;
  int fprotection;
  int freqcnt;
  int fprescale;
  int fsuser;
  int fpidcreat;
}fifo_header;

/*
 * A fifo_entry has the following parts:
 * fev.shmid: The shared memory id in which the event resides. 
 *            Is -1 if in the global data buffer db.
 * fev.dboff: Offset in longwords from the start of the shared
 *            memory.
 * fev.p2da:  Pointer to the data. As the shared memory segment
 *            might be attached at different addresses for 
 *            different processes it is actually calculated
 *            whenever a process gets it (calculated from 
 *            fev.shmid, fev.dboff and the start address of the
 *            shared memory segment.
 * fev.len:   Length of the data in longwords.
 * fev.ctlw1-ctlb2: Control words which are used to route the
 *            event through the fifos.
 */
#endif

#ifdef _LANGUAGE_C
typedef struct fifo_entry{
  int shmid;
  int dboff;
  int *p2da;
  int len;
  int ctlw1;
  int ctlb1;
  int ctlw2;
  int ctlb2;
}fifo_entry;
#else
      STRUCTURE /fifo_entry/
         INTEGER   shmid
         INTEGER   dboff
         INTEGER   p2da
         INTEGER   len
         INTEGER   ctlw1
         INTEGER   ctlb1
         INTEGER   ctlw2
         INTEGER   ctlb2
      END STRUCTURE
      RECORD /fifo_entry/ fev
#endif

#ifdef _LANGUAGE_C
typedef struct a_fifo{
  struct fifo_header fhdr;
  int cnt;			/* no. of events (fentries) CTCTCT */
  int wrptr;			/* this event is currently for writing CTCTCT */
  int rdptr;			/* this event is currently for reading CTCTCT */
  unsigned int noptry;
  unsigned int nopin;
  unsigned int nopout;
  struct fifo_entry fentry[FIFO_DEPTH_MAX];
}a_fifo;


/*
 * fifo_mode structure:
 */
typedef struct fifo_mode{
  int mode;
  int wait;
  int prescale;
  int suser;
  int *p2ctl;
} fifo_mode;


/* 
 * and finally the extern declarations
 */
extern int my_own_fifo;
extern int max_fifo_nb;
extern int fifo_depth;
extern a_fifo *fifo_shm_start;
extern fifo_sys_header *fifo_sys_start;

extern int fifo_mod_lock();
extern int fifo_mod_unlock();
extern int fifo_attach(char *filename, int nfifos, int fifo_depth);
extern int fifo_init(int which_fifo);
extern int fifo_close(int idestroy);
extern int fifo_make(char *fname, char *ftype, struct fifo_mode fmode);
extern int fifo_perm2empty(int which_fifo, int *perm);
extern int fifo_remove(int which_fifo);
extern int fifo_set_status(int which_fifo, int status);
extern int fifo_set_protection(int which_fifo, int protection);
extern int fifo_set_header(int which_fifo, struct fifo_header fhdr);
extern int fifo_set_reqcnt(int which_fifo, int cnt);
extern int fifo_get_header(int which_fifo, struct fifo_header *fhdr);
extern int fifo_get_asemcnt(int which_fifo, int *cnt);
extern int fifo_print_sys();
extern int fifo_print_header(int which_fifo);
extern int fifo_print(int which_fifo, int first, int last);
extern int fifo_sem_status();
extern int fifo_rate(int *fstatus, int *fcnt, int *fdevin, int *fdevout);
extern int lock_input_fifo();
extern int unlock_input_fifo();
extern int get_next_fifo(int which_fifo, struct fifo_entry);
extern int fifo_write(int iwhich_fifo, struct fifo_entry fev);
extern int fifo_read(int iwhich_fifo, struct fifo_entry *fev, int mode );
extern int fifo_exist(char *fifoname);
#endif

#endif
