/*
 * Library for the memory allocation system 
 */
#include "tcl.h"
#include "./mempart.h"
#ifdef VXWORKS
#include <memLib.h>
#endif

#define TCL_PROC(name) int name (void *object, Tcl_Interp *interp, int argc, char **argv)

/* global data */

static DALIST  partList; /* global part list */

libPartInit()
{
  listInit(&partList);
}

/*************************************************************
 * partCreate - create and initialize a memory part 
 */
#define maximum(a,b) (a<b ? b : a)

ROL_MEM_ID partCreate (name, size, c, incr)
     char *name;		/* name of new partition */
     int size;			/* size of a single item */
     int c;			/* initial number of items */
     int incr;			/* number of items to add when enlarging */
     
{
  ROL_MEM_ID pPart;
  int c_alloc = 0;
  
  pPart = (ROL_MEM_ID) ckalloc (sizeof(ROL_MEM_PART));
  bzero ((char *) pPart, sizeof(ROL_MEM_PART));
  if (pPart != NULL)
    {
      listInit (&(pPart->list));
      pPart->size = size + sizeof(DANODE);
      pPart->incr = 0; 
      pPart->total = 0;
      strcpy(pPart->name, name);
      if (name && strlen(name) == 0)
	pPart->name[0] = NULL;
      listAdd (&partList, (DANODE *)pPart);      
      if((partIncr (pPart, c)) != c)
	return (0);
    }
  return pPart;
}

TCL_PROC(partCreate_cmd) 
{
  char tmp[20];
  
  sprintf(tmp,"0x%08x",partCreate(argv[1],atoi(argv[2]),atoi(argv[3]),atoi(argv[4])));
  Tcl_AppendResult(interp,tmp,(char *) NULL);
  return TCL_OK;
}

/************************************************************** 
 * partFindByName - returns a pointer to a memory part
 *                  with the passed Name.
 */

ROL_MEM_ID partFindByName (char *name)
{
  ROL_MEM_ID	pPart;

  pPart = (ROL_MEM_ID) listFirst (&partList);
  
  while (pPart != NULL)
    {
      if (pPart->name && strcmp(pPart->name, name) == 0)break;
      pPart = (ROL_MEM_ID) listNext ((DANODE *)pPart);
    }
  return (pPart);
}


/************************************************************** 
 * partFree - frees all nodes for a given memory part
 *               and removes part from global part list. 
 */

void 
partFree(ROL_MEM_ID pPart)
{
  DANODE *the_node;

  if(pPart == NULL) return;
  printf("free list %s\n",pPart->name);
  
  if (pPart->incr == 1) {
    /* Free all buffers in the partition individually */
    while (pPart->list.c) {
      listGet(&(pPart->list),the_node);
      free(the_node);
      the_node = (DANODE *)0;
    }
    listSnip (&partList, (DANODE *)pPart);
  } else {
    /* Just need to Free the one contiguous block of data */
    listSnip (&partList, (DANODE *)pPart);      
    ckfree( *((char **) &pPart->part[0]));
  }
  ckfree(pPart);
  pPart = 0;
}


/************************************************************** 
 * partFreeAll - frees all memory parts in global part list
 *               and frees all nodes for a given list. 
 */

void 
partFreeAll()
{
  ROL_MEM_ID	pPart = (ROL_MEM_ID) 0;

  if (listCount(&partList)) {
    pPart = (ROL_MEM_ID) listFirst (&partList);    
    while (pPart != NULL)
      {
	partFree(pPart);
	pPart = (ROL_MEM_ID) listFirst (&partList);
      }
  }
}

TCL_PROC(free_all_cmd)
{
  partFreeAll();

  return TCL_OK;
}

/************************************************************** 
 * partIncr - increase part size by c items (min) 
 */

int 
partIncr ( ROL_MEM_ID pPart, int c)
{
/*      ROL_MEM_ID pPart      	partition to be enlarged */
/*      int  c		        min # of items to be added */

  register char *node;
  register long *block;
  unsigned bytes;
  int total_bytes, ii;
  int actual = c;		/* actual # of items added */

  pPart->total += c;

  if ((pPart == NULL)||(c == 0)) return (0);

  total_bytes =  c * pPart->size;

#ifdef VXWORKS
  if (memFindMax() <= total_bytes) {
    daLogMsg("WARN","partIncr: Cannot malloc a single memory Block for data");
    if (memFindMax() < total_bytes/4) {
      daLogMsg("ERROR","partIncr: Memory Fragmentation to severe to allocate buffers - Reboot CPU");
      return (-1);
    } else {
      pPart->incr = 1; /* Create a Fragmented memory Partition */
      while (actual--)
	{
	  if ( memFindMax() <= (pPart->size * actual)/4 ) {
	    printf("bufs %d max %d  needed %d \n",actual, memFindMax(),(pPart->size * (actual+1)));
	    daLogMsg("ERROR","Cannot complete buffer allocation - Reboot CPU");
	    return ((c - actual + 1));
	  }
	  block = (long *) malloc (pPart->size);
	  if (block == NULL)
	    return (-1);
	  bzero((char *) block, pPart->size);
	  ((DANODE *)block)->part = pPart; /* remember where we came from... */
	  listAdd (&pPart->list,(DANODE *)block);
	}
      return (c);
    }
  } else {
    block = (char *) ckalloc (total_bytes);
    if (block == NULL)
      return (-1);
    
    pPart->incr = 0;
    bzero((char *) block, c * pPart->size);
    node = (char *) block;
    *((char **) &pPart->part[0]) = node;

    while (actual--)
      {      
	((DANODE *)node)->part = pPart; /* remember where we came from... */
	listAdd (&pPart->list,(DANODE *)node);
	node += pPart->size;
      }
    return (c);
  }
#else
  block = (long *) ckalloc(total_bytes);
  if (block == NULL) {
    return (-1);
  }

  pPart->incr = 0;
  bzero((char *) block, c * pPart->size);

  node = (char *) block;
  *((char **) &pPart->part[0]) = node;

  while (actual--)
    {      
      ((DANODE *)node)->part = pPart; /* remember where we came from... */
      listAdd (&pPart->list,(DANODE *)node);
      node += pPart->size;
    }
  return (c);
#endif
}

/************************************************************** 
 * partReInit - Initiialize an existing partition 
 */

int 
partReInit (ROL_MEM_ID pPart)
{
  register char *node,*block;
  register DANODE *theNode;
  int actual;

  if (pPart == NULL) return;

  if (pPart->incr == 1) {   /* Does this partition have a Fragmented buffer list */

    /* Check if partition has buffers that do not belong to it
       and return them to their rightful owners */
    if ((pPart->total == 0) && (listCount(&pPart->list) > 0)) {
      while (listCount(&pPart->list) > 0) {
	listGet(&pPart->list,theNode);
	partFreeItem(theNode);
      }
    }
    
  } else {
    /* Cheat to initialize memory partition assuming buffers in one
       contiguous memory bloack */
    bzero(*((char **) &pPart->part[0]), pPart->total * pPart->size);
    
    node = *((char **) &pPart->part[0]);
    
    actual = pPart->total;

    pPart->list.f = pPart->list.l = (DANODE *) (pPart->list.c = 0);
 
    while (actual--)
      {      
	((DANODE *)node)->part = pPart; /* remember where we came from... */
	listAdd (&pPart->list,(DANODE *)node);
	node += pPart->size;
      }
  }
}

int 
partReInitAll()
{
  ROL_MEM_ID	pPart = (ROL_MEM_ID) 0;

  if (listCount(&partList)) {
    pPart = (ROL_MEM_ID) listFirst (&partList);    
    while (pPart != NULL)
      {
	partReInit(pPart);
	pPart = (ROL_MEM_ID) listNext ((DANODE *) pPart);
      }
  }
}

/*************************************************************** 
 * partHdr - Print headings for part statitistics printout 
 */

static void partHdr ()
     
{
  printf("Address    total  free   busy   size  incr  (KBytes)  Name\n");
  printf("-------    -----  ----   ----  -----  ----  --------  ----\n");
}

void TclPartHdr (interp)
     Tcl_Interp *interp;
{
  Tcl_AppendResult(interp,"Address    total  free   busy   size  incr  (KBytes)  Name\n", (char *) NULL);
  Tcl_AppendResult(interp,"-------    -----  ----   ----  -----  ----  --------  ----\n", (char *) NULL);
}


/*************************************************************** 
 * partPrint - Print statitistics for a single part 
 */

static void 
partPrint (pPart)
     ROL_MEM_ID	pPart;    
{
  int freen;

  printf("0x%08x  ",pPart);

  if (pPart != NULL)
    {
      freen = listCount (&pPart->list); 
      printf("%4d  %4d  %4d   %5d  %4d  (%d)       %s\n",
	    pPart->total,
	     freen,
	     pPart->total - freen,
	     pPart->size,
	     pPart->incr,
	     (((pPart->total * pPart->size) + 1023) / 1024),
	     pPart->name
	     );
    }
}

static void 
TclPartPrint (interp,pPart)
     Tcl_Interp *interp;
     ROL_MEM_ID	pPart;    
{
  int freen;
  char tmp[200];
  sprintf(tmp,"0x%08x  ",pPart);
  Tcl_AppendResult(interp,tmp,(char *) NULL);
  
  if (pPart != NULL)
    {
      freen = listCount (&pPart->list); 
      sprintf(tmp,"%4d  %4d  %5d  %5d  %4d  %8d  %s",
	      pPart->total,
	      freen,
	      pPart->total - freen,
	      pPart->size,
	      pPart->incr,
	      (((pPart->total * pPart->size) + 1023) / 1024),
	      pPart->name
	      );
      Tcl_AppendResult(interp,tmp,(char *) NULL);
    }

}


/**************************************************************** 
 * partStats - print statistics on a memory part 
 */

int partStats (pPart)
     ROL_MEM_ID	pPart;
     
{
  partHdr ();
  partPrint (pPart);
  return (0);
}

TCL_PROC(partStats_cmd)
{
  ROL_MEM_ID	pPart;
  
  if (argc != 2 )
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"",argv[0]," <partition name>\"", (char *) NULL);
      return TCL_ERROR ;
    }
  if ((pPart = partFindByName(argv[1])) == NULL ) 
    {
      Tcl_AppendResult(interp, "can't find ",argv[1],(char *) NULL);
      return TCL_ERROR;
    }
  Tcl_ResetResult(interp);
  TclPartHdr (interp);
  TclPartPrint (interp,pPart);
  Tcl_AppendResult(interp, "\n",(char *) NULL);
  return (TCL_OK);
}

/****************************************************************
 *
 * partStatsAll - print statistics on all parts
 *
 * Print int for all memory parts.
 *
 * int includes the total number of items in the part, the
 * number that are in-use and free, the size of a single item,
 * and the total kilobytes of raw memory allocated.
 */

int	partStatsAll ()
     
{
  int lockKey;
  ROL_MEM_ID  pPart;
  
  partHdr ();
#ifdef VXWORKS
  lockKey = intLock();
#endif
  pPart = (ROL_MEM_ID) listFirst (&partList);
  while (pPart != NULL)
    {
      partPrint (pPart);
      pPart = (ROL_MEM_ID) listNext ((DANODE *)pPart);
    }
#ifdef VXWORKS
  intUnlock(lockKey);
#endif
  return (0);
}

TCL_PROC(partStatsAll_cmd)
{
  int lockKey;
  ROL_MEM_ID  pPart;
  
  if (argc == 1)
    TclPartHdr (interp);
#ifdef VXWORKS
  lockKey = intLock();
#endif
  pPart = (ROL_MEM_ID) listFirst (&partList);
  while (pPart != NULL)
    {
      if (argc != 1)
	Tcl_AppendResult(interp, "{",(char *) NULL);
      TclPartPrint (interp,pPart);
      pPart = (ROL_MEM_ID) listNext ((DANODE *)pPart);
      if (argc == 1)
	Tcl_AppendResult(interp, "\n",(char *) NULL);
      else 
	Tcl_AppendResult(interp, "} ",(char *) NULL);
    }
#ifdef VXWORKS
  intUnlock(lockKey);
#endif
  return TCL_OK;
}

TCL_PROC(partReInitAll_cmd)
{
  int lockKey;
  ROL_MEM_ID  pPart;
  
  partReInitAll();

  return TCL_OK;
}

/****************************************************************
 * partPrintList - prints statisics for a given list structure
 */
int partPrintList(alist)
     DALIST *alist;
{
  DANODE *theNode;

  printf("dalist->f         %x\n",alist->f);
  printf("dalist->l         %x\n",alist->l);
  printf("dalist->c         %d\n",alist->c);

  theNode = listFirst(alist);
  while (theNode) {
    printf ("part %x prev %x self %x next %x left %d fd %d\n",
	    theNode->part,
	    theNode->p,
	    theNode,
	    theNode->n,
	    theNode->left,
	    theNode->fd);
    theNode = listNext(theNode);
  }
  return(0);
}

