/* BeginSourceFile fifo_code_lock_invariant.c */

#include <pthread.h>
#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#ifdef __sun
# include <thread.h>
#endif
#include "mq_code.h"
int
check_queue(mq_t mq)
{
	struct mq_elt *elt;

	if (mq->head == NULL && mq->tail == NULL)
		return (1);
	if (mq->head == NULL || mq->tail == NULL)
		return (0);
	if (mq->head->prev != NULL || mq->tail->next != NULL)
		return (0);
	for (elt = mq->tail; elt != mq->head; elt = elt->prev) {
		if (elt->prev->next != elt)
			return (0);
	}
	return (1);
}
/* EndSourceFile */
/* AddSourceFile fifo_code_lock_invariant.c */

/* the msg queue monitor lock */
pthread_mutex_t mq_lock = PTHREAD_MUTEX_INITIALIZER;
/*
 * Internal functions
 */

/*
 * Add a message queue element to the tail of the message queue
 */
static void
add_tail(mq_t mq, struct mq_elt *new)
{
	new->prev = mq->tail;
	new->next = NULL;
	if (mq->tail == NULL)
		mq->head = new;
	else
		mq->tail->next = new;
	mq->tail = new;
	mq->cnt++;
}
/*
 * Delete an element from the head of the message queue
 */
static struct mq_elt *
delete_head(mq_t mq)
{
	struct mq_elt *head;

	head = mq->head;

 	if (head->next == NULL) {
		/* last element on q */
		mq->tail = NULL;
	} else
		head->next->prev = NULL;
	mq->head = head->next;
	mq->cnt--;
	return (head);
}
/*
 * External functions
 */

/*
 * Create a new message queue.
 * Note: no locking is required because no other thread has a
 * pointer to the new message queue yet!
 */
mq_t
mq_new()
{
	mq_t mq;

	mq = (mq_t) malloc(sizeof (struct mq));
	mq->cnt = 0;
	mq->head = mq->tail = NULL;
	pthread_cond_init(&mq->notempty, NULL);
	return (mq);
}
void
mq_flush(mq_t mq)
{
	pthread_mutex_lock(&mq_lock);
	while (mq->head != NULL) {
	  if (mq->head->msg && (mq->head->msg != (void *)0xffffffff))
	    free(mq->head->msg);
	  
	  free(delete_head(mq));
	}
	mq->cnt = 0;
	pthread_mutex_unlock(&mq_lock);
}

/*
 * Put a new message at the tail of the queue.
 */
void
mq_put(mq_t mq, msg_t m)
{
	struct mq_elt *new;

	new = (struct mq_elt *) malloc(sizeof (struct mq_elt));
	new->msg = m;
	pthread_mutex_lock(&mq_lock);
	add_tail(mq, new);
	pthread_cond_signal(&mq->notempty);
	pthread_mutex_unlock(&mq_lock);
}
msg_t
mq_get(mq_t mq)
{
	struct mq_elt *old;
	msg_t m;

	pthread_mutex_lock(&mq_lock);
	assert(check_queue(mq));
	while ((old = mq->head) == NULL) {
		assert(check_queue(mq));
		pthread_cond_wait(&mq->notempty, &mq_lock);
		assert(check_queue(mq));
	}
	old = delete_head(mq);
	m = old->msg;
	assert(check_queue(mq));
	pthread_mutex_unlock(&mq_lock);
	free(old);
	return (m);
}

#define MQEND (msg_t)(-1)

#ifdef NEVER_DEF

mq_t mq;
pthread_t tid1;
pthread_t tid2;

main()
{
	extern void *source(void *);
	extern void *drain(void *);

	mq = mq_new();
#ifdef __sun
	thr_setconcurrency(2);
#endif
	pthread_create(&tid1, NULL, (void *(*)(void *))source, NULL);
	pthread_create(&tid2, NULL, (void *(*)(void *))drain, NULL);
	pthread_join(tid1, NULL);
	pthread_join(tid2, NULL);
	return (0);
}

#define NMSGS 50

void *
source(void *a)
{
	int i;

	for (i = 0; i < NMSGS; i++) {
		printf("source sending message: %d\n", i);
		mq_put(mq, (msg_t)i);
	}
	mq_put(mq, END);
	return (NULL);
}
void *
drain(void *a)
{
	msg_t j;

	while ((j = mq_get(mq)) != END)
		printf("drain read message: %d\n", (int)j);
	printf("drain reads END\n");
	return (NULL);
}
#endif

/* EndSourceFile */
