/******************************************************************************
*
*  sis3320Lib.c  -  Driver library for readout of Struck 3320 Flash ADC
*                  using a VxWorks 5.4 or later based Single Board computer. 
*
*  Author: David Abbott 
*          Jefferson Lab Data Acquisition Group
*          July 2006
*
*  Revision  1.0 - Initial Revision
*
*/

#include "vxWorks.h"
#include "stdio.h"
#include "string.h"
#include "logLib.h"
#include "taskLib.h"
#include "intLib.h"
#include "iv.h"
#include "semLib.h"
#include "vxLib.h"

/* Include ADC definitions */
#include "sis3320.h"

/* Define external Functions */
IMPORT  STATUS sysBusToLocalAdrs(int, char *, char **);
IMPORT  STATUS intDisconnect(int);
IMPORT  STATUS sysIntEnable(int);
IMPORT  STATUS sysIntDisable(int);

IMPORT  STATUS sysVmeDmaDone(int, int);
IMPORT  STATUS sysVmeDmaSend(UINT32, UINT32, int, BOOL);

/* Define Interrupts variables */
BOOL              s3320IntRunning  = FALSE;                    /* running flag */
int               s3320IntID       = -1;                       /* id number of ADC generating interrupts */
LOCAL VOIDFUNCPTR s3320IntRoutine  = NULL;                     /* user interrupt service routine */
LOCAL int         s3320IntArg      = 0;                        /* arg to user routine */
LOCAL int         s3320IntEvCount  = 0;                        /* Number of Events to generate Interrupt */
LOCAL UINT32      s3320IntLevel    = S3320_VME_INT_LEVEL;      /* default VME interrupt level */
LOCAL UINT32      s3320IntVec      = S3320_VME_INT_VEC;        /* default interrupt Vector */


/* Define global variables */
int Ns3320 = 0;                                               /* Number of ADCs in Crate */
int s3320A32Base   = 0x08000000;                              /* Minimum VME A32 Address for use by ADCs */
int s3320A32Offset = 0x08000000;                              /* Difference in CPU A32 Base - VME A32 Base */
volatile struct s3320_struct *s3320p[S3320_MAX_BOARDS];       /* pointers to ADC memory map */
int s3320IntCount = 0;                                        /* Count of interrupts from ADC */
int s3320EventCount[S3320_MAX_BOARDS];                        /* Count of Events taken by ADC (Event Count Register value) */
int s3320EvtReadCnt[S3320_MAX_BOARDS];                        /* Count of events read from specified ADC */
int s3320RunFlag = 0;
int *s3320HistData = 0;    /* Pointer to local memory for histogram data*/

SEM_ID s3320Sem;                               /* Semephore for Task syncronization */

/*******************************************************************************
*
* s3320Init - Initialize SIS 3320 Library. 
*
*
* RETURNS: OK, or ERROR if the address is invalid or board is not present.
*/

STATUS 
s3320Init (UINT32 addr, UINT32 addr_inc, int nadc)
{
  int ii, res, rdata, errFlag = 0;
  int boardID = 0;
  unsigned int laddr;

  
  /* Check for valid address */
  if(addr==0) {
    printf("s3320Init: ERROR: Must specify a Bus (VME-based A32) address for ADC 0\n");
    return(ERROR);
  }else if(addr < 0x00ffffff) { /* A24 Addressing */
    printf("s3320Init: ERROR: A24 Addressing not allowed for the SIS 3320\n");
    return(ERROR);
  }else{ /* A32 Addressing */
    if((addr_inc==0)||(nadc==0))
      nadc = 1; /* assume only one ADC to initialize */

    /* get the ADC address */
    res = sysBusToLocalAdrs(0x09,(char *)addr,(char **)&laddr);
    if (res != 0) {
      printf("s3320Init: ERROR in sysBusToLocalAdrs(0x09,0x%x,&laddr) \n",addr);
      return(ERROR);
    }
  }

  /* Set Max number of supported boards in a crate */
  if(nadc > S3320_MAX_BOARDS) {
    nadc = S3320_MAX_BOARDS;
    printf("s332Init: WARN: Only a maximum of %d ADC boards are supported\n",nadc);
  }

  Ns3320 = 0;
  for (ii=0;ii<nadc;ii++) {
    s3320p[ii] = (struct s3320_struct *)(laddr + ii*addr_inc);
    /* Check if Board exists at that address */
    res = vxMemProbe((char *) &(s3320p[ii]->id),VX_READ,4,(char *)&rdata);
    if(res < 0) {
      printf("s3320Init: ERROR: No addressable board at addr=0x%x\n",(UINT32) s3320p[ii]);
      s3320p[ii] = NULL;
      errFlag = 1;
      break;
    } else {
      /* Check if this is a Model 3320 */
      boardID =  rdata&S3320_BOARD_ID_MASK;
      if(boardID != S3320_BOARD_ID) {
	printf(" ERROR: Board ID does not match: %d \n",boardID);
	return(ERROR);
      }
    }
    Ns3320++;
    printf("Initialized ADC ID %d at address 0x%08x \n",ii,(UINT32) s3320p[ii]);
  }

  /* Calculate the A32 Offset for use in Block Transfers */
  res = sysBusToLocalAdrs(0x09,(char *)s3320A32Base,(char **)&laddr);
  if (res != 0) {
    printf("f1Init: ERROR in sysBusToLocalAdrs(0x09,0x%x,&laddr) \n",s3320A32Base);
    return(ERROR);
  } else {
    s3320A32Offset = laddr - s3320A32Base;
  }

  /* Initialize/Create Semephore */
  if(s3320Sem != 0) {
    semFlush(s3320Sem);
    semDelete(s3320Sem);
  }
  s3320Sem = semBCreate(SEM_Q_PRIORITY,SEM_EMPTY);
  if(s3320Sem <= 0) {
    printf("s3320Init: ERROR: Unable to create Binary Semephore\n");
    return(ERROR);
  }
  
  /* Disable/Clear all ADCs */
  for(ii=0;ii<Ns3320;ii++) {
    s3320p[ii]->reset = 1;
    s3320EventCount[ii] =  0;          /* Initialize the Event Count */
  }
  /* Initialize Interrupt variables */
  s3320IntID = -1;
  s3320IntRunning = FALSE;
  s3320IntLevel = S3320_VME_INT_LEVEL;
  s3320IntVec = S3320_VME_INT_VEC;
  s3320IntRoutine = NULL;
  s3320IntArg = 0;
  s3320IntEvCount = 0;
    

  if(errFlag > 0) {
    printf("s3320Init: ERROR: Unable to initialize all ADC Modules\n");
    if(Ns3320 > 0)
      printf("s3320Init: %d ADC(s) successfully initialized\n",Ns3320);
    return(ERROR);
  } else {
    return(OK);
  }
}


void
s3320Status (int id, int sflag)
{

  int ii;
  unsigned int sreg, iconfReg, clk, acqreg;
  int iLevel, iVec, iEnable, iMode;
  int sampling, armed;

  if((id<0) || (s3320p[id] == NULL)) {
    printf("s3320Status: ERROR : ADC id %d not initialized \n",id);
    return;
  }


  sreg   = s3320p[id]->csr;
  acqreg = s3320p[id]->acqCsr;
  sampling = acqreg&S3320_SAMPLING;
  armed    = acqreg&S3320_ARMED;

  iconfReg = s3320p[id]->intConfig;
  iLevel = (iconfReg&S3320_INT_LEVEL_MASK)>>8;
  iVec = (iconfReg&S3320_INT_VEC_MASK);
  iEnable = (iconfReg&S3320_INT_ENABLE_MASK);
  iMode = (iconfReg&S3320_INT_MODE_MASK)>>12;

  clk =   (acqreg&S3320_CLOCKSOURCE_MASK)>>12;


  printf("\nSTATUS for ADC id %d at base address 0x%x \n",id,(UINT32) s3320p[id]);
  printf("------------------------------------------------ \n");

  if(iEnable) {
    printf(" Interrupts Enabled  - Mode = %d  (0:RORA 1:ROAK)\n",iMode);
    printf(" VME Interrupt Level: %d   Vector: 0x%x \n",iLevel,iVec);
  } else {
    printf(" Interrupts Disabled - Mode = %d  (0:RORA 1:ROAK)\n",iMode);
    printf(" VME Interrupt Level: %d   Vector: 0x%x \n",iLevel,iVec);
  }
  printf("\n");
  printf("  MODULE ID   register = 0x%08x \n",s3320p[id]->id);
  printf("\n");

  printf("  Clock Source         = %s\n",s3320_clksrc_string[clk]);
  printf("  STATUS      register = 0x%08x \n",sreg);
  printf("  INT CONTROL register = 0x%08x \n",s3320p[id]->intControl);
  if(sampling)
    printf("  ACQ CONTROL register = 0x%08x (SAMPLING)\n",acqreg);
  else if(armed)
    printf("  ACQ CONTROL register = 0x%08x (ARMED)\n",acqreg);
  else
    printf("  ACQ CONTROL register = 0x%08x \n",acqreg);

  printf("  MAX Event Count      = %d \n", s3320p[id]->maxNofEvents);
  printf("  Actual Event Count   = %d \n", s3320p[id]->actualNofEvents);
  printf("\n");
  printf("  External Start Delay = %d clocks\n",s3320p[id]->extStartDelay);
  printf("  External Stop  Delay = %d clocks\n",s3320p[id]->extStopDelay);


  printf("\n  Event Configuration Information \n");
  printf("  ------------------------------- \n");
  for (ii=0;ii<4;ii++) {
    printf("  Group %d  ADC %d,%d\n",(ii),(2*ii),(2*ii+1));
    printf("     Event Config   register = 0x%08x\n",s3320p[id]->adcG[ii].eventConfig);
    printf("     Sample Length  register = 0x%08x\n",s3320p[id]->adcG[ii].sampleLength);
    printf("     Sample Start   register = 0x%08x\n",s3320p[id]->adcG[ii].sampleStart);
    printf("     Input Mode     register = 0x%08x\n",s3320p[id]->adcG[ii].adcInputMode);
    printf("     Address Indexes:             ADC %d       ADC %d\n",(2*ii),(2*ii+1));
    printf("           Next Address      = 0x%08x  0x%08x\n",
	   s3320p[id]->adcG[ii].nextSampleAdr[0],s3320p[id]->adcG[ii].nextSampleAdr[1]);
  }


}

/******************************************************
 *
 *   Interrupt related functions
 *
 *
 */


/* Specify the Interrupt Level, Vector, and/or Source */
int
s3320IntSet(int id, int iLevel, int iVec, int iSource)
{
  int iVal=0;

  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320IntSet: ERROR : SIS3320 id %d not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  /* If interrupts are already enabled - don't touch anything */
  if((s3320p[id]->intConfig)&S3320_INT_ENABLE_MASK) {
    logMsg("s3320IntSet: ERROR : Interrupts are already enabled\n",0,0,0,0,0,0);
    return(ERROR);
  }

  if(iLevel)
    iVal = (iLevel<<8);
  else
    iVal = (s3320IntLevel<<8);

  if(iVec)
    iVal |= (iVec&0xff);
  else
    iVal |= s3320IntVec;

  s3320p[id]->intConfig = iVal;

  if(iSource) 
    s3320p[id]->intControl = (iSource&0xf);

  s3320IntID = id;

  return(iVal);
}

void
s3320IntAck(int intMask)
{

  if(s3320IntID<0) {
    logMsg("s3320IntAck: ERROR : Interrupt Module ID not set\n",0,0,0,0,0,0);
    return;
  }

  /* Clear Source */
 s3320p[s3320IntID]->intControl = (intMask<<20);

  /* Enable interrupts */
  S3320_IRQ_ENABLE(s3320IntID);

}


void 
s3320Int (void)
{
  
  /* Disable interrupts */
  sysIntDisable(s3320IntLevel);
  S3320_IRQ_DISABLE(s3320IntID);

  s3320IntCount++;
 
  if (s3320IntRoutine != NULL)  {     /* call user routine */
    (*s3320IntRoutine) (s3320IntArg);
  }else{
    if((s3320IntID<0) || (s3320p[s3320IntID] == NULL)) {
      logMsg("s3320Int: ERROR : SIS3320 id %d not initialized \n",s3320IntID,0,0,0,0,0);
      return;
    }
  }

  /* Acknowledge interrupt - re-enable */
  S3320_IRQ_ENABLE(s3320IntID);

  /* Re-Enable interrupts on CPU */
  sysIntEnable(s3320IntLevel);

}


STATUS 
s3320IntConnect (int id, VOIDFUNCPTR routine, int arg, UINT16 level, UINT16 vector)
{

  if(s3320IntRunning) {
    printf("s3320IntConnect: ERROR : Interrupts already Initialized for SIS3320 id %d\n",
	   s3320IntID);
    return(ERROR);
  }

  s3320IntRoutine = routine;
  s3320IntArg = arg;

  /* Check for user defined VME interrupt level and vector */
  if(level == 0) {
    s3320IntLevel = S3320_VME_INT_LEVEL; /* use default */
  }else if (level > 7) {
    printf("s3320IntConnect: ERROR: Invalid VME interrupt level (%d). Must be (1-7)\n",level);
    return(ERROR);
  } else {
    s3320IntLevel = level;
  }

  if(vector == 0) {
    s3320IntVec = S3320_VME_INT_VEC;  /* use default */
  }else if ((vector < 32)||(vector>255)) {
    printf("s3320IntConnect: ERROR: Invalid interrupt vector (%d). Must be (32<vector<255)\n",vector);
    return(ERROR);
  }else{
    s3320IntVec = (vector&0xf0) + 1;
  }
      
  /* Connect the ISR */
#ifdef VXWORKSPPC
  if((intDisconnect((int)INUM_TO_IVEC(s3320IntVec)) != 0)) {
    printf("s3320IntConnect: ERROR disconnecting Interrupt\n");
    return(ERROR);
  }
#endif
  if((intConnect(INUM_TO_IVEC(s3320IntVec),s3320Int,s3320IntArg)) != 0) {
    printf("s3320IntConnect: ERROR in intConnect()\n");
    return(ERROR);
  }


  s3320IntSet(id, s3320IntLevel, s3320IntVec, 0);
  return (OK);
}



void
s3320IntEnable(int intMask)
{

  if(s3320IntID<0) {
    logMsg("s3320IntEnable: ERROR : Interrupt Module ID not set\n",0,0,0,0,0,0);
    return;
  }

  if(s3320IntRunning) {
    logMsg("s3320IntEnable: ERROR : Interrupts already Enabled for SIS3320 id %d\n",
	   s3320IntID,0,0,0,0,0);
    return;
  }

  sysIntDisable(s3320IntLevel);


  if(intMask == 0) { /* Check if any sources are enabled */
    if( ((s3320p[s3320IntID]->intControl)&S3320_INT_SOURCE_ENABLE_MASK) == 0) {
      logMsg("s3320IntEnable: ERROR : No Interrupt sources are specified\n",0,0,0,0,0,0);
      return;
    }
  } else {
    /* Enable Interrupts */
    s3320p[s3320IntID]->intControl = (intMask&0xf);
  }


  S3320_IRQ_ENABLE(s3320IntID);
  s3320IntRunning = 1;
  sysIntEnable(s3320IntLevel);

  return;
}

void
s3320RORA()
{
  int temp;

  if(s3320IntID<0) {
    logMsg("s3320RORA: ERROR : Interrupt Module ID not set\n",0,0,0,0,0,0);
    return;
  }

  if(s3320IntRunning) {
    logMsg("s3320RORA: ERROR : Interrupts Enabled for SIS3320 id %d\n",
	   s3320IntID,0,0,0,0,0);
    return;
  }

  temp = s3320p[s3320IntID]->intConfig;
  temp = temp&(~S3320_INT_MODE_MASK);
  s3320p[s3320IntID]->intConfig = temp;

  return;
}

void
s3320ROAK()
{

  if(s3320IntID<0) {
    logMsg("s3320ROAK: ERROR : Interrupt Module ID not set\n",0,0,0,0,0,0);
    return;
  }

  if(s3320IntRunning) {
    logMsg("s3320ROAK: ERROR : Interrupts Enabled for SIS3320 id %d\n",
	   s3320IntID,0,0,0,0,0);
    return;
  }

  s3320p[s3320IntID]->intConfig |= S3320_INT_MODE_MASK;

  return;
}


void
s3320IntDisable(int intMask)
{

  if(s3320IntID<0) {
    logMsg("s3320IntEnable: ERROR : Interrupt Module ID not set\n",0,0,0,0,0,0);
    return;
  }

  sysIntDisable(s3320IntLevel);
  S3320_IRQ_DISABLE(s3320IntID);
  s3320IntRunning = 0;

  if(intMask) {
    s3320p[s3320IntID]->csr = (intMask&0xf)<<16;
  }

  return;

}



/* Declare Key Address functions */

void
s3320Reset(int id)
{
  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320Reset: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return;
  }
  s3320p[id]->reset = 1;
}

void
s3320Clear(int id)
{
  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320Clear: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return;
  }

    s3320p[id]->stopSample = 1;
    s3320p[id]->disarmSample = 1;
    s3320p[id]->resetMem = 1;
    s3320p[id]->eventConfig = 0;
}


void
s3320Start(int id)
{
  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320Start: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return;
  }
  s3320p[id]->startSample = 1;
}

void
s3320Stop(int id)
{
  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320Stop: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return;
  }
  s3320p[id]->stopSample = 1;
}

void
s3320Arm(int id)
{
  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320Arm: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return;
  }
  s3320p[id]->armSample = 1;
}

void
s3320Disarm(int id)
{
  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320Disarm: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return;
  }
  s3320p[id]->disarmSample = 1;
}

void
s3320Disable(int id)
{
  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320Disable: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return;
  }

    s3320p[id]->disarmSample = 1;
}

int
s3320Dready(int id)
{
  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320Dready: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return (ERROR);
  }

  return ((int) s3320p[id]->actualNofEvents);
}



/* Acquisition/Mode Functions */

void
s3320Enable(int id, int maxev, int size)
{

  int nsamp=0;

  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320Enable: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return;
  }
  
  if(maxev <= 0) maxev = 1;
  if(maxev > S3320_MAX_EVENTS) maxev = S3320_MAX_EVENTS;

  if((size <=0)||(size>11)) size = 10; /* 128 Samples Default*/
  nsamp = s3320_pagesize[size];

  s3320Disable(id);

  s3320p[id]->maxNofEvents = maxev;
  s3320p[id]->eventConfig = (S3320_PAGE_WRAP_ENABLE | size);
  s3320p[id]->sampleLength = (nsamp-4)&S3320_SAMPLE_LENGTH_MASK;

  if(maxev > 1) {
    logMsg("Enabling ADC %d for %d Max Events and %d sample size\n",id,maxev,nsamp,0,0,0);
    s3320p[id]->extStopDelay = 5;
    s3320p[id]->acqCsr = S3320_ENABLE_FP_STARTSTOP_MODE |
                         S3320_ENABLE_MULTIEVENT |
                         S3320_ENABLE_AUTOSTART;
    s3320Arm(id);

  } else {
    logMsg("Enabling ADC %d for Single Event and %d sample size\n",id,nsamp,0,0,0,0);
    s3320p[id]->acqCsr = S3320_ENABLE_FP_STARTSTOP_MODE;

    s3320Arm(id);
    s3320Start(id);  

  }

}


int
s3320StopDelay(int id, unsigned long delay)
{
  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320StopDelay: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  /* Delay should be passed in clocks */

  s3320p[id]->extStopDelay = delay;

  return((int)(s3320p[id]->extStopDelay));
}

int
s3320StartDelay(int id, unsigned long delay)
{
  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320StartDelay: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  /* Delay should be passed in clocks */

  s3320p[id]->extStartDelay = delay;

  return((int)(s3320p[id]->extStartDelay));
}
 
unsigned int
s3320AcqConfig(int id, unsigned long val)
{
  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320AcqConfig: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return(0xffffffff);
  }

  /* If val is 0 just return current value. Writing a 0 does nothing  */
  if(val)
    s3320p[id]->acqCsr = val;

  return(s3320p[id]->acqCsr);
}

void
s3320EventConfigAll(int id, unsigned long val)
{
  if((id<0) || (s3320p[id] == NULL)) {
    printf("s3320EventConfigAll: ERROR : ADC id %d not initialized \n",id);
    return;
  }

  /* Check to make sure we are not Armed or Sampling */
  if((S3320_IS_ARMED(id))||(S3320_IS_SAMPLING(id))) {
    printf("s3320EventConfigAll: ERROR: Disable ADCs before Configuring!\n");
    return;
  }

  s3320p[id]->eventConfig = val;
}



   /* Readout Functions */

#define S3320_MAX_PRINT_VALS    256


int
s3320PrintEvent(int id, int adc, int evt, int pflag)
{

  int ii;
  int grp, pagesize, tmp;
  int aEvt, ndata, e_offset=0;
  int data, d0, d1;


  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320PrintEvent: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  if((adc<0)||(adc>7)) {
    printf("s3320PrintEvent: ERROR: adc value %d out of range (0-7)\n",adc);
    return(ERROR);
  }

  aEvt = s3320p[id]->actualNofEvents;
  if(aEvt<evt) {
    printf("s3320PrintEvent: ERROR: event value %d out of range (! < %d)\n",evt,aEvt);
    return(ERROR);
  }

  grp = adc>>1;
  tmp= (s3320p[id]->adcG[grp].eventConfig)&S3320_PAGESIZE_MASK;
  pagesize = s3320_pagesize[tmp];
  e_offset = evt*(pagesize>>1);  /* offset in 4 byte words */

  if((pagesize>>1)>S3320_MAX_PRINT_VALS) {
    ndata = S3320_MAX_PRINT_VALS;
    printf("\n  Event data for ADC %d, Event %d, Samples: %d (first %d values):\n",
	   adc,evt,pagesize,ndata);
  }else{
    ndata = pagesize>>1; /* two samples per 4 byte word */
    printf("\n  Event data for ADC %d, Event %d, Samples: %d :\n",adc,evt,pagesize);
  }
  for(ii=0;ii<ndata;ii++) {
    data = s3320p[id]->adcData[adc][ii+e_offset];
    d1 = (data&S3320_ADC_EDATA_MASK)>>16;
    d0 = (data&S3320_ADC_ODATA_MASK);
    if(pflag) {
      if ((ii % 8) == 0) printf("\n    ");
      printf(" 0x%08x",data);
    }else{
      if ((ii % 8) == 0) printf("\n    ");
      printf(" 0x%04x 0x%04x",d0,d1);
    }
  }
  printf("\n");

  printf("\n");

  return(pagesize);

}


int
s3320PrintEventDir(int id, int adc)
{

  int ii;
  int grp, pagesize, tmp;
  int nevents;
  int t1, full, endAddr;


  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320PrintEventDir: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  if((adc<0)||(adc>7)) {
    printf("s3320PrintEventDir: ERROR: adc value %d out of range (0-7)\n",adc);
    return(ERROR);
  }

  grp = adc>>1;
  tmp= (s3320p[id]->adcG[grp].eventConfig)&S3320_PAGESIZE_MASK;
  pagesize = s3320_pagesize[tmp];
  nevents = s3320p[id]->actualNofEvents;

  printf("\n  Event directory data for ADC %d, nevents %d:\n",adc,nevents);

  for(ii=0;ii<nevents;ii++) {
    tmp = s3320p[id]->adcG[grp].eventDir[adc%2][ii];
    endAddr =  tmp&S3320_EDIR_NSAMP_ADDR_MASK;
    full    = (tmp&S3320_EDIR_WRAP)>>28;
    t1      = (tmp&S3320_EDIR_TRIG)>>29;

    printf("      Event %4d: End Addr = 0x%08x   Wrap=%d  Trig=%d \n",ii,endAddr,full,t1);

  }
  printf("\n");

  return(nevents);
}

int
s3320ReadEvent(int id, UINT32 *data, int adc, int evt, int rflag)
{

  int ii, retVal, dummy=0;
  int grp, pagesize, tmp;
  int aEvt, ndata, e_offset=0;
  volatile unsigned int *laddr;
  unsigned int vmeAdr;


  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320ReadEvent: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  if((adc<0)||(adc>7)) {
    logMsg("s3320ReadEvent: ERROR: adc value %d out of range (0-7)\n",adc,0,0,0,0,0);
    return(ERROR);
  }

  aEvt = s3320p[id]->actualNofEvents;
  if(aEvt<evt) {
    logMsg("s3320ReadEvent: ERROR: event value %d out of range (! < %d)\n",evt,aEvt,0,0,0,0);
    return(ERROR);
  }

  /* Get the Event Page size for wordcount */
  grp = adc>>1;
  tmp= (s3320p[id]->adcG[grp].eventConfig)&S3320_PAGESIZE_MASK;
  pagesize = s3320_pagesize[tmp];
  ndata = pagesize>>1;  /* Number of 4 byte words to transfer */
  e_offset = evt*ndata;  /* offset in 4 byte words */


  if(rflag > 0) { /* Reserved for Block Read mode */
    laddr = data; /* set default destination address */

    switch(rflag) {
    case 4:  /* 2eSST */
    case 3:  /* 2eVME */
    case 2:  /* MBLT */
      /* Check for 8 byte boundary for address - insert dummy word - Necessary for Universe chip*/
      if((unsigned long) (data)&0x7) {
	*data = S3320_DUMMY_DATA;
	laddr = (data + 1);
	dummy = 1;
      } else {
	laddr = data;
	dummy = 0;
      }
    default: /* BLK32 */
      vmeAdr = (unsigned int)(&(s3320p[id]->adcData[adc][e_offset])) - s3320A32Offset;
      retVal = sysVmeDmaSend((UINT32)laddr, vmeAdr, (ndata<<2), 0);
      if(retVal != 0) {
	logMsg("s3320ReadEvent: ERROR in DMA transfer Initialization 0x%x\n",retVal,0,0,0,0,0);
	return(retVal);
      }
      /* Wait until Done or Error */
      retVal = sysVmeDmaDone(10000,1);
      if(retVal<0) {
	logMsg("s3320ReadEvent: ERROR in DMA transfer 0x%x\n",retVal,0,0,0,0,0);
	return(ERROR);
      }
    }

  } else { /* Programmed I/O */
    dummy = 0;
    for(ii=0;ii<ndata;ii++) {
      data[ii] = s3320p[id]->adcData[adc][ii+e_offset];
    }

  }

  return((ndata+dummy));
}

int
s3320ReadEventDir(int id, volatile UINT32 *data, int adc, int evt, int rflag)
{

  int grp, aEvt;
  

  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320ReadEventDir: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  if((adc<0)||(adc>7)) {
    logMsg("s3320ReadEventDir: ERROR: adc value %d out of range (0-7)\n",adc,0,0,0,0,0);
    return(ERROR);
  }

  aEvt = s3320p[id]->actualNofEvents;
  if(aEvt<evt) {
    logMsg("s3320ReadEvent: ERROR: event value %d out of range (! < %d)\n",evt,aEvt,0,0,0,0);
    return(ERROR);
  }

  grp = adc>>1;
  *data   = s3320p[id]->adcG[grp].eventDir[adc%2][evt];

  return(OK);
}

UINT32
s3320ReadNextSampleAddr(int id, int adc)
{

  int grp;
  unsigned int val;
  
  if((id<0) || (s3320p[id] == NULL)) {
    logMsg("s3320ReadNextSampleAddr: ERROR : ADC id %d not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  if((adc<0)||(adc>7)) {
    logMsg("s3320ReadReadNextSampleAddr: ERROR: adc value %d out of range (0-7)\n",adc,0,0,0,0,0);
    return(ERROR);
  }

  grp = adc>>1;
  val = s3320p[id]->adcG[grp].nextSampleAdr[adc%2];

  return(val);
}


#define DSWAP(x)        ((((x) & 0x0000ffff) << 16) | \
                         (((x) & 0xffff0000) >> 16))


void
testswap(int count, UINT32 *data) {

  int ii;
  
  for(ii=0;ii<count;ii++) {
    data[ii] = DSWAP(data[ii]);
  }
}


void
s3320Run(int id, int evsize)
{

  int ii, jj, nbytes, totalbytes;
  unsigned int *data;


  if((id<0) || (s3320p[id] == NULL)) {
    printf("s3320Run: ERROR : ADC id %d not initialized \n",id);
    return;
  }

  if((evsize<0)||(evsize > 11)) {
    printf("s3320Run: WARN : Event Page size not in range (0-11) \n");
    printf("                 Set to default 128 Samples   (10)\n");
    evsize = 10;
  }

  /* Get Nbytes needed to store an event = 8 adcs * 2 bytes/sample * pagesize */
  nbytes     = 2*s3320_pagesize[evsize];
  totalbytes = S3320_MAX_ADC_CHANNELS*nbytes;

  if(s3320HistData == 0) {
    s3320HistData = malloc(totalbytes);
  } else {
    free(s3320HistData);
    s3320HistData = malloc(totalbytes);
  }
  printf("s3320Run: Allocated %d bytes for sample data at address 0x%x\n",totalbytes, (UINT32) s3320HistData);
  bzero((char *)s3320HistData,totalbytes);  /* Zero data storage */


  s3320Reset(id);            /* Reset */
  s3320AcqConfig(id,0x110);  /* Program for Single Event, AutoStart, External Stop, 200 MHz */
  s3320EventConfigAll(id,((0x10)|evsize));  /* Program for Wrap mode, user specified pagesize */
  
  s3320RunFlag = 1;          

  while (s3320RunFlag) {

    s3320Arm(id); 
    s3320Start(id);          /* Start Sampling */
    
    while(S3320_IS_SAMPLING(id)) {
      if(s3320RunFlag == 0) 
	break;
    }      

    if(s3320RunFlag) {   /* Readout and histogram the event */

      for (ii=0;ii<S3320_MAX_ADC_CHANNELS;ii++) { /* loop over all ADCs */
	data = (unsigned int *)&s3320HistData[ii*(nbytes>>2)];
	s3320ReadEvent(id,data,ii,0,0);  /* readout all ADC channels ,event 0 */

	/* Need to reorder data here to get samples in proper big endian order */
	for(jj=0;jj<(nbytes>>2);jj++) {
	  data[jj] = DSWAP(data[jj]);
	}
      }

    }

    /* Sleep for half a second before we check again */
    taskDelay(30);
  }


  printf("s3320Run: WARN: Exiting...\n");

}
