/*----------------------------------------------------------------------------*
 *  Copyright (c) 1991, 1992  Southeastern Universities Research Association, *
 *                            Continuous Electron Beam Accelerator Facility   *
 *                                                                            *
 *    This software was developed under a United States Government license    *
 *    described in the NOTICE file included as part of this distribution.     *
 *                                                                            *
 * TJNAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606 *
 *       heyes@cebaf.gov   Tel: (804) 249-7030    Fax: (804) 249-7363         *
 *----------------------------------------------------------------------------*
 * Description: follows this header.
 *
 * Author:
 *	David Abbott
 *	TJNAF Data Acquisition Group
 *
 * Revision History:
 *	  Initial revision
 *
 *
 *----------------------------------------------------------------------------*/

 /* tirLib.c -- Primitive trigger control for VME CPUs using the TJNAF 
               Trigger Supervisor interface card

 File : tirLib.c

------------------------------------------------------------------------------*/

#include <vxWorks.h>
#include <stdio.h>
#include <string.h>
#include <logLib.h>
#include <taskLib.h>
#include <intLib.h>
#include <iv.h>


/* Define TIR Memory structure */
struct vme_tir {
    unsigned short tir_csr;
    unsigned short tir_vec;
    unsigned short tir_dat;
    unsigned short tir_oport;
    unsigned short tir_iport;
  };


/* Define variable for TIR board revision */
static int tirVersion = 0; /* Default is an invalid Version ID */
struct vme_tir *tirPtr;

extern  int sysBusToLocalAdrs(int, char *, char **);
extern  int intDisconnect(int);
extern  int sysIntEnable(int);

/* Define TIR Version IDs (readback of a reset CSR register */
#define TIR_VERSION_MASK 0x00ff
#define TIR_VERSION_1    0x00c0
#define TIR_VERSION_2    0x0080

/* Define TIR Modes of operation:    Ext trigger - Interrupt mode   0
                                     TS  trigger - Interrupt mode   1
                                     Ext trigger - polling  mode    2 
                                     TS  trigger - polling  mode    3  */
#define TIR_EXT_INT    0
#define TIR_TS_INT     1
#define TIR_EXT_POLL   2
#define TIR_TS_POLL    3

#define TIR_EXTERNAL       0x1
#define TIR_ENABLED        0x2
#define TIR_INTERRUPT      0x4
#define TIR_TEST_MODE      0x8
#define TIR_LEVEL_MASK   0x700
#define TIR_INT_PENDING 0x4000
#define TIR_TRIG_STATUS 0x8000


/* Define default Interrupt vector and address for possible sources */
#define TIR_DEFAULT_ADDR 0x0ed0
#define TIR_INT_VEC      0xec


LOCAL BOOL        tirIntRunning  = FALSE;	      /* running flag */
LOCAL VOIDFUNCPTR tirIntRoutine  = NULL;	      /* user interrupt service routine */
LOCAL int         tirIntArg      = 0;	              /* arg to user routine */
LOCAL UINT32      tirIntLevel    = 0;                 /* VME Interrupt Level 1-7 */
LOCAL UINT32      tirIntVec      = TIR_INT_VEC;       /* default interrupt Vector */
LOCAL int         tirDoAck       = 1;                 /* Default - Acknowledge trigger */

/* Globals */
unsigned int      tirIntMode     = 0;
unsigned int      tirIntCount    = 0;
unsigned int      tirSyncFlag    = 0;
unsigned int      tirLateFail    = 0;



/*******************************************************************************
*
* tirInt - default interrupt handler
*
* This rountine handles the tir interrupt.  A user routine is
* called, if one was connected by tirIntConnect().
*
* RETURNS: N/A
*
* SEE ALSO: tirIntConnect()
*/

LOCAL void 
tirInt (void)
{
  tirIntCount++;
  
  if (tirIntRoutine != NULL)	/* call user routine */
    (*tirIntRoutine) (tirIntArg);


  /* Acknowledge trigger */
  if(tirDoAck)
    tirPtr->tir_dat = 0x8000;
}

/* Example User routine */
void
tirIntUser(int arg)
{
  unsigned short tirdata;

  tirPtr->tir_oport = 0x0001;
  tirdata = tirPtr->tir_oport;
  tirPtr->tir_oport = 0x0000;

  return;
}



STATUS
tirIntInit(unsigned int tAddr, unsigned int mode, int force)
{
  int stat = 0, active = 0;
  unsigned long laddr;
  unsigned short rval;

  if (tAddr == 0) {
    tAddr = TIR_DEFAULT_ADDR;
  }

  tirIntCount = 0;
  tirDoAck = 1;

  stat = sysBusToLocalAdrs(0x29,(char *)tAddr,(char **)&laddr);
  if (stat != 0) {
     printf("tirInit: ERROR: Error in sysBusToLocalAdrs res=%d \n",stat);
  } else {
     printf("TIR address = 0x%x\n",laddr);
  }

  /* Check if TIR board is readable */
  stat = vxMemProbe((char *)laddr,0,2,(char *)&rval);
  if (stat != 0) {
    printf("tirInit: ERROR: TIR card not addressable\n");
    return(-1);
  } else {
    if(force == 0) { /* Check if the TIR is active */
      if(rval&TIR_ENABLED) {
	printf("WARN: TIR is currently enabled (set the Force Reset Flag to override)\n");
	return(-1);
      }
    }
    /* Set Up pointer */
    tirPtr = (struct vme_tir *)laddr;
    tirIntRunning = 0;
    tirPtr->tir_csr = 0x80; /* Reset the board */
    rval = (tirPtr->tir_csr)&TIR_VERSION_MASK;
    if(rval == TIR_VERSION_1) {
      tirVersion = 1;
    }else if(rval == TIR_VERSION_2) {
      tirVersion = 2;
    }else{
      tirVersion = 0;
      printf("tirInit: ERROR: Invalid TIR Version register= 0x%04x\n",rval);
      return(-1);
    }
  }

  /* Check on Mode of operations */
  if(mode > 3) {
    printf("tirInit: WARN: Invalid mode id (%d) - Default to External Interrupts\n",mode);
    tirIntMode = TIR_EXT_INT;
  }else{
    tirIntMode = mode;
    switch(tirIntMode) {
    case TIR_EXT_INT:
      printf("TIR (version %d) setup for External Interrupts\n",tirVersion);
      break;
    case TIR_TS_INT:
      printf("TIR (version %d) setup for TS Interrupts\n",tirVersion);
      break;
    case TIR_EXT_POLL:
      printf("TIR (version %d) setup for External polling\n",tirVersion);
      break;
    case TIR_TS_POLL:
      printf("TIR (version %d) setup for TS polling\n",tirVersion);

    }
  }


  return(tirVersion);
  
}


/*******************************************************************************
*
* tirIntConnect - connect a user routine to the TIR interrupt
*
* This routine specifies the user interrupt routine to be called at each
* interrupt
*
* RETURNS: OK, or ERROR .
*/

STATUS 
tirIntConnect ( unsigned int vector, VOIDFUNCPTR routine, int arg)
{

  if(tirPtr == NULL) {
    printf("tirIntConnect: ERROR: TIR not initialized\n");
    return(ERROR);
  }


  /* Disconnect any current interrupts */
  if((intDisconnect(tirIntVec) !=0))
     printf("tirIntConnect: Error disconnecting Interrupt\n");

  tirIntCount = 0;
  tirDoAck = 1;

  /* Set Vector and Level */
  if((vector < 255)&&(vector > 64)) {
    tirIntVec = vector;
  }else{
    tirIntVec = TIR_INT_VEC;
  }
  tirIntLevel = ((tirPtr->tir_csr)&TIR_LEVEL_MASK)>>8;
  printf("tirIntConnect: INFO: Int Vector = 0x%x  Level = %d\n",tirIntVec,tirIntLevel);


  switch (tirIntMode) {
  case TIR_TS_POLL:
  case TIR_EXT_POLL:

       tirPtr->tir_csr = 0x80;
       tirPtr->tir_vec = tirIntVec;

    break;
  case TIR_TS_INT:
  case TIR_EXT_INT:

       tirPtr->tir_csr = 0x80;
       tirPtr->tir_vec = tirIntVec;
       intConnect(INUM_TO_IVEC(tirIntVec),tirInt,0);

    break;
  default:
    printf(" tirIntConnect: ERROR: TIR Mode not defined %d\n",tirIntMode);
    return(ERROR);
  }

  if(routine) {
    tirIntRoutine = routine;
    tirIntArg = arg;
  }else{
    tirIntRoutine = NULL;
    tirIntArg = 0;
  }

  return(OK);
}


void 
tirIntDisconnect()
{
  if(tirPtr == NULL) {
    printf("tirIntDisconnect: ERROR: TIR not initialized\n");
    return;
  }

  if(tirIntRunning) {
    printf("tirIntDisconnect: ERROR: TIR is Enabled - Call tirIntDisable() first\n");
    return;
  }

  /* Reset TIR */
  tirPtr->tir_csr = 0x80;

  /* Disconnect any current interrupts */
  sysIntDisable(tirIntLevel);
  if((intDisconnect(tirIntVec) !=0))
     printf("tirIntConnect: Error disconnecting Interrupt\n");

  return;
}




int
tirIntEnable(int iflag)
{
 int lock_key=0;

  if(tirPtr == NULL) {
    printf("tirIntEnable: ERROR: TIR not initialized\n");
    return(-1);
  }

  if(iflag)
    tirIntCount = 0;

  tirIntRunning = 1;
  tirDoAck      = 1;

  switch (tirIntMode) {
  case TIR_TS_POLL:

    tirPtr->tir_csr = TIR_ENABLED;

    break;
  case TIR_EXT_POLL:

    tirPtr->tir_csr = (TIR_ENABLED | TIR_EXTERNAL);

    break;
  case TIR_TS_INT:

    lock_key = intLock();
    sysIntEnable(tirIntLevel);
    tirPtr->tir_csr = (TIR_ENABLED | TIR_INTERRUPT);

    break;
  case TIR_EXT_INT:

    lock_key = intLock();
    sysIntEnable(tirIntLevel);
    tirPtr->tir_csr = (TIR_ENABLED | TIR_EXTERNAL | TIR_INTERRUPT);

    break;
  default:
    tirIntRunning = 0;
    if(lock_key)
      intUnlock(lock_key);
    printf(" tirIntError: ERROR: TIR Mode not defined %d\n",tirIntMode);
    return(ERROR);
  }


  if(lock_key)
    intUnlock(lock_key);

  return(OK);

}

void 
tirIntDisable()
{

  if(tirPtr == NULL) {
    printf("tirIntDisable: ERROR: TIR not initialized\n");
    return;
  }

  tirPtr->tir_csr = 0;
  tirIntRunning = 0;
}

void 
tirIntReset()
{
  if(tirPtr == NULL) {
    printf("tirIntReset: ERROR: TIR not initialized\n");
    return;
  }

  tirPtr->tir_csr = 0x80;
  tirIntRunning = 0;

}

void 
tirIntAck()
{
  if(tirPtr == NULL) {
    logMsg("tirIntAck: ERROR: TIR not initialized\n",0,0,0,0,0,0);
    return;
  }
  
  tirDoAck = 1;
  tirPtr->tir_dat = 0x8000;
}

void
tirIntPause()
{
  tirDoAck = 0;
}

void
tirIntResume()
{
  tirDoAck = 1;
}


unsigned int
tirIntType()
{
  unsigned short reg;
  unsigned long tt;

  if(tirPtr == NULL) {
    logMsg("tirIntType: ERROR: TIR not initialized\n",0,0,0,0,0,0);
    return(0);
  }
  
  reg = tirPtr->tir_dat;

  if((tirIntMode == TIR_EXT_POLL)||(tirIntMode == TIR_EXT_INT)) {
    if(tirVersion == 1)
      tt = (reg&0x3f);
    else
      tt = (reg&0xfff);

    tirSyncFlag = 0;
    tirLateFail = 0;

  } else if((tirIntMode == TIR_TS_POLL)||(tirIntMode == TIR_TS_INT)) {
    if(tirVersion == 1)
      tt = ((reg&0x3c)>>2);
    else
      tt = ((reg&0xfc)>>2);

    tirSyncFlag = reg&0x01;
    tirLateFail = (reg&0x02)>>1;

  }

  return(tt);
}

int 
tirIntPoll()
{
  unsigned short sval=0;
  unsigned int   lval=0;

  sval = tirPtr->tir_csr;
  if( (sval != 0xffff) && ((sval&0x8000) != 0) ) {
    return (1);
  } else {
    return (0);
  }

}

void
tirIntOutput(unsigned short out)
{
  if(tirPtr == NULL) {
    logMsg("tirIntOutput: ERROR: TIR not initialized\n",0,0,0,0,0,0);
    return;
  }
  
  tirPtr->tir_oport = out;

}


void
tirIntStatus()
{
  unsigned short csr, ivec, data, out, inp;
  int enabled, int_level, ext, test, latch, int_pending, int_mode;

  if(tirPtr == NULL) {
    printf("tirIntStatus: ERROR: TIR not initialized\n");
    return;
  }

  /* Read TIR Registers */
  csr  = tirPtr->tir_csr;
  ivec = tirPtr->tir_vec;
  data = tirPtr->tir_dat;
  out  = tirPtr->tir_oport;
  inp  = tirPtr->tir_iport;

  enabled     = csr&TIR_ENABLED;
  int_level   = (csr&TIR_LEVEL_MASK)>>8;
  ext         = csr&TIR_EXTERNAL;
  int_mode    = csr&TIR_INTERRUPT;
  test        = csr&TIR_TEST_MODE;
  latch       = csr&TIR_TRIG_STATUS;
  int_pending = csr&TIR_INT_PENDING;
  
  /* print out status info */

  printf("STATUS for TIR at base address 0x%x \n",(UINT32) tirPtr);
  printf("----------------------------------------- \n");
  printf("Trigger Count: %d\n",tirIntCount);
  printf("Registers: \n");
  printf("  CSR    = 0x%04x (VME Int Level: %d)\n",csr,int_level);
  printf("  VEC    = 0x%04x \n",ivec);
  printf("  DATA   = 0x%04x \n",data);
  printf("  OUTPUT = 0x%04x \n",out);
  printf("  INPUT  = 0x%04x \n",inp);
  printf("\n");
  if(test)
    printf("Test Mode Enabled\n");

  if(enabled) {
    if(latch)
      printf("State     : Enabled (trigger latched!)\n");
    else
      printf("State     : Enabled\n");

    if(ext)
      printf("Source    : External\n");
    else
      printf("Source    : TS\n");

    if(int_mode) {
      if(int_pending)
	printf("Interrupts: Enabled (pending)\n");
      else
	printf("Interrupts: Enabled\n");
    }
  }else{
    printf("State : Disabled\n");
  }

  printf("\n");

}
