/******************************************************************************
*
*  mollerLib.c  -  Library extension for JLAB FADC with moller readout firmware.
*                  Using a VxWorks 5.4 or later based Single Board computer. 
*
*  Author: David Abbott 
*          Jefferson Lab Data Acquisition Group
*          July 2009
*
*  Revision  1.0 - Initial Revision
*
*/


#define FAMOL_BOARD_REV    0xfadc00a0
#define FAMOL_BOARD_MASK   0xffff00f0

#define FAMOL_NUM_SCALERS  8

#define FAMOL_ENABLE_DATA_INT        0x00080000
#define FAMOL_ENABLE_HELICITY_TEST   0x80000000

/* Define Control 2 Bits */
#define FAMOL_CNTL_GO                     0x1
#define FAMOL_CNTL_GO_MOLLER              0x2
#define FAMOL_FORCE_PAUSE                 0x4
#define FAMOL_CNTL_ENABLE_INT_TRIGGER     0x8
#define FAMOL_CNTL_ENABLED                0xb
#define FAMOL_CNTL_ENABLE_MASK            0xb


#define FAMOL_RESET_MOLFIFO        0x00000400
#define FAMOL_RESET_SCALERS        0x00004000
#define FAMOL_RESET_SCALFIFO       0x00008000

#define FAMOL_SCAL_DATA_AVAILABLE  0x00008000
#define FAMOL_SCAL_DATA_INPROG     0x00004000
#define FAMOL_SCAL_FIFO_EMPTY      0x00000200
#define FAMOL_SCAL_FIFO_FULL       0x00000100
#define FAMOL_SCAL_WORDCNT_MASK    0x000000ff

#define FAMOL_SAMPLE_CNT_MASK      0x0000003f
#define FAMOL_SUM_THRESH_MASK      0x000fffff
#define FAMOL_PULSE_WIDTH_MASK     0x0000ffff
#define FAMOL_DELAY_MASK           0x000000ff
#define FAMOL_PRESCALE_MASK        0x000007ff
#define FAMOL_TRIG_WINDOW_MASK     0x000000ff


/* Define Detector IDs */
#define FAMOL_CAL  0
#define FAMOL_S1   1
#define FAMOL_S2   2
#define FAMOL_S3   3
#define FAMOL_S4   4
#define FAMOL_SLSR 5

#define FAMOL_LEFT   0
#define FAMOL_RIGHT  1
#define FAMOL_LR     2



int
famInit (UINT32 addr, int iFlag)
{
  int ii, res, rdata, errFlag = 0;
  int nadc = 1;
  int boardID = 0;
  int maxSlot = 1;
  int minSlot = 21;
  int trigSrc, clkSrc, srSrc;
  unsigned int laddr, a32addr, a16addr=0;
  volatile struct fadc_struct *fa;
  unsigned short saddr, sdata;


  /* Check if we have already Initialized boards before */
  if((fadcInited>0) && (fadcID[0] != 0)) {
    /* Hard Reset of all FADC boards in the Crate */
    for(ii=0;ii<nfadc;ii++) {
      FAp[fadcID[ii]]->csr  = FA_CSR_HARD_RESET;
    }
    taskDelay(5);
  }
  
  /* Check for valid address */
  if(addr==0) {
    printf("famInit: ERROR: Must specify a Bus (VME-based A24) address for FADC 0\n");
    return(ERROR);
  }else if(addr > 0x00ffffff) { /* A24 Addressing */
    printf("famInit: ERROR: A32 Addressing not allowed for FADC configuration space\n");
    return(ERROR);
  }else{ /* A24 Addressing */
    /* get the FADC address */
    res = sysBusToLocalAdrs(0x39,(char *)addr,(char **)&laddr);
    if (res != 0) {
      printf("famInit: ERROR in sysBusToLocalAdrs(0x39,0x%x,&laddr) \n",addr);
      return(ERROR);
    }
  }

  /* Init Some Global variables */
  fadcSource = iFlag&FA_SOURCE_MASK;
  fadcInited = nfadc = 0;
  fadcUseSDC = 0;
  bzero((char *)fadcChanDisable,sizeof(fadcChanDisable));
  bzero((char *)fadcID,sizeof(fadcID));

  fa = (struct fadc_struct *)(laddr);
  /* Check if Board exists at that address */
  res = vxMemProbe((char *) &(fa->version),VX_READ,4,(char *)&rdata);
  if(res < 0) {
    printf("famInit: ERROR: No addressable board at addr=0x%x\n",(UINT32) fa);
    errFlag = 1;
  } else {
    /* Check that it is a MOLLER FADC board */
    if((rdata&FAMOL_BOARD_MASK) != FAMOL_BOARD_REV) {
      printf(" ERROR: Invalid Board ID: 0x%x\n",rdata);
      return(ERROR);
    }
    /* Check if this is board has a valid slot number */
    boardID =  ((fa->intr)&FA_SLOT_ID_MASK)>>16;
    if((boardID <= 0)||(boardID >21)) {
      printf(" ERROR: Board Slot ID is not in range: %d\n",boardID);
      return(ERROR);
    }
    FAp[boardID] = (struct fadc_struct *)(laddr);
    fadcRev[boardID] = rdata&FA_VERSION_MASK;
  }
  fadcID[nfadc] = boardID;
  if(boardID >= maxSlot) maxSlot = boardID;
  if(boardID <= minSlot) minSlot = boardID;

  nfadc++;
  printf("Initialized MOLLER FADC in Slot # %d at address 0x%08x \n",fadcID[0],(UINT32) FAp[(fadcID[0])]);


  /* Check if we are using a JLAB FADC Signal Distribution Card (SDC) */
  a16addr = iFlag&FA_SDC_ADR_MASK;
  if(a16addr) {
    res = sysBusToLocalAdrs(0x29,(char *)a16addr,(char **)&laddr);
    if (res != 0) {
      printf("famInit: ERROR in sysBusToLocalAdrs(0x29,0x%x,&laddr) \n",a16addr);
      return(ERROR);
    }

    res = vxMemProbe((char *) laddr,VX_READ,2,(char *)&sdata);
    if(res < 0) {
      printf("famInit: ERROR: No addressable SDC board at addr=0x%x\n",(UINT32) laddr);
    } else {
      FASDCp = (struct fadc_sdc_struct *) laddr;
      FASDCp->ctrl = FASDC_CSR_INIT;   /* Reset the Module */

      printf("Using JLAB FADC Signal Distribution Module at address 0x%x\n",
	       (UINT32) FASDCp); 
      printf("famInit: JLAB FADC SDC Used only as a level converter\n");
      fadcUseSDC=0;
      faSDC_Config(2,1);
    }
  }

  /* Hard Reset of Moller FADC board */
  FAp[fadcID[0]]->csr  = FA_CSR_HARD_RESET;
  taskDelay(5);

  /* Initialize Interrupt variables */
  fadcIntID = -1;
  fadcIntRunning = FALSE;
  fadcIntLevel = FA_VME_INT_LEVEL;
  fadcIntVec = FA_VME_INT_VEC;
  fadcIntRoutine = NULL;
  fadcIntArg = 0;

  /* Calculate the A32 Offset for use in Block Transfers */
  res = sysBusToLocalAdrs(0x09,(char *)fadcA32Base,(char **)&laddr);
  if (res != 0) {
    printf("famInit: ERROR in sysBusToLocalAdrs(0x09,0x%x,&laddr) \n",fadcA32Base);
    return(ERROR);
  } else {
    fadcA32Offset = laddr - fadcA32Base;
  }


  /* Establish the Trigger Sync Reset and Clock sources 
       CLOCK = Local
       TRIG  = Internal
       SYNC  = VME (software);  */
  clkSrc  = FA_REF_CLK_INTERNAL;
  trigSrc = FA_TRIG_INTERNAL;
  srSrc   = FA_SRESET_VME | FA_ENABLE_SOFT_SRESET;

  /* Enable Clock source - Internal Clk enabled by default */ 
  FAp[fadcID[0]]->ctrl1 = (clkSrc | FA_ENABLE_INTERNAL_CLK) ;
  taskDelay(20);

  /* Reset FPGAs */
  FAp[fadcID[0]]->reset = (FA_RESET_ADC_FPGA1 | FA_RESET_ADC_FPGA2 | FA_RESET_HITSUM_FPGA |
			    FA_RESET_ADC_FIFO1 | FA_RESET_ADC_FIFO2 | FA_RESET_HITSUM_FIFO |
			    FAMOL_RESET_MOLFIFO | FAMOL_RESET_SCALERS | FAMOL_RESET_SCALFIFO);
  taskDelay(5);

  /* Write configuration registers with default/defined Sources */
  /* Program an A32 access address for this FADC's FIFO */
  a32addr = fadcA32Base;
  res = sysBusToLocalAdrs(0x09,(char *)a32addr,(char **)&laddr);
  if (res != 0) {
    printf("famInit: ERROR in sysBusToLocalAdrs(0x09,0x%x,&laddr) \n",a32addr);
    return(ERROR);
  }
  FAp[fadcID[0]]->adr32 = (a32addr>>16) + 1;  /* Write the register and enable */
  FApd[fadcID[0]] = (unsigned int *)(laddr);  /* Set a pointer to the FIFO */

  /* Set Default Block Level to 1 */
  FAp[fadcID[0]]->blk_level = 1;
  fadcBlockLevel=1;

  /* Setup Trigger and Sync Reset sources */
  FAp[fadcID[0]]->ctrl1 |= (srSrc | trigSrc | FA_ENABLE_CHAN_0_7 | FA_ENABLE_CHAN_8_15);
  

  fadcInited = nfadc;
  if(errFlag > 0) {
    printf("famInit: ERROR: Unable to initialize all FADC Modules\n");
    if(nfadc > 0)
      printf("famInit: %d FADC(s) successfully initialized\n",nfadc );
    return(ERROR);
  } else {
    return(OK);
  }
}



int
famSetSumThresh(int det, unsigned int lr, unsigned int val)
{
  int id=fadcID[0];
  unsigned int v1,v2;

  if(FAp[id] == NULL) {
    printf("famSetSumThresh: ERROR : ADC in slot %d is not initialized \n",id);
    return(ERROR);
  }
  if(det!=FAMOL_CAL) {
    printf("famSetSumThresh: ERROR : Invalid Detector ID (Calorimeter (0) only) \n");
    return(ERROR);
  }
  if(lr>=FAMOL_LR) {
    printf("famSetSumThresh: ERROR : Left(0)/Right(1) id out of range \n");
    return(ERROR);
  }
  if(val>FAMOL_SUM_THRESH_MASK) {
    printf("famSetSumThresh: ERROR : Sum Threshold out of range (>0xfffff)\n");
    return(ERROR);
  }

  v1 = val&0x0000ffff;
  v2 = (val>>16)&0x0000000f;

  if(lr==FAMOL_LEFT) {                /* Left Cal */
    FAp[id]->CL_sum_thresh[0] = v1;
    FAp[id]->CL_sum_thresh[1] = v2;
  }else{                            /* Right Cal */
    FAp[id]->CR_sum_thresh[0] = v1;
    FAp[id]->CR_sum_thresh[1] = v2;
  }
     
  return(OK);
}

int
famSetSampleCount(int det, unsigned int lr, unsigned int val)
{
  int id=fadcID[0];

  if(FAp[id] == NULL) {
    printf("famSetSampleCount: ERROR : ADC in slot %d is not initialized \n",id);
    return(ERROR);
  }
  if(det!=FAMOL_CAL) {
    printf("famSetSampleCount: ERROR : Invalid Detector ID (Calorimeter (0) only) \n");
    return(ERROR);
  }
  if(lr>=FAMOL_LR) {
    printf("famSetSampleCount: ERROR : Left(0)/Right(1) id out of range \n");
    return(ERROR);
  }
  if(val>FAMOL_SAMPLE_CNT_MASK) {
    printf("famSetSampleCount: ERROR : Sample Count out of range (3<=sc<=63)\n");
    return(ERROR);
  }

  if(lr==FAMOL_LEFT) {                /* Left Cal */
    FAp[id]->CL_sample_cnt = (val&FAMOL_SAMPLE_CNT_MASK);
  }else{                            /* Right Cal */
    FAp[id]->CR_sample_cnt = (val&FAMOL_SAMPLE_CNT_MASK);
  }
     
  return(OK);
}

int
famSetPrescale(int det, unsigned int lr, unsigned int val)
{
  int id=fadcID[0];

  if(FAp[id] == NULL) {
    printf("famSetPrescale: ERROR : ADC in slot %d is not initialized \n",id);
    return(ERROR);
  }
  if(det!=FAMOL_CAL) {
    printf("famSetPrescale: ERROR : Invalid Detector ID (Calorimeter (0) only) \n");
    return(ERROR);
  }
  if(lr>FAMOL_LR) {
    printf("famSetPrescale: ERROR : Left(0)/Right(1)/LR(2) id out of range \n");
    return(ERROR);
  }
  if(val>FAMOL_PRESCALE_MASK) {
    printf("famSetPrescale: ERROR : Prescale factor out of range (1<ps<2047)\n");
    return(ERROR);
  }

  if(lr==FAMOL_LEFT) {                /* Left Cal */
    FAp[id]->CL_trig_prescale = (val&FAMOL_PRESCALE_MASK);
  }else if(lr==FAMOL_RIGHT) {          /* Right Cal */
    FAp[id]->CR_trig_prescale = (val&FAMOL_PRESCALE_MASK);
  }else {
    FAp[id]->CLCR_trig_prescale = (val&FAMOL_PRESCALE_MASK);
  }
     
  return(OK);
}



int
famSetPulseWidth(int det, unsigned int lr, unsigned int val)
{
  int id=fadcID[0];

  if(FAp[id] == NULL) {
    printf("famSetPulseWidth: ERROR : ADC in slot %d is not initialized \n",id);
    return(ERROR);
  }
  if(det>FAMOL_S4) {
    printf("famSetPulseWidth: ERROR : Invalid Detector ID (Cal(0),Scint(1,2,3,4)) \n");
    return(ERROR);
  }
  if(lr>=FAMOL_LR) {
    printf("famSetPulseWidth: ERROR : Left(0)/Right(1) id out of range \n");
    return(ERROR);
  }
  if(val>FAMOL_PULSE_WIDTH_MASK) {
    printf("famSetPulseWidth: ERROR : Pulse Width out of range (1<=pw<=65535)\n");
    return(ERROR);
  }

  printf("%d %d %d\n",det,lr,val);

  if(lr==FAMOL_LEFT) {                /* Left */
    switch(det) {
    case FAMOL_CAL:
      FAp[id]->CL_pulse_width = (val&FAMOL_PULSE_WIDTH_MASK);
      break;
    case FAMOL_S1:
    case FAMOL_S2:
    case FAMOL_S3:
    case FAMOL_S4:
      FAp[id]->SL_pulse_width[(det-1)] = (val&FAMOL_PULSE_WIDTH_MASK);
      break;
    }
  }else{                            /* Right */
    switch(det) {
    case FAMOL_CAL:
      FAp[id]->CR_pulse_width = (val&FAMOL_PULSE_WIDTH_MASK);
      break;
    case FAMOL_S1:
    case FAMOL_S2:
    case FAMOL_S3:
    case FAMOL_S4:
      FAp[id]->SR_pulse_width[(det-1)] = (val&FAMOL_PULSE_WIDTH_MASK);
      break;
    }
  }
     
  return(OK);
}


int
famSetDelay(int det, unsigned int lr, unsigned int val)
{
  int id=fadcID[0];

  if(FAp[id] == NULL) {
    printf("famSetDelay: ERROR : ADC in slot %d is not initialized \n",id);
    return(ERROR);
  }
  if(det>FAMOL_SLSR) {
    printf("famSetDelay: ERROR : Invalid Detector ID (Cal(0),Scint(1,2,3,4),SL*SR(5)) \n");
    return(ERROR);
  }
  if(lr>FAMOL_LR) {
    printf("famSetDelay: ERROR : Left(0)/Right(1)/LR(2) id out of range \n");
    return(ERROR);
  }
  if(val>FAMOL_DELAY_MASK) {
    printf("famSetDelay: ERROR : Delay out of range (1<=del<=255)\n");
    return(ERROR);
  }

  if(lr==FAMOL_LEFT) {                /* Left */
    switch(det) {
    case FAMOL_CAL:
      FAp[id]->CL_delay = (val&FAMOL_DELAY_MASK);
      break;
    case FAMOL_S1:
    case FAMOL_S2:
    case FAMOL_S3:
    case FAMOL_S4:
      FAp[id]->SL_delay[(det-1)] = (val&FAMOL_DELAY_MASK);
      break;
    }
  }else if(lr==FAMOL_RIGHT) {                            /* Right */
    switch(det) {
    case FAMOL_CAL:
      FAp[id]->CR_delay = (val&FAMOL_DELAY_MASK);
      break;
    case FAMOL_S1:
    case FAMOL_S2:
    case FAMOL_S3:
    case FAMOL_S4:
      FAp[id]->SR_delay[(det-1)] = (val&FAMOL_DELAY_MASK);
      break;
    }
  } else {
    if(det==FAMOL_SLSR) {
      FAp[id]->SLSR_delay = (val&FAMOL_DELAY_MASK);
    } else {
      printf("famSetDelay: ERROR: Detector ID invalid for Coincidence\n");
      return(ERROR);
    }
  }
     
  return(OK);
}

int
famSetTrigWindow(unsigned int val)
{
  int id=fadcID[0];

  if(FAp[id] == NULL) {
    printf("famSetTrigWindow: ERROR : ADC in slot %d is not initialized \n",id);
    return(ERROR);
  }
  if(val>FAMOL_TRIG_WINDOW_MASK) {
    printf("famSetTrigWindow: ERROR : Calorimeter Trigger Window out of range (>255)\n");
    return(ERROR);
  }

  FAp[id]->moller_trig_window = val&FAMOL_TRIG_WINDOW_MASK;
     
  return(OK);
}


void
famEnable()
{
  int id=fadcID[0];
  unsigned int rval1, rval2;

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("famEnable: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }

  /* Load Internal Trigger Burst Parameters */
  FAp[id]->reset = FA_RESET_ITRIG_BURST;

  rval1 = FAp[id]->ctrl1;
  rval2 = FAp[id]->ctrl2;

  /* Enable All Channels and Moller Interrupts */
  FAp[id]->ctrl1 |= (FA_ENABLE_CHAN_0_7 | FA_ENABLE_CHAN_8_15 | FAMOL_ENABLE_DATA_INT);

  /* Enable FIFOs / Moller Acq and Internal Triggers */
  FAp[id]->ctrl2 = (FAMOL_CNTL_GO | FAMOL_CNTL_GO_MOLLER | FAMOL_CNTL_ENABLE_INT_TRIGGER) ;
}

void
famDisable(int dFlag)
{
  int id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("famDisable: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }

  if(dFlag)
    FAp[id]->ctrl2 = FAMOL_CNTL_GO | FAMOL_FORCE_PAUSE ;   /* Force Moller End */
  else
    FAp[id]->ctrl2 = FAMOL_CNTL_GO;
}


void
famEnableTest()
{
  int id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("famEnableTest: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }

  FAp[id]->ctrl1 |= FAMOL_ENABLE_HELICITY_TEST;

}

void
famDisableTest(int dFlag)
{
  int id=fadcID[0];
  unsigned int reg=0;

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("famDisableTest: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }

  reg = (FAp[id]->ctrl1);

  FAp[id]->ctrl1 = (reg&(~FAMOL_ENABLE_HELICITY_TEST));

}


void
famReset()
{
  int id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("famReset: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }
  
  FAp[id]->reset = (FAMOL_RESET_MOLFIFO | FAMOL_RESET_SCALERS | FAMOL_RESET_SCALFIFO);
}

void
famMollerDataAck()
{
  int id=fadcID[0];

  if((id<=0) || (id>21) || (FApd[id] == NULL)) {
    logMsg("famMollerDataAck: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return;
  }

  /* Write to A32 Data Address to Acknowledge the Interrupt */
  *FApd[id] = 0x80000000;
}




int
famReadScalers(volatile unsigned int *data)
{
  int ii, ntrig, nread=0;
  int jj, maxtrys = 20;   /* each try about 1 microsec */
  unsigned int status=0;
  int id=fadcID[0];

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("famReadScalers: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

  /* Wait for Data available */
  for(jj=0;jj<maxtrys;jj++) {
    status = FAp[id]->moller_scal_status;
    if(status&FAMOL_SCAL_DATA_INPROG) {
      continue;  
    }
    break;
  }

  if(status&FAMOL_SCAL_DATA_AVAILABLE) {
    ntrig = ((status&FAMOL_SCAL_WORDCNT_MASK))/(FAMOL_NUM_SCALERS);
    if(ntrig>=1) {
      nread = FAMOL_NUM_SCALERS*ntrig;
      for(ii=0;ii<nread;ii++) {
	data[ii] = FAp[id]->moller_scal_fifo;
      }
    }else{
      logMsg("famReadScalers: ERROR: invalid word count (status = 0x%x)\n",status,0,0,0,0,0);
      return(ERROR);
    }
  }else{
    if(status&FAMOL_SCAL_DATA_INPROG) {
      return(0);
    } else {
      return(ERROR);
    }
  }

  return(nread);
}

unsigned int
famScalStatus(int sFlag)
{
  int id=fadcID[0];
  unsigned int sreg=0;
  int nwrds;

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    logMsg("famScalStatus: ERROR : ADC in slot %d is not initialized \n",id,0,0,0,0,0);
    return(0);
  }

  sreg = FAp[id]->moller_scal_status;
  nwrds = sreg&FAMOL_SCAL_WORDCNT_MASK;

  if(sFlag) {
    printf("Moller Scaler Status Reg = 0x%08x\n",sreg);
    if(sreg&FAMOL_SCAL_FIFO_FULL) {
      printf("  Scaler FIFO FULL  (Num or Words = %d)\n",nwrds);
    }else if(sreg&FAMOL_SCAL_FIFO_EMPTY) {
      printf("  Scaler FIFO EMPTY\n");
    }else{
      if(sreg&FAMOL_SCAL_DATA_AVAILABLE) 
	printf("  Scaler Data Available (Num of Words = %d)\n",nwrds);
    }       
  }
  
  return(sreg);
}


int
faMollerStatus(int sFlag)
{
  int ii;
  int id=fadcID[0];
  unsigned int r1, r2, r3, r4, rev;
  unsigned int scL, scR, stL, stR, psL, psR, psLR, cw;
  unsigned int pwL[5], pwR[5], dyL[5], dyR[5], dyLR;

  if((id<=0) || (id>21) || (FAp[id] == NULL)) {
    printf("faMollerStatus: ERROR : ADC in slot %d is not initialized \n",id);
    return(0);
  }

  /* Check if Moller FPGA processing is Enabled. IF so we can't read the
     registers */
  if((FAp[id]->ctrl2)&FAMOL_CNTL_GO_MOLLER) {
    printf("faMollerStatus: WARN: Moller DAQ Active!! Cannot access FPGA Registers\n");
    return(0);
  }

  /* Read Moller FPGA Registers */
  rev = (FAp[id]->moller_status[0])&0xffff;

  scL = (FAp[id]->CL_sample_cnt)&FAMOL_SAMPLE_CNT_MASK;
  scR = (FAp[id]->CR_sample_cnt)&FAMOL_SAMPLE_CNT_MASK;

  r1 = (FAp[id]->CL_sum_thresh[0])&0xffff;
  r2 = (FAp[id]->CL_sum_thresh[1])&0xf;
  r3 = (FAp[id]->CR_sum_thresh[0])&0xffff;
  r4 = (FAp[id]->CR_sum_thresh[1])&0xf;
  stL = (r2<<16)|r1;
  stR = (r4<<16)|r3;  

  pwL[0] = (FAp[id]->CL_pulse_width)&FAMOL_PULSE_WIDTH_MASK;
  pwR[0] = (FAp[id]->CR_pulse_width)&FAMOL_PULSE_WIDTH_MASK;
  for(ii=1;ii<5;ii++) {
    pwL[ii] = (FAp[id]->SL_pulse_width[ii-1])&FAMOL_PULSE_WIDTH_MASK;
    pwR[ii] = (FAp[id]->SR_pulse_width[ii-1])&FAMOL_PULSE_WIDTH_MASK;
  }

  dyL[0] = (FAp[id]->CL_delay)&FAMOL_DELAY_MASK;
  dyR[0] = (FAp[id]->CR_delay)&FAMOL_DELAY_MASK;
  for(ii=1;ii<5;ii++) {
    dyL[ii] = (FAp[id]->SL_delay[ii-1])&FAMOL_DELAY_MASK;
    dyR[ii] = (FAp[id]->SR_delay[ii-1])&FAMOL_DELAY_MASK;
  }
  dyLR = (FAp[id]->SLSR_delay)&FAMOL_DELAY_MASK;

  psL  = (FAp[id]->CL_trig_prescale)&FAMOL_PRESCALE_MASK;
  psR  = (FAp[id]->CR_trig_prescale)&FAMOL_PRESCALE_MASK;
  psLR = (FAp[id]->CLCR_trig_prescale)&FAMOL_PRESCALE_MASK;

  cw   = (FAp[id]->moller_trig_window)&FAMOL_TRIG_WINDOW_MASK;

  printf("\nSTATUS for MOLLER FADC in slot %d at base address 0x%x \n",id,(UINT32) FAp[id]);
  printf("---------------------------------------------------------- \n");
  
  printf("Moller FPGA Firmware Rev/ID = 0x%04x \n",rev);
  printf("Note: All time values are in clock ticks (4ns/tick)\n\n");

  printf("                         Left               Right\n");
  printf("Sample Count          %8d             %8d    \n",scL,scR);
  printf("Sum Threshold         %8d             %8d    \n",stL,stR);

  printf("Pulse Widths:\n");
  printf("   Calorimeter        %8d             %8d     \n",pwL[0],pwR[0]);
  printf("   Scint 1            %8d             %8d     \n",pwL[1],pwR[1]);
  printf("   Scint 2            %8d             %8d     \n",pwL[2],pwR[2]);
  printf("   Scint 3            %8d             %8d     \n",pwL[3],pwR[3]);
  printf("   Scint 4            %8d             %8d     \n",pwL[4],pwR[4]);

  printf("Delays     :\n");
  printf("   Calorimeter        %8d             %8d     \n",dyL[0],dyR[0]);
  printf("   Scint 1            %8d             %8d     \n",dyL[1],dyR[1]);
  printf("   Scint 2            %8d             %8d     \n",dyL[2],dyR[2]);
  printf("   Scint 3            %8d             %8d     \n",dyL[3],dyR[3]);
  printf("   Scint 4            %8d             %8d     \n",dyL[4],dyR[4]);
  printf("   SL*SR   : %8d\n",dyLR);
  
  printf("Cal Trigger Prescales:  Left: %d    Right: %d   CR*CL: %d\n",psL,psR,psLR);
  printf("Trigger Window Width :  %d \n",cw);

  return(0);
}
