/* @(#)xdr.h	2.2 88/07/29 4.0 RPCSRC */
/*
 * Sun RPC is a product of Sun Microsystems, Inc. and is provided for
 * unrestricted use provided that this legend is included on all tape
 * media and as a part of the software program in whole or part.  Users
 * may copy or modify Sun RPC without charge, but are not authorized
 * to license or distribute it to anyone else except as part of a product or
 * program developed by the user.
 * 
 * SUN RPC IS PROVIDED AS IS WITH NO WARRANTIES OF ANY KIND INCLUDING THE
 * WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE OR TRADE PRACTICE.
 * 
 * Sun RPC is provided with no support and without any obligation on the
 * part of Sun Microsystems, Inc. to assist in its use, correction,
 * modification or enhancement.
 * 
 * SUN MICROSYSTEMS, INC. SHALL HAVE NO LIABILITY WITH RESPECT TO THE
 * INFRINGEMENT OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY SUN RPC
 * OR ANY PART THEREOF.
 * 
 * In no event will Sun Microsystems, Inc. be liable for any lost revenue
 * or profits or other special, indirect and consequential damages, even if
 * Sun has been advised of the possibility of such damages.
 * 
 * Sun Microsystems, Inc.
 * 2550 Garcia Avenue
 * Mountain View, California  94043
 */
/*      @(#)xdr.h 1.19 87/04/22 SMI      */

/*
 * xdr.h, External Data Representation Serialization Routines.
 *
 * Copyright (C) 1984, Sun Microsystems, Inc.
 */

/*
 * xdr.h porting XDR to Windows NT platform
 *
 * Jie Chen at Jefferson Lab, chen@jlab.org
 */

#ifndef __XDR_HEADER__
#define __XDR_HEADER__

/*
 * XDR provides a conventional way for converting between C data
 * types and an external bit-string representation.  Library supplied
 * routines provide for the conversion on built-in C data types.  These
 * routines and utility routines defined here are used to help implement
 * a type encode/decode routine for each user-defined type.
 *
 * Each data type provides a single procedure which takes two arguments:
 *
 *	bool_t
 *	xdrproc(xdrs, argresp)
 *		XDR *xdrs;
 *		<type> *argresp;
 *
 * xdrs is an instance of a XDR handle, to which or from which the data
 * type is to be converted.  argresp is a pointer to the structure to be
 * converted.  The XDR handle contains an operation field which indicates
 * which of the operations (ENCODE, DECODE * or FREE) is to be performed.
 *
 * XDR_DECODE may allocate space if the pointer argresp is null.  This
 * data can be freed with the XDR_FREE operation.
 *
 * We write only one procedure per data type to make it easy
 * to keep the encode and decode procedures for a data type consistent.
 * In many cases the same code performs all operations on a user defined type,
 * because all the hard work is done in the component type routines.
 * decode as a series of calls on the nested data types.
 */

/* RPC Types: 
 *
 * These types actually are defined in the rpc/types.h in the regular
 * Unix boxes. Copying them here makes life easier
 *
 * Jie Chen: 12/17/98
 */
typedef char * caddr_t;
typedef unsigned int u_int;
typedef unsigned long u_long;
typedef unsigned short u_short;
typedef unsigned char	u_char;
typedef int bool_t;
typedef int enum_t;

#define __dontcare__    -1

#ifndef FALSE
#define FALSE           (0)
#endif

#ifndef TRUE
#define TRUE            (1)
#endif

#ifndef NULL
#define NULL             0
#endif

#define mem_alloc(bsize)     malloc(bsize)
#define mem_free(ptr,bsize)  free(ptr)

#if defined (__STDC__) || defined (__cplusplus)
#define P_(s) s
#else
#define P_(s) ()
#endif

/* 
 * Windows NT specific things are declared here
 */
#if defined (_WIN32)

#if !defined (DLLIMPORT)
#define DLLIMPORT __declspec(dllimport)
#endif

#if !defined (DLL_EXPORT)
#define DLLEXPORT __declspec(dllexport)
#endif

#if defined (_CDEV_CORE_EXPORTS_)
#define XDRAPI DLLEXPORT
#else
#define XDRAPI DLLIMPORT
#endif

#if(_WIN32_WINNT >= 0x0400)
#include <winsock2.h>
#include <mswsock.h>
#else
#include <winsock.h>
#endif /* _WIN32_WINNT >=  0x0400 */

#else

#define XDRAPI
#define DLLIMPORT
#define DLLEXPORT

#endif

/*
 * Xdr operations.  XDR_ENCODE causes the type to be encoded into the
 * stream.  XDR_DECODE causes the type to be extracted from the stream.
 * XDR_FREE can be used to release the space allocated by an XDR_DECODE
 * request.
 */
enum xdr_op {
	XDR_ENCODE=0,
	XDR_DECODE=1,
	XDR_FREE=2
};

/*
 * This is the number of bytes per unit of external data.
 */
#define BYTES_PER_XDR_UNIT	(4)
#define RNDUP(x)  ((((x) + BYTES_PER_XDR_UNIT - 1) / BYTES_PER_XDR_UNIT) \
		    * BYTES_PER_XDR_UNIT)

/*
 * The XDR handle.
 * Contains operation which is being applied to the stream,
 * an operations vector for the paticular implementation (e.g. see xdr_mem.c),
 * and two private fields for the use of the particular impelementation.
 */
typedef struct {
	enum xdr_op	x_op;		/* operation; fast additional param */
        struct xdr_ops *x_ops;
	caddr_t 	x_public;	/* users' data */
	caddr_t		x_private;	/* pointer to private data */
	caddr_t 	x_base;		/* private used for position info */
	int		x_handy;	/* extra private word */
} XDR;

struct xdr_ops {
  bool_t	(*x_getlong)P_((XDR *, long *));	
  /* get a long from underlying stream */
  bool_t	(*x_putlong)P_((XDR *, long *));	
  /* put a long to " */
  bool_t	(*x_getbytes)P_((XDR *, caddr_t, int));
  /* get some bytes from " */
  bool_t	(*x_putbytes)P_((XDR *, caddr_t, int));
  /* put some bytes to " */
  u_int	        (*x_getpostn)P_((XDR *));
  /* returns bytes off from beginning */
  bool_t        (*x_setpostn)P_((XDR *, u_int));
  /* lets you reposition the stream */
  long *	(*x_inline)P_((XDR *, int));
  /* buf quick ptr to buffered data */
  void	        (*x_destroy)P_((XDR *));
  /* free privates of this xdr_stream */
};


/*
 * A xdrproc_t exists for each data type which is to be encoded or decoded.
 *
 * The second argument to the xdrproc_t is a pointer to an opaque pointer.
 * The opaque pointer generally points to a structure of the data type
 * to be decoded.  If this pointer is 0, then the type routines should
 * allocate dynamic storage of the appropriate size and return it.
 * bool_t	(*xdrproc_t)(XDR *, caddr_t *);
 */
#ifdef __cplusplus
typedef	bool_t (*xdrproc_t)(XDR *, void *);
#else
#ifdef __STDC__
typedef	bool_t (*xdrproc_t)();  /* for backword compatablity */
#else
typedef	bool_t (*xdrproc_t)();
#endif
#endif


/*
 * Operations defined on a XDR handle
 *
 * XDR		*xdrs;
 * long		*longp;
 * caddr_t	 addr;
 * u_int	 len;
 * u_int	 pos;
 */
#define XDR_GETLONG(xdrs, longp)			\
	(*(xdrs)->x_ops->x_getlong)(xdrs, longp)
#define xdr_getlong(xdrs, longp)			\
	(*(xdrs)->x_ops->x_getlong)(xdrs, longp)

#define XDR_PUTLONG(xdrs, longp)			\
	(*(xdrs)->x_ops->x_putlong)(xdrs, longp)
#define xdr_putlong(xdrs, longp)			\
	(*(xdrs)->x_ops->x_putlong)(xdrs, longp)

#define XDR_GETBYTES(xdrs, addr, len)			\
	(*(xdrs)->x_ops->x_getbytes)(xdrs, addr, len)
#define xdr_getbytes(xdrs, addr, len)			\
	(*(xdrs)->x_ops->x_getbytes)(xdrs, addr, len)

#define XDR_PUTBYTES(xdrs, addr, len)			\
	(*(xdrs)->x_ops->x_putbytes)(xdrs, addr, len)
#define xdr_putbytes(xdrs, addr, len)			\
	(*(xdrs)->x_ops->x_putbytes)(xdrs, addr, len)

#define XDR_GETPOS(xdrs)				\
	(*(xdrs)->x_ops->x_getpostn)(xdrs)
#define xdr_getpos(xdrs)				\
	(*(xdrs)->x_ops->x_getpostn)(xdrs)

#define XDR_SETPOS(xdrs, pos)				\
	(*(xdrs)->x_ops->x_setpostn)(xdrs, pos)
#define xdr_setpos(xdrs, pos)				\
	(*(xdrs)->x_ops->x_setpostn)(xdrs, pos)

#define	XDR_INLINE(xdrs, len)				\
	(*(xdrs)->x_ops->x_inline)(xdrs, len)
#define	xdr_inline(xdrs, len)				\
	(*(xdrs)->x_ops->x_inline)(xdrs, len)

#define	XDR_DESTROY(xdrs)				\
	if ((xdrs)->x_ops->x_destroy) 			\
		(*(xdrs)->x_ops->x_destroy)(xdrs)
#define	xdr_destroy(xdrs)				\
	if ((xdrs)->x_ops->x_destroy) 			\
		(*(xdrs)->x_ops->x_destroy)(xdrs)

/*
 * Support struct for discriminated unions.
 * You create an array of xdrdiscrim structures, terminated with
 * a entry with a null procedure pointer.  The xdr_union routine gets
 * the discriminant value and then searches the array of structures
 * for a matching value.  If a match is found the associated xdr routine
 * is called to handle that part of the union.  If there is
 * no match, then a default routine may be called.
 * If there is no match and no default routine it is an error.
 */
#define NULL_xdrproc_t ((xdrproc_t)0)
struct xdr_discrim {
	int	value;
	xdrproc_t proc;
};

/*
 * In-line routines for fast encode/decode of primitve data types.
 * Caveat emptor: these use single memory cycles to get the
 * data from the underlying buffer, and will fail to operate
 * properly if the data is not aligned.  The standard way to use these
 * is to say:
 *	if ((buf = XDR_INLINE(xdrs, count)) == NULL)
 *		return (FALSE);
 *	<<< macro calls >>>
 * where ``count'' is the number of bytes of data occupied
 * by the primitive data types.
 *
 * N.B. and frozen for all time: each data type here uses 4 bytes
 * of external representation.
 */
#define IXDR_GET_LONG(buf)		((long)ntohl((u_long)*(buf)++))
#define IXDR_PUT_LONG(buf, v)		(*(buf)++ = (long)htonl((u_long)v))

#define IXDR_GET_BOOL(buf)		((bool_t)IXDR_GET_LONG(buf))
#define IXDR_GET_ENUM(buf, t)		((t)IXDR_GET_LONG(buf))
#define IXDR_GET_U_LONG(buf)		((u_long)IXDR_GET_LONG(buf))
#define IXDR_GET_SHORT(buf)		((short)IXDR_GET_LONG(buf))
#define IXDR_GET_U_SHORT(buf)		((u_short)IXDR_GET_LONG(buf))

#define IXDR_PUT_BOOL(buf, v)		IXDR_PUT_LONG((buf), ((long)(v)))
#define IXDR_PUT_ENUM(buf, v)		IXDR_PUT_LONG((buf), ((long)(v)))
#define IXDR_PUT_U_LONG(buf, v)		IXDR_PUT_LONG((buf), ((long)(v)))
#define IXDR_PUT_SHORT(buf, v)		IXDR_PUT_LONG((buf), ((long)(v)))
#define IXDR_PUT_U_SHORT(buf, v)	IXDR_PUT_LONG((buf), ((long)(v)))

/*
 * These are the "generic" xdr routines.
 */
#if defined (__cplusplus)
extern "C" {
#endif

extern XDRAPI bool_t	xdr_void P_((void));
extern XDRAPI bool_t	xdr_int P_((XDR *, int *));
extern XDRAPI bool_t	xdr_u_int P_((XDR *, u_int *));
extern XDRAPI bool_t	xdr_long P_((XDR *, long *));
extern XDRAPI bool_t	xdr_u_long P_((XDR *, u_long *));
extern XDRAPI bool_t	xdr_short P_((XDR *, short *));
extern XDRAPI bool_t	xdr_u_short P_((XDR *, u_short *));
extern XDRAPI bool_t	xdr_bool P_((XDR *, bool_t *));
extern XDRAPI bool_t	xdr_enum P_((XDR *, enum_t *));
extern XDRAPI bool_t	xdr_array P_((XDR *, caddr_t *, u_int *, const u_int, const u_int, const xdrproc_t));
extern XDRAPI bool_t	xdr_bytes P_((XDR *, char **, u_int *, const u_int));
extern XDRAPI bool_t	xdr_opaque P_((XDR *, caddr_t, const u_int));
extern XDRAPI bool_t	xdr_string P_((XDR *, char **, const u_int));
extern XDRAPI bool_t	xdr_union P_((XDR *, enum_t *, char *, const struct xdr_discrim *, const xdrproc_t));
extern XDRAPI bool_t	xdr_char P_((XDR *, char *));
extern XDRAPI bool_t	xdr_u_char P_((XDR *, u_char *));
extern XDRAPI bool_t	xdr_vector P_((XDR *, char *, const u_int, const u_int, const xdrproc_t));
extern XDRAPI bool_t	xdr_float P_((XDR *, float *));
extern XDRAPI bool_t	xdr_double P_((XDR *, double *));
extern XDRAPI bool_t	xdr_reference P_((XDR *, caddr_t *, u_int, const xdrproc_t));
extern XDRAPI bool_t	xdr_pointer P_((XDR *, char **, u_int, const xdrproc_t));
extern XDRAPI bool_t	xdr_wrapstring P_((XDR *, char **));
extern XDRAPI void     xdr_free P_((xdrproc_t, char *));
#if defined (__cplusplus)
};
#endif

/*
 * Common opaque bytes objects used by many rpc protocols;
 * declared here due to commonality.
 */
#define MAX_NETOBJ_SZ 1024 
struct netobj {
	u_int	n_len;
	char	*n_bytes;
};
typedef struct netobj netobj;
extern bool_t   xdr_netobj();

/*
 * These are the public routines for the various implementations of
 * xdr streams.
 */
#ifdef __cplusplus
extern "C" {
#endif
extern XDRAPI void   xdrmem_create P_((XDR *, const caddr_t, const u_int, const enum xdr_op));		
  /* XDR using memory buffers */
extern XDRAPI void   xdrstdio_create P_((XDR *, FILE *, const enum xdr_op));
  /* XDR using stdio library */
extern XDRAPI void   xdrrec_create P_((XDR *, const u_int, const u_int, const caddr_t, int (*) (void *, caddr_t, int), int (*) (void *, caddr_t, int)));
  /* XDR pseudo records for tcp */
extern XDRAPI bool_t xdrrec_endofrecord P_((XDR *, bool_t));
  /* make end of xdr record */
extern XDRAPI bool_t xdrrec_skiprecord P_((XDR *));	
  /* move to beginning of next record */
extern XDRAPI bool_t xdrrec_eof P_((XDR *));
  /* true if no more input */
#ifdef __cplusplus
};
#endif

#endif !__XDR_HEADER__
