#ifndef _CDEV_COLLECTION_DEFINITION_H_
#define _CDEV_COLLECTION_DEFINITION_H_

#ifndef _CDEV_DIRECTORY_TABLE_H_
	#error "You must include cdevDirectoryTable.h to load cdevCollectionDefinition.h"
#endif

// *****************************************************************************
// * class cdevCollectionDefinition :
// *	This class stores information that identifies a specific device name
// *	and associates it with a collection of devices.
// * 
// *	This class maintains a variety of internal data items for the following
// *	purposes...
// *
// *	name      - This is the name of the device as specified in the DDL file.
// *	directory - This is the cdevDirectoryTable object where the device was
// *                registered.
// *	cdevClass - This is the definition object for the class that the device
// *                belongs to (collection).
// *	devices   - This is the list of devices that make up the collection.
// *	deviceCnt - This is the number of devices in the list.
// *	next      - This is the next device in the linked list of 
// *		    cdevCollectionDefinition objects.  This list is maintained 
// *		    as part of the cdevClassDefinition object.
// *****************************************************************************
class cdevCollectionDefinition 
{
private:
	char                       * name;
	char                      ** devices;
	int                          deviceCnt;
	cdevCollectionDefinition   * next;
		
public:
	inline  cdevCollectionDefinition ( char  * Name, 
				           char ** Devices   = NULL,
				           int     DeviceCnt = 0 );		
	inline ~cdevCollectionDefinition ( void );

	inline void asciiDump ( FILE * fp = stdout );
	
	// *********************************************************************
	// * Member access methods.
	// *********************************************************************
	inline char                     * getName      ( void );
	inline char                    ** getDevices   ( void );
	inline int                        getDeviceCnt ( void );
	inline cdevCollectionDefinition * getNext      ( void );
	inline void                       setNext      ( cdevCollectionDefinition * Next );
};


// *****************************************************************************
// * cdevCollectionDefinition::cdevCollectionDefinition :
// *	This is the constructor for the cdevCollectionDefinition class.
// *****************************************************************************
inline cdevCollectionDefinition::cdevCollectionDefinition 
	( char * Name, char ** Devices, int DeviceCnt )
	: next(NULL), name(Name), devices(Devices), deviceCnt(DeviceCnt)
	{
	}
		
// *****************************************************************************
// * cdevCollectionDefinition::~cdevCollectionDefinition :
// *	This is the destructor for the cdevCollectionDefinition class.
// *****************************************************************************
inline cdevCollectionDefinition::~cdevCollectionDefinition ( void )
	{
	// *********************************************************************
	// * Release strings using free if they were allocated with strdup.
	// *********************************************************************
	if(name) free(name);
	if(devices && deviceCnt)
		{
		while((--deviceCnt)>=0) free(devices[deviceCnt]);
		}
	if(devices) delete[] devices;
	}

// *****************************************************************************
// * cdevCollectionDefinition::asciiDump :
// *	Write the formatted output of the collection to the specified file
// *	pointer.
// *****************************************************************************
inline void cdevCollectionDefinition::asciiDump ( FILE * fp )
	{
	if(devices && deviceCnt)
		{
		fprintf(fp, "\ncollection %s :\n", name);
		for(int i=0; i<deviceCnt; i++)
			{
			fprintf(fp, "\t%s\n", devices[i]);
			}
		fprintf(fp, "\t;\n");
		}
	}

// *****************************************************************************
// * cdevCollectionDefinition::getName :
// *	This method allows the caller to retrieve the device name.
// *****************************************************************************
inline char * cdevCollectionDefinition::getName ( void ) 
	{
	return name;
	}


// *****************************************************************************
// * cdevCollectionDefinition::getDevices :
// *	This method returns the list of devices associated with the 
// *	cdevCollection.
// *****************************************************************************
inline char ** cdevCollectionDefinition::getDevices   ( void )
	{
	return devices;
	}
	

// *****************************************************************************
// * cdevCollectionDefinition::getDevices :
// *	This method returns the number of devices in the device list.
// *****************************************************************************
inline int cdevCollectionDefinition::getDeviceCnt ( void )
	{
	return deviceCnt;
	}

// *****************************************************************************
// * cdevCollectionDefinition::getNext :
// *	This method allows the caller to retrieve the next device in the list.
// *****************************************************************************
inline cdevCollectionDefinition * cdevCollectionDefinition::getNext ( void )
	{
	return next;
	}

// *****************************************************************************
// * cdevCollectionDefinition::setNext :
// *	This method allows the caller to set the next device in the list.
// *****************************************************************************
inline void cdevCollectionDefinition::setNext ( cdevCollectionDefinition * Next )
	{
	next = Next;
	}

#endif
