/*************************************************************************
 *
 *   sfiUserLib.c  - Library of routines for the user to write for 
 *                   Aysyncronous readout and buffering of events
 *                   from FASTBUS. Use RAM LIST PROGRAMMING
 *
 *
 *
 */


#include "sfiLib.h"
#include "dmaPList.h"

#define SFI_ADDR 0xe00000
#define IRQ_SOURCE  0x10

#define ADC_SLOT 15
#define SCANMASK 0x8000
#define RAM_ADDR 0x100

#define MAX_NUM_EVENTS    400
#define MAX_SIZE_EVENTS   2048      /* Size in Bytes */


/* 0: External Mode   1:Trigger Supervisor Mode */
#define TS_MODE  0

/* Input and Output Partions for SFI Readout */
ROL_MEM_ID sfiIN, sfiOUT;


void 
SFIRamLoad () 
{
  SFI_RAM_LOAD_ENABLE(RAM_ADDR); /* Setup Sequencer for RAM List Loading at RAM_ADDR */

  /* Load the List */
  FBWC(ADC_SLOT,0,0x400);        /* Address ADC CSR0 Write 0x400 and release (Load Next Event) */
  FBAD(ADC_SLOT,0);              /* Address ADC  DSR0 */
  FBBR(100);                     /* Start Block Read (And Clear Word Counter) */
  FBREL;                         /* Release the Bus */

  FBSWC;                         /* Store Final Word Count in FIFO */

  FBEND;                         /* End of List */
  SFI_WAIT;                      /* Wait a little */
  SFI_RAM_LOAD_DISABLE;          /* Re-enable Sequencer */
 
}

/* Initialize Memory partitions for managing events */
void
sfiUserInit()
{
  unsigned int res, laddr;

  /* Setup SFI pointers to Enable triggers, and establish
     DMA address offsets */
  res = (unsigned long) sysBusToLocalAdrs(0x39,(char *)SFI_ADDR,(char **)&laddr);
  if (res != 0) {
     printf("sfiUserInit: Error Initializing SFI res=%d \n",res);
  } else {
     printf("sfiUserInit: Calling InitSFI() routine with laddr=0x%x.\n",laddr);
     InitSFI(laddr);
  }


  /* Setup Buffer memory to store events */
  dmaPFreeAll();
  sfiIN  = dmaPCreate("sfiIN",MAX_SIZE_EVENTS,MAX_NUM_EVENTS,0);
  sfiOUT = dmaPCreate("sfiOUT",0,0,0);

  dmaPStatsAll();
}


void
sfiUserDownload()
{

  /* Reinitialize the Buffer memory */
  dmaPReInitAll();

  /* Initialize AUX Card */
  sfiAuxInit();

  /* Initialize SFI Interrupt interface - use defaults */
  sfiIntInit(0,0,0,0);

  /* Connect User Trigger Routine */
  sfiIntConnect(sfiUserTrigger,IRQ_SOURCE);


  printf("sfiUserDownload: User Download Executed\n");

}

void
sfiUserPrestart()
{
  unsigned int adc_id;

  SFI_ENABLE_SEQUENCER;

  /* PRogram ADC */
  fpwc(ADC_SLOT,0,0x40000000);
  fpwc(ADC_SLOT,0,0x104);
  fpwc(ADC_SLOT,1,0x40);
  fprc(ADC_SLOT,0,&adc_id);

  printf("ADC id = 0x%x\n",adc_id);
  
  /* Load the Ram List */
  SFIRamLoad();

  /* Initialize SFI Interrupt variables */
  sfiIntClearCount();

}

void
sfiUserGo()
{
  /* Enable Interrupts */
  sfiIntEnable(IRQ_SOURCE,TS_MODE);
}

void
sfiUserEnd()
{
  int status, dma, count;

  /* Disable Interrupts */
  sfiIntDisable(IRQ_SOURCE,TS_MODE);

  /* Check on last Transfer */
  if(sfiIntCount > 0) {
    status = sfiSeqDone();
    if(status != 0) {
      logMsg("ERROR: in Last Transfer Event NUMBER %d, status = 0x%x\n",sfiIntCount,status,0,0,0,0);
      sfi_error_decode(0);
       *dma_dabufp++ = 0xfb000bad;
    }else{
      dma = sfiReadSeqFifo();
      count = dma&0xffffff;
      printf("Last DMA status = 0x%x\n",dma);
      dma_dabufp += count;
    }
    PUTEVENT(sfiOUT);
  }

  /*Reset AUX Port */
  sfiAuxWrite(0x8000);

  printf("sfiUserEnd: Ended after %d events\n",sfiIntCount);
  
}

void
sfiUserTrigger(int arg)
{
  int ii, status, dma, count;
  unsigned int datascan;


  /* check on Last Transfer */
  if (sfiIntCount > 1) {
    status = sfiSeqDone();
    if(status != 0) {
      logMsg("ERROR: in Last Transfer Event NUMBER %d, status = 0x%x\n",sfiIntCount,status,0,0,0,0);
      sfi_error_decode(0);
      *dma_dabufp++ = 0xfb000bad;
    }else{
      dma = sfiReadSeqFifo();
      count = dma&0xffffff;
      dma_dabufp += count;
    }
    PUTEVENT(sfiOUT);
  }

  /*check for data and start transfer*/
  SFI_ENABLE_SEQUENCER;

  ii=0;
  datascan = 0;
  while ((ii<20) && (SCANMASK != (datascan&SCANMASK))) {
    datascan = 0;
    fb_frcm_1(9,0,&datascan,1,0,1,0,0,0);
    ii++;
  }

 
  if(ii<20) {
    GETEVENT(sfiIN,sfiIntCount);
    *dma_dabufp++ = 0xda0000da;
    sfiRamStart(RAM_ADDR,dma_dabufp);
  }

  if(dmaPEmpty(sfiIN)) {
    sfiNeedAck = 1;
    return;
  } else {
    sfiIntAck(IRQ_SOURCE,TS_MODE);
  }
}

void
sfiUserStatus()
{

}

void
sfiUserTest()
{
  int lock_key;
  DANODE *outEvent;
  
  while(1) {
    if(dmaPEmpty(sfiOUT))
      continue;
    
    /* get event - then free it */
    lock_key = intLock();
    outEvent = dmaPGetItem(sfiOUT);
    dmaPFreeItem(outEvent);
    intUnlock(lock_key);
  }


}
