#
#########################
#			#
#  GLOBAL DEFINITIONS   #
#			#
#########################
global processStates env

image create photo codaImage -file $env(CODA)/common/images/gif/RCLogo.gif
image create photo codaFile -file $env(CODA)/common/images/gif/doc_coda.gif
image create photo binFile -file $env(CODA)/common/images/gif/doc_bin.gif
image create photo trashData -file $env(CODA)/common/images/gif/trash2.gif

set processStates [list \
		       null \
		       booting \
		       booted \
		       configured \
		       downloaded \
		       inited \
		       initing \
		       loading \
		       loaded \
		       prestarting \
		       prestarted \
		       pausing \
		       paused \
		       resuming \
		       activating \
		       active \
		       alive \
		       verifying \
		       verified \
		       ending \
		       terminating \
		       terminated \
		       unknown \
		       dormant \
		       down \
		       dead]

global stateColor
set stateColor(null)        black 
set stateColor(booting)	gray 
set stateColor(booted)      white
set stateColor(configured)  white
set stateColor(downloaded)  white
set stateColor(initing)     gray
set stateColor(inited)      white 
set stateColor(loading)     yellow
set stateColor(loaded)      yellow3
set stateColor(prestarting) cyan
set stateColor(prestarted)  "steel blue"
set stateColor(pausing)     orange
set stateColor(paused)      orange3
set stateColor(resuming)    brown
set stateColor(activating)  "green yellow"
set stateColor(active)      "green"
set stateColor(alive)       "green"
set stateColor(up)       "green"
set stateColor(ending)      pink
set stateColor(verifying)   purple 
set stateColor(verified)    purple2
set stateColor(terminating) pink
set stateColor(terminated)  red
set stateColor(unknown)     red
set stateColor(dormant)     red
set stateColor(down)        red
set stateColor(dead)        red

global linkColor
set linkColor(up)   green
set linkColor(down) red    

#########################
#			#
#   CLASS DEFINITIONS   #
#			#
#########################

#########################
#
# MainWin class 
#
#########################

class MainWin {
    constructor {wid {flags ""}} {}
    destructor  {exit}
    
    private common can      {}
    private common zoom     {}
    private common scale    1.0
    private common loopID   {}
    private common updating 0
    private common host     {}
    private common db       {}
    private common session  {}
    private common period   {3}
    private common position
    private common widget
    private common printer {}
    
    set position(init) {}
    
    public proc chooseHost    {host}
    public proc listDbs       {} 
    public proc chooseDb      {db}
    public proc listSessions  {}
    public proc chooseSession {sess}
    public proc zoomPosition  {x y}
    public proc ScrollMark    {x y}
    public proc ScrollDrag    {x y}
    public proc zoomCursor    {mode x y}
    public proc ScrollDone    {}
    public proc zoom          {scale x y}
    public proc unZoom        {}
    public proc reZoom        {}

    public proc get           {option}
    public proc loop          {}
    public proc stopUpdate    {}
    public proc startUpdate   {}
    public proc clearSession  {}
    public proc clearScreen   {}
    public proc drawScreen    {}
    public proc updateScreen  {}
    public proc print         {}
}

##
## The following procedures enable the scroll operation on the canvas.
##

body MainWin::ScrollMark {x y} {
    set c [MainWin::get -canvas]

    $c configure -cursor hand1
    $c scan mark [expr $x/10] [expr $y/10]
}

body MainWin::ScrollDrag {x y} {
    set c [MainWin::get -canvas]
    $c scan dragto [expr $x/10] [expr $y/10]
}

body MainWin::ScrollDone {} {
    set c  [MainWin::get -canvas]
    $c configure -cursor crosshair
}

body MainWin::print {} {
    set can [MainWin::get -canvas]
    $widget.f1.f3.f1.print config -state disabled
    set dialog Dlog
    
    set fname "/tmp/tkined.log"
    catch {exec /bin/rm -f $fname}

    if {[file exists $fname] && ![file writable $fname]} {
	$dialog acknowledge $widget "Can not write temporary file $fname."
	$widget.f1.f3.f1.print config -state normal
	return
    }

    if {[catch {open $fname w} file]} {
	$dialog acknowledge $widget "Can not open $fname: $file"
	$widget.f1.f3.f1.print config -state normal
	return
    }

    if {[catch {$can postscript -file $fname} err]} {
	$dialog acknowledge $widget "Failed to write $fname: $err"
	$widget.f1.f3.f1.print config -state normal
	return
    }

    catch {close $file}
    $dialog request $widget \
	"Please enter a printer name:" \
	[list [list Printer: $printer]] \
	[list send cancel]
    set printer [$dialog result request]
    
    if {[lindex $printer 0] == "cancel"} {
	$widget.f1.f3.f1.print config -state normal
	return
    }
    catch "exec lp -d [lindex $printer 1] $fname" res
    $widget.f1.f3.f1.print config -state normal

}

body MainWin::get {option} {
    # return values of private variables
    switch -- $option {
	-canvas   {return $can}
	-zoom     {return $zoom}
	-scale    {return $scale}
	-loopID   {return $loopID}
	-updating {return $updating}
	-host     {return $host}
	-db       {return $db}
	-session  {return $session}
    }
    error "Process::get bad option \"$option\""
}

body MainWin::zoomPosition {x y} {
    # Routine bound to mouse motion in canvas to record position
    set position(x) [$can canvasx $x]
    set position(y) [$can canvasx $y]
    # move cursor-text (tag z)
    catch {$can move z \
	       [expr $position(x) - $position(oldx)] \
	       [expr $position(y) - $position(oldy)]}
    set position(oldx) $position(x)
    set position(oldy) $position(y)
}

body MainWin::zoomCursor {mode X Y} {
    set x [$can canvasx $X]
    set y [$can canvasx $Y]
    if {"$mode" == "zoom"} {	   
	$can config -cursor {crosshair red white}
	$can create text $x [expr $y - 20] -fill red -text ZOOM  -tags z
    } elseif {"$mode" == "shrink"} {
	$can config -cursor {circle blue white}
	$can create text $x [expr $y - 20] -fill blue -text SHRINK -tags z
    } elseif {"$mode" == "reset"} {
	$can config -cursor {gumby purple white}
	$can create text $x [expr $y - 20] -fill purple -text RESET  -tags z
    } else {}
    
    return
}

body MainWin::zoom {scale_ X Y} {
    if { [expr $scale == 0.8 ] && [expr $scale_ < 1.0] } {
	$can config -cursor {arrow black white}
	$can delete z
	return
    }
    set x [$can canvasx $X]
    set y [$can canvasx $Y]
    $can config -cursor {arrow black white}
    $can delete z
    $can scale all $x $y $scale_ $scale_
    lappend zoom [list $scale_ $x $y]
    set scale [expr $scale * $scale_]

    return
}

body MainWin::unZoom {} {
    set length [llength $zoom]

    for {set i $length} {$i > 0} {incr i -1} {
	set lastZoom [lindex $zoom [expr $i - 1]]
	set exp [expr 1./[lindex $lastZoom 0]]
	set x   [lindex $lastZoom 1]
	set y   [lindex $lastZoom 2]
	$can scale all $x $y $exp $exp
    }
    set zoom {}
    $can config -cursor {arrow black white}
    $can delete z
    return
}

body MainWin::reZoom {} {
    set length [llength $zoom]
    for {set i 0} {$i < $length} {incr i} {
	set Zoom [lindex $zoom $i]
	set exp  [lindex $Zoom 0]
	set x    [lindex $Zoom 1]
	set y    [lindex $Zoom 2]
	$can scale all $x $y $exp $exp
    }
    return
}


body MainWin::constructor {w {flags ""}} {

    global env argv processStates stateColor linkColor
    #wm withdraw .
    set widget $w
    #toplevel    $widget
    #wm title    $widget "ProcView"
    #wm geometry $widget 800x600

    ################ $widget.f1
    frame $widget.f1

    ################ $widget.f1.f1
    #frame $widget.f1.f1

    #label $widget.f1.f1.label2 \
	#	  -background white \
	#	  -foreground blue \
	#	  -anchor center \
	#	  -relief ridge \
	#	  -borderwidth 2 \
	#	  -font "-*-helvetica-bold-r-normal--12-*-*-*-*-*-*-*" \
	#	  -text "PROCESS VIEWTOOL"
    #if { "$flags" == ""} {
    #  pack  $widget.f1.f1.label3 -side left
    #}
    ################ $widget.f1.f2 The main canvas
    frame $widget.f1.f2
    frame $widget.f1.f2.f
    
    canvas $widget.f1.f2.f.canvas \
	-xscrollcommand "$widget.f1.f2.f.scrollx set" \
	-yscrollcommand "$widget.f1.f2.scrolly   set" \
	-scrollregion {0 0 2500 2500} \
	-background grey \
	-confine false \
	-cursor crosshair


    scrollbar $widget.f1.f2.f.scrollx -command "$widget.f1.f2.f.canvas xview" -orient horizontal
    scrollbar $widget.f1.f2.scrolly   -command "$widget.f1.f2.f.canvas yview" -orient vertical

    ################ $widget.f1.f3 the control bar...
    frame  $widget.f1.f3 -borderwidth 4 -relief groove

    ################ $widget.f1.f3.f1
    frame  $widget.f1.f3.f1 -borderwidth 2 -relief groove

    tixComboBox $widget.f1.f3.f1.host \
	-command "MainWin::chooseHost" \
	-editable true \
	-label "HOST: " \
	-history false \
	-prunehistory true

    tixComboBox $widget.f1.f3.f1.db \
	-background white \
	-state disabled \
	-command "MainWin::chooseDb" \
	-listcmd "MainWin::listDbs" \
	-label "DATABASE: " \
	-history false

    tixComboBox $widget.f1.f3.f1.session \
	-background white \
	-state disabled \
	-command "MainWin::chooseSession" \
	-listcmd "MainWin::listSessions" \
	-label "SESSION: " \
	-history false
    
    button $widget.f1.f3.f1.print -command MainWin::print -text Print
    
    if {"[array names env RCDATABASE]" == "RCDATABASE"} {
	if {[file readable $env(RCDATABASE)/codaDB.hosts] == 1} {
	    set fid [open $env(RCDATABASE)/codaDB.hosts r]
	    set hostdata [read -nonewline $fid]
	    set hostlist [split $hostdata \n]
	    foreach i $hostlist {
		$widget.f1.f3.f1.host insert end $i
	    }
	} else {
	    $widget.f1.f3.f1.host insert end alcor
	    $widget.f1.f3.f1.host insert end codahp
	}
    } else {
	$widget.f1.f3.f1.host insert end alcor
	$widget.f1.f3.f1.host insert end codahp
    }
    ################ $widget.f1.f3.f2
    #frame  $widget.f1.f3.f2 -borderwidth 2 -relief groove

    #button $widget.f1.f3.f2.button1 \
	# -state disabled \
	# -text "STOP" \
	# -width 6 \
	# -command [code stopUpdate]

    #button $widget.f1.f3.f2.button2 \
	# -state disabled \
	# -text "START" \
	# -width 6 \
	# -command [code startUpdate]

    #tixControl $widget.f1.f3.f2.period \
	#		-integer 1 -max 999 -min 1 -label "Seconds: " \
	#		-variable [scope period]
    #$widget.f1.f3.f2.period subwidget entry config -width 3

    ###################
    # SETTING GEOMETRY
    ###################
    pack $widget.f1 \
        -anchor center -expand 1 -fill both -side top 
    #pack $widget.f1.f1 \
	#    -anchor center -expand 0 -fill x -side top 
    #pack $widget.f1.f1.label2 \
	#    -anchor center -expand 1 -fill x -side left 
    pack $widget.f1.f3 \
        -anchor center -expand 0 -fill x -side top 
    pack $widget.f1.f3.f1 \
        -anchor nw -expand 0 -fill x -side left 
    #pack $widget.f1.f3.f1.host \
	#    -anchor nw -expand 1 -fill x -side left 
    pack $widget.f1.f3.f1.host.label \
        -anchor center -expand 0 -fill y -side left 
    pack $widget.f1.f3.f1.host.frame \
        -anchor center -expand 1 -fill both -side left 
    pack $widget.f1.f3.f1.host.frame.arrow \
        -anchor center -expand 0 -fill none -padx 1 -side right 
    pack $widget.f1.f3.f1.host.frame.entry \
        -anchor center -expand 1 -fill x -padx 1 -side right 
    pack $widget.f1.f3.f1.db \
        -anchor nw -expand 1 -fill x -side left 
    pack $widget.f1.f3.f1.db.label \
        -anchor center -expand 0 -fill y -side left 
    pack $widget.f1.f3.f1.db.frame \
        -anchor center -expand 1 -fill both -side left 
    pack $widget.f1.f3.f1.db.frame.arrow \
        -anchor center -expand 0 -fill none -padx 1 -side right 
    pack $widget.f1.f3.f1.db.frame.entry \
        -anchor center -expand 1 -fill x -padx 1 -side right 
    pack $widget.f1.f3.f1.session \
        -anchor nw -expand 1 -fill x -side left 
    pack $widget.f1.f3.f1.session.label \
        -anchor center -expand 0 -fill y -side left 
    pack $widget.f1.f3.f1.session.frame \
        -anchor center -expand 1 -fill both -side left 
    pack $widget.f1.f3.f1.session.frame.arrow \
        -anchor center -expand 0 -fill none -padx 1 -side right 
    pack $widget.f1.f3.f1.session.frame.entry \
        -anchor center -expand 1 -fill x -padx 1 -side right 
    pack $widget.f1.f3.f1.print \
        -anchor center -expand 1 -fill x -padx 1 -side right 
    #pack $widget.f1.f3.f2 \
	#    -anchor nw -expand 1 -fill both -side left 
    #pack $widget.f1.f3.f2.button1 \
	#    -anchor center -expand 0 -fill none -side left 
    #pack $widget.f1.f3.f2.button2 \
	#    -anchor center -expand 0 -fill none -side left 
    #pack $widget.f1.f3.f2.period \
	#    -anchor center -expand 1 -fill x -side left 
    #pack $widget.f1.f3.f2.period.label \
	#    -anchor center -expand 0 -fill y -side left 
    #pack $widget.f1.f3.f2.period.frame \
	#    -anchor center -expand 1 -fill both -side left 
    #pack $widget.f1.f3.f2.period.frame.entry \
	#    -anchor center -expand 1 -fill both -side left 
    #pack $widget.f1.f3.f2.period.frame.decr \
	#    -anchor center -expand 1 -fill both -side bottom 
    #pack $widget.f1.f3.f2.period.frame.incr \
	#    -anchor center -expand 1 -fill both -side top 

    pack $widget.f1.f2 \
        -anchor center -expand 1 -fill both -side top 
    pack $widget.f1.f2.scrolly \
        -anchor center -expand 0 -fill y -side right 
    pack $widget.f1.f2.f \
        -anchor center -expand 1 -fill both -side left 
    pack $widget.f1.f2.f.scrollx \
        -anchor center -expand 0 -fill x -side bottom 
    pack $widget.f1.f2.f.canvas \
	-anchor center -expand 1 -fill both -side top
    
    # set the private variable "can"
    set can $widget.f1.f2.f.canvas
    
    ################ bindings
    bind $can <Shift-ButtonPress-1>   {+ MainWin::zoomCursor zoom   %x %y}
    bind $can <Shift-ButtonPress-2>   {+ MainWin::zoomCursor shrink %x %y}
    bind $can <Shift-ButtonPress-3>   {+ MainWin::zoomCursor reset  %x %y}
    bind $can <Shift-ButtonRelease-1> {+ MainWin::zoom 1.25 %x %y}
    bind $can <Shift-ButtonRelease-2> {+ MainWin::zoom 0.80 %x %y}
    bind $can <Shift-ButtonRelease-3> {+ MainWin::unZoom}
    bind $can <Motion> {+ MainWin::zoomPosition %x %y}
    bind $can <2> {+ MainWin::ScrollMark %x %y}
    bind $can <B1-Motion> {+ MainWin::ScrollDrag %x %y}
    bind $can <ButtonRelease-2> {+ MainWin::ScrollDone}
    ####################
    # Color Key widget #
    ####################
    tixBalloon $widget.key -initwait 5000 
    $widget.key bind $can -balloonmsg "key to color code"
    ################ $widget.win
    
    tixScrolledWindow $widget.key.win -scrollbar "auto -x"
    set f [$widget.key.win subwidget window]
    
    set indx 0
    foreach i $processStates {
	set fcolor black
	set bcolor $stateColor($i)
	if {"$bcolor" == "black"} {set fcolor white}
	label $f.$indx -text "$i" -background "$bcolor" -foreground $fcolor -width 2
	pack  $f.$indx -fill both -expand 1
	incr  indx
    }

    pack $widget.key.win -side top -fill both -expand 1
    return
}

#################################

body MainWin::startUpdate {} {
    #puts "startUpdate:"
    incr updating;
    if {$updating >1} {
	set updating 1
	return
    }
    loop
    #    $widget.f1.f1.label2 config -text "UPDATING" -foreground blue
    return
}

body MainWin::stopUpdate {} {
    #puts "stopUpdate:"
    set updating 0;
    if {[catch {dp_after cancel $loopID} msg] != 1} {
	# $widget.f1.f1.label2 config -text "STOPPED UPDATING" -foreground red
    }
    return
}

#################################

body MainWin::chooseHost {_host} {
    #puts "In chooseHost"    
    clearScreen

    # "msq" is the command name
    catch {msq disconnect}
    if { "$_host" != "" } {
	if {[catch {uplevel #0 msql connect msq $_host} msg] == 1} {
	    tk_dialog .err ERROR "Cannot connect with msql database on host $_host: $msg" {error} 0 "DISMISS"
	    $widget.f1.f3.f1.host subwidget entry delete 0 end
	    return
	}
    } else {
	if {[catch {uplevel #0 msql connect msq} msg] == 1} {
	    tk_dialog .err ERROR "Cannot connect with msql databaset: $msg" {error} 0 "DISMISS"
	    $widget.f1.f3.f1.host subwidget entry delete 0 end
	    return
	}
    }
    
    set host $_host
    
    # disable update buttons
    #$widget.f1.f3.f2.button1 config -state disabled
    #$widget.f1.f3.f2.button2 config -state disabled

    # stop the automatic drawing of screens
    set updating 0

    catch {dp_after cancel $loopID}

    # add host to HOST: widget's list (prunehistory option is on)
    $widget.f1.f3.f1.host config -history true
    $widget.f1.f3.f1.host config -history false

    # Since host is chosen, turn on database choice widget
    $widget.f1.f3.f1.db config -state normal
    $widget.f1.f3.f1.db subwidget entry delete 0 end

    # Turn off session choice widget
    $widget.f1.f3.f1.session subwidget entry delete 0 end
    $widget.f1.f3.f1.session config -state disabled

    return
}

#################################

body MainWin::listDbs {} {
    #puts "In listDbs"    
    # get list of databases

    if {[catch {msq get databases} msg] == 1} {
	tk_dialog .err ERROR "$msg" {error} 0 "DISMISS"
	return
    } else {
	set dblist $msg
    }

    # Stick databases into widget's listbox
    $widget.f1.f3.f1.db subwidget listbox delete 0 end
    foreach i $dblist {
	$widget.f1.f3.f1.db insert end $i
    }

    #puts "dblist = $dblist"
    return
}

#################################

body MainWin::chooseDb {_db} {
    #puts "In chooseDb"    
    clearScreen

    if {[catch {msq set database $_db} msg] == 1} {
	tk_dialog .err ERROR "msg" {error} 0 "DISMISS"
	$widget.f1.f3.f1.db subwidget entry delete 0 end
	return
    }

    if {[catch {msq get tables} msg] == 1} {
	tk_dialog .err ERROR "msg" {error} 0 "DISMISS"
	$widget.f1.f3.f1.db subwidget entry delete 0 end
	return
    } else {
	set tables $msg
    }
    #puts "tables = $tables"

    # make sure the sessions table exists
    if {[lsearch -exact $tables sessions] == -1} {
	tk_dialog .err ERROR "The \"sessions\" table does NOT exist in $_db" {error} 0 "DISMISS"
	$widget.f1.f3.f1.db subwidget entry delete 0 end
	return
    }

    set db $_db

    # disable update buttons
    #$widget.f1.f3.f2.button1 config -state disabled
    #$widget.f1.f3.f2.button2 config -state disabled

    # stop the automatic drawing of screens
    set updating 0
    
    catch {dp_after cancel $loopID}

    # Since db is chosen, turn on session choice widget
    $widget.f1.f3.f1.session config -state normal
    $widget.f1.f3.f1.session subwidget entry delete 0 end

    return
}

#################################

body MainWin::listSessions {} {
    #puts "In listSessions"    
    # get list of sessions

    set qcmd "SELECT name FROM sessions"

    if {[catch {msq query $qcmd} msg] == 1} {
	tk_dialog .err ERROR "$msg" {error} 0 "DISMISS"
	$widget.f1.f3.f1.session subwidget entry delete 0 end
	clearScreen
	return
    }

    if {[catch {msq get rest} msg] == 1} {
	tk_dialog .err ERROR "$msg" {error} 0 "DISMISS"
	$widget.f1.f3.f1.session subwidget entry delete 0 end
	clearScreen
	return
    } else {
	set names $msg
    }
    
    #puts "session names = $names"

    # Stick names into widget's listbox
    $widget.f1.f3.f1.session subwidget listbox delete 0 end
    foreach i $names {
	$widget.f1.f3.f1.session insert end $i
    }

    return
}

#################################

body MainWin::chooseSession {sess} {
    global errorInfo
    set session $sess
    #puts "chooseSession:  ..."

    # After picking a "session" in the sessions table,
    # look at its "config" field for a table.
    # Take this table and from the "inputs" and
    # "outputs" lists, construct the graphics.
    
    # activate update buttons
    #$widget.f1.f3.f2.button1 config -state normal
    #$widget.f1.f3.f2.button2 config -state normal

    catch "monitor setSession $session"

    # undo all previous zooming
    set length [llength $zoom]
    for {set i $length} {$i > 0} {incr i -1} {
	set lastZoom [lindex $zoom [expr $i - 1]]
	set exp [expr 1./[lindex $lastZoom 0]]
	set x   [lindex $lastZoom 1]
	set y   [lindex $lastZoom 2]
	$can scale all $x $y $exp $exp
    }
    set zoom {}

    # cancel any plans to update screen
    
    catch {dp_after cancel $loopID}
    # tell everyone we're in updating mode
    set updating 1
    # change label
    # $widget.f1.f1.label2 config -text "UPDATING" -foreground blue
    # draw screen
    if {[catch {drawScreen} msg] == 1} {
	global errorInfo
	tk_dialog .err ERROR "chooseSession: $msg \n$errorInfo" {error} 0 "DISMISS"
	clearSession
	return
    }
    # wait 10 sec before going into infinite update loop
    #$widget.f1.f3.f2.period update
    #puts "chooseSession: period = $period, call loop"
    #set loopID [dp_after [expr 1000*$period] catch \"itcl_context win.obj ::MainWin loop\" res]
    
    return
}
#################################

body MainWin::clearSession {} {
    #puts "clearSession"
    dp_after cancel $loopID
    #puts "clearSession: cancelling $loopID"
    set updating 0
    clearScreen
    $widget.f1.f3.f1.session subwidget entry delete 0 end
    #$widget.f1.f3.f2.button1 config -state disabled
    #$widget.f1.f3.f2.button2 config -state disabled
    # $widget.f1.f1.label2 config -text "STOPPED UPDATING" -foreground red
    return
}

#################################

body MainWin::clearScreen {} {
    #puts "clearScreen"
    # erase canvas
    $can delete all
    
    # delete processes
    set   items [info objects -class Process]
    catch {eval delete object $items}
    
    # delete links
    catch {delete object link.obj}
    
    # set lists holding state info to {}
    currentState.obj clean

    return
}

#################################

body MainWin::drawScreen {} {
    global errorInfo
    #puts "drawScreen"
    clearScreen

    if {[catch {currentState.obj getData $session} msg] == 1} {
	return -code error -errorinfo $errorInfo "drawScreen: $msg"
    } else {
	if {$msg!="none"} {
	    if {[catch {currentState.obj digestData} msg] == 1} {
		return -code error -errorinfo $errorInfo "drawScreen: $msg"
	    } else {
		if {[catch {currentState.obj createObjects} msg] == 1} {
		    return -code error -errorinfo $errorInfo "drawScreen: $msg"
		}
	    }
	}
    }
    reZoom
    return
}

#################################

body MainWin::updateScreen {} {
    global errorInfo

    currentState.obj clean
    if {![catch {currentState.obj cget -config_db} old_config]} {

	if {[catch {currentState.obj getData $session} msg] == 1} {
	    return -code error -errorinfo $errorInfo "updateScreen: $msg"
	} else {

	    if {[catch {currentState.obj digestData} msg] == 1} {
		return -code error -errorinfo $errorInfo "updateScreen: $msg"
	    } else {

		if {"$old_config" == "[currentState.obj cget -config_db]"} {
		    currentState.obj updateIcons
		    if {[catch {link.obj updateLinks} msg] == 1} {
			return -code error -errorinfo $errorInfo "updateScreen: $msg"
		    }
		} else {
		    if {[catch {currentState.obj createObjects} msg] == 1} {
			return -code error -errorinfo $errorInfo "drawScreen: $msg"
		    }
		}
	    }
	}
    }
    return
}

#################################

body MainWin::loop {} {
    #$widget.f1.f3.f2.period update
    #set loopID [dp_after [expr 1000*$period] catch \"itcl_context win.obj ::MainWin loop\" res]
    # puts "\nloop:  newID = $loopID, period = $period"
    updateScreen  
    return
}

#########################
#
# Process class 
#
#########################

class Process {
    constructor { _name _type _inputs _outputs _state x y _tags _size} {}
    destructor  {[MainWin::get -canvas] delete $name}
    
    protected variable name
    protected variable type
    protected variable inputs
    protected variable outputs
    protected variable size
    protected variable state
    protected variable tags
    protected variable xpos
    protected variable ypos
    protected variable text1 {}
    protected variable text2 {}
    protected variable text3 {}
    protected variable font  {}
    protected variable canvasIDs {}
    public    variable config {}
    
    public method get      {option}
    public method move     {X Y}
    public method setText1 {txt ft}
    public method setText2 {txt ft}
    public method setText3 {txt ft}
    public method setFont  {ft}
    public method setState {_state}
    public method binding1 {}
    public method binding2 {}
}

body Process::constructor { _name _type _inputs _outputs { _state null} {x 0} {y 0} {_tags {}} { _size {1 1}}} {
    global stateColor

    set name    $_name
    set type    $_type
    set inputs  $_inputs
    set outputs $_outputs
    set tags    $_tags
    set state   $_state
    set size    $_size
    set xpos    0
    set ypos    0
    set color   black
    set sizex [lindex $size 0]
    set sizey [lindex $size 1]
    
    set can [MainWin::get -canvas]

    set id0 [$can create rectangle 0 0 ${sizex}i ${sizey}i \
		 -outline black \
		 -fill lightgrey \
		 -width 2 \
		 -tags [concat $name rec $tags]]
    
    set id1 [$can create rectangle 0 [expr ${sizey} - 0.2]i ${sizex}i ${sizey}i \
		 -fill "$stateColor($state)" \
		 -outline black \
		 -width 1 \
		 -tags [concat $name rec $tags]]
    
    set id2 [$can create rectangle 0 0 ${sizex}i 0.2i \
		 -outline black \
		 -width 1 \
		 -tags [concat $name rec $tags]]
    
    #if {"$stateColor($state)" == "black"} {
    #	set color "white"
    #}

    set id3 [$can create text 0.05i 0.2i \
		 -anchor sw \
		 -text "$name" \
		 -fill black \
		 -tags [concat $name txt $tags]]

    set id4 [$can create text [expr ${sizex} -0.05]i 0.2i \
		 -anchor se \
		 -text "$text1" \
		 -fill black \
		 -tags [concat $name txt $tags]]	

    set id5 [$can create text [expr ${sizex}/2.0]i [expr ${sizey} - 0.1]i \
		 -anchor center \
		 -fill $color \
		 -text "$text2" \
		 -tags [concat $name txt $tags]]

    set id6 [$can create text 0.05i 0.25i \
		 -anchor nw \
		 -fill black \
		 -text "$text3" \
		 -tags [concat $name txt $tags]]
    
    set canvasIDs [list $id0 $id1 $id2 $id3 $id4 $id5 $id6]
    
    setText1 [string tolower $type] "-*-courier-medium-r-normal--*-120-*"
    setText2 $_state "-*-courier-medium-r-normal-*-*-120-*"
    setText3 "no info" "-*-courier-medium-r-normal-*-*-100-*"
    
    move $x $y
    
    # Bindings are to id's and not the tag $name!
    # Binding to $name cause the same script to be executed
    # for each of the items with tag $name (+1)!
    $can bind $id0 <Button-1> "+ $this binding1"
    $can bind $id0 <Button-2> "+ $this binding2"
    $can bind $id1 <Button-1> "+ $this binding1"
    $can bind $id1 <Button-2> "+ $this binding2"
    $can bind $id2 <Button-1> "+ $this binding1"
    $can bind $id2 <Button-2> "+ $this binding2"
    $can bind $id3 <Button-1> "+ $this binding1"
    $can bind $id3 <Button-2> "+ $this binding2"
    $can bind $id4 <Button-1> "+ $this binding1"
    $can bind $id4 <Button-2> "+ $this binding2"
    $can bind $id5 <Button-1> "+ $this binding1"
    $can bind $id5 <Button-2> "+ $this binding2"
    $can bind $id6 <Button-1> "+ $this binding1"
    $can bind $id6 <Button-2> "+ $this binding2"

    return
}

body Process::binding1 {} {
    if {[catch {DP_ask $name pwd} msg] == 1} {
	tk_dialog .err ERROR "Cannot connect to \"$name\"" {error} 0 "DISMISS"
	return
    }
    set objs [info objects -class Shell]
    foreach i $objs {
	if {"[$i get -process]" == "$name"} {return}
    }
    #puts "create shell$name.obj in b1"
    catch {Shell shell$name.obj $name}
    return
}

body Process::binding2 {} {
    if {[catch {DP_ask $name pwd} msg] == 1} {
	tk_dialog .err ERROR "Cannot connect to \"$name\"" {error} 0 "DISMISS"
	return
    }
    if {[winfo exists .win$name] == 1} {return}
    set obj [ConfigWin config$name.obj .win$name]
    $obj setProcess $name
    #puts "b2: created $obj, connected to $name"
    return
}

body Process::get {{option -name}} {
    # return values of private variables
    switch -- $option {
	-name    {return $name}
	-type    {return $type}
	-inputs  {return $inputs}
	-outputs {return $outputs}
	-size    {return $size}
	-state   {return $state}
	-tags    {return $tags}
	-xpos    {return $xpos}
	-ypos    {return $ypos}
	-text1   {return $text1}
	-text2   {return $text2}
	-font    {return $font}
	-ids     {return $canvasIDs}
    }
    error "Process::get bad option \"$option\""
}


body Process::move {X Y} {
    [MainWin::get -canvas] move $name ${X}i ${Y}i
    set xpos [expr $xpos + $X]
    set ypos [expr $ypos + $Y]
}

body Process::setText1 {txt {ft {}}} {
    global stateColor
    set can [MainWin::get -canvas]
    set color black
    $can itemconfigure [lindex $canvasIDs 4] -text $txt -fill $color
    if {"$ft" != ""} {$can itemconfigure [lindex $canvasIDs 4] -font $ft}
}

body Process::setText2 {txt {ft {}}} {
    global stateColor
    set can [MainWin::get -canvas]
    set color black
    if {"$stateColor($state)" == "black"} {
	set color "white"
    }
    $can itemconfigure [lindex $canvasIDs 5] -text $txt -fill $color
    if {"$ft" != ""} {$can itemconfigure [lindex $canvasIDs 5] -font $ft}
}

body Process::setText3 {txt {ft {}}} {
    global stateColor
    set can [MainWin::get -canvas]
    $can itemconfigure [lindex $canvasIDs 6] -text $txt
    if {"$ft" != ""} {$can itemconfigure [lindex $canvasIDs 6] -font $ft}
}

body Process::setFont {ft} {
    set can [MainWin::get -canvas]
    $can itemconfigure [lindex $canvasIDs 3] -font $ft
    $can itemconfigure [lindex $canvasIDs 4] -font $ft
    $can itemconfigure [lindex $canvasIDs 5] -font $ft
}

body Process::setState {_state} {
    global stateColor

    set state $_state
    set can   [MainWin::get -canvas]
    set color black
    if {"$stateColor($state)" == "black"} {
	set color "white"
    }

    $can itemconfigure [lindex $canvasIDs 1] -fill "$stateColor($state)"  
    $can itemconfigure [lindex $canvasIDs 5] -fill $color
    setText2 $state

}

#########################
#
# Link class 
#
#########################

class Link {
    constructor {} {}
    destructor  {[MainWin::get -canvas] delete link}
    
    private variable links {} 
    private variable states   
    
    public method updateLinks {}
    public method get         {option}
    public method add         {from to}
}

body Link::add {from to} {
    global linkColor

    
    # draw lines from process to process, representing links
    set can [MainWin::get -canvas]
    set index 0
    #puts "constructing link $i"

    lappend links ${to}.${from}

    # find position of "from" process icon
    set fromSize  [$from.obj get -size]
    set fromSizeX [lindex $fromSize 0]
    set fromSizeY [lindex $fromSize 1]
    set fromX     [expr [$from.obj get -xpos] + $fromSizeX]
    set fromY     [expr [$from.obj get -ypos] + $fromSizeY/2.]
    
    # find position of "to" process icon
    set toSize    [$to.obj get -size]
    set toSizeY   [lindex $toSize 1]
    set toX       [$to.obj get -xpos]
    set toY       [expr [$to.obj get -ypos] + $toSizeY/2.]
    
    #draw line
    $can create line ${fromX}i ${fromY}i ${toX}i ${toY}i \
	-fill "$linkColor(down)" \
	-width 2 \
	-tags [list ${to}.${from} link] \
	-arrow last
    
    return
}

body Link::get {{option -links}} {
    # return values of private variables
    switch -- $option {
	-links  {return $links}
	-states {return $states}
    }
    error "Link::get bad option \"$option\""
}

body Link::updateLinks {} {
    global linkColor errorInfo

    set can [MainWin::get -canvas]

    # get link state data from the LINK table
    set qcmd "SELECT name,state FROM links"
    if {[catch {msq query $qcmd} msg] == 1} {
	return -code error -errorinfo $errorInfo $msg
    }
    if {[catch {msq get rest} msg] == 1} {
	return -code error -errorinfo $errorInfo $msg
    }

    set linkdata $msg

    # only add links that are relevant to our current state
    foreach i $linkdata {
	#puts "     $i"
	# decode link data into "to", "from", & "state" parts
	regexp {^(.+)->(.+) +(up|down)$} $i all from to state
	#puts "       to = $to, from = $from, state = $state"
	if {("$state" != "down") && ("$state" != "up")} {
	    set state down
	}
	set linkitem [$can find withtag ${to}.$from]
	if {"$linkitem" != ""} {
	    $can itemconfig $linkitem -fill "$linkColor($state)"
	}
    }
    
}

#########################
#
# State class 
#
#########################

class State {
    constructor {} {}
    destructor  {}

    private variable names   {}
    private variable types   {}
    private variable status  {}
    private variable inputs  {}
    private variable outputs {}
    private variable data
    public variable config_db {}
    set data(init) {}

    public method getData       {session}
    public method digestData    {}
    public method createObjects {}

    public method get           {option}
    public method dump          {}
    public method clean         {}
    public method updateIcons   {}
    public method addProcess    {_name _type _inputs _outputs _status}
    public method changeStatus  {_name _status}
}

##########

body State::get {{option -names}} {
    # return values of private variables
    switch -- $option {
	-data    {return $data}
	-names   {return $names}
	-types   {return $types}
	-status  {return $status}
	-inputs  {return $inputs}
	-outputs {return $outputs}
    }
    error "State::get bad option \"$option\""
}
##########

body State::clean {} {    
    set names   {}
    set types   {}
    set status  {}
    set inputs  {}
    set outputs {}
    return
}
##########

body State::dump {} {
    puts "processes = $names"
    puts "types     = $types"
    puts "status    = $status"
    puts "inputs    = $inputs"
    puts "outputs   = $outputs"
    return
}
##########

body State::updateIcons {} {    
    set length [llength $names]
    for {set i 0} {$i < $length} {incr i} {
	#puts "status is \"$status\""
	[lindex $names $i].obj setState [lindex $status $i]
    }
    return
}
##########

body State::changeStatus {_name _status} {
    set index [lsearch -exact $names $_name]
    if {$index == -1} {
	tk_dialog .err ERROR "changeStatus: no such process as \"$_name\"" {error} 0 "DISMISS"
    } else {
	set status [lreplace $status $index $index $_status]
    }
    return
}
##########

body State::addProcess {_name _type _inputs _outputs _status} {
    lappend names   $_name
    lappend types   $_type
    lappend status  $_status
    lappend inputs  $_inputs 
    lappend outputs $_outputs
    return
}

##########

body State::getData {session} {
    global errorInfo
    #puts "getData" 

    # read the config field
    set qcmd "SELECT config FROM sessions WHERE name = '$session'"

    if {[catch {msq query $qcmd} msg] == 1} {
	return -code error -errorinfo $errorInfo "getData: $msg"
    }
    if {[catch {msq get rest} msg] == 1} {
	return -code error -errorinfo $errorInfo "getData: $msg"
    } else {
	set config_db [lindex $msg 0]
    }
    if {"$config_db"=="{}"} {
	return "none"
    }
    #    puts "config_db = $config_db"

    # get process status/type data from the process table
    set qcmd "SELECT name,state,type FROM process"

    if {[catch {msq query $qcmd} msg] == 1} {
	return -code error -errorinfo $errorInfo "getData: $msg"
    }
    if {[catch {msq get rest} msg] == 1} {
	return -code error -errorinfo $errorInfo "getData: $msg"
    } else {
	set data(process) $msg
    }
    #puts "process data = \n$data(process)"

    # get process connection data from the $config_db table
    set qcmd "SELECT name,inputs,outputs FROM $config_db WHERE name not like 'file%'"

    if {[catch {msq query $qcmd} msg] == 1} {
	return -code error -errorinfo $errorInfo "getData: $msg"
    }
    if {[catch {msq get rest} msg] == 1} {
	return -code error -errorinfo $errorInfo "getData: $msg"
    } else {
	set data(config) $msg
    }
    #puts "config data = \n$data(config)"

    # get icon position data from the ${config_db}_pos table
    set qcmd "SELECT name,row,col FROM [set config_db]_pos"

    if {[catch {msq query $qcmd} msg] == 1} {
	return -code error -errorinfo $errorInfo "getData: $msg"
    }
    if {[catch {msq get rest} msg] == 1} {
	return -code error -errorinfo $errorInfo "getData: $msg"
    } else {
	set data(position) $msg
    }
    #puts "position data = \n$data(position)"

    return
}
##########

body State::digestData {} {
    global errorInfo
    # Use capital letters for vars to avoid collision with object vars  
    #puts "digestData"

    # Process the process (state) data
    foreach i $data(process) {
	lappend Names  [lindex $i 0]
	lappend States [lindex $i 1]
	lappend Types  [lindex $i 2]
    }
    #puts "  names  = $Names"
    #puts "  states = $States"
    #puts "  types  = $Types\n"

    # Process the config (connection) data
    # Extract the component names from the config data

    set count [llength $data(config)]
    for {set i 0} {$i < $count} {incr i} {
	set Name [lindex [lindex $data(config) $i] 0]
	set Ins  [lindex [lindex $data(config) $i] 1]
	set Outs [lindex [lindex $data(config) $i] 2]
	set Inputs  {}
	set Outputs {}

	# For each item from the config table, store data in the State object.
	set index  [lsearch -exact $Names $Name]
	if {$index == -1} {
	    return -code error -errorinfo $errorInfo "digestData: Can't find process data on \"$Name\""
	} else {
	    set Status [lindex $States $index]
	    set Type   [lindex $Types  $index]
	}

	# decode ins and outs by discarding stuff after colon
	foreach j $Ins {
	    if {[regexp {^(.+)\:.+$} $j all in] == 1} {
		lappend Inputs $in
	    } else {
		return -code error -errorinfo $errorInfo "digestData: Problem decoding config table input data"
	    }
	}

	if {"$Ins"=="" && "$Type"=="ER"} {
	    lappend Inputs DD
	}
	case $Outs {
	    "" {
		if {"$Type"=="EB"} {
		    set Outputs DD
		} else {
		    set Outputs ""
		}
	    }
	    none {
		set Outputs none
	    }
	    debug {
		set Outputs debug
	    }
	    file* {
		set Outputs binaryfile
	    }
	    coda* {
		set Outputs codaFile
	    }
	    default {
		foreach j $Outs {
		    if {[regexp {^(.+)\:.+$} $j all out] == 1} {
			lappend Outputs $out
		    } else {
			return -code error -errorinfo $errorInfo "digestData: Problem decoding config table output data"
		    }
		}
	    }
	}

	#puts "digestData: $i $Name $Type $Inputs $Outputs $Status"
	addProcess $Name $Type $Inputs $Outputs $Status
    }

    return
}
##########

body State::createObjects {} {
    global errorInfo
    #puts "createObjects:"
    # Create a Link object
    if {[catch {Link ::link.obj} msg] == 1 } {
	return -code error -errorinfo $errorInfo $createObjects: msg
    }

    # Process position data
    foreach i $data(position) {
	lappend names_pos [lindex $i 0]
	lappend rows      [lindex $i 1]
	lappend cols      [lindex $i 2]
    }

    set count [llength $names]

    # For each object in the state, create an icon (Process object)
    for {set i 0} {$i < $count} {incr i} {
	set name [lindex $names   $i]
	set type [lindex $types   $i]
	set in   [lindex $inputs  $i]
	set out  [lindex $outputs $i]
	set stat [lindex $status  $i]
	
	set index [lsearch -exact $names_pos $name]
	if {$index == -1} {
	    return -code error -errorinfo $errorInfo "createObjects: Problem moving \"$name\""
	}
	set x [expr 1.25*[lindex $cols $index] - 2.0]
	set y [expr      [lindex $rows $index] - 1.5]
	
	# Create Process object. Create the object in global
	# space so its name is $name (& not garbage::name).
	#puts "    $name: $type, $in, $out, $stat, ($x, $y)"
	Process ::$name.obj $name $type $in $out $stat $x $y
    }
    set can [MainWin::get -canvas]
    set ddInputs {}
    for {set i 0} {$i < $count} {incr i} {
	set name [lindex $names   $i]
	set outputs_  [lindex $outputs $i]
	set in   [lindex $inputs  $i]

	set x [${name}.obj get -xpos]
	set y [${name}.obj get -ypos]
	puts "input is $in"
	if {"$in" == "DD"} {
	    lappend ddInputs $name
	}
	case $outputs_ {
	    default {
		foreach output $outputs_ {
		    link.obj add $name $output
		}
	    }
	    {} {}
	    binary* {
		$can create image 1.25i 0.25i \
		    -image binFile \
		    -anchor nw \
		    -tags [concat $name file.$name] 
		$can create line 1.0i 0.5i 1.25i 0.5i -fill black -width 2 -arrow last -tags \
		    [concat $name file.$name] 
		$can move file.$name ${x}i ${y}i
	    }
	    coda* {
		$can create image 1.25i 0.25i \
		    -image codaFile \
		    -anchor nw \
		    -tags [concat $name file.$name] 
		$can create line 1.0i 0.5i 1.25i 0.5i -fill black -width 2 -arrow last -tags \
		    [concat $name file.$name] 
		$can move file.$name ${x}i ${y}i
	    }
	    none {
		$can create image 1.25i 0.25i \
		    -image trashData \
		    -anchor nw \
		    -tags [concat $name file.$name] 
		$can create line 1.0i 0.5i 1.25i 0.5i -fill black -width 2 -arrow last -tags \
		    [concat $name file.$name] 
		$can move file.$name ${x}i ${y}i
	    }
	    DD {
		if {![catch "Process ::dd.obj DD DD \"\" \"\" down $x $y" res]} {
		    dd.obj move 1.25 0.0
		} else {
		    puts "result $res"
		}
		
		link.obj add $name dd
		#$can create image 1.25i 0.25i \
		    # -image binFile \
		    # -anchor nw \
		    # -tags [concat $name file.$name] 
		#$can create line 1.0i 0.5i 1.25i 0.5i -fill black -width 2 -arrow last -tags \
		    #[concat $name file.$name] 
		#$can move file.$name $x $y
	    }
	}
	
    }
    foreach inp $ddInputs {
	link.obj add dd $inp
    }
    return
}

#########################
#
# ShellText class:
#	This class implements a toplevel window with a
#	text widget whose bindings (see shellTextBinding)
#	behave as a dp shell run on the interpreter "$process".
#	This is done by using "DP_ask $process". Output of the
#	command is sent back to the shell.
#
#########################

class Shell {
    constructor { _process} {}
    destructor  {destroy $top}
    
    protected variable process	;# ROC2, EB5, etc.
    protected variable top	;# top level window name
    protected variable widget	;# text widget address
    protected variable cmd {}	;# last entered command
    protected variable oldcmd {}  ;# previous command
    protected variable complete 1 ;# =1 if last entered command complete
    
    private method findCmd1 {}
    private method findCmd2 {}
    private method execCmd  {}
    private method reset    {}
    public  method get      {option}
}

#---------------------
body Shell::constructor { _process} {
    set process    $_process
    # window can't start with uppercase letter
    set top     .[string tolower $process]shell
    toplevel    $top
    wm geometry $top 500x400
    wm title    $top "$process ShellText"
    wm protocol $top WM_DELETE_WINDOW "delete object $this"

    ################ .$top.f1
    frame $top.f1

    tixScrolledText $top.f1.text -scrollbar auto
    
    pack $top.f1.text -fill both -expand 1
    pack $top.f1 -fill both -expand 1

    ###########################
    
    set widget [$top.f1.text subwidget text]
    $widget insert "1.0" "% "
    
    # don't change order of bindtags list
    bindtags $widget [list ShellText $widget $top all]
    bind $widget <Return> [code $this findCmd1]
    bind $widget <ButtonRelease-2> [code $this findCmd2]
    bind $widget <Control-c> [code $this reset]
    bind $widget <Control-d> [code delete object $this]
    return
}

#---------------------
body Shell::get {option} {
    switch -- $option {
	-process  {return $process}
	-top      {return $top}
	-widget   {return $widget}
	-cmd      {return $cmd}
	-oldcmd   {return $oldcmd}
	-complete {return $complete}
    }
    error "Shell::get bad option \"$option\""
}


#---------------------
body Shell::reset {} {
    set cmd ""
    set complete 1
    $widget insert "end -1c" "\n% "
    return
}

#---------------------
body Shell::findCmd1 {} {
    if {$complete == 0} {
	set c [$widget get "insert -1l linestart" "insert -1l lineend"]
    } else {
	set c [$widget get "insert -1l linestart +2c" "insert -1l lineend"]
    }
    #puts "c1 = $c"
    if {$complete == 0} {set c "$cmd$c"}
    if {[command complete $c] == 0} {
	set cmd "${c}\n"
	set complete 0
    } else {
	set cmd "$c"
	set complete 1
    }
    #puts "cmd1 = $cmd, $complete"
    execCmd
    return
}

#---------------------
body Shell::findCmd2 {} {
    # add stuff already on line to front of selection
    if {$complete == 0} {
	set c [$widget get "place linestart" "place +1c"]
    } else {
	set c [$widget get "place linestart +2c" "place +1c"]
    }
    #puts "front = $c, place = [$widget index place]"
    set c "$c[$widget get sel.first sel.last]"
    #puts "c2 = $c"
    if {$complete == 0} {set cmd "$cmd$c"} else {set cmd $c}
    if {[command complete $cmd] == 0} {set complete 0} else {set complete 1}
    #puts "cmd2 = $cmd, $complete"
    # If last char of selection isn't "\n", backup 1 line (text binding
    # inserts "\n") and remove last line from cmd if incomplete.
    if {"[string index $c [expr [string length $c]-1]]" != "\n"} {
	if {$complete == 0} {
	    $widget delete "insert linestart -1c" "insert lineend"
	    set cmd [string range $cmd 0 [string last "\n" $cmd]]
	    return
	}
    }
    execCmd
    return
}

#---------------------
body Shell::execCmd {} {
    if {"[string trim $cmd]" == "clear"} {
	$widget delete 1.0 end
	$widget insert 1.0 "% "
    } elseif {"[string trim $cmd]" == ""} {
    } elseif {"[string trim $cmd]" == "!!"} {
	catch {eval DP_ask $process $oldcmd} msg
	$widget delete "insert linestart" "insert lineend"
	$widget insert insert "${oldcmd}\n"
	if {"$msg" == ""} {
	    $widget insert insert "% "
	} else {
	    $widget insert insert "${msg}\n% "
	} 
	$widget see "end -1c"
    } elseif {$complete == 0} {
	$widget delete "insert linestart" "insert lineend"
	$widget insert insert "  "
    } else {
	catch {eval DP_ask $process $cmd} msg
	$widget delete "insert linestart" "insert lineend"
	if {"$msg" == ""} {
	    $widget insert insert "% "
	} else {
	    $widget insert insert "${msg}\n% "
	} 
	$widget see "end -1c"
	set oldcmd $cmd
    }
    return
}

####################
# This procedure defines the shell-like bindings for Tk text widgets
# & provides the procedures that help in implementing the bindings.
###################

proc shellTextBinding {} {
    # Changed:
    bind ShellText <1> {
	global tkPriv
	set tkPriv(selectMode) char
	set tkPriv(mouseMoved) 0
	set tkPriv(pressX) %x
	
	if {[%q compare @%x,%y >= "end -1c linestart +2c"]} {
	    %q mark set insert @%x,%y
	    %q mark set anchor insert
	} else {
	    %q mark set anchor @%x,%y
	}
	if {[%q cget -state] == "normal"} {focus %W}
	
	%q tag remove sel 0.0 end
    }
    bind ShellText <B1-Motion> {
	set tkPriv(x) %x
	set tkPriv(y) %y
	tkTextSelectTo %W %x %y
    }
    # Changed:
    bind ShellText <Double-1> {
	set tkPriv(selectMode) word
	tkTextSelectTo %W %x %y
    }
    # Changed:
    bind ShellText <Triple-1> {
	set tkPriv(selectMode) line
	tkTextSelectTo %W %x %y
    }
    bind ShellText <Shift-1> {
	tkTextResetAnchor %W @%x,%y
	set tkPriv(selectMode) char
	tkTextSelectTo %W %x %y
    }
    bind ShellText <Double-Shift-1> {
	set tkPriv(selectMode) word
	tkTextSelectTo %W %x %y
    }
    bind ShellText <Triple-Shift-1> {
	set tkPriv(selectMode) line
	tkTextSelectTo %W %x %y
    }
    bind ShellText <B1-Leave> {
	set tkPriv(x) %x
	set tkPriv(y) %y
	tkTextAutoScan %W
    }
    bind ShellText <B1-Enter> {
	tkCancelRepeat
    }
    bind ShellText <ButtonRelease-1> {
	tkCancelRepeat
    }
    # Changed:
    bind ShellText <Left> {
	if {[%q compare insert > "insert linestart +2c"]} {
	    tkTextSetCursor %W [%q index {insert - 1c}]
	}
    }
    # Changed:
    bind ShellText <Right> {
	if {[%q compare insert != "insert lineend"]} {
	    tkTextSetCursor %W [%q index {insert + 1c}]
	}
    }
    # Changed:
    bind ShellText <Shift-Left> {
	if {[%q compare insert > "insert linestart +2c"]} {
	    tkTextKeySelect %W [%q index {insert - 1c}]
	}
    }
    bind ShellText <Shift-Right> {
	tkTextKeySelect %W [%q index {insert + 1c}]
    }
    bind ShellText <Control-Down> {
	tkTextSetCursor %W [tkTextNextPara %W insert]
    }
    # Changed:
    bind ShellText <Shift-Control-Left> {
	if {[%q compare insert > "end -1c linestart +2c"]} {
	    tkTextKeySelect %W [%q index {insert - 1c wordstart}]
	}
    }
    bind ShellText <Shift-Control-Right> {
	tkTextKeySelect %W [%q index {insert wordend}]
    }
    bind ShellText <Shift-Control-Down> {
	tkTextKeySelect %W [tkTextNextPara %W insert]
    }
    # Changed:
    bind ShellText <Home> {
	tkTextSetCursor %W {insert linestart +2c}
    }
    # Changed:
    bind ShellText <Shift-Home> {
	tkTextKeySelect %W {insert linestart +2c}
    }
    bind ShellText <End> {
	tkTextSetCursor %W {insert lineend}
    }
    bind ShellText <Shift-End> {
	tkTextKeySelect %W {insert lineend}
    }
    # Changed:
    bind ShellText <Return> {
	%q mark set insert "end -1c"
	new_tkTextInsert %W "\n%% "
    }
    # Changed:
    bind ShellText <Delete> {
	if {([%q compare insert > "end -1c linestart"]) &&
	    ([%q compare insert < "end -1c lineend"])} {
	    %q delete insert
	    %q see insert
	}
    }
    # Changed:
    bind ShellText <BackSpace> {
	if {[%q compare insert > "end -1c linestart +2c"]} {
	    %q delete insert-1c
	    %q see insert
	}
    }
    bind ShellText <Control-space> {
	%q mark set anchor insert
    }
    bind ShellText <Select> {
	%q mark set anchor insert
    }
    bind ShellText <Control-Shift-space> {
	set tkPriv(selectMode) char
	tkTextKeyExtend %W insert
    }
    bind ShellText <Shift-Select> {
	set tkPriv(selectMode) char
	tkTextKeyExtend %W insert
    }
    bind ShellText <Control-slash> {
	%q tag add sel 1.0 end
    }
    bind ShellText <Control-backslash> {
	%q tag remove sel 1.0 end
    }
    tkTextClipboardKeysyms F16 F20 F18
    bind ShellText <Insert> {
	catch {new_tkTextInsert %W [selection get -displayof %W]}
    }
    bind ShellText <KeyPress> {
	new_tkTextInsert %W %A
    }

    # Ignore all Alt, Meta, and Control keypresses unless explicitly bound.
    # Otherwise, if a widget binding for one of these is defined, the
    # <KeyPress> class binding will also fire and insert the character,
    # which is wrong.  Ditto for <Escape>.

    bind ShellText <Alt-KeyPress> {
	# nothing 
    }
    bind ShellText <Meta-KeyPress> {
	# nothing
    }
    bind ShellText <Control-KeyPress> {
	# nothing
    }
    bind ShellText <Escape> {
	# nothing
    }
    bind ShellText <KP_Enter> {
	# nothing
    }
    
    # Changed:
    bind ShellText <Control-h> {
	if [%q compare insert > "end -1c linestart +2c"] {
	    %q delete insert-1c
	    %q see insert
	}
    }
    bind ShellText <B2-Motion> {
	if {(%x != $tkPriv(x)) || (%y != $tkPriv(y))} {
	    set tkPriv(mouseMoved) 1
	}
	if $tkPriv(mouseMoved) {
	    %q scan dragto %x %y
	}
    }
    # Changed:
    bind ShellText <ButtonRelease-2> {
	set sel [selection get -displayof %W]
	# mark one char before insertion of sel
	%q mark set place "end -2c"
	catch {%q insert "end -1c" $sel}
	%q mark set insert "end -1c"
	if {[string first "\n" $sel] != -1} {
	    if {"[string index $sel [expr [string length $sel]-1]]" ==  "\n"} {
		%q insert insert "%% "
	    } else {
		%q insert insert "\n%% "
	    }
	}
	%q see "end -1c"
    }
}
# Changed tkTextInsert procedure
# Only insert text on last line.
# If selection in last line, delete selection.
proc new_tkTextInsert {w s} {
    set q [winfo command $w]
    if {($s == "") || ([$q cget -state] == "disabled")} {
	return
    }
    catch {
	if {[$q compare sel.first <= insert]
	    && [$q compare sel.last >= insert]
	    && [$q compare sel.first >= "end -1c linestart +2c"]
	    && [$q compare insert >= "end -1c linestart +2c"]} {
	    $q delete sel.first sel.last
	}
    }
    if [$q compare insert < "end -1c linestart"] {
	$q insert "end -1c" $s
    } else {
	$q insert insert $s
    }
    $q see "end -1c"

    return
}

############################
# Provide binding for text widget for shell-like operation
shellTextBinding
############################ 


#########################
#
# Option class
#	A result of the configure command to an dp object (eg. EB2)
#	is a list of options, initial values, and current values.
#	This class is a displayed frame (row) containing a single
#	option (name), initial value (initval), and current value
#	(val).
#
#########################

class Option {
    constructor {_name _initval _val _frame cmdobj winobj} {}
    destructor  {pack forget $frame; destroy $frame}

    private variable name
    private variable initval
    private variable val
    private variable frame
    
    public method get       {option}
    public method clear     {}
    public method setVal    {newVal}
    public method updateOpt {} {set val [$frame.entry3 get]; $frame.entry3 config -foreground red}
}

###########
body Option::constructor {_name _initval _val _frame cmdobj winobj} {
    set name    $_name
    set initval $_initval
    set val     $_val
    set frame   $_frame
    
    # place data into entry widgets & pack
    frame $frame
    entry $frame.entry1 \
	-width 20 \
	-background {gray80} \
	-highlightcolor red
    $frame.entry1 insert end "$name"
    $frame.entry1 config -state disabled
    
    entry $frame.entry2 \
	-width 20 \
	-background {gray80} \
	-highlightcolor red
    $frame.entry2 insert end "$initval"
    $frame.entry2 config -state disabled
    
    set widget "$frame.entry3"
    entry $widget \
	-width 20 \
	-highlightcolor red
    $widget insert end "$val"	
    bind $widget <Return> "+ $this updateOpt; $cmdobj addOption $this
			     $cmdobj formCmd $winobj"
    
    pack $frame.entry1 $frame.entry2 $widget -side left -anchor nw
    return
}
###########
body Option::clear {} {
    $frame.entry3 config -foreground black
    return
}

###########
body Option::get {option} {
    switch -- $option {
	-name    {return $name}
	-initval {return $initval}
	-val     {return $val}
	-frame   {return $frame}
	-pair    {return "$name $val"}
    }
    error "Option::get bad option \"$option\""
}

###########
body Option::setVal {newVal} {
    set val $newVal
    $frame.entry3 config -foreground black
    $frame.entry3 delete 0 end
    $frame.entry3 insert end "$val"
    return
}


#########################
#
# Command class
#	A command class object contains a constructed configure command
#	with various user-defined options and values to be sent to a dp
#	object (eg. EB2). 
#
#########################

class Command {
    constructor {} {}
    destructor  {}
    
    private variable object    {}  ;# object of interest (i.e. EB2, ROC5)
    private variable objectOld {}  ;# previous object
    private variable command   {}  ;# new configure command
    private variable cmdOpts   {}  ;# string of configure option-val pairs
    # constructed by user
    public method get          {option}
    public method clear        {args} 
    public method setObject    {winobj args}
    public method addOption    {optobj}
    public method formCmd      {winobj}
    public method execCmd      {} {eval $command}
}

###########
body Command::get {option} {
    switch -- $option {
	-object    {return $object}
	-objectOld {return $objectOld}
	-command   {return $command}
	-cmdOpts   {return $cmdOpts}
    }
    error "Command::get bad option \"$option\""
}

###########
body Command::clear {args} {
    if {"$args" == ""} {
	set command ""
	set cmdOpts ""
	return
    }
    set option [lindex $args 0]
    switch -- $option {
	-command   {set command ""}
	-cmdOpts   {set cmdOpts ""}
    }
    error "Command::clear bad option \"$option\""
}

###########
body Command::setObject {winobj args} {
    # When called with the "-command" option of a widget there is an
    # extra argument given (object) than when it's called with a binding.
    # This allows the calling widget's value to be updated when the
    # widget is left by <Leave> as well as when <Return> or a
    # menu-selection is used.
    set window [$winobj get -window]
    set objectOld $object
    if {"$args" == ""} {
	set object [[$window.f1.f2.lf1 subwidget frame].object cget -selection]
    } else {
	set object [lindex $args 0]
    }
    if {"$objectOld" == "$object"} {
	return
    }
    if {"$object" == ""} {
	$winobj deleteOptions
	return
    }
    if {[catch {$winobj updateWindow} msg] == 1} {
	tk_dialog .err ERROR "$msg" {error} 0 "DISMISS"
	$winobj deleteOptions
	return
    }
    
    # clear the command display widget
    $window.f1.f4.command config -text ""
    # re-enable updating widgets
    set win [$window.f1.f2.lf2 subwidget frame]
    $win.button    configure -state normal
    $win.f1.period configure -state normal
    return
}

###########
body Command::formCmd {winobj} {
    # create the new command 
    set command "DP_ask $object configure $cmdOpts"

    # update the command display widget
    set window [$winobj get -window]
    $window.f1.f4.command config -text ""
    $window.f1.f4.command config -text "$command"

    return
}

###########
body Command::addOption {optobj} {
    set opt    [$optobj get -name]
    set optVal [$optobj get -pair]
    # replace the "-option val" in cmdOpts with current val
    if {[regsub -- "$opt \[\^-\]+" $cmdOpts "$optVal " x] == 0} {
	set cmdOpts "$cmdOpts $optVal"
    } else {
	set cmdOpts "$x"
    }
    return
}


#########################
#
# ConfigWin class
#	This class creates a toplevel window with all the attendant
#	functionality to view a list of all the options, initial values,
#	and current values of a dp object (eg. EB2). It allows one to
#	change the current values by issuing a new "configure" command.
#
#########################

class ConfigWin {
    constructor {{winName .win}} {}
    destructor  {}
    
    private variable window ""  ;# toplevel window name
    private common period 	;# Seconds between automatic updates.
    # By making period common, it can be global
    # and thus become a widget's "variable"
    set period(init) ""
    private variable loopID ""  ;# id of the "after" update command
    private variable optionObjects {} ;# list of option objects
    private variable optionCount   0  ;#  no. of option objects

    public method get           {option}
    public method dismiss       {}
    public method setProcess    {process}
    public method setPeriod     {time}
    public method resetOptions  {}
    public method deleteOptions {}
    public method createOptions {}
    public method updateOptions {}
    public method updateWindow  {}
    public method displayData   {}
    public method updateLoop    {args}
}

###########
body ConfigWin::get {option} {
    switch -- $option {
	-window  {return $window}
	-period  {return $period($window)}
	-objects {return $optionObjects}
	-count   {return $optionCount}
	-process {return [cmd$window.obj get -object]}
    }
    error "ConfigWin::get bad option \"$option\""
}

###########
body ConfigWin::dismiss {} {
    delete object $this
    #if {"[info objects -class ConfigWin]" == ""} {exit}
    return
}

###########
body ConfigWin::setProcess {process} {
    [$window.f1.f2.lf1 subwidget frame].object configure -value $process
    return
}

###########
body ConfigWin::setPeriod {time} {
    [$window.f1.f2.lf2 subwidget frame].f1.period configure -value $time
    return
}

###########
body ConfigWin::resetOptions {} {
    foreach i $optionObjects {$i clear}
    return
}

###########
body ConfigWin::deleteOptions {} {
    catch {
	dp_after cancel $loopID
	setPeriod 0
	#clear object widget
	setProcess ""
	# delete option objects
	eval delete object $optionObjects
	set optionObjects ""
	set optionCount 0
	#clear command object
	cmd$window.obj clear
	# clear command display widget
	$window.f1.f4.command config -text ""
	# disable window updating widgets
	set win [$window.f1.f2.lf2 subwidget frame]
	$win.button    configure -state disabled
	$win.f1.period configure -state disabled
    }
    return
}

###########
body ConfigWin::updateWindow {} {
    global errorInfo
    #get option data & create option objects
    if {[catch {createOptions} msg] == 1} {
	return -code error -errorinfo $errorInfo "updateWindow:$msg"
    }
    # display main window object
    displayData
    return
}

###########
body ConfigWin::updateLoop {args} {

    catch {dp_after cancel $loopID}
    if {$period($window) == 0} {
	return
    }
    set loopID [dp_after [expr 1000*$period($window)] [code $this updateLoop]]
    #puts "update  ID = $loopID, period = $period($window)"
    updateOptions
    return
}

###########
body ConfigWin::createOptions {} {
    global errorInfo
    if { [catch "DP_ask [cmd$window.obj get -object] configure" result] } {
	return -code error -errorinfo $errorInfo "createOptions:$result"
    }	
    # delete all option objects
    catch {eval "delete object $optionObjects"}
    
    set optionCount [llength $result]
    set optionObjects ""
    set win [$window.f1.tswin subwidget window].f
    catch {frame $win}

    # create Option objects
    for {set i 0} {$i < $optionCount} {incr i} {
	set line  [lindex $result $i]
	set name  [lindex $line 0]
	set init  [lindex $line 1]
	set val   [lindex $line 2]
	set frame $win.f$i
	set obj [Option #auto.obj $name $init $val $frame cmd$window.obj $this]
	lappend optionObjects $obj
    }
    return
}

###########
body ConfigWin::updateOptions {} {
    global errorInfo
    if { [catch "DP_ask [cmd$window.obj get -object] configure" result] } {
	return -code error -errorinfo $errorInfo "createOptions:$result"
    }	

    # update Option objects' values
    for {set i 0} {$i < $optionCount} {incr i} {
	set line  [lindex $result $i]
	set name  [lindex $line 0]
	set val   [lindex $line 2]
	foreach j $optionObjects {
	    if {"$name" != "[$j get -name]"} {
		continue
	    }
	    $j setVal $val
	}
    }
    return
}

###########
body ConfigWin::displayData {} {
    set win [$window.f1.tswin subwidget window].f
    set counter 0
    set revcounter $optionCount
    set colorincr  [expr int((255.)/$optionCount)]
    
    # arrange options alphabetically
    foreach i $optionObjects {
	set name [string trim [$i get -name] -]
	lappend objlist [list $name $i]
    }
    set newobjlist [lsort -ascii $objlist]; # sorts on first element
    
    foreach i $newobjlist {
	set obj [lindex $i 1]
	incr counter
	incr revcounter -1
	#set bcolor "gray[expr 10 + $counter*80/$count]"
	
	#set red [format %02x [expr $revcounter*$colorincr]]
	#set color1 "#${red}00ff"
	#set color2 "#${red}00af"
	#set color3 "#${red}0064"
	
	set frame  [$obj get -frame]
	#$frame.entry1 config -highlightbackground "$color1"
	#$frame.entry2 config -highlightbackground "$color2"
	#$frame.entry3 config -highlightbackground "$color3"
	pack $frame
    }
    pack $win -anchor nw
    return
}

###########
body ConfigWin::destructor {} {
    catch {

	dp_after cancel $loopID
	eval delete object $optionObjects cmd$window.obj
	destroy $window
	unset period($window)
    }
}

###########
body ConfigWin::constructor {{winName .win}} {
    global env argv
    
    if {[winfo exists $winName] == 1} {
	error "ConfigWin: toplevel window \"$winName\" already exists"
    }
    set window  $winName
    # create a command object
    Command ::cmd$window.obj
    set period($window) 0
    
    toplevel    $window
    wm geometry $window 550x400
    wm title    $window "CODA 2.0 Command ConfigTool"

    ################ $window.f1

    frame $window.f1

    ################ $window.f1.f1

    frame $window.f1.f1


    label $window.f1.f1.label1 -image codaImage
    label $window.f1.f1.label2 \
	-background white \
	-foreground blue \
	-anchor center \
	-relief ridge \
	-borderwidth 2 \
	-font "-*-helvetica-bold-r-normal--12-*-*-*-*-*-*-*" \
	-text "COMMAND CONFIGTOOL"
    label $window.f1.f1.label3 -image codaImage

    pack  $window.f1.f1.label1 -side left
    pack  $window.f1.f1.label2 -side left -fill x -expand 1
    pack  $window.f1.f1.label3 -side left
    
    ################ $window.f1.tswin

    tixScrolledWindow $window.f1.tswin -scrollbar "auto -x" -borderwidth 4 -relief groove
    set win [$window.f1.tswin subwidget window].header
    frame $win

    label $win.label1 -anchor center -width 21 -text "Option Name"   -height 2
    label $win.label2 -anchor center -width 21 -text "Initial Value"
    label $win.label3 -anchor center -width 21 -text "Current Value"
    
    pack  $win.label1 $win.label2 $win.label3 -side left   
    pack  $win -anchor nw

    ################ $window.f1.f2

    frame $window.f1.f2
    
    tixLabelFrame $window.f1.f2.lf1 -label Command -labelside top
    tixLabelFrame $window.f1.f2.lf2 -label Update  -labelside top
    set frame1 [$window.f1.f2.lf1 subwidget frame]
    set frame2 [$window.f1.f2.lf2 subwidget frame]
    pack $window.f1.f2.lf1 $window.f1.f2.lf2 -side left -fill x -expand 1

    ################ 
    
    tixComboBox $frame1.object \
	-command "cmd$window.obj setObject $this" \
	-editable true \
	-label "Object " \
	-labelside left \
	-history true \
	-prunehistory true
    $frame1.object subwidget entry config -width 5
    bind [$frame1.object subwidget entry] <Leave>  "+ cmd$window.obj setObject $this"

    button $frame1.button \
	-width 11 \
	-text "CLEAR CMD" \
	-command "cmd$window.obj clear
		  $this resetOptions
		  $window.f1.f4.command config -text {}"

    pack  $frame1.object
    pack  $frame1.button
    
    ################ $window.f1.f2.f2
    
    button $frame2.button \
	-state disabled \
	-width 11 \
	-text "NOW" \
	-command "$this updateOptions"

    frame  $frame2.f1
    tixControl $frame2.f1.period \
	-state disabled \
	-integer 1 -max 999 -min 0 -label "Every " \
	-variable [scope period($window)]
    $frame2.f1.period subwidget entry config -width 3

    # use this trace as a method to update the screen when
    # the period value changes
    trace variable period($window) w "$this updateLoop"

    label  $frame2.f1.label -text " sec"
    pack   $frame2.f1.period $frame2.f1.label -side left

    pack   $frame2.f1
    pack   $frame2.button

    pack   $frame1 $frame2 -side left -fill x -expand 1

    ################ $window.f1.f4
    
    frame  $window.f1.f4
    label  $window.f1.f4.label -text "[lindex $argv 0] Configure Command"
    label  $window.f1.f4.command \
	-foreground blue -background white \
	-relief sunken -borderwidth 2
    button $window.f1.f4.button1 \
	-width 11 \
	-text "EXECUTE" \
	-command "cmd$window.obj execCmd"

    button $window.f1.f4.button2 \
	-width 11 \
	-text "DISMISS" \
	-command "$this dismiss"

    pack  $window.f1.f4.label
    pack  $window.f1.f4.command -fill x
    pack  $window.f1.f4.button1 $window.f1.f4.button2 -fill x -expand 1 -side left
    
    ################ final packing
    
    pack  $window.f1.f1    -fill x
    pack  $window.f1.f4    -fill x -side bottom
    pack  $window.f1.f2    -fill x -side bottom
    pack  $window.f1.tswin -fill both
    pack  $window.f1 -fill both -expand 1
    
    return
}






