/*----------------------------------------------------------------------------*
 *  Copyright (c) 1991, 1992  Southeastern Universities Research Association, *
 *                            Continuous Electron Beam Accelerator Facility   *
 *                                                                            *
 *    This software was developed under a United States Government license    *
 *    described in the NOTICE file included as part of this distribution.     *
 *                                                                            *
 * CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606 *
 *      heyes@cebaf.gov   Tel: (804) 249-7030    Fax: (804) 249-7363          *
 *----------------------------------------------------------------------------*
 * Module: libts.c
 *
 * Description: Trigger Supervisor Program
 *              Function Library. 
 *
 * Author:
 *	David Abbott
 *	CEBAF Data Acquisition Group
 *
 * Revision History:
 *	  Revision 1.0  1996/05/1  17:38:00  abbottd
 *
 */


/* Include Files */
#ifdef VXWORKS
#include <vxWorks.h>
#endif
#include <stdio.h>
#include <stdlib.h>

#include "ts.h"
#include "ts_macros.h"

/* Global Definitions  */
volatile struct vme_ts *tsP = 0;
volatile unsigned long *tsMemP = 0;


/* external definitions */
#ifdef VXWORKS
extern STATUS sysBusToLocalAdrs();
#endif


/* Calculate the least common Multiple of two numbers */
static int 
lcm(int a, int b)
{
  int lcmval;
  int ii;

  if (a >= b) {
    if(a%b) {
      for(ii=2;ii<=b;ii++) {
	if( (a*ii)%b ) {
	  continue;
	}else{
	  lcmval = a*ii;
	  break;
	}
      }
    }else{
      lcmval = a;
    }
  }else{
    if(b%a) {
      for(ii=2;ii<=a;ii++) {
	if( (b*ii)%a ) {
	  continue;
	}else{
	  lcmval = b*ii;
	  break;
	}
      }
    }else{
      lcmval = b;
    }
  }
  
  return(lcmval);
}

/*******************************************************************
 *   Function : ts_init_1                                
 *                                                    
 *   Function : Initialize Trigger Supervisor         
 *                                                    
 *   Parameters :  int laddr - Local Address of TS      
 *                             if 0 then use default
 *                 int iflag - if non-zero then do not
 *                             initialize TS and Memory.
 *                                0 - Initialize TS and MLU
 *                                1 - No Initialization of MLU
 *                               >1 - No Initialization of TS or MLU
 *                                                    
 *   returns -1 if error, 0 if ok.                    
 *                                                    
 *******************************************************************/

int 
ts_init_1 (unsigned long laddr, int iflag)
{
  unsigned long addr;
  int stat1, stat2;

  if (laddr) {
    stat1 = sysBusToLocalAdrs(VME_STD_USR_DAT,(char *)laddr,&addr);
    if(stat1) {
      stat2 = sysBusToLocalAdrs(VME_STD_SUP_DAT,(char *)laddr,&addr);
      if (stat2) {
	printf("ts_init_1: ERROR in address translation for address=0x%x\n",laddr);
	return(ERROR);
      } else {
	tsP = (struct vme_ts *)(addr);
	printf("ts_init_1: Trigger Supervisor Base address Pointer: tsP = 0x%x\n",tsP);
      }
    }else{
      tsP = (struct vme_ts *)(addr);
      printf("ts_init_1: Trigger Supervisor Base address Pointer: tsP = 0x%x\n",tsP);
    }
  }else{
    stat1 = sysBusToLocalAdrs(VME_STD_USR_DAT,(char *)TS_BASE_ADDR,&addr);
    if(stat1) {
      stat2 = sysBusToLocalAdrs(VME_STD_SUP_DAT,(char *)TS_BASE_ADDR,&addr);
      if (stat2) {
	laddr = TS_BASE_ADDR;
	printf("ts_init_1: ERROR in address translation for address=0x%x\n",laddr);
	return(ERROR);
      } else {
	tsP = (struct vme_ts *)(addr);
	printf("ts_init_1: Trigger Supervisor Base address Pointer: tsP = 0x%x\n",tsP);
      }
    }else{
      tsP = (struct vme_ts *)(addr);
      printf("ts_init_1: Trigger Supervisor Base address Pointer: tsP = 0x%x\n",tsP);
    }
  }

  /* Setup pointer to Trigger Supervisor Memory Lookup table */
  tsMemP = (unsigned long *)((unsigned long)tsP + TS_MEM);
  printf("ts_init_1: Trigger Supervisor MLU Pointer: tsMemP = 0x%x\n",tsMemP);

  /* Initialize Trigger Supervisor if flag is not set */
  if(iflag == 0) {
    tsP->csr = TS_CSR_INIT;
    printf("ts_init_1: Trigger Supervisor -- Initialized\n");
    tsMemInit();
  } else if(iflag == 1) {
    tsP->csr = TS_CSR_INIT;
    printf("ts_init_1: WARN : Trigger Supervisor MLU -- Not Initalized\n");
  }else if(iflag >= 2) {
    printf("ts_init_1: WARN : Trigger Supervisor and MLU -- Not Initalized\n");
  }

  return(OK);
}


/*********************************************************************
 *   Function : tsMemInit                                
 *                                                    
 *   Function : Initialize Trigger Supervisor Memory to its Default
 *              configuration:
 *                     For any single trigger input 1-12 latched the
 *                     corresponding trigger type is 1-12. For any
 *                     multiple trigger input latch the trigger type
 *                     will be 14. Trigger types 0,13,15 are not used.
 *                     For any trigger all level 1 Accept outputs fire
 *                     and all triggers are defined as class 1.
 *                                                    
 *   Parameters :  None
 *
 *                                                    
 *   returns -1 if error, 0 if ok.                    
 *                                                    
 *********************************************************************/
int 
tsMemInit ()
{
  unsigned long laddr, mem_value;
  unsigned long *ts_memory;
  int jj;
  int tserror=0;

  if(tsMemP == 0) {
    printf("tsMemInit: ERROR : Trigger Supervisor MLU Pointer not Initialized\n");
    return(-1);
  }
  if(TSENABLED) {
    printf("tsMemInit: ERROR : Cannot Write to MLU while TS is enabled\n");
    return(-1);
  }

  ts_memory = (unsigned long *) malloc(4096*sizeof(ts_memory));
  if(ts_memory == NULL) {
    printf("tsMemInit: ERROR : cannot malloc array to store MLU Programming\n");
    return(-1);
  }

  ts_memory[0] = 0;
  for( laddr = 1; laddr <= 4095;  laddr++ )    /* Assign default data to all memory addresses  */
	ts_memory[laddr] = 0xFFE3;             /* All L1A fire, Class 1, Type = 14             */

  jj = 0;
  for( laddr = 1; laddr <= 4095;  laddr = 2*laddr )   /*  Fix ROC code for single hit patterns  */
     {
        jj++;
        ts_memory[laddr] = 0xFF03 + (0x10)*jj;  /* Trigger type = input number */
     }

  /* Load and readback TS memory */
  for( laddr = 0; laddr <= 4095;  laddr++ )
     {
        tsMemP[laddr] = ts_memory[laddr];
        mem_value = tsMemP[laddr];
        if( ts_memory[laddr] != ( TS_MEM_MASK & mem_value ) ) {
                  printf("tsMemInit: ***** TS memory error ***** Program Value = 0x%x, Read Back = 0x%x\n",
			 ts_memory[laddr],(TS_MEM_MASK & mem_value));
		  tserror++;
		}
     }

  free(ts_memory);

  if(tserror) {
    return(-1);
  }else{
    printf("Trigger Supervisor Memory Lookup Table -- Initialized\n");
    return(OK);
  }
}

/*********************************************************************
 *   Function : tsCsr                                
 *                                                    
 *   Function   : Write to TS CSR Register
 *   Parameters : Val - value to write into CSR
 *                      if 0 then readback CSR only
 *                                                    
 *   Returns    : Readback of CSR register                    
 *                                                    
 *********************************************************************/
unsigned long 
tsCsr(unsigned long Val)
{
  if(Val) TSCSR(Val);
  return((tsP->csr)&TS_CSR_MASK);
}

/*********************************************************************
 *   Function : tsTrig                                
 *                                                    
 *   Function   : Write to TS TRIG Register
 *   Parameters : Val - bitmask of inputs to enable 0-11
 *                      bit 0 (Non-strobe) is set automatically
 *                      if 0 then readback TRIG only
 *                                                    
 *   Returns    : Readback of TRIG register                    
 *                                                    
 *********************************************************************/
unsigned long 
tsTrig(unsigned long Val)
{
  unsigned long trigval;

  if(Val) {
    trigval = 0x0001|(Val<<1);
    TSTRIG(trigval);
  }
  return((tsP->trig)&TS_TRIG_MASK);
}

/*********************************************************************
 *   Function : tsRoc                                
 *                                                    
 *   Function   : Write to TS ROC enable Register
 *   Parameters : b4,b3,b2,b1 - bitmask of enabled ROCs for each branch
 *
 *   Returns    : Readback of ROC register                    
 *                                                    
 *********************************************************************/
unsigned long 
tsRoc(unsigned char b4,unsigned char b3,unsigned char b2,unsigned char b1)
{
  TSROC(b4,b3,b2,b1);
  return((tsP->roc)&TS_ROC_MASK);
}

/*********************************************************************
 *   Function : tsSync
 *                                                    
 *   Function   : Write to TS SYNC Register
 *   Parameters : Val - Value of syncronization event interval
 *                                                    
 *   Returns    : Readback of SYNC register                    
 *                                                    
 *********************************************************************/
unsigned long 
tsSync(unsigned long Val)
{
  unsigned long sval;

  sval = (Val > TS_SYNC_MASK) ? TS_SYNC_MASK : Val;
  TSSYNC(sval);
  return((tsP->sync)&TS_SYNC_MASK);
}

/*********************************************************************
 *   Function : tsTest
 *                                                    
 *   Function   : Write to TS Test Register
 *   Parameters : Val - value to write into TEST Register
 *                      if 0 then readback TEST only
 *                                                    
 *   Returns    : Readback of TEST register                    
 *                                                    
 *********************************************************************/
unsigned long 
tsTest(unsigned long Val)
{
  if(Val) TSTEST(Val);
  return((tsP->test)&TS_TEST_MASK);
}


/*********************************************************************
 *   Function : tsPrescale
 *                                                    
 *   Function   : Write to TS Prescale Registers
 *   Parameters : Num - number of prescale input to set 1-8
 *                Val - Value to set the prescale register
 *                      if 0 then readback PRESCALE only
 *                                                    
 *   Returns    : Readback of PRESCALE register                    
 *                                                    
 *********************************************************************/
unsigned long 
tsPrescale(int Num, unsigned long Val)
{
  int pnum;
  unsigned long pval,pmask;
 
  if((Num<=0)||(Num>12)) {
    logMsg("tsPrescale: ERROR : Invalid Input number %d  (1-12 only)\n",Num,0,0,0,0,0);
    return(0);
  }
  if((Num>8)&&(Num<=12)) {
    logMsg("tsPrescale: WARN : Input %d not Prescalable (set to 1)\n",Num,0,0,0,0,0);
    return(1);
  }

  pnum  = Num-1;
  pmask = (Num < 5) ? TS_PRESCALE1_4MASK : TS_PRESCALE5_8MASK;
  pval  = (Val >= (pmask+1)) ? pmask : (Val-1);

  if(Val > 0) TSPRESCALE(pnum,pval);   /* Only Write to Prescale register if Val > 0 */
  return(((tsP->prescale[pnum])&pmask)+1);
}

/*********************************************************************
 *   Function : tsTimer
 *                                                    
 *   Function   : Write to TS Timer Registers
 *   Parameters : Num - number of timer to set 1-5
 *                Val - timer value to set
 *                                                    
 *   Returns    : Readback of Timer register  
 *                                                    
 *********************************************************************/
unsigned long 
tsTimer(int Num, unsigned long Val)
{
  int tnum;
  unsigned long tval,tmask;
 
  if((Num<=0)||(Num>5)) {
    logMsg("tsTimer: ERROR : Timer Number %d not Valid (1-5 only)\n",Num,0,0,0,0,0);
    return(0);
  }

  tnum  = Num-1;
  tmask = (Num < 5) ? TS_TIMER1_4MASK : TS_TIMER5_MASK;
  tval  = (Val >= tmask) ? tmask : Val;

  TSTIMER(tnum,tval);
  return((tsP->timer[tnum])&tmask);
}

/*********************************************************************
 *   Function : tsMemory
 *                                                    
 *   Function   : Write to TS Memory Lookup table
 *   Parameters : Num - memory address 0-4095
 *                Val - Value to write into memory
 *                                                    
 *   Returns    : Readback of Memory register specified                    
 *                                                    
 *********************************************************************/
unsigned long 
tsMemory(int Num, unsigned long Val)
{
  if(TSENABLED) {
    logMsg("tsMemory: ERROR : Cannot Access MLU while TS is enabled\n");
    return(0);
  }
  if((Num<0)||(Num>4095)) {
    logMsg("tsMemory: ERROR : TS MLU address %d not Valid\n",Num,0,0,0,0,0);
    return(0);
  }

  tsMemP[Num] = Val;
  return(tsMemP[Num]&TS_MEM_MASK);
}

/*********************************************************************
 *   Functions : tsGo , tsStop, tsInit                               
 *                                                    
 *   Function   : Utilities to Enable, Disable and Initialize the TS
 *   Parameters : iflag -
 *                      for tsGo if 1 then set L1 Enable bit as well
 *                      for tsStop if 1 then clear L1 Enable bit as well
 *                      for tsInit if 1 then Reset only (No Initialize)
 *                                                    
 *   Returns    : None
 *                                                    
 *********************************************************************/
void 
tsGo(int iflag)
{
  if(iflag)
    tsP->csr = 0x21;  /* Enable Go and L1 Hardware */
  else
    tsP->csr = 0x01;  /* Enable Go only */
}

void 
tsStop(int iflag)
{
  if(iflag)
    tsP->csr = 0x210000; /* Disable Go and L1 Hardware */
  else
    tsP->csr = 0x10000;  /* Disable Go only */
}

void 
tsInit(int iflag)
{
  if(iflag)
    tsP->csr = 0x4000; /* Reset Latched status only*/
  else
    tsP->csr = 0x8000;  /* Initialize TS */
}

/*********************************************************************
 *   Function : tsRead
 *                                                    
 *   Function   : General TS Read utilitiy
 *   Parameters : Reg - local address of Regisiter to Access
 *                Num - for Memory table specify address entry 0-4095
 *                                                    
 *   Returns    : Masked Readback of Register specified
 *                                                    
 *********************************************************************/
unsigned long
tsRead(unsigned long Reg, int Num)
{
  unsigned long rval;

  switch (Reg) {
  case TS_CSR:
    rval = tsP->csr&TS_CSR_MASK;
    break;
  case TS_TRIG:
    rval = tsP->trig&TS_TRIG_MASK;
    break;
  case TS_ROC:
    rval = tsP->roc&TS_ROC_MASK;
    break;
  case TS_SYNC:
    rval = tsP->sync&TS_SYNC_MASK;
    break;
  case TS_TEST:
    rval = tsP->test&TS_TEST_MASK;
    break;
  case TS_STATE:
    rval = tsP->state;
    break;
  case TS_PRESCALE1:
    rval = (tsP->prescale[0]&TS_PRESCALE1_4MASK)+1;
    break;
  case TS_PRESCALE2:
    rval = (tsP->prescale[1]&TS_PRESCALE1_4MASK)+1;
    break;
  case TS_PRESCALE3:
    rval = (tsP->prescale[2]&TS_PRESCALE1_4MASK)+1;
    break;
  case TS_PRESCALE4:
    rval = (tsP->prescale[3]&TS_PRESCALE1_4MASK)+1;
    break;
  case TS_PRESCALE5:
    rval = (tsP->prescale[4]&TS_PRESCALE5_8MASK)+1;
    break;
  case TS_PRESCALE6:
    rval = (tsP->prescale[5]&TS_PRESCALE5_8MASK)+1;
    break;
  case TS_PRESCALE7:
    rval = (tsP->prescale[6]&TS_PRESCALE5_8MASK)+1;
    break;
  case TS_PRESCALE8:
    rval = (tsP->prescale[7]&TS_PRESCALE5_8MASK)+1;
    break;
  case TS_TIMER1:
    rval = tsP->timer[0]&TS_TIMER1_4MASK;
    break;
  case TS_TIMER2:
    rval = tsP->timer[1]&TS_TIMER1_4MASK;
    break;
  case TS_TIMER3:
    rval = tsP->timer[2]&TS_TIMER1_4MASK;
    break;
  case TS_TIMER4:
    rval = tsP->timer[3]&TS_TIMER1_4MASK;
    break;
  case TS_TIMER5:
    rval = tsP->timer[4]&TS_TIMER5_MASK;
    break;
  case TS_SCALER0A:
    rval = tsP->scale_0a;
    break;
  case TS_SCALER0B:
    rval = tsP->scale_0b;
    break;
  case TS_SCALER1A:
    rval = tsP->scale_1a;
    break;
  case TS_SCALER1B:
    rval = tsP->scale_1b;
    break;
  case TS_MEM:
    if(TSENABLED) {
      logMsg("tsRead: ERROR : Cannot Access MLU while TS is enabled\n",0,0,0,0,0,0);
      rval = 0;
    }else{
      if((Num>=0)&&(Num<4095))
	rval = tsMemP[Num]&TS_MEM_MASK;
      else
	rval = tsMemP[0]&TS_MEM_MASK;
    }
    break;
  default:
    rval = 0;
    logMsg("tsRead: ERROR : Invalid Register Access, Reg = 0x%x\n",Reg,0,0,0,0,0);
  }

  return(rval);
}

/*********************************************************************
 *   Function : tsWrite                                
 *                                                    
 *   Function   : General TS Write Utility
 *   Parameters : Reg - Local address of Register to access
 *                Num - for Memory table specify entry 0-4095
 *                Val - Value to write into register
 *                                                    
 *   Returns    : Readback of register specified                   
 *                                                    
 *********************************************************************/
unsigned long
tsWrite(unsigned long Reg, int Num, unsigned long Val)
{
  unsigned long rval;
  unsigned char b1,b2,b3,b4;

  switch (Reg) {
  case TS_CSR:
    rval = tsCsr(Val);
    break;
  case TS_TRIG:
    rval = tsTrig(Val);
    break;
  case TS_ROC:
    b1 =    (Val)&0xff;
    b2 = (Val>>8)&0xff;
    b3 = (Val>>16)&0xff;
    b4 = (Val>>24)&0xff;
    rval = tsRoc(b4,b3,b2,b1);
    break;
  case TS_SYNC:
    rval = tsSync(Val);
    break;
  case TS_TEST:
    rval = tsTest(Val);
    break;
  case TS_PRESCALE:
    rval = tsPrescale(Num,Val);
    break;
  case TS_TIMER:
    rval = tsTimer(Num,Val);
    break;
  case TS_SCALER0A:
    tsP->scale_0a = Val;
    rval = tsP->scale_0a;
    break;
  case TS_SCALER0B:
    tsP->scale_0b = Val;
    rval = tsP->scale_0b;
    break;
  case TS_SCALER1A:
    tsP->scale_1a = Val;
    rval = tsP->scale_1a;
    break;
  case TS_SCALER1B:
    tsP->scale_1b = Val;
    rval = tsP->scale_1b;
    break;
  case TS_MEM:
    rval = tsMemory(Num,Val);
    break;
  default:
    rval = 0;
    logMsg("tsWrite: ERROR : Invalid Register Access, Reg = 0x%x\n",Reg,0,0,0,0,0);
  }

  return(rval);
}


/*********************************************************************
 *   Function : tsTimerStatus
 *                                                    
 *   Function   : Get status of TS timers
 *   Parameters : iflag - not currently used
 *                                                    
 *   Returns    : none                    
 *                                                    
 *********************************************************************/
void
tsTimerStatus(int iflag)
{
  int ii;
  unsigned long csr;
  float tval[5];

  csr = tsP->csr&TS_CSR_MASK;

  for(ii=0;ii<4;ii++) {
    tval[ii] = ((tsP->timer[ii]&TS_TIMER1_4MASK)*TS_TIMER1_4RES)/1000.0;
  }
  tval[4] = ((tsP->timer[4]&TS_TIMER5_MASK)*TS_TIMER5_RES)/1000.0;
  
  printf("\n");
  printf("  *** Trigger Supervisor Timers ***\n\n");

  printf("     Level 2          : Enabled    = %11.3f microSec\n",tval[1]);
  
  printf("     Level 3          : Enabled    = %11.3f microSec\n",tval[2]);
  
  if (csr&TS_CSR_ENFBT)
    printf("     Front-End Busy   : Enabled    = %11.3f microSec\n",tval[3]);
  else
    printf("     Front-End Busy   : Disabled   = %11.3f microSec\n",tval[3]);
  
  if (csr&TS_CSR_ENCPT)
    printf("     Clear Permit     : Enabled    = %11.3f microSec\n",tval[0]);
  else
    printf("     Clear Permit     : Disabled   = %11.3f microSec\n",tval[0]);
  
  if (csr&TS_CSR_ENCHT)
    printf("     Clear Hold       : Enabled    = %11.3f microSec\n",tval[4]);
  else
    printf("     Clear Hold       : Disabled   = %11.3f microSec\n",tval[4]);

}

/*********************************************************************
 *   Function : tsTriggerStatus
 *                                                    
 *   Function   : Return information on TS trigger programming
 *   Parameters : tmask - specific trigger input pattern to get info
 *                        on. if 0 then return info on single inputs
 *                        (1-12) only. 
 *                                                    
 *   Returns    : none
 *                                                    
 *********************************************************************/
void
tsTriggerStatus(unsigned long tmask)
{
  int ii;
  unsigned short mval, jj;
  struct ts_trigger tstm, tst[12];
  int tsEnable = 0, trigEnable = 1;
  int ps1=0,ps=1;
  int inputs[12] = {0,0,0,0,0,0,0,0,0,0,0,0};

  tsEnable = TSENABLED;
  if(tsEnable){
    printf("\n       ********** WARNING : WARNING : WARNING : WARNING **********\n");
    printf("    Trigger Class and L1 Accept Pattern info is invalid when GO is enabled\n");
  } 

  /* If tmask is set then we assume that the User wants info on a a specific trigger pattern 
     Decode the trigger pattern and lookup the information */
  if(tmask) {
    if((tmask<1)||(tmask>4095)) {
      printf("tsTriggerStatus: ERROR : Trigger Pattern 0x%x not Valid\n",tmask);
      return;
    }else{
      for(ii=0;ii<12;ii++) {
	if(tmask&(1<<ii)) {
	  inputs[ii] = 1;
	  if(ps1==0) { 
	    ps1 = tsPrescale((ii+1),0);
	    ps = ps1;
	  }else{
	    ps1 = tsPrescale((ii+1),0);
	    ps = lcm(ps,ps1);
	  }
	  trigEnable &= ((tsTrig(0))>>(ii+1)&1);
	}
      }
      tstm.num = tmask;
      tstm.enabled = trigEnable;
      tstm.prescale = ps;
      if(tsEnable) {
	mval = 0;
      }else{
	mval = tsMemP[tmask] & TS_MEM_MASK;
      }
      
      tstm.type = (int) (mval&0xf0)>>4;
      if(mval&0x1) {
	if(mval&0x8) {
	  tstm.class = 3;
	}else if(mval&0x4) {
	  tstm.class = 2;
	}else if(mval&0x2) {
	  tstm.class = 1;
	}else{
	  tstm.class = 0;
	}
      }else{
	tstm.class = -1;
      }
      
      for(jj=0;jj<8;jj++) {
	tstm.l1out[jj] = (mval&(1<<(jj+8))) ? (unsigned char) 1 : (unsigned char) 0;
      }

    }
  }

    for(ii=0;ii<12;ii++) {
      tst[ii].num       = ii+1;  /* Input Number */
      tst[ii].enabled   = ((tsP->trig)>>(ii+1))&1;
      if(ii<8) {
	tst[ii].prescale  = tsPrescale(ii+1,0);
      }else{
	tst[ii].prescale  = 1;
      }
      
      if(tsEnable) {
	mval = 0;
      }else{
	mval = tsMemP[(1<<(ii))] & TS_MEM_MASK;
      }
      tst[ii].type = (int) (mval&0xf0)>>4;
      if(mval&0x1) {
	if(mval&0x8) {
	  tst[ii].class = 3;
	}else if(mval&0x4) {
	  tst[ii].class = 2;
	}else if(mval&0x2) {
	  tst[ii].class = 1;
	}else{
	  tst[ii].class = 0;
	}
      }else{
      tst[ii].class = -1;
    }
      
      for(jj=0;jj<8;jj++) {
	tst[ii].l1out[jj] = (mval&(1<<(jj+8))) ? (unsigned char) 1 : (unsigned char) 0;
      }
    }
    

  /* Print out results for trigger inputs */

  printf("\n");
  printf("  ******  Trigger Supervisor Trigger Inputs  ******\n\n");

  if (tmask) {
    printf("                         Prescale                   Level 1 Accept Pattern\n");
    printf("   Pattern  Enabled?      Factor     Type   Class   8  7  6  5  4  3  2  1\n");
    printf("   -------  --------     --------    ----   -----   ----------------------\n");
    printf(" *%6d  %6d         %8d   %4d   %4d     %1d  %1d  %1d  %1d  %1d  %1d  %1d  %1d\n",
	   tstm.num,
	   tstm.enabled,
	   tstm.prescale,
	   tstm.type,
	   tstm.class,
	   tstm.l1out[7],
	   tstm.l1out[6],
	   tstm.l1out[5],
	   tstm.l1out[4],
	   tstm.l1out[3],
	   tstm.l1out[2],
	   tstm.l1out[1],
	   tstm.l1out[0]);
    printf("Requires Inputs:\n");
  }else{
    printf("                         Prescale                   Level 1 Accept Pattern\n");
    printf("     Input  Enabled?      Factor     Type   Class   8  7  6  5  4  3  2  1\n");
    printf("     -----  --------     --------    ----   -----   ----------------------\n");
  }
  for(ii=0;ii<12;ii++) {
    if(tmask&&(!inputs[ii]))
      continue;
    printf("%8d  %6d         %8d   %4d   %4d     %1d  %1d  %1d  %1d  %1d  %1d  %1d  %1d\n",
	   tst[ii].num,
	   tst[ii].enabled,
	   tst[ii].prescale,
	   tst[ii].type,
	   tst[ii].class,
	   tst[ii].l1out[7],
	   tst[ii].l1out[6],
	   tst[ii].l1out[5],
	   tst[ii].l1out[4],
	   tst[ii].l1out[3],
	   tst[ii].l1out[2],
	   tst[ii].l1out[1],
	   tst[ii].l1out[0]);
  }

}



/*********************************************************************
 *   Function : tsStatus
 *                                                    
 *   Function   : General status info on TS
 *   Parameters : iflag - General level of retruned status info
 *                      = 0 then return value of CSR register only
 *                      = 1 then print general CSR info
 *                      > 1 then print Trigger and Timer status as well
 *                                                    
 *   Returns    : Readback of CSR Register
 *                                                    
 *********************************************************************/
unsigned long 
tsStatus (int iflag)
{
  int tsEnable = 0;
  unsigned long csr = 0;

  if(tsP == 0) {
    printf("tsStatus: ERROR: TS Library Not Initialized -- Call ts_init_1() \n");
    return(0);
  }else{
    csr = tsP->csr;
  }

  if (iflag) {
    printf("*** Trigger Supervisor Status ***\n");
    if (csr&TS_CSR_GO) {
      printf("  TS Go                : Enabled\n");
      tsEnable = 1;
    }else{
      printf("  TS Go                : Disabled\n");
      tsEnable = 0;
    }

    if (csr&TS_CSR_ENL1) {
      printf("  TS Level 1 Hardware  : Enabled\n");
    }else{
      printf("  TS Level 1 Hardware  : Disabled\n");
    }

    if (csr&TS_CSR_ENSYNC) {
      printf("  TS Sync Events       : Enabled\n");
      if(csr&TS_CSR_FS)
	printf("    Forced Sync        : Pending\n");
    }else{
      printf("  TS Sync Events       : Disabled\n");
    }

    if (csr&TS_CSR_ROCLOCK)
      printf("  TS Buffered Mode     : Disabled\n");
    else
      printf("  TS Buffered Mode     : Enabled\n");
    
    if (csr&TS_CSR_ROCLOCK4)
      printf("     Branch 4          : Unbuffered\n");
    
    
    if (iflag > 1) {
      tsTimerStatus(iflag);
      tsTriggerStatus(0);
    }
    
    printf("\n");
  }

  return(csr);
}

/*********************************************************************
 *   Function : tsState
 *                                                    
 *   Function   : Current TS state info
 *   Parameters : iflag - General level of putput returned
 *                      = 0 state Register value only
 *                      > 0 print out useful diagnostics of state info
 *                                                    
 *   Returns    : Readback of STATE Register
 *                                                    
 *********************************************************************/
unsigned long 
tsState(int iflag)
{
  unsigned long sval, roc, l1;
  struct ts_state tss;
  int tsEnabled = 0;
  int ii, jj, kk;

  if(TSENABLED) {
    tsEnabled = 1;
    strncpy((char *) &tss.go[0],"Enabled \0",9);
  }else{
    strncpy((char *) &tss.go[0],"Disabled\0",9);
  }
  tss.l1e = (L1ENABLED) ? 1 : 0;

  roc = tsP->roc&TS_ROC_MASK;        /* Read ROC Register   */
  sval = tsP->state&TS_STATE_MASK;   /* Read State Register */

  /* Count # of ROCs per Branch */
  for (ii=0; ii<4; ii++) {
    kk = 0;
    for(jj=0; jj<8; jj++) {
      if(roc&(1<<(jj+(ii*8))))
	kk++;
    }
    tss.branch[ii] = kk;
    tss.strobe[ii] = 0;
    tss.ack[ii] = 0;
  }
  if(sval&TS_STATE_B1S) tss.strobe[0] = 1;
  if(sval&TS_STATE_B2S) tss.strobe[1] = 1;
  if(sval&TS_STATE_B3S) tss.strobe[2] = 1;
  if(sval&TS_STATE_B4S) tss.strobe[3] = 1;
  if(sval&TS_STATE_B1OA) tss.ack[0] = 1;
  if(sval&TS_STATE_B2OA) tss.ack[1] = 1;
  if(sval&TS_STATE_B3OA) tss.ack[2] = 1;
  if(sval&TS_STATE_B4OA) tss.ack[3] = 1;

  tss.feb     = (sval&TS_STATE_FEB) ? 1 : 0;
  tss.inhibit = (sval&TS_STATE_EI) ? 1 : 0;
  tss.clear   = (sval&TS_STATE_CLEAR) ? 1 : 0;
  tss.sync    = (sval&TS_STATE_SYNC) ? 1 : 0;
  tss.ready   = (sval&TS_STATE_READY) ? 1 : 0;
  tss.busy    = (sval&TS_STATE_BUSY) ? 1 : 0;
  tss.seq     = (sval&TS_STATE_SEQ) ? 1 : 0;

  if (tss.ready) {
    strncpy((char *) &tss.trigger[0],"Ready   \0",9);
    tss.feb = 0;  /* Front End Busy is Invalid when no Trigger is latched */ 
  }else if (sval&TS_STATE_TRIG) {
    strncpy((char *) &tss.trigger[0],"Latched \0",9);
  }else{
    strncpy((char *) &tss.trigger[0],"Disabled\0",9);
  }
        
  if (sval&TS_STATE_BUF_FULL) {
    strncpy((char *) &tss.buffers[0],"Full   \0",8);
  }else if (sval&TS_STATE_BUF_EMPTY) {
    strncpy((char *) &tss.buffers[0],"Empty  \0",8);
  }else{
    strncpy((char *) &tss.buffers[0],"Partial\0",8);
  }

  /* Print out info in Useful format */  
  if (iflag) {

    printf("\n      *******  Trigger Supervisor - Current State information  *******\n\n");
    printf("            TS Go  : %9s          TS Busy: %d\n",tss.go,tss.busy);
    printf("            TRIGGER: %9s          TS Sync: %d\n",tss.trigger,tss.sync);
    printf("            Buffers: %s\n",tss.buffers);

    printf("\n            Front Panel:   Front-End Busy    : %d\n",tss.feb);
    printf("                           External Inhibit  : %d\n",tss.inhibit);
    printf("                           Enable L1 Hardware: %d\n",tss.l1e);
    printf("                           Clear             : %d\n",tss.clear);

    printf("\n      ******* Branch Status ********\n");
    printf("            Branch    # ROCs    Strobe    ACK(at least one)\n");
    printf("            ------    ------    ------    -----------------\n");
    for (ii=0;ii<4;ii++) {
      printf("                %d        %d        %d         %d\n",
	     (ii+1),tss.branch[ii],tss.strobe[ii],tss.ack[ii]);
    }
    printf("\n");    
  }

  return(sval);
}
