/*
 * ns.c --
 *
 *  Introduction -- This file implements the extra Tcl/Tk commands
 *	needed by the name server.
 *
 * Copyright (c) 1995 Cornell University
 * All rights reserved.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for research and educational purposes, without fee, and
 * without written agreement is hereby granted, provided that the above
 * copyright notice and the following two paragraphs appear in all copies
 * of this software.
 *
 * IN NO EVENT SHALL CORNELL UNIVERSITY BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF CORNELL
 * UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * CORNELL UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND CORNELL UNIVERSITY HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
 
/* Carl Timmer removed everything except the time stuff */

#include <tcl.h>
#include <tclInt.h>
#include <tk.h>
#include <stdio.h>

#include <time.h>
#include <sys/types.h>
#include <sys/time.h>

/*
 *--------------------------------------------------------------
 *
 * Tns_strptimeCmd --
 *
 *	Implement the Tcl "ns_ptime" command.  This returns an
 *	integer (seconds since 00:00:00 UTC, January 1, 1970)
 *	that indicates the time of "timeval" which is, in turn,
 *	decoded using the "format" string. Be careful of daylight
 *	saving time or "dst".
 *
 *   -- A "isdst flag" positive value interprets "timeval" to be in
 *	  dst. This results in the subtraction of one hour if this
 *	  routine is called during main timezone time.
 *   -- A value of 0 forces main timezone time, and thus results in
 *	  the addition of one hour if called during dst.
 *   -- A negative value does NOT adjust the "timeval" time.
 *	  Ie., "timeval" is assumed to be the current dst status.
 *   -- With no given "isdst flag", a negative value is assumed.
 *
 *--------------------------------------------------------------
 */
Tns_strptimeCmd(clientData, interp, argc, argv)
ClientData clientData;         /* ignored */
Tcl_Interp *interp;            /* tcl interpreter */
int argc;                      /* Number of arguments */
char **argv;                   /* Arg list */
{
  struct tm tm;
  char *fmt;
  int dstFlag=-2;
  
  if ((argc == 1)||(argc > 4)) {
    Tcl_AppendResult(interp, argv[0], " <timeval> [<format>] [<isdst flag>]", NULL);
    return TCL_ERROR;
  }
  
  if (argc == 2) {
    fmt = "";
  } else {
    fmt = argv[2];
  }
  
  if (argc == 4) {
    dstFlag = atoi(argv[3]);
  }
    
  /* printf("ns_ptime: %s, format=%s, dstFlag=%d\n",argv[1],fmt,dstFlag); */
  strptime (argv[1], fmt, &tm);
  tm.tm_isdst = dstFlag;
  
  sprintf (interp->result, "%d", mktime(&tm));
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Tns_SystimeCmd --
 *
 *	Implement the Tcl "ns_systime" command.  This returns a
 *	string that indicates the current value on the system clock
 *	as a double precision value.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
int
Tns_SystimeCmd (clientData, interp, argc, argv)
    ClientData clientData;         /* ignored */
    Tcl_Interp *interp;            /* tcl interpreter */
    int argc;                      /* Number of arguments */
    char **argv;                   /* Arg list */
{
#ifndef __linux
    struct timespec tv;
#else 
    struct timeval tv;
#endif
    if (argc != 1) {
        Tcl_AppendResult(interp, argv[0], " doesn't take any args", NULL);
        return TCL_ERROR;
    }
#ifndef __linux
    clock_gettime(0, &tv);
    sprintf (interp->result, "%d.%09d", tv.tv_sec, tv.tv_nsec);
#else 
    gettimeofday(&tv, NULL);
    sprintf (interp->result, "%d.%06d", tv.tv_sec, tv.tv_usec);
#endif

    return TCL_OK;
}

Tns_CtimeCmd (clientData, interp, argc, argv)
    ClientData clientData;         /* ignored */
    Tcl_Interp *interp;            /* tcl interpreter */
    int argc;                      /* Number of arguments */
    char **argv;                   /* Arg list */
{
  struct timeval tv;

  if (argc != 3) {
    Tcl_AppendResult(interp, argv[0], " <timeval> <format>", NULL);
    return TCL_ERROR;
  }

  sscanf (argv[1],"%d.%06d", &tv.tv_sec, &tv.tv_usec);
  strftime (interp->result, 50, argv[2], localtime((time_t *) &tv.tv_sec));

  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Tns_SystimeCmd --
 *
 *	Implement the Tcl "ns_systime" command.  This returns a
 *	string that indicates the current value on the system clock
 *	as a double precision value.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
int
Tns_TrueSystimeCmd (clientData, interp, argc, argv)
ClientData clientData;         /* ignored */
Tcl_Interp *interp;            /* tcl interpreter */
int argc;                      /* Number of arguments */
char **argv;                   /* Arg list */
{
#ifndef __linux
  struct timespec tv;
#else
  struct timeval tv;
#endif
  double true_time;
  if (argc != 1) {
    Tcl_AppendResult(interp, argv[0], " doesn't take any args", NULL);
    return TCL_ERROR;
  }
#ifndef __linux
  (void) clock_gettime(0, &tv);
  true_time = tv.tv_sec + tv.tv_nsec/1000000000.0;  
#else
  gettimeofday(&tv,NULL);
  true_time = tv.tv_sec + tv.tv_usec/1000000.0;  
#endif
  sprintf (interp->result, "%f", true_time);
  
  return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * Tns_Init --
 *
 *	Initialize the time portion of Tcl-ISIS package.
 *
 * Results:
 *	None
 *
 * Side effects:
 *	ISIS related commands are bound to the interpreter.
 *
 *--------------------------------------------------------------
 */
int
Tns_Init (interp)
    Tcl_Interp *interp;
{
    Tcl_CreateCommand(interp, "ns_ptime", Tns_strptimeCmd,
		      (ClientData) NULL, (void (*)()) NULL);
    Tcl_CreateCommand(interp, "ns_systime", Tns_SystimeCmd,
		      (ClientData) NULL, (void (*)()) NULL);
    Tcl_CreateCommand(interp, "ns_time", Tns_TrueSystimeCmd,
		      (ClientData) NULL, (void (*)()) NULL);
    Tcl_CreateCommand(interp, "ns_ctime", Tns_CtimeCmd,
		      (ClientData) NULL, (void (*)()) NULL);
    return TCL_OK;
}
