/************************************************************************
 *									*
 *	Copyright 1992 by Motorola Mobile Data Division,		*
 *			  Bothell, WA					*
 *									*
 *	Motorola hereby grants permission to use, copy, modify and	*
 *	distribute  this software and its documentation for any		*
 *	purpose and without fee, provided that you retain this		*
 *	copyright notice in all copies.  Motorola makes no		*
 *	representations about the suitability of this software for any	*
 *	purpose.  Motorola provides this software ``as is'' without	*
 *	express or implied warranty.					*
 *									*
 ************************************************************************/
#ifndef lint
static char RCSid[] = "$Header: /usr/local/source/coda_source/Tcl/src7.4/svipc/svipcSem.c,v 1.5 1999/11/30 22:10:06 rwm Exp $";
#endif /* lint */

/*
 * File: SVipc/svipcSem.c
 * Facility: Tcl C Routines
 * Author: Joe Kelsey
 * Description:
 *	Tcl commands to interact with semaphores.
 *
 * Global Functions:
 *	Svipc_SemgetCmd
 *	Svipc_SemopCmd
 *	Svipc_SemrmidCmd
 *	Svipc_SemsetCmd
 *	Svipc_SemstatCmd
 *	Svipc_SemvalCmd
 *
 * $Log: svipcSem.c,v $
 * Revision 1.5  1999/11/30 22:10:06  rwm
 * def of union semun now valid for Linux.
 *
 * Revision 1.4  1999/10/25 17:33:40  abbottd
 * Fix for RedHat Linux 6.0 compiles
 *
 * Revision 1.3  1998/11/06 15:25:05  timmer
 * Linux port
 *
 * Revision 1.2  1998/01/21 20:02:06  heyes
 * fix for linux the semun bug
 *
 * Revision 1.1.1.1  1996/08/21 19:30:09  heyes
 * Imported sources
 *
 * Revision 1.6  1993/10/14  21:31:47  kelsey
 * Define union semun if sys/sem.h doesn't do it for us.
 *
 * Revision 1.5  1993/10/01  23:06:41  kelsey
 * Fix all of the AppendElement calls to add trailing 0 parameter.
 * Fix the aggregate return from status.
 *
 * Revision 1.4  1993/08/18  21:34:41  kelsey
 * Use Svipc_ prefix instead of Tcl_.
 * Update for Tcl 7.0.
 *
 * Revision 1.3  1993/04/09  01:31:49  kelsey
 * Make keyword arguments abbreviatable.
 * Make stat return structured result even if variable specified.
 *
 * Revision 1.2  1993/04/02  22:18:03  kelsey
 * Change copyright notice.
 * Remove unused functions.
 * Standardize function calls.
 *
 * Revision 1.1  1992/06/02  14:09:34  kelsey
 * Initial revision
 *
 */

#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include <tcl.h>

#ifdef SVIPC_NO_UNION_SEMUN
/*
 * SVR4 (at least) does not define this in sem.h!
 */
union semun {
  int val;
  struct semid_ds *buf;
  ushort *array;
# ifdef LINUX
  struct seminfo *__buf;	/*	<= buffer for IPC_INFO */
# endif
};

#endif /* SVIPC_NO_UNION_SEMUN */


/*
 * Function: Svipc_SemgetCmd
 * Description:
 *	Acquire a semaphore id.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments (3-6).
 * char **	argv		Vector of command arguments.
 *				[ semget key nsems mode create excl ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 */
/* ARGSUSED */
int
Svipc_SemgetCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  key_t key;
  int nsems;
  int semflg;
  int semid;
  char r_semid[20];

  if (argc < 3 || 6 < argc)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0],
			" key nsems ?mode? ?create? ?excl?", (char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], (int*)&key) != TCL_OK)
    {
      if (strncmp (argv[1], "private", strlen (argv[1])))
	{
	  return TCL_ERROR;
	}
      key = IPC_PRIVATE;
    }

  if (Tcl_GetInt (interp, argv[2], &nsems) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (argc > 3)
    {
      char **pargv = argv + 3;

      if (Tcl_GetInt (interp, *pargv, &semflg) != TCL_OK)
	{
	  semflg = 0666;
	}
      else
	{
	  pargv++;
	}

      while (*pargv)
	{
	  int plen = strlen (*pargv);

	  if (strncmp (*pargv, "create", plen) == 0)
	    {
	      semflg |= IPC_CREAT;
	    }
	  else if (strncmp (*pargv, "excl", plen) == 0)
	    {
	      semflg |= IPC_EXCL;
	    }
	  else
	    {
	      (void) strcpy (interp->result,
			     "flag argument must be \"create\" or \"excl\".");
	      return TCL_ERROR;
	    }
	  pargv++;
	}
    }
  else
    {
      semflg = 0666;
    }

  if ((semid = semget (key, nsems, semflg)) == -1)
    {
      int i;
      char flgstr[20];

      (void) sprintf (flgstr, "0%o", semflg);
      Tcl_AppendResult (interp, "cannot get semaphore id for \"", argv[1], " ",
			argv[2], " ", flgstr, (char *)0);
      for (i = 3; i < argc; i++)
	{
	  Tcl_AppendResult (interp, i == 3 ? " (" : " ", argv[i], (char *)0);
	}
      Tcl_AppendResult (interp, argc == 3 ? "\": " : ")\": ",
			Tcl_PosixError (interp), (char *)0);
      return TCL_ERROR;
    }

  (void) sprintf (interp->result, "%d", semid);

  return TCL_OK;
}

/*
 * Function: Svipc_Semgetall
 * Description:
 *	Get all semaphore values, possibly stuffing them into a variable.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * Tcl_Interp *	interp		Interpreter.
 * int		semid		Semaphore identifier.
 * char *	varName		Variable name to stuff values into.
 *
 * Return type: 
 * Side Effects:
 */

static int
Svipc_Semgetall (interp, semid, varName)
     Tcl_Interp *interp;
     int semid;
     char *varName;

/* Detail Design:

 */

{
  union semun semun;
  struct semid_ds sem_data;
  int flags = 0;
  int i;
  
  semun.buf = &sem_data;
  if (semctl (semid, 0, IPC_STAT, semun) == -1)
    {
      char idstr[20];

      (void) sprintf (idstr, "%d", semid);
      Tcl_AppendResult (interp, "cannot stat semaphore id ", idstr, ": ",
			Tcl_PosixError (interp), (char *)0);
      return TCL_ERROR;
    }
  
  semun.array = (ushort *)ckalloc (sem_data.sem_nsems * sizeof (ushort));
  
  if (semctl (semid, 0, GETALL, semun) == -1)
    {
      char idstr[20];

      (void) sprintf (idstr, "%d", semid);
      Tcl_AppendResult (interp, "cannot get all values for semaphore id ",
			idstr, ": ", Tcl_PosixError (interp),
			(char *)0);
      return TCL_ERROR;
    }
  
  for (i = 0; i < sem_data.sem_nsems; i++)
    {
      char r_semval[20];
      
      (void) sprintf (r_semval, "%d", semun.array[i]);
      
      if (varName)
	{
	  (void) Tcl_SetVar (interp, varName, r_semval, flags);
	  /* First value resets variable.  Subsequent values get
	     appended.  */
	  if (!flags)
	    {
	      flags = TCL_APPEND_VALUE|TCL_LIST_ELEMENT;
	    }
	}
      Tcl_AppendElement (interp, r_semval);
    }
  return TCL_OK;
}

/*
 * Function: Svipc_Semgetset
 * Description:
 *	Get or set a semaphore value.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * Tcl_Interp *	interp		Interpreter.
 * int		semid		Semaphore identifier.
 * char *	numstr		Semaphore number to set or get.
 * char *	valstr		Valus to set into semaphore.
 *
 * Return type: 
 * Side Effects:
 */

static int
Svipc_Semgetset (interp, semid, numstr, valstr)
     Tcl_Interp *interp;
     int semid;
     char *numstr;
     char *valstr;

/* Detail Design:

 */

{
  int semnum;
  union semun semun;

  if (Tcl_GetInt (interp, numstr, &semnum) != TCL_OK)
    {
      return -1;
    }
  
  if (valstr && Tcl_GetInt (interp, valstr, &semun.val) != TCL_OK)
    {
      return -1;
    }
  
  if (!valstr)
    {
      if ((semun.val = semctl (semid, semnum, GETVAL, semun)) == -1)
	{
	  char idstr[20];
	  
	  (void) sprintf (idstr, "%d", semid);
	  Tcl_AppendResult (interp, "cannot get semaphore id ", idstr, " #",
			    numstr, " value: ", Tcl_PosixError (interp),
			    (char *)0);
	  return -1;
	}
    }
  else
    {
      if (semctl (semid, semnum, SETVAL, semun) == -1)
	{
	  char idstr[20];
	  
	  (void) sprintf (idstr, "%d", semid);
	  Tcl_AppendResult (interp, "cannot set semaphore id ", idstr, " #",
			    numstr, " to value ", valstr, ": ",
			    Tcl_PosixError (interp), (char *)0);
	  return -1;
	}
    }
  return semun.val;
}

/*
 * Function: Svipc_SemopCmd
 * Description:
 *	Perform counting operations on semaphores.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments.
 * char **	argv		Vector of command arguments.
 *				[ semop semid sop ... ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 *	Modifies semaphore.
 */
/* ARGSUSED */
int
Svipc_SemopCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  int semid;
  struct sembuf *sops;
  int nsops = argc - 2;
  int i;

  if (argc < 3)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0], " semid, sop ...",
			(char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], &semid) != TCL_OK)
    {
      return TCL_ERROR;
    }

  sops = (struct sembuf *)ckalloc (sizeof *sops * nsops);

  argv += 2;
  for (i = 0; i < nsops; i++)
    {
      int temp;
      int largc;
      char ** largv;

      if (Tcl_SplitList (interp, argv[i], &largc, &largv) != TCL_OK)
	{
	  ckfree ((char *)sops);
	  return TCL_ERROR;
	}

      if (largc < 2 || largc > 4)
	{
	  (void) strcpy (interp->result,
			 "semop lists need 2-4 elements: sem_num sem_op ?undo? ?nowait?");
	  ckfree ((char *)largv);
	  ckfree ((char *)sops);
	  return TCL_ERROR;
	}

      if (Tcl_GetInt (interp, largv[0], &temp) != TCL_OK)
	{
	  ckfree ((char *)largv);
	  ckfree ((char *)sops);
	  return TCL_ERROR;
	}
      sops[i].sem_num = temp;

      if (Tcl_GetInt (interp, largv[1], &temp) != TCL_OK)
	{
	  ckfree ((char *)largv);
	  ckfree ((char *)sops);
	  return TCL_ERROR;
	}
      sops[i].sem_op = temp;

      sops[i].sem_flg = 0;
      if (largc > 2)
	{
	  char **list = largv + 2;

	  while (*list)
	    {
	      int llen = strlen (*list);

	      if (strncmp (*list, "undo", llen) == 0)
		{
		  sops[i].sem_flg |= SEM_UNDO;
		}
	      else if (strncmp (*list, "nowait", llen) == 0)
		{
		  sops[i].sem_flg |= IPC_NOWAIT;
		}
	      else
		{
		  (void) strcpy (interp->result,
				 "sem_flg must be \"undo\" or \"nowait\".");
		  
		  ckfree ((char *)largv);
		  ckfree ((char *)sops);
		  return TCL_ERROR;
		}
	      list++;
	    }
	}
      ckfree ((char *)largv);
    }
  
  while (semop (semid, sops, nsops) == -1)
    {
      if (errno == EINTR)
	{
	  (void) strcpy (interp->result, "eintr");
	  ckfree ((char *)sops);
	  return TCL_OK;
	}
      else if (errno == EAGAIN)
	{
	  (void) strcpy (interp->result, "eagain");
	  ckfree ((char *)sops);
	  return TCL_OK;
	}
      else
	{
	  Tcl_AppendResult (interp,
			    "cannot perform operation on semaphore id ",
			    argv[1], ": ", Tcl_PosixError (interp), (char *)0);
	  ckfree ((char *)sops);
	  return TCL_ERROR;
	}
    }
  ckfree ((char *)sops);
  return TCL_OK;
}

/*
 * Function: Svipc_SemrmidCmd
 * Description:
 *	Remove the semaphore from the system.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments.
 * char **	argv		Vector of command arguments.
 *				[ semrmid semid ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 *	Modifies semaphore.
 */
/* ARGSUSED */
int
Svipc_SemrmidCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;

/* Detail Design:
   
 */

{
  int semid;
  union semun semun;
  
  if (argc != 2)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0], " semid", (char *)0);
      return TCL_ERROR;
    }
  
  if (Tcl_GetInt (interp, argv[1], &semid) != TCL_OK)
    {
      return TCL_ERROR;
    }
  
  semun.val = 0;
  if (semctl (semid, 0, IPC_RMID, semun) == -1)
    {
      Tcl_AppendResult (interp, "cannot remove semaphore id ", argv[1], ": ",
			Tcl_PosixError (interp), (char *)0);
      return TCL_ERROR;
    }
  return TCL_OK;
}

/*
 * Function: Svipc_SemsetCmd
 * Description:
 *	Perform the IPC_SET function on the semid.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments.
 * char **	argv		Vector of command arguments.
 *				[ semset semid sem_perm ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 *	Modifies semaphore.
 */
/* ARGSUSED */
int
Svipc_SemsetCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;

/* Detail Design:
   
 */

{
  int semid;
  int temp;
  int largc;
  char **largv;
  struct semid_ds semid_ds;
  union semun semun;
  
  if (argc != 3)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0], " semid sem_perm",
			(char *)0);
      return TCL_ERROR;
    }
  
  if (Tcl_GetInt (interp, argv[1], &semid) != TCL_OK)
    {
      return TCL_ERROR;
    }
  
  if (Tcl_SplitList (interp, argv[2], &largc, &largv) != TCL_OK)
    {
      return TCL_ERROR;
    }
  
  if (largc < 3)
    {
      (void) strcpy (interp->result,
		     "sem_perm list needs 3 arguments: uid gid mode.");
      ckfree ((char *)largv);
      return TCL_ERROR;
    }
  
  if (Tcl_GetInt (interp, largv[0], &temp) != TCL_OK)
    {
      ckfree ((char *)largv);
      return TCL_ERROR;
    }
  semid_ds.sem_perm.uid = temp;
  
  if (Tcl_GetInt (interp, largv[1], &temp) != TCL_OK)
    {
      ckfree ((char *)largv);
      return TCL_ERROR;
    }
  semid_ds.sem_perm.gid = temp;
  
  if (Tcl_GetInt (interp, largv[2], &temp) != TCL_OK)
    {
      ckfree ((char *)largv);
      return TCL_ERROR;
    }
  semid_ds.sem_perm.mode = temp;
  
  ckfree ((char *)largv);
  
  semun.buf = &semid_ds;
  if (semctl (semid, 0, IPC_SET, semun) == -1)
    {
      Tcl_AppendResult (interp, "cannot modify permissions for semaphore id ",
			argv[1], " {", argv[2], "}: ", Tcl_PosixError (interp),
			(char *)0);
      return TCL_ERROR;
    }
  return TCL_OK;
}

/*
 * Function: Svipc_SemstatCmd
 * Description:
 *	Return a vector of information about a semaphore.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments.
 * char **	argv		Vector of command arguments.
 *				[ semstat semid ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 */
/* ARGSUSED */
int
Svipc_SemstatCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;

/* Detail Design:
   
 */

{
  int semid;
  struct semid_ds sem_data;
  union semun semun;
  int i;
  char rstr[7][20];
  char *sem_list;
  char * margv[7];

  if (argc < 2 || 3 < argc)
    {
      Tcl_AppendResult (interp, "usage: ", argv[0], " semid ?varName?",
			(char *)0);
      return TCL_ERROR;
    }
  
  if (Tcl_GetInt (interp, argv[1], &semid) != TCL_OK)
    {
      return TCL_ERROR;
    }
  
  semun.buf = &sem_data;
  if (semctl (semid, 0, IPC_STAT, semun) == -1)
    {
      Tcl_AppendResult (interp, "cannot stat semaphore id ", argv[1], ": ",
			Tcl_PosixError (interp), (char *)0);
      return TCL_ERROR;
    }
  
  (void) sprintf (rstr[0], "%d", sem_data.sem_perm.cuid);
  (void) sprintf (rstr[1], "%d", sem_data.sem_perm.cgid);
  (void) sprintf (rstr[2], "%d", sem_data.sem_perm.uid);
  (void) sprintf (rstr[3], "%d", sem_data.sem_perm.gid);
  (void) sprintf (rstr[4], "%07o", sem_data.sem_perm.mode);
#ifdef LINUX
  (void) sprintf (rstr[5], "%d", sem_data.sem_perm.__seq);
  (void) sprintf (rstr[6], "%d", sem_data.sem_perm.__key);
#else
  (void) sprintf (rstr[5], "%d", sem_data.sem_perm.seq);
  (void) sprintf (rstr[6], "%d", sem_data.sem_perm.key);
#endif

  margv[0] = rstr[0];
  margv[1] = rstr[1];
  margv[2] = rstr[2];
  margv[3] = rstr[3];
  margv[4] = rstr[4];
  margv[5] = rstr[5];
  margv[6] = rstr[6];
  
  sem_list = Tcl_Merge (7, margv);
  Tcl_AppendElement (interp, sem_list);
  ckfree (sem_list);

  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "cuid", rstr[0], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "cgid", rstr[1], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "uid", rstr[2], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "gid", rstr[3], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "mode", rstr[4], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "seq", rstr[5], (int)0);
      (void) Tcl_SetVar2 (interp, argv[2], "key", rstr[6], (int)0);
    }

  (void) sprintf (rstr[0], "%d", sem_data.sem_nsems);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "nsems", rstr[0], (int)0);
    }

  (void) sprintf (rstr[0], "%d", sem_data.sem_otime);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "otime", rstr[0], (int)0);
    }

  (void) sprintf (rstr[0], "%d", sem_data.sem_ctime);
  Tcl_AppendElement (interp, rstr[0]);
  if (argc == 3)
    {
      (void) Tcl_SetVar2 (interp, argv[2], "ctime", rstr[0], (int)0);
    }

  for (i = 0; i < sem_data.sem_nsems; i++)
    {
      int ctl;
      char indstr[20];
      int tmp;

      if ((ctl = semctl (semid, i, GETVAL, semun)) < 0)
	{
	  Tcl_ResetResult (interp);
	  (void) sprintf (indstr, "%d", i);
	  Tcl_AppendResult (interp, "cannot get value for semaphore id ",
			    argv[1], " #", indstr, ": ",
			    Tcl_PosixError (interp), (char *)0);
	  return TCL_ERROR;
	}
      
      (void) sprintf (rstr[0], "%d", ctl);

      if ((tmp = semctl (semid, i, GETPID, semun)) < 0)
	{
	  Tcl_ResetResult (interp);
	  (void) sprintf (indstr, "%d", i);
	  Tcl_AppendResult (interp, "cannot get pid for semaphore id ",
			    argv[1], " #", indstr, ": ",
			    Tcl_PosixError (interp), (char *)0);
	  return TCL_ERROR;
	}
      (void) sprintf (rstr[1], "%d", tmp);

      if ((ctl = semctl (semid, i, GETNCNT, semun)) < 0)
	{
	  Tcl_ResetResult (interp);
	  (void) sprintf (indstr, "%d", i);
	  Tcl_AppendResult (interp, "cannot get n-count for semaphore id ",
			    argv[1], " #", indstr, ": ",
			    Tcl_PosixError (interp), (char *)0);
	  return TCL_ERROR;
	}

      (void) sprintf (rstr[2], "%d", ctl);

      if ((ctl = semctl (semid, i, GETZCNT, semun)) < 0)
	{
	  Tcl_ResetResult (interp);
	  (void) sprintf (indstr, "%d", i);
	  Tcl_AppendResult (interp, "cannot get z-count for semaphore id ",
			    argv[1], " #", indstr, ": ",
			    Tcl_PosixError (interp), (char *)0);
	  return TCL_ERROR;
	}

      (void) sprintf (rstr[3], "%d", ctl);

      margv[0] = rstr[0];
      margv[1] = rstr[1];
      margv[2] = rstr[2];
      margv[3] = rstr[3];
      sem_list = Tcl_Merge (4, margv);
      
      Tcl_AppendElement (interp, sem_list);
      ckfree (sem_list);
      if (argc == 3)
	{
	  char arrind[10];

	  (void) sprintf (arrind, "val%d", i);
	  (void) Tcl_SetVar2 (interp, argv[2], arrind, rstr[0], (int)0);
	  (void) sprintf (arrind, "pid%d", i);
	  (void) Tcl_SetVar2 (interp, argv[2], arrind, rstr[1], (int)0);
	  (void) sprintf (arrind, "ncnt%d", i);
	  (void) Tcl_SetVar2 (interp, argv[2], arrind, rstr[2], (int)0);
	  (void) sprintf (arrind, "zcnt%d", i);
	  (void) Tcl_SetVar2 (interp, argv[2], arrind, rstr[3], (int)0);
	}
    }
  return TCL_OK;
}

/*
 * Function: Svipc_SemvalCmd
 * Description:
 *	Set the values of a group of semaphores.
 *
 * Parameters:
 * Type		Name		Description
 * ----------------------------------------------------------------------
 * ClientData	clientData	Global command data (not used).
 * Tcl_Interp * interp		Tcl interpreter.
 * int		argc		Number of command arguments.
 * char **	argv		Vector of command arguments.
 *				[ semsetval semid ?{semnum ?val|varName?} ...? ]
 *
 * Return type: int
 * Side Effects:
 *	Sets interp->result.
 *	Modifies semaphore.
 */
/* ARGSUSED */
int
Svipc_SemvalCmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char ** argv;

/* Detail Design:

 */

{
  int semid;
  char rstr[20];

  if (argc < 2)
    {
      Tcl_AppendResult (interp,
			"usage: ", argv[0],
			" semid ?varName|{semnum ?val|varName?} ...?",
			(char *)0);
      return TCL_ERROR;
    }

  if (Tcl_GetInt (interp, argv[1], &semid) != TCL_OK)
    {
      return TCL_ERROR;
    }

  if (argc == 2 ||
      (argc == 3 && !isdigit(*argv[2])))
    {
      /* argv[2] contains either a pointer to a variable name or a zero.  */
      return Svipc_Semgetall (interp, semid, argv[2]);
    }
  else
    {
      char **pargv = argv + 2;

      while (*pargv)
	{
	  int semnum;
	  int semval;
	  int largc;
	  char **largv;
	  char *numstr;
	  char *valstr = 0;
	  char *varstr = 0;

	  if (Tcl_SplitList (interp, *pargv, &largc, &largv) != TCL_OK)
	    {
	      return TCL_ERROR;
	    }

	  numstr = largc == 1 ? pargv[0] : largv[0];
	  if (largc == 2)
	    {
	      if (isdigit(*largv[1]))
		{
		  valstr = largv[1];
		}
	      else
		{
		  varstr = largv[1];
		}
	    }
	  else if (largc == 3)
	    {
	      valstr = largv[1];
	      varstr = largv[2];
	    }
	  else if (largc != 1)
	    {
	      Tcl_AppendResult (interp, "wrong # args in list {", pargv,
				"}: format: {semnum ?val|varName?}",
				(char *)0);
	      ckfree ((char *) largv);
	      return TCL_ERROR;
	    }

	  if ((semval = Svipc_Semgetset (interp, semid, numstr, valstr)) == -1)
	    {
	      ckfree ((char *)largv);
	      return TCL_ERROR;
	    }

	  (void) sprintf (rstr, "%d", semval);

	  if (varstr)
	    {
	      (void) Tcl_SetVar (interp, varstr, rstr, (int)0);
	    }
	  Tcl_AppendElement (interp, rstr);
	  ckfree ((char *) largv);

	  pargv++;
	}
    }
  return TCL_OK;
}
