/*
   header file for the memory allocation system . Keep all structure
   elements (for danode) aligned on 4-byte boundaries. This insures that malloc()
   will create event buffer nodes that are 4-byte aligned.
*/

#ifndef  __ROM_MEM_PART__
#define __ROM_MEM_PART__

#ifndef NULL
#define NULL ((void *) 0)
#endif

#define maximum(a,b) (a<b ? b : a)

extern char	*malloc();

#include "mempart.h"

/* global data */

static DALIST  partList; /* global part list */

libPartInit()
{
  listInit(&partList);
}

/************************************************************** 
 * partIncr - increase part size by c items (min) 
 */

void partIncr (pPart,c)
     ROL_MEM_ID pPart;		/* part to be enlarged */
     int	c;		/* min # of items to be added */
{
  register long *block;
  unsigned bytes;
  int actual = c;		/* actual # of items added */
  pPart->total += c;
  if ((pPart == NULL)||(c == 0)) return;
  
  while (actual--)
    {
      block = (long *) malloc (pPart->size);
      if (block == NULL)
	return;
      bzero((char *) block, pPart->size);
      ((DANODE *)block)->part = pPart; /* remember where we came from... */
      listAdd (&pPart->list,(DANODE *)block);
    }
}


/*************************************************************
 * partCreate - create and initialize a memory part 
 */

ROL_MEM_ID partCreate (name, size, c, incr)
     char *name;		/* name of new partition */
     int size;			/* size of a single item (lw) */
     int c;			/* initial number of items */
     int incr;			/* number of items to add when enlarging */
     
{
  ROL_MEM_ID pPart;
  
  pPart = (ROL_MEM_ID) malloc (sizeof(ROL_MEM_PART));
  bzero ((char *) pPart, sizeof(ROL_MEM_PART));
  if (pPart != NULL)
    {
      listInit (&(pPart->list));
      pPart->size = maximum (size, sizeof(DANODE));
      pPart->incr = maximum (0, incr); 
      pPart->total = 0;
      strcpy(pPart->name, name);
      if (name && strlen(name) == 0)
	pPart->name[0] = NULL;
      listAdd (&partList, (DANODE *)pPart);      
      partIncr (pPart, c);
      printf("Created new partition -- %s\n",pPart->name);
    }
  return pPart;
}

/************************************************************** 
 * partFindByName - returns a pointer to a memory part
 *                  with the passed Name.
 */

ROL_MEM_ID partFindByName (char *name)
{
  ROL_MEM_ID	pPart = (ROL_MEM_ID) listFirst (&partList);
  
  while (pPart != NULL)
    {
      if (pPart->name && strcmp(pPart->name, name) == 0)break;
      pPart = (ROL_MEM_ID) listNext ((DANODE *)pPart);
    }
  return (pPart);
}


/************************************************************** 
 * partFree - frees all nodes for a given memory part
 *               and removes part from global part list. 
 */

void partFree(ROL_MEM_ID pPart)
{
  DANODE *the_node;

  while (pPart->list.c) {
    listGet(&(pPart->list),the_node);
    free(the_node);
    the_node = (DANODE *)0;
  }
  listSnip (&partList, (DANODE *)pPart);      
  free(pPart);
  pPart = 0;
}


/************************************************************** 
 * partFreeAll - frees all memory parts in global part list
 *               and frees all nodes for a given list. 
 */

void partFreeAll()
{
  ROL_MEM_ID	pPart = (ROL_MEM_ID) 0;

  if (listCount(&partList)) {
    pPart = (ROL_MEM_ID) listFirst (&partList);    
    while (pPart != NULL)
      {
	printf("partFreeAll - Freeing Memory Partition %s \n",pPart->name);
	partFree(pPart);
	pPart = (ROL_MEM_ID) listFirst (&partList);
      }
  } else {
    printf("partFreeAll - No Memory Partitions to Free\n");
  }
}


/*************************************************************** 
 * partHdr - Print headings for part statitistics printout 
 */

static void partHdr ()
     
{
  printf("Address    total  free  busy   size  incr  (KBytes)  Name\n");
  printf("-------    -----  ----  ----   ----  ----  --------  ----\n");
}


/*************************************************************** 
 * partPrint - Print statitistics for a single part 
 */

static void partPrint (pPart)
     ROL_MEM_ID	pPart;    
{
  int freen;

  printf("0x%08x  ",pPart);

  if (pPart != NULL)
    {
      freen = listCount (&pPart->list); 
      printf("%4d  %4d  %4d   %4d  %4d  (%d)       %s\n",
	    pPart->total,
	     freen,
	     pPart->total - freen,
	     pPart->size,
	     pPart->incr,
	     (((pPart->total * pPart->size) + 1023) / 1024),
	     pPart->name
	     );
    }
}


/**************************************************************** 
 * partStats - print statistics on a memory part 
 */

int partStats (pPart)
     ROL_MEM_ID	pPart;
     
{
  partHdr ();
  partPrint (pPart);
  return (0);
}

/****************************************************************
 *
 * partStatsAll - print statistics on all parts
 *
 * Print int for all memory parts.
 *
 * int includes the total number of items in the part, the
 * number that are in-use and free, the size of a single item,
 * and the total kilobytes of raw memory allocated.
 */

int	partStatsAll ()
     
{
  int lockKey;
  ROL_MEM_ID  pPart;
  
  partHdr ();
  lockKey = intLock();
  pPart = (ROL_MEM_ID) listFirst (&partList);
  while (pPart != NULL)
    {
      partPrint (pPart);
      pPart = (ROL_MEM_ID) listNext ((DANODE *)pPart);
    }
  intUnlock(lockKey);
  return (0);
}

/****************************************************************
 * partPrintList - prints statisics for a given list structure
 */
int partPrintList(alist)
     DALIST *alist;
{
  DANODE *theNode;

  printf("dalist->f         %x\n",alist->f);
  printf("dalist->l         %x\n",alist->l);
  printf("dalist->c         %d\n",alist->c);

  theNode = listFirst(alist);
  while (theNode) {
    printf ("part %x prev %x self %x next %x left %d fd %d\n",
	    theNode->part,
	    theNode->p,
	    theNode,
	    theNode->n,
	    theNode->left,
	    theNode->fd);
    theNode = listNext(theNode);
  }
  return(0);
}

#endif
