%{
  /*-----------------------------------------------------------------------------
   * Copyright (c) 1991,1992 Southeastern Universities Research Association,
   *                         Continuous Electron Beam Accelerator Facility
   *
   * This software was developed under a United States Government license
   * described in the NOTICE file included as part of this distribution.
   *
   * CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
   * Email: coda@cebaf.gov  Tel: (804) 249-7101  Fax: (804) 249-7363
   *-----------------------------------------------------------------------------
   * 
   * Description:
   *	CODA readout langauge parser
   *	
   * Author:  Jie Chen, CEBAF Data Acquisition Group
   *
   */
#include <stdio.h>
#include <string.h>
#include "crlHash.h"

#define OLD_SYNTAX 10

#define CAMAC_TY   1
#define VME_TY     2
#define FASTBUS_TY 3
#define SFI_TY     4
#define EVENT_TY   5
#define TEST_TY    6
#define GEN_TY     7
  
char *log_content;		/* dalogMessge content  */
char tx[256];			/* temparary expression */
char aline[256];
char *prefix,*eg;
int  newpad = 0;
int  newsad = 0;
int  syntax_style = 0;         /* new syntax here       */
int  insideBlock = 0;          /* inside user block     */
int  type_define = 0;
int  GdoneFlag = 0;           /* Global done routine flag */
  
char *rol_ty_names[9] = {
  "undefined",
  "camac",
  "vme",
  "fastbus",
  "sfi",
  "event",
  "test",
  "gen",
  "undefined",
};

extern char *rolf_name;
extern char *strdup();
extern char *strToLower ();
extern FILE *fout;
extern int  mode, type;
extern int  errorline,lineno,tokenpos;
extern char errorbuf[],lastbuf[];
  
void badContext()
{
  strcpy(errorbuf,lastbuf);
  fprintf(stderr,"line %d: Command not allowed in \"%s\" context\n%s\n",
	  lineno,rol_ty_names[type],errorbuf);
  fprintf(stderr,"%*s\n",tokenpos,"^");
}
%}

%union{
  char  *string;
  int   i_value;
  float r_value;
}

%token <i_value> CAMAC_ROL  VME_ROL  FASTBUS_ROL  SFI_ROL  EVENT_ROL  TEST_ROL GEN_ROL POLLING
%token <i_value> CAMAC_TYPE VME_TYPE FASTBUS_TYPE SFI_TYPE EVENT_TYPE TEST_TYPE GEN_TYPE
%token <i_value> TRIG_SUPERVISOR PARALLEL_LINK
%token <i_value> START_DOWNLOAD END_DOWNLOAD
%token <i_value> START_PRESTART END_PRESTART 
%token <i_value> START_END END_END
%token <i_value> START_PAUSE END_PAUSE
%token <i_value> START_GO END_GO
%token <i_value> START_TRIGGER END_TRIGGER
%token <i_value> START_DONE END_DONE
%token <i_value> START_STATUS END_STATUS 
%token <i_value> STATUS LIMIT FASTBUS SFI INLINE NOCHECK FAST
%token <i_value> RESET CRATE
%token <i_value> CLEAR CONTROL LINK READ FROM WRIT OUTPUT RANGE
%token <i_value> VARDEC HEX_NUM VME_N STD SUPV EXT DATA_N INTO_N MASK_N
%token <i_value> ULONG_DEC USHORT_DEC LONG_DEC SHORT_DEC
%token <i_value> EXPLICIT_HEX_NUM
%token <i_value> USER ASC PGM IO SHR
%token <i_value> ARBITRATE ADDRESS GEOGRAPHIC SECONDARY RELEASE BLOCK BROADCAST
%token <i_value> ACCESS_LOG INFORM WARN ALARM INTERRUPT
%token <i_value> INT_DEC INT_VAL IF ELSE ELSEIF ENDIF THEN
%token <i_value> WHILE ENDLOOP IS LESS THAN GREATER EQUAL NOT TO
%token <i_value> INCREMENT DECREMENT OR AND BY
%token <i_value> END_SELECT SELECT ON CASE DEFAULT BREAK
%token <i_value> OPEN BANK CLOSE NCLOSE OF MAX CONST WRITE
%token <i_value> USER_BEGIN USER_END CALL CODE ENABLE DISABLE START STOP TRIG TSINIT
%token <i_value> TSLINKA TSLINKS SOURCE WTYPE ETYPE EVENT WRAPEV
%token <i_value> GET_EVENT COPY_EVENT PASS_EVENT
%token <i_value> CONNECT RLIST INHIBIT SET
%token <i_value> DECLARE ATTACHED CINCLUDE
%token <string>  QSTRING VAR_NAME C_LINE BEGIN_C COMP_MATHOP 


%left '-' '+'
%left '*' '/'
%left '&' '|'
%nonassoc UMINUS
%nonassoc THEN
%nonassoc ELSE
%type <string> arith_exp var_list comma_list print_list init_dec 
%type <string> address_exp fastbus_address fastbus_secondary_addr pure_fastbus_addr
%start prog

%%
prog: preconditions global_type comp_code
      | preconditions global_type misc_dec comp_code
      | global_type comp_code
      | global_type misc_dec comp_code

global_type: type_mode {
            if (!type_define) {
              fprintf(fout,"#define INIT_NAME %s__init\n",rolf_name);
              fprintf(fout,"#include <rol.h>\n");
	      if (type == FASTBUS_TY) 
		fprintf(fout,"#include <FASTBUS_source.h>\n");
	      else if (type == SFI_TY)
		fprintf(fout,"#include <SFI_source.h>\n");
	      else if (type == CAMAC_TY)
		fprintf(fout,"#include <CAMAC_source.h>\n");
	      else if (type == VME_TY)
		fprintf(fout,"#include <VME_source.h>\n");
	      else if (type == EVENT_TY)
		fprintf(fout,"#include <EVENT_source.h>\n");
	      else if (type == TEST_TY)
		fprintf(fout,"#include <TEST_source.h>\n");
	      else if (type == GEN_TY)
		fprintf(fout,"#include <GEN_source.h>\n");
	      type_define = 1;
                         }
            }
      | type_mode global_type {
            if (!type_define) {
              fprintf(fout,"#define INIT_NAME %s__init\n",rolf_name);
              fprintf(fout,"#include <rol.h>\n");
	      if (type == FASTBUS_TY) 
		fprintf(fout,"#include <FASTBUS_source.h>\n");
	      else if (type == SFI_TY)
		fprintf(fout,"#include <SFI_source.h>\n");
	      else if (type == CAMAC_TY)
		fprintf(fout,"#include <CAMAC_source.h>\n");
	      else if (type == VME_TY)
		fprintf(fout,"#include <VME_source.h>\n");
	      else if (type == EVENT_TY)
		fprintf(fout,"#include <EVENT_source.h>\n");
	      else if (type == TEST_TY)
		fprintf(fout,"#include <TEST_source.h>\n");
	      else if (type == GEN_TY)
		fprintf(fout,"#include <GEN_source.h>\n");
	      type_define = 1;
                         }
            }
type_mode: type_def
      | mode_def
      | ts_control
      | parallel_link
      | fb_status_def
      | fb_inline_def
      | fb_nocheck_def

type_def: camac_rol
      | vme_rol
      | fastbus_rol
      | sfi_rol
      | event_rol
      | test_rol
      | gen_rol
      | camac_type
      | vme_type
      | fastbus_type
      | sfi_type
      | event_type
      | test_type
      | gen_type


preconditions: list_tag
      |        list_tag max_def

list_tag:  RLIST VAR_NAME { fprintf(fout,"#define ROL_NAME__ \"%s\"\n",$2);
                            free ($2);
                          }

max_def:  MAX arith_exp ',' arith_exp  { fprintf(fout,"#define MAX_EVENT_LENGTH %s\n",$2);
                                         fprintf(fout,"#define MAX_EVENT_POOL   %s\n",$4);
                                         free ($2); free($4);
                                        }

camac_rol: CAMAC_ROL { fprintf(fout,"\n/* CAMAC */\n");type = CAMAC_TY;}

camac_type: CAMAC_TYPE { fprintf(fout,"#define CAMAC\n"); type = CAMAC_TY;
		         syntax_style = OLD_SYNTAX;
		       }

vme_rol: VME_ROL {fprintf(fout,"\n/* VME */\n");type = VME_TY;}

vme_type: VME_TYPE     {fprintf(fout,"#define VME\n"); type = VME_TY;
		        syntax_style = OLD_SYNTAX;
		      }

fastbus_type: FASTBUS_TYPE {fprintf(fout,"#define FASTBUS\n"); type = FASTBUS_TY;
			    syntax_style = OLD_SYNTAX;
			  }
fastbus_rol: FASTBUS_ROL {fprintf(fout,"\n/* FASTBUS */\n");type = FASTBUS_TY;}


sfi_type: SFI_TYPE {fprintf(fout,"#define SFI\n"); type = SFI_TY;
			    syntax_style = OLD_SYNTAX;
			  }
sfi_rol: SFI_ROL {fprintf(fout,"\n/* SFI */\n");type = SFI_TY;}


event_type: EVENT_TYPE  {fprintf(fout,"#define EVENT_MODE\n"); type = EVENT_TY;
		        syntax_style = OLD_SYNTAX;
		      }
event_rol:   EVENT_ROL   {fprintf(fout,"\n/* EVENT */\n");type = EVENT_TY;}


test_type: TEST_TYPE  {fprintf(fout,"#define TEST_MODE\n"); type = TEST_TY;
		        syntax_style = OLD_SYNTAX;
		      }
test_rol: TEST_ROL {fprintf(fout,"\n/* TEST */\n");type = TEST_TY;}


gen_type: GEN_TYPE  {fprintf(fout,"#define GEN_MODE\n"); type = GEN_TY;
		        syntax_style = OLD_SYNTAX;
		      }
gen_rol: GEN_ROL {fprintf(fout,"\n/* GENERAL */\n");type = GEN_TY;}


mode_def: polling_mode

ts_control: TRIG_SUPERVISOR {fprintf(fout,"#define TRIG_SUPV\n");}

parallel_link: PARALLEL_LINK {fprintf(fout,"#define PARALLEL_LINK\n");}

polling_mode: POLLING { fprintf(fout,"/* POLLING_MODE */\n");
                        fprintf(fout,"#define POLLING___\n");
			fprintf(fout,"#define POLLING_MODE\n");
                       }

fb_status_def:LIMIT STATUS     {if (type == FASTBUS_TY) 
				  fprintf(fout,"#define FB_LIMITED_STATUS\n"); 
                                else 
			          badContext();
			       }

fb_inline_def:INLINE FASTBUS   {if (type == FASTBUS_TY) 
				  fprintf(fout,"#define FB_BUILD_INLINE\n");
                                else 
				  badContext();
			       }

fb_nocheck_def:NOCHECK FASTBUS {if (type == FASTBUS_TY) 
				  fprintf(fout,"#define FB_BUILD_NOCHECK 1\n"); 
                                else 
				  badContext();
			      }

misc_dec: real_misc_code
      | real_misc_code misc_dec

real_misc_code: int_var_dec 
      | var_dec
      | const_assignment
      | c_include

real_misc_decs:real_misc_dec
      | real_misc_dec real_misc_decs

 
real_misc_dec:int_var_dec
      | var_dec
      | const_assignment

int_var_dec: INT_DEC var_list {fprintf(fout,"int %s ;\n",$2);
			       free ($2); }
      | INT_DEC var_list ';'  {fprintf(fout,"int %s ;\n",$2);
			       free ($2); }
      ;

var_list: init_dec             {strcpy(tx,$1);$$ = strdup(tx); free ($1);}
      | init_dec ',' var_list  {strcpy(tx,$1);strcat(tx,", ");
				strcat(tx,$3);$$ = strdup(tx);
                                free ($1); free ($3);
                               }
      ;

var_dec: VARDEC var_list       {fprintf(fout,"unsigned long %s;\n",$2); free ($2);}
      | VARDEC var_list ';'    {fprintf(fout,"unsigned long %s;\n",$2); free ($2);}
      | ULONG_DEC var_list     {fprintf(fout,"unsigned long %s;\n",$2); free ($2);}
      | ULONG_DEC var_list ';' {fprintf(fout,"unsigned long %s;\n",$2); free ($2);}
      | USHORT_DEC var_list    {fprintf(fout,"unsigned short %s;\n",$2);free ($2);}
      | USHORT_DEC var_list ';' {fprintf(fout,"unsigned short %s;\n",$2);free ($2);} 
      | LONG_DEC var_list      {fprintf(fout,"long %s;\n",$2); free ($2);}
      | LONG_DEC var_list ';'  {fprintf(fout,"long %s;\n",$2); free ($2);}     
      | SHORT_DEC var_list     {fprintf(fout,"short %s;\n",$2); free ($2);}
      | SHORT_DEC var_list ';' {fprintf(fout,"short %s;\n",$2); free ($2);}     
      ;
 
init_dec:   VAR_NAME           {crlAddSymbols($1,0);$$ = strdup($1); free ($1);}
      | VAR_NAME '=' arith_exp {crlAddSymbols($1,0);
				strcpy(tx,$1);strcat(tx,"=");
				strcat(tx,$3);$$ = strdup(tx);
			        free ($1); free ($3);}
      ;

const_assignment: CONST VAR_NAME '=' arith_exp {crlAddSymbols($2,0);
			      fprintf(fout,"#define %s %s\n",$2, $4);
			      free ($2); free ($4);}
      ;

c_include: CINCLUDE QSTRING { fprintf(fout,"#include \"%s\"\n",$2);
			       free ($2);}
      ;


arith_assignment: VAR_NAME '=' arith_exp {isSymbolFound($1);
					  fprintf(fout,"%s = %s ;\n",$1, $3);
					  free ($3); free ($1);}
      | VAR_NAME IS arith_exp            {isSymbolFound($1);
					  fprintf(fout,"%s = %s ;\n",$1, $3);
				          free ($1); free ($3);}
      ;

arith_exp: arith_exp '+' arith_exp {strcpy(tx,$1);strcat(tx,"+");
				    strcat(tx,$3);$$ = strdup(tx);
				    free ($1); free ($3);}
      |    arith_exp '-' arith_exp {strcpy(tx,$1);strcat(tx,"-");
				    strcat(tx,$3);$$ = strdup(tx);
				    free ($1); free ($3); }
      |    arith_exp '*' arith_exp {strcpy(tx,$1);strcat(tx,"*");
				    strcat(tx,$3);$$ = strdup(tx);
				    free ($1); free ($3); }
      |    arith_exp '/' arith_exp {strcpy(tx,$1);strcat(tx,"/");
				    strcat(tx,$3);$$ = strdup(tx);
				    free ($1); free ($3);}
      |    arith_exp '&' arith_exp {strcpy(tx,$1);strcat(tx,"&");
				    strcat(tx,$3);$$ = strdup(tx);
				    free ($1); free ($3);}
      |    arith_exp '|' arith_exp {strcpy(tx,$1);strcat(tx,"|");
				    strcat(tx,$3);$$ = strdup(tx);
				    free ($1); free ($3);}
      |    '(' arith_exp ')'       {strcpy(tx,"(");strcat(tx,$2);
				    strcat(tx,")");$$ = strdup(tx);
				    free ($2); }
      |    INT_VAL                 {sprintf(tx,"%d",$1); $$ = strdup(tx);}
      |    HEX_NUM                 {sprintf(tx,"0x%s",$1); $$ = strdup(tx);}
      |    EXPLICIT_HEX_NUM        {sprintf(tx,"%s",$1);  $$ = strdup(tx);}
      |    VAR_NAME                {sprintf(tx,"%s",$1); $$ = strdup(tx);}

comma_list: arith_exp              {sprintf(tx,"%s",$1); $$ = strdup(tx);
                                    free ($1);
                                   }
      | arith_exp ',' comma_list   {strcpy(tx,$1);strcat(tx,",");
                                    strcat(tx,$3);$$ = strdup(tx);
                                    free ($1); free ($3);
                                   }

print_list: arith_exp              {sprintf(tx,"%s",$1); $$ = strdup(tx);
                                    free ($1);
                                   }
      | QSTRING                    {sprintf(tx,"%s\"",$1); $$ = strdup(tx);
                                    free ($1);
                                   }
      | arith_exp ',' print_list   {strcpy(tx,$1);strcat(tx,"\",");strcat(tx,$3);
                                    $$ = strdup(tx);
                                    free ($1); free ($3);
                                   }
      | QSTRING ',' print_list     {sprintf(tx,"%s\",%s",$1,$3);
                                    $$ = strdup(tx);
                                    free ($1); free ($3);
                                   }

comp_code: single_code
      | single_code comp_code

single_code:clear_var
      | var_increment
      | var_decrement
      | arith_assignment
      | c_line
      | access_log
      | reset
      | clear
      | cntl
      | read
      | write
      | network_io
      | fastbus_arbitrate
      | fastbus_address
      | fastbus_secondary_addr
      | fastbus_block_read 
      | fastbus_release
      | user_block
      | download_block
      | prestart_block
      | end_block
      | pause_block
      | go_block
      | trig_block
      | done_block
      | status_block
      | while_block
      | if_else_block
      | select_block
      | begin_c
      | bank_io
      | call_routine
      | trigger_source

clear_var: CLEAR VAR_NAME          {isSymbolFound($2);
                                    fprintf(fout,"    %s = 0 ;\n", $2);
                                    free ($2);
                                   }

var_increment: INCREMENT VAR_NAME          {isSymbolFound($2);
                                            fprintf(fout,"    %s++ ;\n",$2); 
                                            free ($2);
                                           }
       | INCREMENT VAR_NAME BY arith_exp   {isSymbolFound($2);
                                            fprintf(fout,"    %s += %s ;\n",$2,$4); 
                                            free ($2);free($4);
                                           }


var_decrement: DECREMENT VAR_NAME          {isSymbolFound($2);
                                            fprintf(fout,"    %s-- ;\n",$2); 
                                            free ($2);
                                           }
       | DECREMENT VAR_NAME BY arith_exp   {isSymbolFound($2);
                                            fprintf(fout,"    %s -= %s ;\n",$2,$4); 
                                            free ($2);free($4);
                                           }


download_block:start_dnld real_misc_decs comp_code end_dnld
      | start_dnld comp_code end_dnld
      | start_dnld real_misc_decs end_dnld
      | start_dnld end_dnld

start_dnld: START_DOWNLOAD  {insideBlock = 1;
                             fprintf(fout,"static void __download()\n");
			     fprintf(fout,"{\n");
                             fprintf(fout,"    daLogMsg(\"INFO\",\"Readout list compiled %%s\", DAYTIME);\n");
                             fprintf(fout,"#ifdef POLLING___\n");
                             fprintf(fout,"   rol->poll = 1;\n");
                             fprintf(fout,"#endif\n");
                             if(type == FASTBUS_TY) {
			       fprintf(fout,"    {\n");
			       fprintf(fout,"       static int ourTaskId=0;  \n");  
			       fprintf(fout,"       if (taskIdSelf() != ourTaskId)\n");
			       fprintf(fout,"         {\n");
			       fprintf(fout,"           ourTaskId = taskIdSelf();\n");
			       fprintf(fout,"           fb_task_init_1();\n");
			       fprintf(fout,"         }\n");
			       fprintf(fout,"    }\n");
			     } 
			     else if (type == SFI_TY) {
			       fprintf(fout,"/* Get offset for local DRAM as seen from VME bus */\n");
			       fprintf(fout,"  if (sysLocalToBusAdrs(0x09,0,&sfi_cpu_mem_offset)) { \n");
			       fprintf(fout,"     printf(\"**ERROR** in sysLocalToBusAdrs() call \\n\"); \n");
			       fprintf(fout,"     printf(\"sfi_cpu_mem_offset=0 FB Block Reads may fail \\n\"); \n");
			       fprintf(fout,"  } else { \n");
			       fprintf(fout,"     printf(\"sfi_cpu_mem_offset = 0x%%x \\n\",sfi_cpu_mem_offset); \n");
			       fprintf(fout,"  } \n");
			     } 
			     else if (type == VME_TY) {
			     }
			     else if (type == CAMAC_TY) {
			     }
			     fprintf(fout,"    *(rol->async_roc) = 0; /* Normal ROC */\n");
			     fprintf(fout,"  {  /* begin user */\n");
                            }


end_dnld: END_DOWNLOAD      {
			     fprintf(fout,"  }  /* end user */\n");
                             if (type == FASTBUS_TY) {
			        fprintf(fout,"    return;\n");
			        fprintf(fout,"   fooy: \n");
			        fprintf(fout,"    FB_REPORT_ERROR;\n");
			        fprintf(fout,"    RECOVER;\n");
                             }
                             else if (type == SFI_TY) {
			        fprintf(fout,"    return;\n");
			        fprintf(fout,"   fooy: \n");
			        fprintf(fout,"    SFI_REPORT_ERROR;\n");
			        fprintf(fout,"    RECOVER;\n");
                             }
                             else if (type == VME_TY) {
                             } 
                             else if (type == CAMAC_TY) {
                             }
                             fprintf(fout,"} /*end download */     \n"); 
                             fprintf(fout,"\n");
                             insideBlock = 0;
                            }

prestart_block: start_prestart real_misc_decs comp_code end_prestart
      | start_prestart comp_code end_prestart
      | start_prestart real_misc_decs list_comms comp_code end_prestart
      | start_prestart list_comms comp_code end_prestart
      ;

start_prestart: START_PRESTART {insideBlock = 1;
                                fprintf(fout,"static void __prestart()\n");
                                fprintf(fout,"{\n");
                                fprintf(fout,"CTRIGINIT;\n");
                                if (type == FASTBUS_TY) {
				  fprintf(fout,"    fb_init_1 ();\n");
				  fprintf(fout,"    fb_frd_1(0,0,0,0,0,1,1,1,1,1); /* arb */\n");
				  fprintf(fout,"    IFERROR(\"hidden arbitrate\");\n");
				  fprintf(fout,"    fptdisable();\n");
                                } 
				else if (type == VME_TY) {
                                } 
				else if (type == CAMAC_TY) {
                                }
                                fprintf(fout,"    *(rol->nevents) = 0;\n");
	  		        fprintf(fout,"  {  /* begin user */\n");
 	 		      }

end_prestart: END_PRESTART {
			    fprintf(fout,"  }  /* end user */\n");
                            fprintf(fout,"    if (__the_event__) WRITE_EVENT_;\n");
                            fprintf(fout,"    *(rol->nevents) = 0;\n");
                            fprintf(fout,"    rol->recNb = 0;\n");
                            if (type == FASTBUS_TY) {
			      fprintf(fout,"    APORT_DR = 0x81;\n");
			      fprintf(fout,"    return;\n");
			      fprintf(fout,"   fooy: \n");
			      fprintf(fout,"    FB_REPORT_ERROR;\n");
			      fprintf(fout,"    RECOVER;\n");
                            } 
                            else if (type == SFI_TY) {
			       fprintf(fout,"    return;\n");
			       fprintf(fout,"   fooy: \n");
			       fprintf(fout,"    SFI_REPORT_ERROR;\n");
			       fprintf(fout,"    RECOVER;\n");
                            }
			    else if (type == VME_TY) {
                            } 
			    else if (type == CAMAC_TY) {
                            }
			    fprintf(fout,"} /*end prestart */     \n"); 
			    fprintf(fout,"\n");
                            insideBlock = 0;
			  }

end_block: start_end real_misc_decs comp_code end_end
      | start_end comp_code end_end
      | start_end real_misc_decs end_end
      | start_end end_end

start_end: START_END {insideBlock = 1;
                      fprintf(fout,"static void __end()\n{\n");
                      if (type == FASTBUS_TY) {
			 fprintf(fout,"#ifndef TRIG_SUPV\n");
			 fprintf(fout,"    APORT_DR = 0x80;\n");
			 fprintf(fout,"#endif\n");
			 fprintf(fout,"    fptdisable();\n");
		      } 
		      else if (type == VME_TY) {
		      }
		      else if (type == CAMAC_TY) {
		      }
		      fprintf(fout,"  {  /* begin user */\n");
		     }

end_end: END_END     {
		      fprintf(fout,"  }  /* end user */\n");
                      fprintf(fout,"    if (__the_event__) WRITE_EVENT_;\n");
                      if (type == FASTBUS_TY) {
			fprintf(fout,"    return;\n");
			fprintf(fout,"   fooy: \n");
			fprintf(fout,"    FB_REPORT_ERROR;\n");
			fprintf(fout,"    RECOVER;\n");
		      } 
		      else if (type == SFI_TY) {
			fprintf(fout,"    return;\n");
			fprintf(fout,"   fooy: \n");
			fprintf(fout,"    SFI_REPORT_ERROR;\n");
			fprintf(fout,"    RECOVER;\n");
		      }
		      else if (type == VME_TY) {
		      } 
		      else if (type == CAMAC_TY) {
		      }
                      fprintf(fout,"} /* end end block */\n");
		      fprintf(fout,"\n");
                      insideBlock = 0;
		     }

pause_block :start_pause real_misc_decs comp_code end_pause
      | start_pause comp_code end_pause
      | start_pause real_misc_decs end_pause
      | start_pause end_pause

start_pause: START_PAUSE {insideBlock = 1;
                          fprintf(fout,"static void __pause()\n{\n");
                          if (type == FASTBUS_TY) {
			     fprintf(fout,"#ifndef TRIG_SUPV\n");
			     fprintf(fout,"    APORT_DR = 0x80;\n");
			     fprintf(fout,"#endif\n");
			     fprintf(fout,"    fptdisable();\n");
			  }
			  else if (type == VME_TY) {
			  } 
			  else if (type == CAMAC_TY) {
			  }
			  fprintf(fout,"  {  /* begin user */\n");
			 }

end_pause: END_PAUSE {
			  fprintf(fout,"  }  /* end user */\n");
                          fprintf(fout,"    if (__the_event__) WRITE_EVENT_;\n");
                          if (type == FASTBUS_TY) {
                            fprintf(fout,"    return;\n");
                            fprintf(fout,"   fooy: \n");
                            fprintf(fout,"    FB_REPORT_ERROR;\n");
                            fprintf(fout,"    RECOVER;\n");
			  } 
			  else if (type == SFI_TY) {
			    fprintf(fout,"    return;\n");
			    fprintf(fout,"   fooy: \n");
			    fprintf(fout,"    SFI_REPORT_ERROR;\n");
			    fprintf(fout,"    RECOVER;\n");
			  }
			  else if (type == VME_TY) {
			  } 
			  else if (type == CAMAC_TY) {
			  }
		          fprintf(fout,"} /*end pause */\n");
                          insideBlock = 0;
                     }

go_block: start_go real_misc_decs comp_code end_go
      | start_go comp_code end_go
      | start_go real_misc_decs end_go
      | start_go end_go

start_go: START_GO   {    insideBlock = 1;
                          fprintf(fout,"static void __go()\n{\n");
		          fprintf(fout,"\n");
                          if (type == FASTBUS_TY) {
			     fprintf(fout,"    fpttoggle();\n");
			     fprintf(fout,"#ifdef PARALLEL_LINK\n");
			     fprintf(fout,"    OPORT_CSR1 = 0xe8;\n");
			     fprintf(fout,"    OPORT_CSR1 = 0xe1;\n");
			     fprintf(fout,"#endif\n");
			  } 
			  else if (type == VME_TY) {
			  }
			  else if (type == CAMAC_TY) {
			  }
			  fprintf(fout,"  {  /* begin user */\n");
		     }

end_go: END_GO       {
			  fprintf(fout,"  }  /* end user */\n");
                          fprintf(fout,"    if (__the_event__) WRITE_EVENT_;\n");
                          if (type == FASTBUS_TY) {
			     fprintf(fout,"#ifndef POLLING_MODE\n");
			     fprintf(fout,"    fptenable();\n");
			     fprintf(fout,"#endif\n");
			     fprintf(fout,"    fpttoggle();\n");
			     fprintf(fout,"#ifndef TRIG_SUPV\n");
			     fprintf(fout,"    APORT_DR = 0x82;\n");
			     fprintf(fout,"#else\n");
			     fprintf(fout,"    APORT_DR = 0x00;\n");
			     fprintf(fout,"    APORT_DR = 0x00;\n");
			     fprintf(fout,"#endif\n");
                             fprintf(fout,"    return;\n");
                             fprintf(fout,"   fooy: \n");
                             fprintf(fout,"    FB_REPORT_ERROR;\n");
                             fprintf(fout,"    RECOVER;\n");
			  } 
			  else if (type == SFI_TY) {
			    fprintf(fout,"    return;\n");
			    fprintf(fout,"   fooy: \n");
			    fprintf(fout,"    SFI_REPORT_ERROR;\n");
			    fprintf(fout,"    RECOVER;\n");
			  }
			  else if (type == VME_TY) {
			  } 
			  else if (type == CAMAC_TY) {
			  }
                          fprintf(fout,"}\n\n");
                          insideBlock = 0;
                    }


trig_block: start_trig real_misc_decs comp_code end_trig 
      | start_trig comp_code end_trig 
      | start_trig real_misc_decs end_trig
      | start_trig end_trig
      | start_trig real_misc_decs event_rol comp_code end_trig 
      | start_trig real_misc_decs event_rol end_trig
      | start_trig event_rol comp_code end_trig 
      | start_trig event_rol end_trig
      ;

start_trig: START_TRIGGER VAR_NAME {insideBlock = 1;
                                    crlAddSymbols($2,0);
                                    fprintf(fout,"void %s(unsigned long EVTYPE,unsigned long EVSOURCE)\n",$2);
                                    fprintf(fout,"{\n");
			            fprintf(fout,"    long EVENT_LENGTH;\n");
                                    if (type == FASTBUS_TY) {
				      fprintf(fout,"#ifndef POLLING_MODE\n");
				      fprintf(fout,"#define daLogMsg printf\n");
				      fprintf(fout,"    fptdisable ();\n");
				      fprintf(fout,"#endif\n");
				    } 
				    else if (type == VME_TY) {
				    } 
				    else if (type == CAMAC_TY) {
				    }
                                    free ($2);
			            fprintf(fout,"  {  /* begin user */\n");
			           }

end_trig: END_TRIGGER {             
			            fprintf(fout,"  }  /* end user */\n");
                                    if (type == FASTBUS_TY) {
				      fprintf(fout,"    return;\n");
				      fprintf(fout,"  fooy: \n");
                                      fprintf(fout,"    *rol->dabufp = 0xFB000BAD;\n");
				      fprintf(fout,"    *rol->dabufp++;\n");
				      fprintf(fout,"    FB_REPORT_ERROR;\n");
				      fprintf(fout,"    RECOVER;\n");
				    } 
				    else if (type == SFI_TY) {
				      fprintf(fout,"    return;\n");
				      fprintf(fout,"   fooy: \n");
				      fprintf(fout,"    SFI_REPORT_ERROR;\n");
				      fprintf(fout,"    RECOVER;\n");
				    }
				    else if (type == VME_TY) {
				    }
				    else if (type == CAMAC_TY) {
				    }
                                    fprintf(fout,"} /*end trigger */\n");
                                    fprintf(fout,"\n");
                                    insideBlock = 0;
        	     }

done_block: start_done real_misc_decs comp_code end_done
      | start_done comp_code end_done
      | start_done real_misc_decs end_done
      | start_done end_done

start_done: START_DONE          {insideBlock = 1;
                                 GdoneFlag = 1;
                                 fprintf(fout,"void __done()\n");
			         fprintf(fout,"{\n");
                                 if (type == FASTBUS_TY) {
				 }
                                 if (type == SFI_TY) {
				 }
				 else if (type == VME_TY) {
                                 }
				 else if (type == CAMAC_TY) {
                                 }
                                 fprintf(fout,"poolEmpty = 0; /* global Done, Buffers have been freed */\n");
			         fprintf(fout,"  {  /* begin user */\n");}
      | START_DONE VAR_NAME     {insideBlock = 1;
                                 crlAddSymbols($2,0);
                                 fprintf(fout,"void %s_done()\n",$2);
			         fprintf(fout,"{\n");
                                 if (type == FASTBUS_TY) {
				 }
                                 if (type == SFI_TY) {
				 }
				 else if (type == VME_TY) {
                                 }
				 else if (type == CAMAC_TY) {
                                 }
                                 free ($2);
			         fprintf(fout,"  {  /* begin user */\n");}




end_done: END_DONE {
			        fprintf(fout,"  }  /* end user */\n");
                                if (type == FASTBUS_TY) {
				   if (GdoneFlag) {
				     fprintf(fout,"#ifndef POLLING_MODE\n");
				     fprintf(fout,"    fptenable();\n");
				     fprintf(fout,"#endif\n");
				     fprintf(fout,"    fpttoggle();\n");
				     fprintf(fout,"#ifndef TRIG_SUPV\n");
				     fprintf(fout,"    APORT_DR = 0xc2;\n");
				     fprintf(fout,"    APORT_DR = 0x82;\n");
				     fprintf(fout,"#else\n");
				     fprintf(fout,"    APORT_DR = 0x40;\n");
				     fprintf(fout,"    APORT_DR = 0;\n");
				     fprintf(fout,"#endif\n");
				     GdoneFlag = 0;
				   }
				   fprintf(fout,"    return;\n");
				   fprintf(fout,"   fooy: \n");
				   fprintf(fout,"    FB_REPORT_ERROR;\n");
				   fprintf(fout,"    RECOVER;\n");
                                } 
				else if (type == SFI_TY) {
				  fprintf(fout,"    return;\n");
				  fprintf(fout,"   fooy: \n");
				  fprintf(fout,"    SFI_REPORT_ERROR;\n");
				  fprintf(fout,"    RECOVER;\n");
				}
				else if (type == VME_TY) {
				   if (GdoneFlag) {
				     GdoneFlag = 0;
				   }
                                }
				else if (type == CAMAC_TY) {
				   if (GdoneFlag) {
				     GdoneFlag = 0;
				   }
                                }
                                fprintf(fout,"} /*end done */\n");
		                fprintf(fout,"\n");
                                insideBlock = 0;
		  }

status_block: start_status real_misc_decs comp_code end_status
      | start_status comp_code end_status
      | start_status real_misc_decs end_status
      | start_status end_status
      ;

start_status: START_STATUS {insideBlock = 1;
                            fprintf(fout,"static void __status()\n");
			    fprintf(fout,"{\n");
			    fprintf(fout,"  {  /* begin user */\n");
			   }
      ;

end_status: END_STATUS {
			fprintf(fout,"  }  /* end user */\n");
                        fprintf(fout,"} /* end status */\n");
			fprintf(fout,"\n");
                        insideBlock = 0;
		      }
      ;

user_block:  start_user_block real_misc_decs comp_code end_user_block
      | start_user_block comp_code end_user_block
      | start_user_block real_misc_decs end_user_block
      | start_user_block end_user_block

start_user_block: USER_BEGIN VAR_NAME {insideBlock = 1;
                                       fprintf(fout,"void %s () \n", $2);
				       fprintf(fout,"{\n");
                                       free ($2);
				     }
end_user_block: USER_END VAR_NAME {fprintf(fout,"} /* end of user codes */\n");
				    fprintf(fout,"\n");
                                    free ($2);
                                    insideBlock = 0;
				  }

while_block: while_statement comp_code  end_statement

while_statement: WHILE {fprintf(fout,"while( ");} comp_logical_exp 
                 {fprintf(fout,"){ \n"); }
      ;

comp_logical_exp: logical_exp
      | '('{fprintf(fout,"(");} comp_logical_exp ')'{fprintf(fout,")");} logic_op 
        '('{fprintf(fout,"(");} comp_logical_exp ')'{fprintf(fout,")");} 
      ;

logic_op: AND {fprintf(fout,"&&");}
      | OR    {fprintf(fout,"||");}
      | COMP_MATHOP {fprintf(fout," %s ",$1);}
      ;

logical_exp: VAR_NAME IS arith_exp                      {isSymbolFound($1);
                                                         fprintf(fout,"( %s == %s) ",$1, $3);
                                                         free ($1); free ($3);}
      | VAR_NAME COMP_MATHOP arith_exp                  {isSymbolFound($1);
                                                         fprintf(fout,"( %s %s %s) ",$1, $2, $3);
                                                         free ($1); free ($2); free ($3);}
      | VAR_NAME IS LESS THAN arith_exp                 {isSymbolFound($1);
                                                         fprintf(fout,"( %s <  %s) ",$1, $5);
                                                         free ($1); free ($5);}
      | VAR_NAME IS GREATER THAN arith_exp              {isSymbolFound($1);
                                                         fprintf(fout,"( %s >  %s) ",$1, $5);
                                                         free ($1); free ($5);}
      | VAR_NAME IS EQUAL TO arith_exp                  {isSymbolFound($1);
                                                         fprintf(fout,"( %s == %s) ",$1, $5);
                                                         free ($1); free ($5);}
      | VAR_NAME IS NOT EQUAL TO arith_exp              {isSymbolFound($1);
                                                         fprintf(fout,"( %s != %s) ",$1, $6);
                                                         free ($1); free ($6);}
      | VAR_NAME IS LESS THAN OR EQUAL TO arith_exp     {isSymbolFound($1);
                                                         fprintf(fout,"( %s <= %s) ",$1, $8);
                                                         free ($1); free ($8);}
      | VAR_NAME IS GREATER THAN OR EQUAL TO arith_exp  {isSymbolFound($1);
                                                         fprintf(fout,"( %s >= %s) ",$1, $8);
                                                         free ($1); free ($8);}
      | '!' VAR_NAME                                    {isSymbolFound($2);
                                                         fprintf(fout,"(!%s) ",$2);
                                                         free ($2);}
      ;

end_statement:ENDLOOP   {fprintf(fout,"}/* end while loop */\n");
			 fprintf(fout,"\n");
		       }

if_else_block: if_block elseif_block else_block endif_statement
      | if_block elseif_block endif_statement
      | if_block else_block endif_statement
      | if_block endif_statement

if_block:if_statement comp_code

if_statement: IF {fprintf(fout,"if(");} comp_logical_exp {fprintf(fout,") {\n");} THEN

elseif_block: single_elseif
      | elseif_block single_elseif

single_elseif:ELSEIF {fprintf(fout,"}\nelse if(");} comp_logical_exp {fprintf(fout,"){\n");}
              comp_code 

else_block:ELSE {fprintf(fout,"}\nelse{\n");} comp_code 
      | ELSE {fprintf(fout,"}\nelse{ \n ;\n");}

endif_statement:ENDIF {fprintf(fout,"\n}/*endif whole if block*/\n");
		       fprintf(fout,"\n");
		      }

select_block: select_statement case_blocks default_block endselect_statement
      | select_statement case_blocks endselect_statement
      | select_statement default_block endselect_statement
      | select_statement endselect_statement

select_statement: SELECT ON arith_exp {fprintf(fout,"switch (%s) {\n",$3);
                                       free ($3);}

case_blocks:case_block
      | case_blocks case_block

case_block:case_statement comp_code {fprintf(fout,"    break;\n");}

case_statement: CASE arith_exp {fprintf(fout,"case %s:\n",$2); free ($2);}

default_block: DEFAULT {fprintf(fout,"default: \n");} comp_code {fprintf(fout,"    break;\n");}

endselect_statement: END_SELECT {fprintf(fout,"}/*end switch*/\n");}

c_line: C_LINE {fprintf(fout,"%s\n",$1); free ($1);} 

begin_c: BEGIN_C {
                   if (insideBlock){                  
		     fprintf(fout,"{/* inline c-code */\n %s \n }/*end inline c-code */\n",$1); 
		   }
                   else {  /* user function */
		     fprintf(fout,"/* inline c-code */\n %s \n /*end inline c-code */\n",$1); 
		   } 
                   free ($1);
                 }

access_log: ACCESS_LOG INFORM print_list {log_content = strdup($3);
                 fprintf(fout,"    daLogMsg(\"INFO\",\"%s);\n",log_content);
  		 fprintf(fout,"\n"); free ($3);
                                         }
      | ACCESS_LOG WARN print_list       {log_content = strdup($3);
                 fprintf(fout,"    daLogMsg(\"WARN\",\"%s);\n",log_content);
                 fprintf(fout,"\n"); free ($3);
			                 }
      | ACCESS_LOG ALARM print_list      {log_content = strdup($3);
                 fprintf(fout,"    daLogMsg(\"ALRM\",\"%s);\n",log_content);
	         fprintf(fout,"\n"); free ($3);
			                 }
      | ACCESS_LOG INTERRUPT print_list      {log_content = strdup($3);
                 fprintf(fout,"    daLogMsgI(\"INTR\",\"%s);\n",log_content);
	         fprintf(fout,"\n"); free ($3);
			                 }
      | ACCESS_LOG print_list            {log_content = strdup($2);
                 fprintf(fout,"    daLogMsg(\"%s);\n",log_content);
                 fprintf(fout,"\n"); free ($2);
			                 }

reset: RESET CRATE arith_exp {switch(type) {
                                  case CAMAC_TY: 
                                    fprintf(fout,"    ccinit(0) ;\n",$3);
                                    break;
		                  case FASTBUS_TY: 
                                    fprintf(fout,"    fb_init_1();\n");
                                    break;
		                  case SFI_TY: 
                                    fprintf(fout,"    fb_init_1(%s);\n",$3);
                                    break;
			          default:
                                    fprintf(fout,"/* no reset routine */ \n");
                                    badContext();
                                    break;
				  }
                              free ($3);
                             }

clear: CLEAR CRATE INHIBIT arith_exp {
                                     if (type == CAMAC_TY  ) { 
                                      fprintf(fout,"    cdreg(&address___, 0, %s,0, 0);\n",$4);
				      fprintf(fout,"    ccci(address___,0);\n");
			             } else {
                                      badContext();
				     }	
                                     free ($4);
                                    }
       | SET CRATE INHIBIT arith_exp {
                                     if (type == CAMAC_TY  ) { 
                                        fprintf(fout,"    cdreg(&address___, 0, %s,0, 0);\n",$4);
				        fprintf(fout,"    ccci(address___,1);\n");
			             } else {
                                        badContext();
				     }	
                                     free ($4);
                                   }			      

cntl: CONTROL arith_exp ',' arith_exp ',' arith_exp ',' arith_exp ',' arith_exp   
                          {
                           if (type == CAMAC_TY  ) { 
			     fprintf(fout,"    cdreg(&address___,%s, %s, %s, %s);\n",$2,$4,$6,$8);
			     fprintf(fout,"    cssa(%s,address___,&dumstat,&dumstat);\n",$10);
			     fprintf(fout,"\n");
			   } else {
			     badContext();
			   }
                           free ($2); free ($4); free ($6); free ($8); free ($10);
                          }      				      

fastbus_address: BROADCAST DATA_N arith_exp {
               if ((type == FASTBUS_TY)||(type == SFI_TY)) {
                 sprintf(tx,"broadcast data %s",$3);$$=strdup(tx);
                 fprintf(fout,"    padr   = %s ;\n", $3 );
                 prefix = "dm";
                 eg     = "";
		 newpad = 1;
	       } else {
                 badContext();
               }
               free ($3);
              }           
      | BROADCAST CONTROL arith_exp {
               if ((type == FASTBUS_TY)||(type == SFI_TY)) {
                 sprintf(tx,"broadcast control %s",$3);$$=strdup(tx);
	         fprintf(fout,"    padr   = %s;\n", $3 );
		 prefix = "cm";
                 eg     = "";
 		 newpad = 1;
	       } else {
                 badContext();
               }	
               free ($3);
              }	       
      | GEOGRAPHIC DATA_N arith_exp {
               if ((type == FASTBUS_TY)||(type == SFI_TY)) {
                 fprintf(fout,"    padr   = %s ;\n",$3);
                 sprintf(tx,"geographic data %s",$3);$$=strdup(tx);
		 prefix = "d";
                 eg     = ",1";
		 newpad = 1;
	       } else {
                 badContext();
               }
               free ($3);
            }
      | GEOGRAPHIC CONTROL arith_exp {
               if ((type == FASTBUS_TY)||(type == SFI_TY)) {
	         fprintf(fout,"    padr   = %s; \n",$3);
                 sprintf(tx,"geographic control %s",$3);$$=strdup(tx);
		 prefix = "c";
                 eg     = ",1";
		 newpad = 1;
	       } else {
                 badContext();
               }
               free ($3);
            }

fastbus_secondary_addr:SECONDARY ADDRESS arith_exp {
                if ((type == FASTBUS_TY)||(type == SFI_TY)) {
                 sprintf(tx,"secondary address %s",$3);$$=strdup(tx);
                 fprintf(fout,"    sadr = %s ;\n",$3);
                 newsad = 1;	 
	       } else {
                 badContext();
               }
              }
      ;

pure_fastbus_addr: fastbus_address fastbus_secondary_addr {
                      sprintf(tx,"%s %s",$1,$2);$$=strdup(tx);
                      free ($1); free ($2);
               }

address_exp:  pure_fastbus_addr {
                      $$ = $1;
                      fprintf(fout,"    /* here */\n"); 
               }
      | fastbus_address {
                      $$ = $1;
                      fprintf(fout,"    /* here */\n");
               }
      | arith_exp ',' arith_exp ',' arith_exp ',' arith_exp ',' arith_exp {
                      fprintf(fout,"    cdreg(&address___, %s, %s, %s, %s);\n",$1,$3,$5,$7);
                      sprintf(tx,"%s, %s, %s, %s, %s",$1,$3,$5,$7,$9);
                      $$=strdup(tx);
                      fprintf(fout,"    fcode = %s;\n",$9);
                      free ($1); free ($3); free ($5); free ($7); free ($9);
               }
       | STD SUPV ASC VME_N arith_exp {
	              fprintf(fout,"    sysBusToLocalAdrs(VME_STD_SUP_ASC, %s, &address___);\n",$5);
                      free ($5);}     
       | STD SUPV PGM VME_N arith_exp {
                      fprintf(fout,"    sysBusToLocalAdrs(VME_STD_SUP_PGM, %s, &address___);\n",$5);
                      free ($5);}     
       | STD SUPV DATA_N VME_N arith_exp {
                      fprintf(fout,"    sysBusToLocalAdrs(VME_STD_SUP_DAT, %s, &address___);\n",$5);
                      free ($5);}
       | STD USER ASC VME_N arith_exp {
                      fprintf(fout,"    sysBusToLocalAdrs(VME_STD_USR_ASC, %s, &address___);\n",$5);
                      free ($5);}     
       | STD USER PGM VME_N arith_exp {
                      fprintf(fout,"    sysBusToLocalAdrs(VME_STD_USR_PGM, %s, &address___);\n",$5);
                      free ($5);}		
       | STD USER DATA_N VME_N arith_exp {
                      fprintf(fout,"    sysBusToLocalAdrs(VME_STD_USR_DAT, %s, &address___);\n",$5);
                      free ($5);}
       | SHR SUPV IO  VME_N arith_exp {
                      fprintf(fout,"    sysBusToLocalAdrs(VME_SHR_SUP_IO,  %s, &address___);\n",$5);
                      free ($5);}
       | SHR USER IO  VME_N arith_exp {
                      fprintf(fout,"    sysBusToLocalAdrs(VME_SHR_USR_IO,  %s, &address___);\n",$5);
                      free ($5);}
       | EXT SUPV ASC VME_N arith_exp {
                      fprintf(fout,"    sysBusToLocalAdrs(VME_EXT_SUP_ASC, %s, &address___);\n",$5);
                      free ($5);}
       | EXT SUPV PGM VME_N arith_exp {
                      fprintf(fout,"    sysBusToLocalAdrs(VME_EXT_SUP_PGM, %s, &address___);\n",$5);
                      free ($5);}
       | EXT SUPV DATA_N VME_N arith_exp {
                      fprintf(fout,"    sysBusToLocalAdrs(VME_EXT_SUP_DAT, %s, &address___);\n",$5);
                      free ($5);} 	
       | EXT USER ASC VME_N arith_exp {
                      fprintf(fout,"    sysBusToLocalAdrs(VME_EXT_USR_ASC, %s, &address___);\n",$5);
                      free ($5);}  	
       | EXT USER PGM VME_N arith_exp {
                      fprintf(fout,"    sysBusToLocalAdrs(VME_EXT_USR_PGM, %s, &address___);\n",$5);
                      free ($5);}	
       | EXT USER DATA_N VME_N arith_exp {
                      fprintf(fout,"    sysBusToLocalAdrs(VME_EXT_USR_DAT, %s, &address___);\n",$5);
                      free ($5);}	
       | VME_N arith_exp             {
                      fprintf(fout,"    sysBusToLocalAdrs(defaultAm, %s, &address___);\n",$2);
                      free ($2);}

read: READ ATTACHED ADDRESS {
               if (type == CAMAC_TY  ) { 
		 badContext ();
	       } 
	       else if (type == VME_TY) {
		 badContext ();
	       } 
	       else if (type == EVENT_TY) { 
		 fprintf(fout,"   *rol->dabufp++ = *rol->dabufpi++;\n");
	       } 
	       else if ((type == FASTBUS_TY)||(type == SFI_TY)) {
		 if ((newpad) && (newsad)) {
		   newpad = newsad = 0;
		   fprintf(fout,"    fb_fr%s_1(padr,sadr,rol->dabufp%s,1,0,0,0,1,1);\n",prefix,eg);
		 } 
		 else if (newpad) {
		   newpad = 0;
		   fprintf(fout,"    fb_fr%s_1(padr,0,rol->dabufp%s,1,0,1,0,1,1);\n",prefix,eg);
		 } 
		 else if (newsad) {
		   newsad = 0;
		   fprintf(fout,"    fb_fr%s_1(0,sadr,rol->dabufp%s,1,1,0,0,1,1);\n",prefix,eg);
		 } 
		 else {
		   fprintf(fout,"    fb_fr%s_1(0,0,rol->dabufp%s,1,1,1,0,1,1);\n",prefix,eg);
		 }
		 fprintf(fout,"    rol->dabufp += 1;\n");
		 if (type == FASTBUS_TY) {
		   fprintf(fout,"    if (fb_errno != 0x800BC119){ \n");
		   fprintf(fout,"        fb_rtn = \"read\" ;\n");
		   fprintf(fout,"        goto fooy;\n");
		   fprintf(fout,"    }\n");
		 }
	       } else {
		 badContext();
	       }
           }
      | READ FROM address_exp {
               if (type == CAMAC_TY  ) { 
		 fprintf(fout,"    cfsa(fcode, address___, rol->dabufp, &dumstat);\n");
		 fprintf(fout,"    rol->dabufp += 1;\n");
		 fprintf(fout,"\n");
	       }
               else if (type == VME_TY) {
		 fprintf(fout,"   *rol->dabufp  = *address___;\n");
		 fprintf(fout,"    rol->dabufp += 1;\n");
	       } 
	       else if (type == EVENT_TY) { 
		 fprintf(fout,"   *rol->dabufp++ = *rol->dabufpi++;\n");
	       } 
	       else if ((type == FASTBUS_TY)||(type == SFI_TY)) {
		 if ((newpad) && (newsad)) {
		   newpad = newsad = 0;
		   fprintf(fout,"    fb_fr%s_1(padr,sadr,rol->dabufp%s,1,0,0,0,0,0);\n",prefix,eg);
		 } 
		 else if (newpad) {
		   newpad = 0;
		   fprintf(fout,"    fb_fr%s_1(padr,0,rol->dabufp%s,1,0,1,0,0,0);\n",prefix,eg);
		 } 
		 else if (newsad) {
		   newsad = 0;
		   fprintf(fout,"    fb_fr%s_1(0,sadr,rol->dabufp%s,1,1,0,0,0,0);\n",prefix,eg);
		 } else {
		   fprintf(fout,"    fb_fr%s_1(0,0,rol->dabufp%s,1,1,1,0,0,0);\n",prefix,eg);
		 }
		 fprintf(fout,"    rol->dabufp += 1;\n");
		 if (type == FASTBUS_TY) {
		   fprintf(fout,"    if (fb_errno != 0x800BC119){ \n");
		   fprintf(fout,"        fb_rtn = \"read %s\" ;\n",$3);
		   fprintf(fout,"        goto fooy;\n");
		   fprintf(fout,"    }\n");
		 }
	       } 
	       else {
		 badContext();
	       }
               free ($3);
            }
      |  READ STATUS INTO_N VAR_NAME {
               if (type == CAMAC_TY  ) { 
		 fprintf(fout,"    ctstat(&%s); \n",$4);
	       } 
	       else {
		 badContext();
	       }
               free ($4);
             } 
      |  READ FROM address_exp INTO_N VAR_NAME  {
               isSymbolFound($5);
               if (type == CAMAC_TY  ) { 
		 fprintf(fout,"    cfsa(fcode, address___, &%s, &dumstat); \n",$5);
		 fprintf(fout,"\n");
	       } 
	       else if (type == VME_TY) {
		 fprintf(fout,"    %s = *address___;\n",$5);
	       } 
	       else if (type == EVENT_TY) { 
		 fprintf(fout,"    %s = *rol->dabufpi++;\n",$5);
	       } 
	       else if ((type == FASTBUS_TY)||(type == SFI_TY)) {
		 if ((newpad) && (newsad)) {
		   newpad = newsad = 0;
		   fprintf(fout,"    fb_fr%s_1(padr,sadr,&%s%s,1,0,0,0,0,0);\n",prefix,$5,eg);
		 } 
		 else  if (newpad) {
		   newpad = 0;
		   fprintf(fout,"    fb_fr%s_1(padr,0,&%s%s,1,0,1,0,0,0);\n",prefix,$5,eg);
		 } 
		 else if (newsad) {
		   newsad = 0;
		   fprintf(fout,"    fb_fr%s_1(0,sadr,&%s%s,1,1,0,0,0,0);\n",prefix,$5,eg);
		 } 
		 else {
		   fprintf(fout,"    fb_fr%s_1(0,0,&%s%s,1,1,1,0,0,0);\n",prefix,$5,eg);
		 }
		 if (type == FASTBUS_TY) {
		   fprintf(fout,"    if (fb_errno != 0x800BC119){ \n");
		   fprintf(fout,"        fb_rtn = \"read %s into %s\" ;\n",$3,$5);
		   fprintf(fout,"        goto fooy;\n");
		   fprintf(fout,"    }\n");
		 }
	       } else {
		 badContext();
	       }
               free ($3); free ($5);
             }
      |  READ INTO_N VAR_NAME  {
               if ((type == FASTBUS_TY)||(type == SFI_TY)) {
		 if ((newpad) && (newsad)) {
		   newpad = newsad = 0;
		   fprintf(fout,"    fb_fr%s_1(padr,sadr,&%s%s,1,0,0,0,1,1);\n",prefix,$3,eg);
		 } 
		 else if (newpad) {
		   newpad = 0;
		   fprintf(fout,"    fb_fr%s_1(padr,0,&%s%s,1,0,1,0,1,1);\n",prefix,$3,eg);
		 } 
		 else if (newsad) {
		   newsad = 0;
		   fprintf(fout,"    fb_fr%s_1(0,sadr,&%s%s,1,1,0,0,1,1);\n",prefix,$3,eg);
		 } 
		 else {
		   fprintf(fout,"    fb_fr%s_1(0,0,&%s%s,1,1,1,0,1,1);\n",prefix,$3,eg);
		 }
		 if (type == FASTBUS_TY) {
		   fprintf(fout,"    if (fb_errno != 0x800BC119){ \n");
		   fprintf(fout,"        fb_rtn = \"read into %s\" ;\n",$3);
		   fprintf(fout,"        goto fooy;\n");
		   fprintf(fout,"    }\n");
		 }
	       } else if (type == EVENT_TY) { 
		 fprintf(fout,"    %s = *rol->dabufpi++;\n",$3);
	       } else {
		 badContext();
	       }
               free ($3);
         }

write: WRITE arith_exp {
               if (type == CAMAC_TY  ) { 
		 fprintf(fout,"    {int dd__ = %s; \n",$2);
		 fprintf(fout,"      cfsa(fcode, address___, &dd__, &dumstat);\n");
		 fprintf(fout,"      rol->dabufp += 1;\n");
		 fprintf(fout,"    }\n");
		 fprintf(fout,"\n");
	       } 
	       else if ((type == FASTBUS_TY)||(type == SFI_TY)) {
		 if ((newpad) && (newsad)) {
		   newpad = newsad = 0;
		   fprintf(fout,"    fb_fw%s_1(padr,sadr,%s%s,1,0,0,0,1,1);\n",prefix,$2,eg);
		 } 
		 else if (newpad) {
		   newpad = 0;
		   fprintf(fout,"    fb_fw%s_1(padr,0,%s%s,1,0,1,0,1,1);\n",prefix,$2,eg);
		 } 
		 else if (newsad) {
		   newsad = 0;
		   fprintf(fout,"    fb_fw%s_1(0,sadr,%s%s,1,1,0,0,1,1);\n",prefix,$2,eg);
		 } 
		 else {
		   fprintf(fout,"    fb_fw%s_1(0,0,%s%s,1,1,1,0,1,1);\n",prefix,$2,eg);
		 }
		 if (type == FASTBUS_TY) {
		   fprintf(fout,"      if (fb_errno != 0x800BC119){ \n");
		   fprintf(fout,"          fb_rtn = \"write %s\" ;\n",$2);
		   fprintf(fout,"          goto fooy;\n");
		   fprintf(fout,"      }\n");
		 }
		 fprintf(fout," \n");
	       } else {
		 badContext();
	       }
               free ($2);
             }
      | WRITE  arith_exp INTO_N address_exp {
               if (type == CAMAC_TY  ) { 
		 fprintf(fout,"    {int dd__ = %s; \n",$2);
		 fprintf(fout,"      cfsa(fcode, address___, &dd__, &dumstat); \n");
		 fprintf(fout,"    }\n");
	       } 
	       else if (type == VME_TY) {
		 fprintf(fout,"    *address___, = %s;\n",$2);
	       } 
	       else if (type == EVENT_TY) { 
		 fprintf(fout,"    *rol->dabufpi++ = %s;\n",$2);
	       } 
	       else if ((type == FASTBUS_TY)||(type == SFI_TY)) {
		 if ((newpad) && (newsad)) {
		   newpad = newsad = 0;
		   fprintf(fout,"    fb_fw%s_1(padr,sadr,%s%s,1,0,0,0,0,0);\n",prefix,$2,eg);
		 } 
		 else if (newpad) {
		   newpad = 0;
		   fprintf(fout,"    fb_fw%s_1(padr,0,%s%s,1,0,1,0,0,0);\n",prefix,$2,eg);
		 } else if (newsad) {
		   newsad = 0;
		   fprintf(fout,"    fb_fw%s_1(0,sadr,%s%s,1,1,0,0,0,0);\n",prefix,$2,eg);
		 } 
		 else {
		   fprintf(fout,"    fb_fw%s_1(0,0,%s%s,1,1,1,0,0,0);\n",prefix,$2,eg);
		 }
		 if (type == FASTBUS_TY) {
		   fprintf(fout,"      if (fb_errno != 0x800BC119){ \n");
		   fprintf(fout,"          fb_rtn = \"write %s into %s\" ;\n",$2,$4);
		   fprintf(fout,"          goto fooy;\n");
		   fprintf(fout,"      }\n");
		 }
		 fprintf(fout," \n");
	       } 
	       else {
		 badContext();
	       }
               free ($2); free ($4);
          }


fastbus_arbitrate: ARBITRATE  {
               if (type == FASTBUS_TY) {
		 fprintf(fout,"    fb_frd_1(0,0,0,0,0,1,1,1,1,1);\n");
		 fprintf(fout,"    if (fb_errno != 0x800BC119){ \n");
		 fprintf(fout,"        fb_rtn = \"arbitrate\" ;\n");
		 fprintf(fout,"        goto fooy;\n");
		 fprintf(fout,"        }\n");
		 fprintf(fout," \n");
	       } 
	       else {
		 badContext();
	       }
             }
      | BROADCAST ARBITRATE  {
               if (type == FASTBUS_TY) {
		 fprintf(fout,"    fb_frdm_1(0,0,0,0,1,1,1,1,1);\n");
		 fprintf(fout,"    if (fb_errno != 0x800BC119){ \n");
		 fprintf(fout,"        fb_rtn = \"broadcast arbitrate\" ;\n");
		 fprintf(fout,"        goto fooy;\n");
		 fprintf(fout,"        }\n");
		 fprintf(fout," \n");
	       }
	       else {
		 badContext();
	       }
             }

fastbus_block_read: BLOCK READ {
               if (type == CAMAC_TY  ) { 
		 badContext ();
	       } 
	       else if (type == VME_TY) {
		 badContext ();
	       } 
	       else if (type == EVENT_TY) { 
		 fprintf(fout,"   *rol->dabufp++ = *rol->dabufpi++;\n");
	       } 
	       else if (type == SFI_TY) {
		 fprintf(fout,"   fpbr(-1,0);\n");
	       } 
	       else if (type == FASTBUS_TY) {
		 if (newpad) {
		   newpad = newsad = 0;
		   fprintf(fout,"    fb_fr%s_1(padr,0,0%s,1,0,1,1,1,1);\n",prefix,eg);
		   fprintf(fout,"    if (fb_errno != 0x800BC119){ \n");
		   fprintf(fout,"        fb_rtn = \"address slave\" ;\n");
		   fprintf(fout,"        goto fooy;\n");
		   fprintf(fout,"    }\n");
		   fprintf(fout,"    fpbr(-1,0); \n");
		 } else {
		   newpad = newsad = 0;
		   badContext ();
		 }
		 fprintf(fout,"    if (fb_errno != 0x800BC119){ \n");
		 fprintf(fout,"        fb_rtn = \"block read\" ;\n");
		 fprintf(fout,"        goto fooy;\n");
		 fprintf(fout,"    }\n");
	       } else {
		 badContext();
	       }
             }
      | BLOCK READ arith_exp {
               if (type == CAMAC_TY  ) { 
		 badContext ();
	       } 
	       else if (type == VME_TY) {
		 badContext ();
	       } 
	       else if (type == EVENT_TY) { 
		 fprintf(fout,"   *rol->dabufp++ = *rol->dabufpi++;\n");
	       } 
	       else if (type == SFI_TY) {
		 fprintf(fout,"   fpbr(-1,%s);\n",$3);
	       } 
	       else if (type == FASTBUS_TY) {
		 if (newpad) {
		   newpad = newsad = 0;
		   fprintf(fout,"    fb_fr%s_1(padr,0,0%s,1,0,1,1,1,1);\n",prefix,eg);
		   fprintf(fout,"    if (fb_errno != 0x800BC119){ \n");
		   fprintf(fout,"        fb_rtn = \"address slave\" ;\n");
		   fprintf(fout,"        goto fooy;\n");
		   fprintf(fout,"    }\n");
		   fprintf(fout,"    fpbr(-1,%s); \n",$3);
		 } else {
		   newpad = newsad = 0;
		   badContext ();
		 }
		 fprintf(fout,"    if (fb_errno != 0x800BC119){ \n");
		 fprintf(fout,"        fb_rtn = \"block read\" ;\n");
		 fprintf(fout,"        goto fooy;\n");
		 fprintf(fout,"    }\n");
	       } else {
		 badContext();
	       }
               free ($3);
             }
      | BLOCK READ arith_exp FROM address_exp {
               if (type == FASTBUS_TY) {
		 fprintf(fout,"    fpbr(padr,%s); \n",$3);
	       } 
	       else if (type == SFI_TY) {
		 fprintf(fout,"    fpbr(padr,%s); \n",$3);
	       } else {
		 badContext();
	       }
               free ($3);
             }
      | BLOCK READ FROM address_exp {
               if (type == FASTBUS_TY) {
		 fprintf(fout,"    fpbr(padr,0); \n");
	       } 
	       else if (type == SFI_TY) {
		 fprintf(fout,"    fpbr(padr,0); \n");
	       } else {
		 badContext();
	       }
             }
      | FAST BLOCK READ {
               if (type == CAMAC_TY  ) { 
		 badContext ();
	       } 
	       else if (type == VME_TY) {
		 badContext ();
	       } 
	       else if (type == EVENT_TY) { 
		 fprintf(fout,"   *rol->dabufp++ = *rol->dabufpi++;\n");
	       } 
	       else if (type == SFI_TY) {
		 fprintf(fout,"    fpbrf(-1,0); \n");
	       }
	       else if (type == FASTBUS_TY) {
		 if (newpad) {
		   newpad = newsad = 0;
		   fprintf(fout,"    fb_fr%s_1(padr,0,0%s,1,0,1,1,1,1);\n",prefix,eg);
		   fprintf(fout,"    if (fb_errno != 0x800BC119){ \n");
		   fprintf(fout,"        fb_rtn = \"address slave\" ;\n");
		   fprintf(fout,"        goto fooy;\n");
		   fprintf(fout,"    }\n");
		   fprintf(fout,"    fpbrf(-1,0); \n");
		 } else {
		   newpad = newsad = 0;
		   badContext ();
		 }
		 fprintf(fout,"    if (fb_errno != 0x800BC119){ \n");
		 fprintf(fout,"        fb_rtn = \"block read\" ;\n");
		 fprintf(fout,"        goto fooy;\n");
		 fprintf(fout,"    }\n");
	       } else {
		 badContext();
	       }
             }
      | FAST BLOCK READ arith_exp {
               if (type == CAMAC_TY  ) { 
		 badContext ();
	       } 
	       else if (type == VME_TY) {
		 badContext ();
	       } 
	       else if (type == EVENT_TY) { 
		 fprintf(fout,"   *rol->dabufp++ = *rol->dabufpi++;\n");
	       } 
	       else if (type == SFI_TY) {
		 fprintf(fout,"    fpbrf(-1,%s); \n",$4);
	       }
	       else if (type == FASTBUS_TY) {
		 if (newpad) {
		   newpad = newsad = 0;
		   fprintf(fout,"    fb_fr%s_1(padr,0,0%s,1,0,1,1,1,1);\n",prefix,eg);
		   fprintf(fout,"    if (fb_errno != 0x800BC119){ \n");
		   fprintf(fout,"        fb_rtn = \"address slave\" ;\n");
		   fprintf(fout,"        goto fooy;\n");
		   fprintf(fout,"    }\n");
		   fprintf(fout,"    fpbrf(-1,%s); \n",$4);
		 } else {
		   newpad = newsad = 0;
		   badContext ();
		 }
		 fprintf(fout,"    if (fb_errno != 0x800BC119){ \n");
		 fprintf(fout,"        fb_rtn = \"block read\" ;\n");
		 fprintf(fout,"        goto fooy;\n");
		 fprintf(fout,"    }\n");
	       } else {
		 badContext();
	       }
               free ($4);
             }
      | FAST BLOCK READ arith_exp FROM address_exp {
               if (type == FASTBUS_TY) {
		 fprintf(fout,"    fpbrf(padr,%s); \n",$4);
	       }
	       else if (type == SFI_TY) {
		 fprintf(fout,"    fpbrf(padr,%s); \n",$4);
	       }
	       else {
		 badContext();
	       }
               free ($4);
             }
      | FAST BLOCK READ FROM address_exp {
               if (type == FASTBUS_TY) {
		 fprintf(fout,"    fpbrf(padr,0); \n");
	       }
	       else if (type == SFI_TY) {
		 fprintf(fout,"    fpbrf(padr,0); \n");
	       }
	       else {
		 badContext();
	       }
             }


fastbus_release: RELEASE {
                if (type == FASTBUS_TY) {
		  fprintf(fout,"    fb_frd_1(0,0,0,1,1,1,1,1,0,0); \n");
		  fprintf(fout,"    if (fb_errno != 0x800BC119){ \n");
		  fprintf(fout,"        fb_rtn = \"release\" ;\n");
		  fprintf(fout,"        goto fooy;\n");
		  fprintf(fout,"        }\n");
		  fprintf(fout," \n");		 	
		} else if (type == SFI_TY) {
		  fprintf(fout,"    fprel(); \n");
		} else {
		  badContext();
		}
              }
      | BROADCAST RELEASE {
                  if (type == FASTBUS_TY) {
		    fprintf(fout,"    fb_frdm_1(0,0,0,1,1,1,1,0,0); \n");
		    fprintf(fout,"    if (fb_errno != 0x800BC119){ \n");
		    fprintf(fout,"        fb_rtn = \"broadcast release\" ;\n");
		    fprintf(fout,"        goto fooy;\n");
		    fprintf(fout,"        }\n");
		    fprintf(fout," \n");		 	
		  } else if (type == SFI_TY) {
		    fprintf(fout,"    fprel(); \n");
		  } else {
		    badContext();
		  }
               }

call_routine: CALL VAR_NAME {fprintf(fout,"    %s ( );\n", $2); free ($2);}

network_io: OPEN QSTRING {fprintf(fout,"    stream_fd__ = stream_io_open__(\"%s\");\n",$2);
                          free ($2);}
      | NCLOSE VAR_NAME  {fprintf(fout,"    stream_io_close__(%s);\n",$2);
                          free ($2);}
      | DECLARE QSTRING VAR_NAME {fprintf(fout,"    strcpy (temp_string__,rol->name);\n");
                                  fprintf(fout,"    strcat (temp_string__,\":\");\n");
                                  fprintf(fout,"    strcat (temp_string__,\"%s\");\n",$2);
                                  fprintf(fout,"    cm_declare(temp_string__,CM_ROLE_NONXWRITER,&%s,4,CM_VAR_LONG);\n",$3);
                                  free ($2); free ($3);}
      | DECLARE QSTRING ULONG_DEC VAR_NAME {
                                  fprintf(fout,"    strcpy (temp_string__,rol->name);\n");
                                  fprintf(fout,"    strcat (temp_string__,\":\");\n");
                                  fprintf(fout,"    strcat (temp_string__,\"%s\");\n",$2);
                                  fprintf(fout,"    cm_declare(temp_string__,CM_ROLE_NONXWRITER,&%s,4,CM_VAR_LONG);\n",$3);
                                  free ($2); free ($4);}
      | DECLARE QSTRING LONG_DEC VAR_NAME {
                                  fprintf(fout,"    strcpy (temp_string__,rol->name);\n");
                                  fprintf(fout,"    strcat (temp_string__,\":\");\n");
                                  fprintf(fout,"    strcat (temp_string__,\"%s\");\n",$2);
                                  fprintf(fout,"    cm_declare(temp_string__,CM_ROLE_NONXWRITER,&%s,4,CM_VAR_LONG);\n",$3);
                                  free ($2); free ($4);}
      | DECLARE QSTRING USHORT_DEC VAR_NAME {
                                  fprintf(fout,"    strcpy (temp_string__,rol->name);\n");
                                  fprintf(fout,"    strcat (temp_string__,\":\");\n");
                                  fprintf(fout,"    strcat (temp_string__,\"%s\");\n",$2);
                                  fprintf(fout,"    cm_declare(temp_string__,CM_ROLE_NONXWRITER,&%s,2,CM_VAR_SHORT);\n",$3);
                                  free ($2); free ($4);}
      | DECLARE QSTRING SHORT_DEC VAR_NAME {
                                  fprintf(fout,"    strcpy (temp_string__,rol->name);\n");
                                  fprintf(fout,"    strcat (temp_string__,\":\");\n");
                                  fprintf(fout,"    strcat (temp_string__,\"%s\");\n",$2);
                                  fprintf(fout,"    cm_declare(temp_string__,CM_ROLE_NONXWRITER,&%s,2,CM_VAR_SHORT);\n",$3);
                                  free ($2); free ($4);}

bank_io: OPEN EVENT {
                                  fprintf(fout,"    rol->dabufpi = (long *) &input_event__->data[1];\n");
                                  fprintf(fout,"    EVENT_LENGTH = *rol->dabufpi;\n");
                    }
      | OPEN EVENT OF VAR_NAME {
                                  fprintf(fout,"    CEOPEN(currType, %s);\n", $4);
                                  free ($4);
                    }
      | OPEN ETYPE arith_exp OF VAR_NAME {
                                  fprintf(fout,"    CEOPEN(%s, %s);\n", $3, $5);
                                  free ($3); free ($5);
                    }                                 
      | OPEN BANK arith_exp OF VAR_NAME {
                                  fprintf(fout,"    CBOPEN(%s, %s, 0);\n", $3, $5);
                                  free ($3); free ($5);
                    }
      | OPEN BANK arith_exp OF VAR_NAME CODE arith_exp {
                                  fprintf(fout,"    CBOPEN(%s, %s, %s);\n", $3, $5, $7);
                                  free ($3); free ($5); free ($7);
                    }
      | CLOSE BANK {
                                  fprintf(fout,"    CBCLOSE;\n");
                    }
      | CLOSE EVENT {
                                  fprintf(fout,"    CECLOSE;\n");
                    }
      | WRAPEV VAR_NAME  {
                                  fprintf(fout,"    {void %s();CRWRAPP(%s);}\n",$2,$2);
                                  free ($2);
                    }

trigger_source: TSINIT VAR_NAME { 
                                  fprintf(fout,"    %s_INIT;\n",$2);
                                  free ($2);
                    }
      | WTYPE arith_exp {
                                  fprintf(fout,"    CETWRITE(%s);\n",$2); 
                                  free ($2);
                    }
      | ETYPE arith_exp THEN READ VAR_NAME comma_list {
                                  fprintf(fout,"    CRTTYPE(%s,%s,%s);\n",$2,$5,$6); 
                                  free ($2); free ($5); free ($6);
                    }
      | TSLINKA VAR_NAME comma_list TO VAR_NAME AND VAR_NAME {
                                  fprintf(fout,"    CTRIGRSA(%s,%s,%s,%s);\n",$2,$3,$5,$7);
                                  free ($2); free ($3); free ($5); free ($7);
                    } 
      | TSLINKS VAR_NAME comma_list TO VAR_NAME AND VAR_NAME {
                                  fprintf(fout,"    CTRIGRSS(%s,%s,%s,%s);\n",$2,$3,$5,$7);
                                  free ($2); free ($3); free ($5); free ($7);
                     } 
      | ENABLE VAR_NAME comma_list{
                                  fprintf(fout,"    %s_ENA(%s);\n",$2,$3); 
                                  free ($2); free ($3);
                     }
      | DISABLE VAR_NAME comma_list {
                                  fprintf(fout,"    %s_DIS(%s);\n",$2,$3); 
                                  free ($2); free ($3);
                       }
      | START VAR_NAME comma_list {
                                  fprintf(fout,"    %s_START(%s);\n",$2,$3); 
                                  free ($2); free ($3);
                      }
      | STOP  VAR_NAME comma_list {
                                  fprintf(fout,"    %s_STOP(%s);\n",$2,$3); 
                                  free ($2); free ($3);
                      }
      | OUTPUT RANGE arith_exp arith_exp VAR_NAME {
                                  fprintf(fout,"    {int ix; for (ix=%s;ix<%s;ix++); CBWRITE32(%s[ix]);} \n",$3,$4,$5);
                                  free ($5); free ($3); free ($4);
                      }
      | OUTPUT SHR arith_exp {
                                  fprintf(fout,"    CBWRITE16(%s); \n",$3);
                                  free ($3);
                      }
      | OUTPUT arith_exp {
                                  fprintf(fout,"    CBWRITE32(%s); \n",$2);
                                  free ($2);
                      }
      | GET_EVENT {
                                  fprintf(fout,"    EVENT_GET; \n");
                      }
      | COPY_EVENT {
                                  fprintf(fout,"    COPYEVENT; \n");
                      }
      | PASS_EVENT {
                                  fprintf(fout,"    PASSEVENT; \n");
                      }

list_comms: CONNECT TO VAR_NAME {
           fprintf(fout,"    rol->output = partFindByName(\"%s\");\n",$3);
           free ($3);
       }


%%







