//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      RunControl Status Panel
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: rcRunStatusPanel.cc,v $
//   Revision 1.3  1997/08/21 19:28:46  heyes
//   fix borders and color problems
//
//   Revision 1.2  1997/07/11 14:07:53  heyes
//   almost working
//
//   Revision 1.1.1.1  1996/10/11 13:39:25  chen
//   run control source
//
//
#include <Xm/Frame.h>
#include <daqNetData.h>
#include <rcSvcProtocol.h>
#include "rcRunStatusPanel.h"
#include <cgFrameRec.h>
// number of vme crates in the graph
const int     RC_ST_NUM_CRATES = 3;
// number of lines
const int     RC_ST_NUM_LINES = 4;
// window bounday
const double  RC_ST_XMIN = 0.0;
const double  RC_ST_XMAX = 100.0;
const double  RC_ST_YMIN = 0.0;
const double  RC_ST_YMAX = 100.0;

rcRunStatusPanel::rcRunStatusPanel (Widget parent,
				    char* name,
				    rcClientHandler& handler,
				    Dimension width,
				    Dimension height)
: netHandler_ (handler), armed_ (0), 
  autoTimerId_ (0), counter_ (0), dataLogged_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("Create rcRunStatusPanel Class Object\n");
#endif
  Arg arg[20];
  int ac = 0;

  appContext_ = XtWidgetToApplicationContext (parent);
  
  // create display device
  disp_ = new cgDispDev (parent, name, width, height,
			 CG_SIMPLE_MODE);
  disp_->manage ();
  // the following is a must
  disp_->createColorCxt ();

  scene_ = new cgScene (RC_ST_XMIN, RC_ST_XMAX, RC_ST_YMIN, RC_ST_YMAX,
			0.1, 0.9, 0.0, 1.0, *disp_);

  disp_->registerScene (scene_);

  //cgFrameRec *framerec = new cgFrameRec (RC_ST_XMIN, RC_ST_YMIN, RC_ST_XMAX, RC_ST_YMAX, 2, CG_SHADOW_IN );
  //framerec->createCgCxt (*disp_);
  //framerec->attach (scene_);

  // create all components
  crates_ = new cgVmeCrate*[RC_ST_NUM_CRATES];
  
  double dh = (RC_ST_YMAX - RC_ST_YMIN)/10.0;
  for (int i = 0; i < RC_ST_NUM_CRATES; i++) {
    crates_[i] = new cgVmeCrate (RC_ST_XMIN + 1, 
				 RC_ST_YMIN - dh/2.0 + 2.5*(i+1)*dh, dh, 10);
    crates_[i]->createCgCxt (*disp_);
    crates_[i]->setCrateColor ("white");
    crates_[i]->setCrateFgColor ("black");
    crates_[i]->attach (scene_);
  }

  // create computer
  double compht = (RC_ST_YMAX - RC_ST_YMIN)/6.0;
  double midvpt = RC_ST_YMIN + (RC_ST_YMAX - RC_ST_YMIN)/2.0;
  double midhpt = RC_ST_XMIN + (RC_ST_XMAX - RC_ST_XMIN)/2.0;
  computer_ = new cgComputer (midhpt - compht, 
			      midvpt - compht/2.0, compht);
  computer_->createCgCxt (*disp_);
  computer_->setComputerColor ("white");
  computer_->setComputerFgColor ("black");
  computer_->attach (scene_);

  // create a tape drive
  drive_ = new cgTapeDrive (RC_ST_XMAX - compht*2 - 5.0,
 		    midhpt - compht/2.0, compht);
  drive_->createCgCxt (*disp_);
  drive_->setTapeDriveColor ("white");
  drive_->setTapeDriveFgColor ("black");

  // create a trash can
  can_ = new cgTrashCan (RC_ST_XMAX - compht*2 - 5.0,
			 midhpt - compht/2.0, compht);
  can_->createCgCxt (*disp_);
  can_->setCanColor ("white");
  can_->setCanFgColor ("black");
  can_->attach (scene_);

  // create all lines
  lines_ = new cgLine*[RC_ST_NUM_LINES];
  
  lines_[0] = new cgLine (RC_ST_XMIN + 1 + 2*dh, RC_ST_YMIN + dh*2.5,
			  midhpt - compht, midvpt);
  lines_[0]->createCgCxt (*disp_);
  cgContext* lcxt = lines_[0]->cgCxt ();
  lcxt->setLineWidth (2);
  lcxt->getForeground (lineFg_);
  lcxt->getBackground (bg_);

  lines_[1] = new cgLine (RC_ST_XMIN + 1 + 2*dh, midvpt,
			  midhpt - compht, midvpt);
  lines_[1]->setCgCxt (*lcxt);
  
  lines_[2] = new cgLine (RC_ST_XMIN + 1 + 2*dh, RC_ST_YMIN + 7.5*dh,
			  midhpt - compht, midvpt);
  lines_[2]->setCgCxt (*lcxt);
  
  lines_[3] = new cgLine (midhpt, midvpt,
			  RC_ST_XMAX - compht*2 - 5.0, midvpt);
  lines_[3]->setCgCxt (*lcxt);

  for (i = 0; i < RC_ST_NUM_LINES; i++)
    lines_[i]->attach (scene_);
}

rcRunStatusPanel::~rcRunStatusPanel (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete rcRunStatusPanel Class Object\n");
#endif
  // cgDispDev will be automatically destroyed by Xt Widget
  // destroy callback

  // cgDisplay also will delete every single scene, so no need to delete
  // scene_  

  // depending whether tape drive or trash can is mapped, delete
  // the other one
  if (dataLogged_)
    delete can_;
  else
    delete drive_;

  delete []lines_;
  delete []crates_;
}

const Widget
rcRunStatusPanel::baseWidget (void)
{
  return disp_->baseWidget ();
}

void
rcRunStatusPanel::config (int status)
{
  switch (status) {
  case DA_DORMANT:
  case DA_BOOTING:
  case DA_BOOTED:
  case DA_CONFIGURED:
  case DA_CONFIGURING:
  case DA_DOWNLOADING:
  case DA_TERMINATING:
    endDataTaking ();
    disconnectComponents ();
    break;
  case DA_DOWNLOADED:
  case DA_PRESTARTING:
    endDataTaking ();
    connectComponents ();
    break;
  case DA_PAUSED:
  case DA_PRESTARTED:
  case DA_ACTIVATING:
    endDataTaking ();
    readyToGo ();
    break;
  case DA_ACTIVE:
    startDataTaking ();
    break;
  case DA_ENDING:
  case DA_PAUSING:
    endDataTaking ();
    break;
  default:
    break;
  }
}

void
rcRunStatusPanel::anaLogChanged (daqNetData* info, int added)
{
  int oldlog = dataLogged_;
  if (netHandler_.anaLogToRealFiles ())
    dataLogged_ = 1;
  else
    dataLogged_ = 0;

  if (oldlog != dataLogged_) {
    if (dataLogged_) {
      can_->erase (scene_);
      can_->detach (scene_);
      drive_->draw (scene_);
      drive_->attach (scene_);
    }
    else {
      drive_->erase (scene_);
      drive_->detach (scene_);
      can_->draw (scene_);
      can_->attach (scene_);
    }
  }
}

void
rcRunStatusPanel::disconnectComponents (void)
{
  cgContext* lcxt = lines_[0]->cgCxt ();
  // set line color to green
  lcxt->setForeground (bg_);

  for (int i = 0; i < RC_ST_NUM_LINES; i++)
    lines_[i]->draw (scene_);
}

void
rcRunStatusPanel::connectComponents (void)
{
  int i = 0;

  cgContext* lcxt = lines_[0]->cgCxt ();
  // set line color to green
  lcxt->setForeground ("green");

  for (i = 0; i < RC_ST_NUM_LINES; i++)
    lines_[i]->draw (scene_);  
}

void
rcRunStatusPanel::readyToGo (void)
{
  cgContext* lcxt = lines_[0]->cgCxt ();
  // set line color to yellow
  lcxt->setForeground ("yellow");

  for (int i = 0; i < RC_ST_NUM_LINES; i++)
    lines_[i]->draw (scene_);  

}

void
rcRunStatusPanel::startDataTaking (void)
{
  if (!armed_) {
    if (!dataLogged_) 
      can_->open (scene_);

    armed_ = 1;
    counter_ = 0;
    autoTimerId_ = XtAppAddTimeOut (appContext_, 500,
	    (XtTimerCallbackProc)&(rcRunStatusPanel::timerCallback),
	    (XtPointer)this);
  }
}

void
rcRunStatusPanel::endDataTaking (void)
{
  if (armed_) {
    XtRemoveTimeOut (autoTimerId_);
    armed_ = 0;
    autoTimerId_ = 0;


    cgContext* lcxt = lines_[0]->cgCxt ();
    // set line color to original color
    lcxt->setForeground (lineFg_);    

    for (int i = 0; i < RC_ST_NUM_LINES; i++)
      lines_[i]->draw (scene_);  

    if (!dataLogged_) 
      can_->close (scene_);
  }
}

void
rcRunStatusPanel::timerCallback (XtPointer clientData,
				 XtIntervalId * )
{
  rcRunStatusPanel* obj = (rcRunStatusPanel *)clientData;
  cgContext* lcxt = obj->lines_[0]->cgCxt ();

  // if data is really logged into tape or file, show tape
  if (obj->dataLogged_)
    obj->drive_->rotateTape (obj->scene_);

  if (obj->counter_ == 0) {
    // set first 3 line color to red
    lcxt->setForeground ("red");

    for (int i = 0; i < RC_ST_NUM_LINES - 1; i++)
      obj->lines_[i]->draw (obj->scene_);
    lcxt->setForeground (obj->lineFg_);
    obj->lines_[i]->draw (obj->scene_);
    obj->counter_ += 1;
  }
  else {
    for (int i = 0; i < RC_ST_NUM_LINES - 1; i++)
      obj->lines_[i]->draw (obj->scene_);
    lcxt->setForeground ("red");
    obj->lines_[i]->draw (obj->scene_);
    obj->counter_ = 0;
  }
  obj->autoTimerId_ = XtAppAddTimeOut (obj->appContext_,
				       500,
				       &(rcRunStatusPanel::timerCallback),
				       (XtPointer)obj);
}
