#include <msql.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#ifdef VXWORKS
#  include <time.h>
#  include <sys/times.h>
#  include <taskLib.h>
   extern int gettimeofday();
#else
#  include <sys/types.h>
#  include <unistd.h>
#  include <sys/time.h>
#  include <string.h>
#  include <netinet/in.h>
#  include <netdb.h>
#endif

#ifdef VXWORKS
#  define PRINT logMsg
#else
#  define PRINT printf
#endif


/******************************************************************
 *
 * eMsgLog
 * 
 * PURPOSE:
 *	This routine enters an error message into an msql table.
 *	The index and time information is generated by this routine.
 *	Normally, error messages are generated from "official" coda
 *	processes through tcl/dp channels. This routine provides a
 *	C interface for any other processes the user deems necessary.
 *
 * ARGUMENTS:
 *	msqlhost - machine on which the msql server resides,
 *		   (defaults to the local host if NULL)
 *	db       - msql database name
 *		   (defaults to EXPID env variable if NULL)
 *	tbl	 - msql table name in which to store error msg
 *	proc	 - name of process that is generating the error
 *		   (eg. ROC10, EB45, EBANA3)
 *	sev	 - error severity (ERROR,WARN,INFO,DEBUG,START,STOP,STATUS)
 *	text	 - error message (limited to 135 chars)
 *	code	 - error id number
 *	daemon	 - id number of the msql daemon being used
 *
 *	(The rest of the arguments, below, are currently ignored.
 *	 They are included for future implementation.)
 *
 *	host	 - the machine hosting the error-generating process,
 *		   (used for vxworks where it's the host machine)
 *	facility - used for anything
 *
 * RETURNS:
 *	 0 if OK
 *
 *	-1 if cannot get the time
 *	-2 if the table name is null 
 *	-3 if severity is improper value 
 *	-4 if cannot connnect to msql server 
 *	-5 if the database name is not defined 
 *	-6 if cannot find the database 
 *	-7 if cannot select the database 
 *	-8 if cannot find the table 
 *	-9 if cannot insert message into the database 
 *	-10 if id number of daemon < 1 or > 10 
 *	-11 if bad port number for msql daemon 
 *
 *******************************************************************/

int
eMsgLog(char *msqlhost, char *db, char *tbl,
	char *proc, char *sev, char *text,
	char *host, char *facility, int code, int daemon)

{
    char *dbase, *mhost, *envVar;
    char  query[400];
    char  time[9];
    char  severity[11];
    char  msg[135];
    char  envarHost[30], envarPort[30], service[7];
    int   connId,
	  match,
	  tcpPort=-1,
	  pid;
    m_result *result;
    m_row row;
    struct timeval	tv;
    struct servent	*serv_ptr;
    
    /* Find the time */
    if (gettimeofday(&tv, (void *) 0) == -1) {
#ifdef DEBUG
	PRINT("eMsgLog: Cannot get time\n");
#endif
 	return -1;
    }

    /* reject null input */
    if (tbl == NULL) {
#ifdef DEBUG
	PRINT("eMsgLog: table argument CANNOT be null\n");
#endif
	return -2;
    }
    
    /* Find the pid */
#ifdef VXWORKS
    pid = taskIdSelf();
#else
    pid = getpid();
#endif

    /* check severity input */
/*
    if ((sev != "ERROR") && (sev != "error") &&
	(sev != "START") && (sev != "STOP")  &&
	(sev != "WARN")  && (sev != "INFO")  &&
	(sev != "DEBUG") && (sev != "STATUS"))   {
#ifdef DEBUG
	PRINT("eMsgLog: Severity must be either ERROR, error, WARN, START, STOP, INFO, DEBUG, or STATUS\n");
#endif
	return -3;
    }
*/
    if (strlen(sev) > 10) {
	strncpy(severity,sev,10);
	severity[10] = '\0';
    } else {
	strcpy(severity, sev);
    }
    
    /* depending on which daemon is being talked to, use different env vars */
    if (daemon == 1) {
	sprintf(envarHost, "MSQL_TCP_HOST");
    } else if ((daemon > 1) && (daemon < 11)) {
	sprintf(envarHost, "MSQL%d_TCP_HOST", daemon);
	sprintf(envarPort, "MSQL%d_TCP_PORT", daemon);
	sprintf(service, "msql%d", daemon);
    } else {
#ifdef DEBUG
	PRINT("eMsgLog: Improper id specified for msql daemon\n");
#endif
	return -10;
    }
    
    /* if host = NULL, connection is made to MSQLX_TCP_HOST. If that's NULL, */
    /* it tries to connect to a server on the local host */
    if (msqlhost == NULL)
	mhost = getenv(envarHost);
    else
	mhost = msqlhost;

    if (daemon == 1) {
	if ((connId = msqlConnect(mhost)) == -1) {
#ifdef DEBUG
	    PRINT("eMsgLog: Cannot connect to host\n");
#endif
	    return -4;
	}
    } else {
	 /* Check port numbers */
	if ((envVar = getenv(envarPort)) == NULL) {
#ifndef VXWORKS
	    if (serv_ptr = getservbyname(service, "tcp"))
		 tcpPort = ntohs(serv_ptr->s_port);
#endif
	} else {
	    tcpPort = atoi(envVar);
	}
	
	if (tcpPort < 0) {
#ifdef DEBUG
	    PRINT("eMsgLog: Cannot connect to host, bad port number\n");
#endif
	    return -11;
	}
		
	if ((connId = msqlConnectPort(mhost,tcpPort)) == -1) {
#ifdef DEBUG
	    PRINT("eMsgLog: Cannot connect to host\n");
#endif
	    return -4;
	}
    }
	    
    
    /* The env var EXPID should contain database name */
    if (db == NULL) {
	if (dbase = getenv("EXPID")) {
#ifdef DEBUG
	    PRINT("eMsgLog: Your environmental variable , EXPID, needs definition\n");
#endif
	    msqlClose(connId);
	    return -5;
	}
    } else {
	dbase = db;
    }

    /* check list of databases against database name */
    match  = 0;
    result = msqlListDBs(connId);
    if (result != NULL) {
	while (row = msqlFetchRow(result)) {
	    if (strcmp(row[0],dbase) == 0) {
		match = 1;
		break;
	    }
	}
    }

    if (!match) {
#ifdef DEBUG
	PRINT("eMsgLog: Cannot find the database \"%s\"\n",dbase);
#endif
	msqlFreeResult(result);
	msqlClose(connId);
 	return -6;
    }
    msqlFreeResult(result);
    
    /* select db for queries */
    if (msqlSelectDB(connId, dbase) == -1) {
#ifdef DEBUG
	PRINT("eMsgLog: Can't select database\"%s\"", dbase);
	if (msqlErrMsg != NULL)
	    PRINT(":  %s\n", msqlErrMsg);
	else
	    PRINT("\n");
#endif
	msqlClose(connId);
	return -7;
    }
    
    /* check list of tables against table name */
    match  = 0;
    result = msqlListTables(connId);
    if (result != NULL) {
	while (row = msqlFetchRow(result)) {
	    if (strcmp(row[0],tbl) == 0) {
		match = 1;
		break;
	    }
	}
    }

    if (!match) {
#ifdef DEBUG
	PRINT("eMsgLog: Cannot find the table \"%s\"\n",tbl);
#endif
	msqlFreeResult(result);
	msqlClose(connId);
 	return -8;
    }
    msqlFreeResult(result);
    
    /* Put stuff into table */
    strftime (time, 9,"%H:%M:%S", localtime((time_t *) &tv.tv_sec));
    if (strlen(text) > 134) {
	strncpy(msg,text,134);
	msg[134] = '\0';
	sprintf(query, "INSERT INTO %s (index,time,process,severity,text,code,pid) VALUES (%ld.%06ld, '%s','%s','%s','%s',%d,%d)",
	    tbl, tv.tv_sec, tv.tv_usec, time, proc, severity, msg, code, pid);
    } else {
	sprintf(query, "INSERT INTO %s (index,time,process,severity,text,code,pid) VALUES (%ld.%06ld, '%s','%s','%s','%s',%d,%d)",
	    tbl, tv.tv_sec, tv.tv_usec, time, proc, severity, text, code, pid);
    }
	
    if (msqlQuery(connId, query) == -1) {
#ifdef DEBUG
	PRINT("eMsgLog: Can't insert message into database");
	if (msqlErrMsg != NULL)
	    PRINT(":  %s\n", msqlErrMsg);
	else
	    PRINT("\n");
#endif
	msqlClose(connId);
	return -9;
    }

    msqlClose(connId);
    return 0;
}



/* FORTRAN callable version */
#ifndef VXWORKS

char *strFtoC(char *fortranStr, int len)
{
    char *s;
    int len_occ;

    /* find occupied length, not including trailing blanks */
    for(len_occ=len; (len_occ>0)&&(fortranStr[len_occ-1]==' '); len_occ--)
	;
    
    s = (char *)malloc(len_occ+1);
    strncpy(s,fortranStr,(size_t)len_occ);
    s[len_occ]='\0';
  
    return s;
}

int *
eMsgLog_(char *msqlF, char *dbF,  char *tblF,
	 char *procF, char *sevF, char *textF,
	 char *hostF, char *facF, int *codeF, int *daemonF,
	 int len1, int len2, int len3, int len4,
	 int len5, int len6, int len7, int len8)

{
    static int	ret;
    int		i;
    char	*s[8];
    
    /* change Fortran strings to C strings (generally longer) */
    s[0]=strFtoC(msqlF, len1);
    s[1]=strFtoC(dbF,   len2);
    s[2]=strFtoC(tblF,  len3);
    s[3]=strFtoC(procF, len4);
    s[4]=strFtoC(sevF,  len5);
    s[5]=strFtoC(textF, len6);
    s[6]=strFtoC(hostF, len7);
    s[7]=strFtoC(facF,  len8);

    ret = eMsgLog(s[0], s[1], s[2], s[3], s[4], s[5], s[6], s[7], *codeF, *daemonF);
    
    for(i=0; i<8; i++)
	free(s[i]);
    
    return &ret;
}

#endif /* !VXWORKS */
