/*
 * ------------------------------------------------------------------------
 *      PACKAGE:  [incr Tk]
 *  DESCRIPTION:  Building mega-widgets with [incr Tcl]
 *
 *  [incr Tk] provides a framework for building composite "mega-widgets"
 *  using [incr Tcl] classes.  It defines a set of base classes that are
 *  specialized to create all other widgets.
 *
 *  This file defines the initialization and facilities common to all
 *  mega-widgets.
 *
 * ========================================================================
 *  AUTHOR:  Michael J. McLennan       Phone: (610)712-2842
 *           AT&T Bell Laboratories   E-mail: michael.mclennan@att.com
 *     RCS:  $Id: itk_cmds.c,v 1.1.1.1 1996/08/21 19:25:43 heyes Exp $
 * ========================================================================
 *             Copyright (c) 1993-1995  AT&T Bell Laboratories
 * ------------------------------------------------------------------------
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */
#include <assert.h>
#include "itkInt.h"

#ifndef lint
static char rcsid[] = "$Id: itk_cmds.c,v 1.1.1.1 1996/08/21 19:25:43 heyes Exp $";
#endif


/*
 * ------------------------------------------------------------------------
 *  Itk_Init()
 *
 *  Should be invoked whenever a new interpeter is created to add
 *  [incr Tk] facilities.  Creates an "::itk" namespace and adds
 *  commands needed by this package.
 * ------------------------------------------------------------------------
 */
int
Itk_Init(interp)
    Tcl_Interp *interp;  /* interpreter to be updated */
{
    static char initCmd[] =
    "if [file exists ${itk::library}/init.itk] {\n\
        source ${itk::library}/init.itk\n\
    } else {\n\
        set msg \"can't find ${itk::library}/init.itk\\n\"\n\
        append msg \"Perhaps you need to install \\[incr Tk\\] \\n\"\n\
        append msg \"or set your ITK_LIBRARY environment variable?\"\n\
        error $msg\n\
    }";

    Itcl_Namespace itkNs, parserNs;
    ClientData parserInfo;
    char *libDir;
    int pLevel;

    /*
     *  Install [incr Tk] facilities if not already installed.
     */
    if (Itcl_FindNamesp(interp, "::itk", 0, &itkNs) == TCL_OK &&
        itkNs != NULL) {
        Tcl_SetResult(interp, "already installed: [incr Tk]", TCL_STATIC);
        return TCL_ERROR;
    }

    /*
     *  Add the "itk_option" ensemble to the itcl class definition parser.
     */
    if (Itcl_FindNamesp(interp, "::itcl::parser", 0, &parserNs) != TCL_OK ||
        parserNs == NULL) {
        Tcl_AppendResult(interp, "cannot initialize [incr Tk]: ",
            "[incr Tcl] has not been installed\n",
            "Make sure that Itcl_Init() is called before Itk_Init()",
            (char*)NULL);
        return TCL_ERROR;
    }
    parserInfo = Itcl_GetNamespData(parserNs);

    if (Itcl_CreateEnsemble(interp, "::itcl::parser::itk_option") != TCL_OK) {
        return TCL_ERROR;
    }
    if (Itcl_AddEnsembleOption(interp, "::itcl::parser::itk_option", "define",
            4, 5, "-switch resourceName resourceClass init ?config?",
            Itk_ClassOptionDefineCmd,
            Itcl_PreserveData(parserInfo), Itcl_ReleaseData)
            != TCL_OK ||
        Itcl_AddEnsembleOption(interp, "::itcl::parser::itk_option", "add",
            1, ITCL_VAR_ARGS, "name ?name name...?",
            Itk_ClassOptionIllegalCmd,
            (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL)
            != TCL_OK ||
        Itcl_AddEnsembleOption(interp, "::itcl::parser::itk_option", "remove",
            1, ITCL_VAR_ARGS, "name ?name name...?",
            Itk_ClassOptionIllegalCmd,
            (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL)
            != TCL_OK) {
        return TCL_ERROR;
    }

    /*
     *  Add the "bind" command to the itcl class definition parser.
     */
    Tcl_CreateCommand(interp, "::itcl::parser::bind", Tk_BindCmd,
        (ClientData)Tk_MainWindow(interp), (Tcl_CmdDeleteProc*)NULL);

    /*
     *  Create the "itk" namespace.
     */
    if (Itcl_CreateNamesp(interp, "::itk", (ClientData)NULL,
        (Itcl_DeleteProc*)NULL, &itkNs) != TCL_OK) {
        Tcl_AppendResult(interp, " (cannot initialize ::itk namespace)",
            (char*)NULL);
        return TCL_ERROR;
    }

    /*
     *  Setup things for itk::Archetype base class.
     */
    if (Itk_ArchetypeInit(interp) != TCL_OK) {
        return TCL_ERROR;
    }

    /*
     *  Fix the "itcl::configbody" command to recognize mega-widget
     *  options.
     */
    Tcl_CreateCommand(interp, "::itcl::configbody", Itk_ConfigBodyCmd,
        (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);

    /*
     *  Set up the library and load the "init.itk" file.
     */
    pLevel = Itcl_VarProtection(interp, ITCL_PUBLIC);

    libDir = Tcl_GetVar2(interp, "env", "ITK_LIBRARY", TCL_GLOBAL_ONLY);
    if (libDir == NULL) {
        libDir = ITK_LIBRARY;
    }
    Tcl_SetVar(interp, "itk::library", libDir, ITCL_GLOBAL_VAR);

    Itcl_VarProtection(interp, pLevel);

    return Tcl_Eval(interp, initCmd);
}


/*
 * ------------------------------------------------------------------------
 *  Itk_ConfigBodyCmd()
 *
 *  Replacement for the usual "itcl::configbody" command.  Recognizes
 *  mega-widget options included in a class definition.  Options are
 *  identified by their "switch" name, but without the "-" prefix:
 *
 *    itcl::configbody <class>::<itkOption> <body>
 *
 *  Handles bodies for public variables as well:
 *
 *    itcl::configbody <class>::<publicVar> <body>
 *
 *  If an <itkOption> is found, it has priority over public variables.
 *  If <body> has the form "@name" then it is treated as a reference
 *  to a C handling procedure; otherwise, it is taken as a body of
 *  Tcl statements.
 *
 *  Returns TCL_OK/TCL_ERROR to indicate success/failure.
 * ------------------------------------------------------------------------
 */
/* ARGSUSED */
int
Itk_ConfigBodyCmd(dummy, interp, argc, argv)
    ClientData dummy;        /* unused */
    Tcl_Interp *interp;      /* current interpreter */
    int argc;                /* number of arguments */
    char **argv;             /* argument strings */
{
    char *head, *tail;
    Itcl_Namespace ns;
    Itcl_Class *cdefn;
    ItclCmdImplement *mimpl;
    ItkClassOptTable *optTable;
    Tcl_HashEntry *entry;
    ItkClassOption *opt;
    Tcl_DString buffer;

    if (argc != 3) {
        Tcl_AppendResult(interp, "wrong # args: should be \"",
            argv[0], " class::option body\"",
            (char*)NULL);
        return TCL_ERROR;
    }

    /*
     *  Parse the member name "namesp::namesp::class::option".
     *  Make sure that a class name was specified, and that the
     *  class exists.
     */
    Itcl_ParseNamespPath(argv[1], &head, &tail);

    if (!head || *head == '\0') {
        Tcl_AppendResult(interp, "missing class specifier ",
            "for body declaration \"", argv[1], "\"",
            (char*)NULL);
        return TCL_ERROR;
    }
    if (Itcl_FindClass(interp, head, &ns) != TCL_OK) {
        return TCL_ERROR;
    }
    if (ns == NULL) {
        Tcl_AppendResult(interp, "class \"",
            head, "\" not found in context \"",
            Itcl_GetNamespPath(Itcl_GetActiveNamesp(interp)), "\"",
            (char*)NULL);
        return TCL_ERROR;
    }
    cdefn = (Itcl_Class*)Itcl_GetNamespData(ns);

    /*
     *  Look first for a configuration option with that name.
     *  If it is not found, assume the reference is for a public
     *  variable, and use the usual "configbody" implementation
     *  to handle it.
     */
    optTable = Itk_FindClassOptTable(cdefn);
    opt = NULL;

    if (optTable) {
        Tcl_DStringInit(&buffer);
        Tcl_DStringAppend(&buffer, "-", -1);
        Tcl_DStringAppend(&buffer, tail, -1);
        entry = Tcl_FindHashEntry(&optTable->options,
            Tcl_DStringValue(&buffer));

        if (entry) {
            opt = (ItkClassOption*)Tcl_GetHashValue(entry);
        }
        Tcl_DStringFree(&buffer);
    }

    if (opt == NULL) {
        return Itcl_ConfigBodyCmd(dummy, interp, argc, argv);
    }

    /*
     *  Otherwise, change the implementation for this option.
     */
    if (Itcl_CreateCmdImplement(interp,(char*)NULL,argv[2],&mimpl)
        != TCL_OK) {
        return TCL_ERROR;
    }
    Itcl_PreserveData((ClientData)mimpl);
    Itcl_EventuallyFree((ClientData)mimpl, Itcl_DeleteCmdImplement);

    if (opt->config) {
        Itcl_ReleaseData((ClientData)opt->config);
    }
    opt->config = mimpl;

    return TCL_OK;
}
