/*-----------------------------------------------------------------------------
 * Copyright (c) 1991,1992 Southeastern Universities Research Association,
 *                         Continuous Electron Beam Accelerator Facility
 *
 * This software was developed under a United States Government license
 * described in the NOTICE file included as part of this distribution.
 *
 * CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
 * Email: coda@cebaf.gov  Tel: (804) 249-7101  Fax: (804) 249-7363
 *-----------------------------------------------------------------------------
 * SCCS release:
 * @(#) fb_fscc_macro.h 1.20 Delta: 92/10/02 09:48:57 Extraction: 92/10/05 17:22:42 @(#)
 * 
 * Description:
 *	Fastbus Standard Routines Library, level 1 macros
 *	
 * Author:  Chip Watson, CEBAF Data Acquisition Group
 *
 * Revision History:
 *   $Log: fb_fscc_macro.h,v $
 *   Revision 1.1.1.1  1996/08/21 19:20:38  heyes
 *   Imported sources
 *
*	  Revision 1.1  94/03/15  12:01:36  12:01:36  heyes (Graham Heyes)
*	  Initial revision
*	  
*	  Revision 1.3  93/12/01  12:14:20  12:14:20  abbottd (David Abbott)
*	  Fixed bug in fb_frlen_1 for ACTION_FIFO and Replaced Multiplication with shifts
*	  
*	  Revision 1.2  93/11/22  08:53:25  08:53:25  abbottd (David Abbott)
*	  Fixed bug with FB_BUILD_NOCHECK option. fb_errno set normal in FB_FINISH_ACTION routine."
*	  
*	  Revision 1.1  93/10/27  09:26:24  09:26:24  heyes (Graham Heyes)
*	  Initial revision
*	  
 *	  Revision 1.4  93/07/26  10:20:26  10:20:26  heyes (Graham Heyes)
 *	  fix problem with nleft (not initialised properly).
 *	  
 *	  Revision 1.3  93/05/10  14:03:25  14:03:25  watson (Chip Watson)
 *	  change "temp" to "fb_fscc_temp"
 *	  
 *	  Revision 1.2  1993/04/28  16:41:13  heyes
 *	  correct bus error handling
 *
 *	  Revision 1.1  1993/04/14  20:42:31  watson
 *	  Initial revision
 *
 *	  Revision 1.1  1993/03/16  13:59:16  heyes
 *	  Initial revision
 *
 *	  Revision 1.6  1992/09/03  18:33:22  watson
 *	  add secondary address read macro
 *
 *	  Revision 1.5  1992/08/03  13:07:34  watson
 *	  replace ... w/ ....., === w/ =====
 *
 *	  Revision 1.4  1992/07/17  19:21:23  watson
 *	  added type cast to dp
 *
 *	  Revision 1.3  1992/07/17  14:59:22  watson
 *	  added NOP, fixed write block length calculation
 *
 *	  Revision 1.2  1992/07/16  14:57:34  watson
 *	  added "noda" to block xfer; added write block macro
 *
 *	  Revision 1.1  1992/07/15  19:55:03  watson
 *	  Initial revision
 *
 *
 *	This file contains macro versions of many of the level 1 routines.
 *	Because each macro serves many addressing modes, they are invoked
 *	with an additional leading argument indicating the addressing mode.
 *
 *  Compile Options:
 *  ----------------
 *	FB_BUILD_NOIPL    assume code is being complied to run only at an
 *			  elevated interrupt priority level (service routine);
 *			  ISR invokes FB_ISR_ENTER and FB_ISR_EXIT
 *	FB_BUILD_NEWSEQ   new sequencer code (CEBAF version)
 *	FB_BUILD_NOCHECK  remove all error checking
 *
 *  Public Macros:
 *  -------------
 *	FSCC_RD_SINGLE()	read single word from fastbus
 *	FSCC_WR_SINGLE()	write single word to fastbus
 *	FSCC_RD_BLOCK()		block read from fastbus
 *	FSCC_WR_BLOCK()		write block to fastbus
 *	FSCC_FRLEN		get length of last seq buffer transfer
 *
 *  Private Macros:
 *  --------------
 *	FSCC_ADDRESS()		arb,pa,sa for action routines 
 *	FSCC_RELEASE()		drop AS, GK for action routines 
 *	FSCC_START_ACTION()	everything at the beginning of action
 *	FSCC_ABORT_ACTION()	for error exit from macro
 *	FSCC_FINISH_ACTION()	everything at the end of an action
 */

#ifndef INCfbfsccmacro
#define INCfbfsccmacro

#include "fb_fscc_def.h"
#include "fb_error.h"

extern int fb_errno;

#ifndef FB_DATA_PATH
#define FB_DATA_PATH		-1
#endif

#ifndef FB_DATA_PATH_ADDR
/************************************************************************/
/* The FSCC uses this as a flag to determine Data FIFO, or Data Buffer.	*/
/* If the data argument is "FB_DATA_PATH_ADDR" then the Data FIFO path	*/
/* is used, otherwise; the address of the data argument is filled with	*/
/* data from a FASTBUS read operation.					*/
#define FB_DATA_PATH_ADDR	FB_DATA_PATH
#endif

/*************** FSCC Layer 1 Support Macro's *******************************/

void fscc_error_decode(int noarb,int nopa,int nosa,
		       int noda,int holdas,int hold);

#define CAT(a,b) a##b		/* defer concatenation until after each */
#define XCAT(a,b) CAT(a,b)	/*   argument is expanded */
#define RETURN goto XCAT(return,__LINE__) /* pseudo return from macro */


#define FSCC_START_ACTION \
=====if FB_BUILD_NEWSEQ==0\
=====if FB_BUILD_NOCHECK==0\
.....  fscc_error = 0;		/* clear flag for isr */\
=====endif\
=====endif\
.....

#define FSCC_ABORT_ACTION(holdas,hold) \
/* Termination: can either drop GK and AS, drop AS, or drop none */ \
.....  if ((holdas)==0) { \
.....    FPAREL();		/* release AS */ \
.....    if ((hold)==0) \
.....      FPREL(); 		/* release bus */ \
.....  }\
.....

#define FSCC_FINISH_ACTION(noarb,nopa,nosa,noda,holdas,hold,nowait) \
.....  if ((nowait)==0) { \
/* Termination: can either drop GK and AS, drop AS, or drop none */ \
.....    if ((holdas)==0) { \
.....      FPAREL();		/* release AS */\
.....      if ((hold)==0) \
.....        FPREL();\
.....    }\
=====if FB_BUILD_NOCHECK==0\
.....    FPNOP;			/* synchronize with sequencer */\
.....    FPNOP;			/* synchronize with sequencer */\
.....    FPNOP;			/* synchronize with sequencer */\
=====endif\
=====if FB_BUILD_NEWSEQ==0\
=====if FB_BUILD_NOCHECK==0\
.....    if (fscc_error==0) {\
.....      fb_errno = FB_ERR_NORMAL;\
.....      RETURN;\
.....    } else {\
.....      fscc_error_decode(noarb,nopa,nosa,noda,holdas,hold);\
.....    }\
=====endif\
=====endif\
.....  }

/****************************  FSCC_ADDRESS  *********************************
 *
 * This macro handles arbitration, primary, and secondary addressing for all
 * macros that follow.
 *
 * 1. The first argument selects the underlying address mode:
 *	AD (data)  AC (control)  ADM (data multiple)  ACM (control multiple)
 * 2. error checking may be disabled to improve speed w/ old sequencer code
 *
 *****************************************************************************/

#define FSCC_ADDRESS(am,pa,sa,eg,noarb,nopa,nosa,holdas,hold)\
.....  if ((noarb)==0) {\
.....    FPARB(fscc_csr[8]);	/* synchronous: may hang till bus timeout */\
.....  }\
.....  if ((nopa)==0) {	/* primary address cycle? */\
.....    if ((eg)!=0) {\
.....      FPG##am (pa);		/* geographical address */\
.....    } else {\
.....      FP##am (pa);		/* logical address */\
.....    }\
.....  }\
=====if FB_BUILD_NEWSEQ==0\
=====if FB_BUILD_NOCHECK==0\
.....  FPNOP;			/* synchronize with sequencer */\
.....  FSCC_SEQS_RESET;		/* reset sequencer status fifo */\
=====endif\
=====endif\
.....  if ((nosa)==0) {	/* secondary address cycle? */\
.....    FPSAW(sa);\
.....  }

/****************************  FB_RD_SINGLE  *******************************
 *
 * This macro handles all flavors of single word reads from FASTBUS.
 *
 * 1. The first argument selects the underlying address mode:
 *	AD (data)  AC (control)  ADM (data multiple)  ACM (control multiple)
 * 2. error checking may be disabled to improve speed w/ old sequencer code
 * 3. data must be the address at which to place the data (int *)
 *    OR the value FB_DATA_PATH_ADDR (data is moved through the data FIFO)
 *
 *****************************************************************************/

#define FB_RD_SINGLE(am,pa,sa,data,eg,noarb,nopa,nosa,noda,holdas,hold)\
.....{\
.....  FSCC_START_ACTION;		/* disable interrupts, clear errors */\
.....  FSCC_ADDRESS(am,pa,sa,eg,noarb,nopa,nosa,holdas,hold);\
.....  if ((noda)==0) {	/* data cycle? */\
.....    register int fb_fscc_temp;\
.....    FPR(fb_fscc_temp);		/* get data into temp register */\
.....    if ((int)(data)==FB_DATA_PATH_ADDR) {/* to data fifo? */\
.....      FPWFF(fb_fscc_temp);		/* write to fifo */\
.....    } else {			/* not FB_DATA_PATH */\
.....	 *(int *)(data)=fb_fscc_temp;\
.....    }\
.....  } \
.....  FSCC_FINISH_ACTION(noarb,nopa,nosa,noda,holdas,hold,0); \
.....XCAT(return,__LINE__): 	/* put label here for RETURN statements */\
.....;\
.....}



/*************************  FB_WR_SINGLE  **********************************
 *
 * This macro handles all flavors of single word writes to FASTBUS.
 *
 * 1. The first argument selects the underlying address mode:
 *	AD (data)  AC (control)  ADM (data multiple)  ACM (control multiple)
 * 2. error checking may be disabled to improve speed w/ old sequencer code
 * 3. data contains the data value to write
 *
 *****************************************************************************/

#define FB_WR_SINGLE(am,pa,sa,data,eg,noarb,nopa,nosa,noda,holdas,hold)\
.....{\
.....  FSCC_START_ACTION;		/* disable interrupts, clear errors */\
.....  FSCC_ADDRESS(am,pa,sa,eg,noarb,nopa,nosa,holdas,hold);\
.....  if ((noda)==0) {	/* data cycle? */\
.....    FPW(data);\
.....  }\
.....  FSCC_FINISH_ACTION(noarb,nopa,nosa,noda,holdas,hold,0);\
.....XCAT(return,__LINE__): 	/* put label here for RETURN statements */\
.....;\
.....}



/**************************** FB_RD_BLOCK ***********************************
 *
 * This macro handles all flavors of block reads from FASTBUS
 *
 * 1. The first argument selects the underlying address mode:
 *	AD (data)  AC (control)  ADM (data multiple)  ACM (control multiple)
 * 2. error checking may be disabled to improve speed w/ old sequencer code
 * 3. data is the adress at which to place the data, OR
 *    the value FB_DATA_PATH_ADDR (data is moved through the data FIFO)
 * 4. maxbytes sets upper limit on transfer if non-zero
 * 5. pipe is only used for transfers to the data fifo, and takes the following
 *    values:  0 = not pipe, <100 = 100ns, <200 = 200ns, else 400ns
 *
 *****************************************************************************/

#define FB_RD_BLOCK(am,pa,sa,data,maxbytes,eg,noarb,nopa,nosa,noda,pipe,holdas,hold,nowait)\
.....{\
.....  FSCC_START_ACTION;		/* disable interrupts, clear errors */\
.....  FSCC_ADDRESS(am,pa,sa,eg,noarb,nopa,nosa,holdas,hold);\
.....  if ((noda)==0) {	/* data cycle? */\
.....    if ((int)(data)==FB_DATA_PATH_ADDR) {	/* to data fifo? */\
.....      if (maxbytes!=0) {	/* if 0, don't reset counter */\
.....        FPWLC(maxbytes/4);	/* write local counter */\
.....        FSCC_LCEN;		/* local counter enable */\
.....      } else { \
.....        FSCC_LCDIS;		/* local counter disable */\
.....      }\
.....      fscc_frlen_info = HCR_COUNT; /* get global word cnt for frlen_1 */\
.....      if (pipe==0)\
.....        FPRBF;		      /* start block transfer to data fifo */\
.....      else if (pipe<=100)\
.....        FPRP100;		/* 100ns pipeline transfer to data fifo */\
.....      else if (pipe<=200)\
.....        FPRP200;		/* 200ns pipeline transfer to data fifo */\
.....      else\
.....        FPRP400;		/* 400ns pipeline transfer to data fifo */\
.....      fscc_seq_buf_action = ACTION_FIFO_BLOCK;\
.....    } else {			/* not FB_DATA_PATH */\
.....      register int *dp,nleft=0;\
.....      FSCC_FCLERR;		/* reset SS latches to watch for SS2 */\
.....      dp = (int *) (data);\
.....      if (maxbytes!=0) {\
.....        for(nleft=maxbytes/4;nleft>0;nleft--) {\
.....	     FPRB(*dp++);\
.....	     if (FSCC_SS!=FSCC_M_SS0) break;\
..... 	     if (fscc_error!=0) break;\
.....        }\
.....      } else {\
.....        for(;;) {\
.....	     FPRB(*dp++);\
.....	     if (FSCC_SS!=FSCC_M_SS0) break;\
..... 	     if (fscc_error!=0) break;\
.....        }\
.....      }\
.....      FPBTT;			/* terminate block transfer */\
.....      fscc_seq_buf_action = ACTION_READ_BLOCK;\
.....      fscc_frlen_info = (maxbytes/4) - nleft;\
.....    }			/* FB_DATA_FIFO */\
.....  }				/* (noda) */\
.....  FSCC_FINISH_ACTION(noarb,nopa,nosa,noda,holdas,hold,nowait);\
.....XCAT(return,__LINE__): 	/* put label here for RETURN statements */\
.....;\
.....}

/**************************** FB_WR_BLOCK ***********************************
 *
 * This macro handles all flavors of block writes to FASTBUS
 *
 * 1. The first argument selects the underlying address mode:
 *	AD (data)  AC (control)  ADM (data multiple)  ACM (control multiple)
 * 2. error checking may be disabled to improve speed w/ old sequencer code
 * 3. data is the adress of an array containing the data
 * 4. maxbytes is the number of bytes to transfer
 * 5. pipe is ignored 
 *
 *****************************************************************************/


#define FB_WR_BLOCK(am,pa,sa,data,maxbytes,eg,noarb,nopa,nosa,noda,pipe,holdas,hold,nowait)\
.....{\
.....  FSCC_START_ACTION;		/* disable interrupts, clear errors */\
.....  FSCC_ADDRESS(am,pa,sa,eg,noarb,nopa,nosa,holdas,hold);\
.....  if ((noda)==0) {	/* data cycle? */\
.....    register int *dp,nleft=0;\
.....    FSCC_FCLERR;		/* reset SS latches to watch for SS2 */\
.....    dp = (int *) (data);\
.....    for(nleft=maxbytes/4;nleft>0;nleft--) {\
.....	 FPWB(*dp++);\
.....	 if (FSCC_SS!=FSCC_M_SS0) break;\
..... 	 if (fscc_error!=0) break;\
.....    }\
.....    FPBTT;			/* terminate block transfer */\
.....    fscc_seq_buf_action = ACTION_WRITE_BLOCK;\
.....    fscc_frlen_info = maxbytes - 4*nleft;\
.....  }				/* (noda) */\
.....  FSCC_FINISH_ACTION(noarb,nopa,nosa,noda,holdas,hold,nowait);\
.....XCAT(return,__LINE__): 	/* put label here for RETURN statements */\
.....;\
.....}


/****************************  fb_frlen_1  ************************************
 *
 *  return length of last sequential buffer transfer
 *
 *****************************************************************************/

#define FB_FRLEN_1(data) \
.....{\
.....  fb_errno = FB_ERR_NORMAL;\
.....  switch(fscc_seq_buf_action) {\
.....  case ACTION_READ_BLOCK:\
.....    *(data) = (fscc_frlen_info<<2);\
.....    break;\
.....  case ACTION_WRITE_BLOCK:\
.....    *(data) = fscc_frlen_info;\
.....    break;\
.....  case ACTION_FIFO_BLOCK:\
.....    FPNOP;\
.....    *(data) = ((HCR_COUNT - fscc_frlen_info)<<2);\
.....    if (*(data)<0) *(data) += 0x4000; /* overflowed 12 bit counter */\
.....    break;\
.....  default:\
.....    fb_errno = FB_ERR_ERROR;\
.....    *(data) = 0;\
.....  }\
.....}

/****************************  FB_RD_SA  **********************************
 *
 * 1. The first argument selects the underlying address mode:
 *	AD (data)  AC (control)
 * 2. error checking may be disabled to improve speed w/ old sequencer code
 * 3. sa must be the address at which to place the data (int *)
 *    OR the value FB_DATA_PATH_ADDR (data is moved through the data FIFO)
 *
 *****************************************************************************/

#define FB_RD_SA(am,pa,sa,eg,noarb,nopa,nosa,holdas,hold)\
.....{\
.....  FSCC_START_ACTION;		/* disable interrupts, clear errors */\
.....  FSCC_ADDRESS(am,pa,0,eg,noarb,nopa,1,holdas,hold); /* no sa */\
.....  if ((nosa)==0) {	/* secondary address cycle? */\
.....    register int fb_fscc_temp;\
.....    FPSAR(fb_fscc_temp);		/* get data into temp register */\
.....    if ((int)(sa)==FB_DATA_PATH_ADDR) {/* to data fifo? */\
.....      FPWFF(fb_fscc_temp);		/* write to fifo */\
.....    } else {			/* not FB_DATA_PATH */\
.....	 *(int *)(sa)=fb_fscc_temp;\
.....    }\
.....  } \
.....  FSCC_FINISH_ACTION(noarb,nopa,nosa,1,holdas,hold,0); \
.....XCAT(return,__LINE__): 	/* put label here for RETURN statements */\
.....;\
.....}


/****************************  fb_fbprst_1  ***********************************
 *
 *  reset FSCC FASTBUS port
 *
 *****************************************************************************/

#define FB_FBPRST_1 \
.....  { \
.....  fb_errno = FB_ERR_NORMAL; \
.....  FSCC_SRESET;                  /* reset sequencer & errors */ \
.....  FSCC_SEQS_RESET;              /* reset sequencer status */ \
.....  FPNOP;                        /* check sequencer w/ null instruction */ \
.....  FSCC_WAIT; \
.....  if (FSCC_STATUS!=12) {        /* does sequencer respond OK? */ \
.....    fb_errno = FB_ERR_HARDWARE_ERROR; \
.....    } \
.....  if (FSCC_STATUS!=0) {         /* is sequencer now idle? */ \
.....    fb_errno = FB_ERR_HARDWARE_ERROR; \
.....    } \
.....  }


#endif  /* ifndef INCfbfsccmacro */









