/*-----------------------------------------------------------------------------
 * Copyright (c) 1991,1992 Southeastern Universities Research Association,
 *                         Continuous Electron Beam Accelerator Facility
 *
 * This software was developed under a United States Government license
 * described in the NOTICE file included as part of this distribution.
 *
 * CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
 * Email: coda@cebaf.gov  Tel: (804) 249-7101  Fax: (804) 249-7363
 *-----------------------------------------------------------------------------
 * 
 * Description:
 *	Xcefdmp value mask dialog and related routines
 *	
 * Author:  Jie Chen, CEBAF Data Acquisition Group
 *
 * Revision History:
 *   $Log: xcef_value_mask.c,v $
 *   Revision 1.2  1997/11/25 15:33:10  rwm
 *   Zapped compiler warnings.
 *
 *   Revision 1.1.1.1  1996/09/19 18:26:24  chen
 *   original port to solaris
 *
 *	  Revision 1.2  1995/02/28  21:09:32  chen
 *	  update mask value upon changing shift bit
 *
 *        Revision 1.1  1994/12/08  19:58:40  chen
 *        Initial revision
 *
 */
#include <stdio.h>
#include <string.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/DialogS.h>
#include <Xm/ArrowB.h>
#include <Xm/PushB.h>
#include <Xm/Separator.h>
#include <Xm/Label.h>
#include <Xm/Frame.h>
#include <Xm/TextF.h>
#include <Xm/ToggleB.h>

#include "ValueMask.h"

#include "xcef_layout.h"

static Widget createShiftWindow();
static Widget createModeWindow();
static Widget createMaskWindow();
static void   toggle_callback();
static void   arrow_callback();
static void   bit_select_callback();
static void   ok_new_mask();
static void   apply_new_mask();
static void   unset_mask();
static void   cancel_new_mask();
static void   destroy_callback();

char   *applyMaskToIntBuffer();

typedef struct _widget_list{
  Widget shell;
  Widget text_w;
  Widget value_mask_w;
  Widget right_shift_w;
  Widget and_mode_w;
  Widget or_mode_w;
  DIALOG *value_dialog;
}MaskWidgets;

/*****************************************************************
 *          void pop_value_mask( )                               *
 * Description:                                                  *
 *     Popup dialog box containing value mask                    *
 ****************************************************************/
void pop_value_mask(parent, parent_shell, cbs)
Widget parent;
DIALOG *parent_shell;
XmAnyCallbackStruct *cbs;
{
  MaskWidgets *widget_list;
  Widget      shell_form, mask_w,action_form;
  Widget      pushb0, pushb1, pushb2, pushb3;
  Widget      sep;
  Widget      top_label;
  Widget      sub_form1, sub_form2, sub_form3;
  Arg         arg[20];
  int         ac = 0;
  XmString    str;
  char        temp[256];
  Pixel       bg;
  
  widget_list = (MaskWidgets *)malloc(sizeof(MaskWidgets));
  if(widget_list == NULL){
    fprintf(stderr,"Cannot allocate memory for widget list.\n");
    exit(1);
  }
  widget_list->value_dialog = parent_shell;
  if(parent_shell->mask_shell != NULL){
    XtMapWidget (parent_shell->mask_shell);
    XtPopup (parent_shell->mask_shell, XtGrabNone);
    return;
  }
  XtSetArg (arg[ac], XmNtitle, "ValueMask Dialog"); ac++;
  XtSetArg (arg[ac], XmNresizePolicy, XmRESIZE_NONE); ac++;
  XtSetArg (arg[ac], XmNnoResize, True); ac++;
  XtSetArg (arg[ac], XmNautoUnmanage, False); ac++;
  shell_form = XmCreateFormDialog(parent, "value_mask_form", arg, ac);
  ac = 0;
  widget_list->shell = XtParent(shell_form);
  XtSetArg (arg[ac], XmNbackground, &bg); ac++;
  XtGetValues (shell_form, arg, ac); 
  ac = 0;
  XtSetArg (arg[ac], XmNbackground, bg); ac++;
  XtSetValues (widget_list->shell, arg, ac);
  ac = 0;
  
  strcpy(temp,"Integer value mask will apply to all events\n");
  strcat(temp,"with integer data type for this node only !");
  str = XmStringCreateLtoR(temp, XmSTRING_DEFAULT_CHARSET);
  
  XtSetArg (arg[ac], XmNlabelString, str); ac++;
  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNtopOffset, 10); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftOffset, 10); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNrightOffset, 10); ac++;
  top_label = XtCreateManagedWidget ("value_top_label", xmLabelWidgetClass,
				     shell_form, arg, ac);
  ac = 0;
  XmStringFree (str);

  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
  XtSetArg (arg[ac], XmNtopWidget, top_label); ac++;
  XtSetArg (arg[ac], XmNtopOffset, 10); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftOffset, 2); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNrightOffset, 2); ac++;
  sep = XtCreateManagedWidget ("sep", xmSeparatorWidgetClass,
			       shell_form, arg, ac);
  ac = 0;

  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
  XtSetArg (arg[ac], XmNtopWidget, sep); ac++;
  XtSetArg (arg[ac], XmNtopOffset, 10); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftOffset, 10); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNrightOffset, 15); ac++;
  XtSetArg (arg[ac], XmNborderWidth, 0); ac++;
  mask_w = XtCreateManagedWidget("mask_w", xcodaValueMaskWidgetClass,
				 shell_form, arg, ac);
  ac = 0;
  widget_list->value_mask_w = mask_w;
  
  sub_form1 = createMaskWindow (shell_form, mask_w, widget_list);

  sub_form2 = createShiftWindow (shell_form, sub_form1, widget_list);

  sub_form3 = createModeWindow (shell_form, sub_form2, widget_list);

  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
  XtSetArg (arg[ac], XmNtopWidget, sub_form3); ac++;
  XtSetArg (arg[ac], XmNtopOffset, 10); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftOffset, 2); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNrightOffset, 2); ac++;
  sep = XtCreateManagedWidget ("sep", xmSeparatorWidgetClass,
			       shell_form, arg, ac);
  ac = 0;  

  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
  XtSetArg (arg[ac], XmNtopWidget, sep); ac++;
  XtSetArg (arg[ac], XmNtopOffset, 10); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftOffset, 2); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNrightOffset, 2); ac++;
  XtSetArg (arg[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNbottomOffset, 10); ac++;
  action_form = XtCreateWidget ("mask_action_form", xmFormWidgetClass,
				shell_form, arg, ac);
  ac = 0;

  str = XmStringCreateSimple (" Ok ");
  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_POSITION); ac++;
  XtSetArg (arg[ac], XmNleftPosition, 3); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_POSITION); ac++;
  XtSetArg (arg[ac], XmNrightPosition, 22); ac++;
  XtSetArg (arg[ac], XmNlabelString, str); ac++;
  pushb0 = XtCreateManagedWidget("mask_ok", xmPushButtonWidgetClass,
				 action_form, arg, ac);
  ac = 0;
  XmStringFree (str);

  str = XmStringCreateSimple ("Apply");
  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_POSITION); ac++;
  XtSetArg (arg[ac], XmNleftPosition, 28); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_POSITION); ac++;
  XtSetArg (arg[ac], XmNrightPosition, 47); ac++;
  XtSetArg (arg[ac], XmNlabelString, str); ac++;
  pushb1 = XtCreateManagedWidget("mask_apply", xmPushButtonWidgetClass,
				 action_form, arg, ac);
  ac = 0;
  XmStringFree (str);

  str = XmStringCreateSimple ("Cancel");
  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_POSITION); ac++;
  XtSetArg (arg[ac], XmNleftPosition, 53); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_POSITION); ac++;
  XtSetArg (arg[ac], XmNrightPosition, 72); ac++;
  XtSetArg (arg[ac], XmNlabelString, str); ac++;
  pushb2 = XtCreateManagedWidget("mask_cancel", xmPushButtonWidgetClass,
				 action_form, arg, ac);
  ac = 0;
  XmStringFree (str);

  str = XmStringCreateSimple ("Unmask");
  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_POSITION); ac++;
  XtSetArg (arg[ac], XmNleftPosition, 78); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_POSITION); ac++;
  XtSetArg (arg[ac], XmNrightPosition, 97); ac++;
  XtSetArg (arg[ac], XmNlabelString, str); ac++;
  pushb3 = XtCreateManagedWidget("mask_unmask", xmPushButtonWidgetClass,
				 action_form, arg, ac);
  ac = 0;
  XmStringFree (str);

  XtAddCallback(pushb0, XmNactivateCallback, ok_new_mask, widget_list);
  XtAddCallback(pushb1, XmNactivateCallback, apply_new_mask, widget_list);
  XtAddCallback(pushb2, XmNactivateCallback, cancel_new_mask, widget_list);
  XtAddCallback(pushb3, XmNactivateCallback, unset_mask, widget_list);

  XtAddCallback(mask_w, XcodaNbitSelectCallback, bit_select_callback, 
		widget_list);
  XtAddCallback(shell_form, XmNdestroyCallback, destroy_callback,
		widget_list);

  XtManageChild (sub_form1);
  XtManageChild (sub_form2);
  XtManageChild (sub_form3);
  XtManageChild (action_form);
  XtManageChild (shell_form);
  
  widget_list->value_dialog->mask_shell = widget_list->shell;
  XtPopup (widget_list->shell, XtGrabNone);
}
  
static Widget createMaskWindow(form, top_widget, list)
Widget form, top_widget;
MaskWidgets *list;
{
  Widget sub_form, label, text_w;
  Arg    arg[20];
  int    ac = 0;
  XmString str;

  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
  XtSetArg (arg[ac], XmNtopWidget, top_widget); ac++;
  XtSetArg (arg[ac], XmNtopOffset, 10); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftOffset, 2); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNrightOffset, 2); ac++;
  sub_form = XtCreateWidget("mask_text_subform", xmFormWidgetClass,
			    form, arg, ac);
  ac = 0;

  str = XmStringCreateSimple ("Value Mask:");
  XtSetArg (arg[ac], XmNlabelString, str); ac++;
  XtSetArg (arg[ac], XmNalignment, XmALIGNMENT_END); ac++;
  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_POSITION); ac++;
  XtSetArg (arg[ac], XmNrightPosition, 30); ac++;
  label = XtCreateManagedWidget("value_label", xmLabelWidgetClass,
				sub_form, arg, ac);
  ac = 0;
  XmStringFree (str);

  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNrightOffset, 20); ac++;
  XtSetArg (arg[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_POSITION); ac++;
  XtSetArg (arg[ac], XmNleftPosition, 42); ac++;  
  XtSetArg (arg[ac], XmNvalue,"0xffffffff"); ac++;
  XtSetArg (arg[ac], XmNeditable, False); ac++;
  list->text_w = XtCreateManagedWidget("valuemask_text_w", xmTextFieldWidgetClass,
				       sub_form, arg, ac);
  ac = 0;

  return sub_form;
}

static Widget createShiftWindow(form, top_widget, list)
Widget form, top_widget;
MaskWidgets *list;
{
  Widget sub_form, label, right_arrow, left_arrow, frame;
  Arg    arg[20];
  int    ac = 0;
  XmString str;

  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
  XtSetArg (arg[ac], XmNtopWidget, top_widget); ac++;
  XtSetArg (arg[ac], XmNtopOffset, 10); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftOffset, 2); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNrightOffset, 2); ac++;
  sub_form = XtCreateWidget("mask_shift_subform", xmFormWidgetClass,
			    form, arg, ac);
  ac = 0;

  str = XmStringCreateSimple ("Shift Right:");
  XtSetArg (arg[ac], XmNlabelString, str); ac++;
  XtSetArg (arg[ac], XmNalignment, XmALIGNMENT_END); ac++;
  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_POSITION); ac++;
  XtSetArg (arg[ac], XmNrightPosition, 30); ac++;
  label = XtCreateManagedWidget("value_label", xmLabelWidgetClass,
				sub_form, arg, ac);
  ac = 0;
  XmStringFree (str);  
  
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_POSITION); ac++;
  XtSetArg (arg[ac], XmNleftPosition, 42); ac++;
  XtSetArg (arg[ac], XmNarrowDirection, XmARROW_RIGHT); ac++;
  XtSetArg (arg[ac], XmNuserData, 0); ac++;
  right_arrow = XtCreateManagedWidget("right_arrow", xmArrowButtonWidgetClass,
				      sub_form, arg, ac);
  ac = 0;

  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNrightOffset, 20); ac++;
  XtSetArg (arg[ac], XmNarrowDirection, XmARROW_LEFT); ac++;
  XtSetArg (arg[ac], XmNuserData, 1); ac++;
  left_arrow = XtCreateManagedWidget("left_arrow", xmArrowButtonWidgetClass,
				      sub_form, arg, ac);
  ac = 0;

  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_WIDGET); ac++;
  XtSetArg (arg[ac], XmNleftWidget, right_arrow); ac++;
  XtSetArg (arg[ac], XmNleftOffset, 20); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_WIDGET); ac++;
  XtSetArg (arg[ac], XmNrightWidget, left_arrow); ac++;
  XtSetArg (arg[ac], XmNrightOffset, 20); ac++;
  XtSetArg (arg[ac], XmNshadowType, XmSHADOW_IN); ac++;
  frame = XtCreateWidget ("shift_frame", xmFrameWidgetClass,
			  sub_form, arg, ac);
  ac = 0;

  str = XmStringCreateSimple("0");
  XtSetArg (arg[ac], XmNlabelString, str); ac++;
  XtSetArg (arg[ac], XmNuserData, 0); ac++;
  list->right_shift_w = XtCreateManagedWidget("right_shift_bit", 
					      xmLabelWidgetClass,
					      frame, arg, ac);
  ac = 0;
  XmStringFree (str);
  
  XtAddCallback(right_arrow, XmNactivateCallback, arrow_callback, list);
  XtAddCallback(left_arrow, XmNactivateCallback, arrow_callback, list);

  XtManageChild (frame);

  return sub_form;
}

static Widget createModeWindow(form, top_widget, list)
Widget form, top_widget;
MaskWidgets *list;
{
  Widget sub_form, label;
  Arg    arg[20];
  int    ac = 0;
  XmString str;

  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_WIDGET); ac++;
  XtSetArg (arg[ac], XmNtopWidget, top_widget); ac++;
  XtSetArg (arg[ac], XmNtopOffset, 10); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftOffset, 2); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNrightOffset, 2); ac++;
  sub_form = XtCreateWidget("mode_subform", xmFormWidgetClass,
			    form, arg, ac);
  ac = 0;

  str = XmStringCreateSimple ("Mask Mode:");
  XtSetArg (arg[ac], XmNlabelString, str); ac++;
  XtSetArg (arg[ac], XmNalignment, XmALIGNMENT_END); ac++;
  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_POSITION); ac++;
  XtSetArg (arg[ac], XmNrightPosition, 30); ac++;
  label = XtCreateManagedWidget("value_label", xmLabelWidgetClass,
				sub_form, arg, ac);
  ac = 0;
  XmStringFree (str);  

  str = XmStringCreateSimple ("And");
  XtSetArg (arg[ac], XmNindicatorType, XmONE_OF_MANY); ac++;
  XtSetArg (arg[ac], XmNlabelString, str); ac++;
  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNleftAttachment, XmATTACH_POSITION); ac++;
  XtSetArg (arg[ac], XmNleftPosition, 42); ac++;
  XtSetArg (arg[ac], XmNset, True); ac++;
  list->and_mode_w = XtCreateManagedWidget("and_window", xmToggleButtonWidgetClass,
					   sub_form, arg, ac);
  ac = 0;
  XmStringFree (str);

  str = XmStringCreateSimple ("Or");
  XtSetArg (arg[ac], XmNindicatorType, XmONE_OF_MANY); ac++;
  XtSetArg (arg[ac], XmNlabelString, str); ac++;
  XtSetArg (arg[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNbottomAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
  XtSetArg (arg[ac], XmNrightOffset, 20); ac++;
  XtSetArg (arg[ac], XmNset, False); ac++;
  list->or_mode_w = XtCreateManagedWidget("and_window", xmToggleButtonWidgetClass,
					  sub_form, arg, ac);
  ac = 0;
  XmStringFree (str);

  XtAddCallback(list->and_mode_w, XmNvalueChangedCallback, toggle_callback,
		list);
  XtAddCallback(list->or_mode_w, XmNvalueChangedCallback, toggle_callback,
		list);

  return sub_form;
}

/*******************************************************************
 *        static void toggle_callback()                            *
 * Description:                                                    *
 *       Mode toggle button callback routine                       *
 ******************************************************************/
static void toggle_callback(w, list, cbs)
Widget w;
MaskWidgets *list;
XmToggleButtonCallbackStruct *cbs;
{
  if(w == list->and_mode_w){ /*and mode button */
    if(cbs->set)
      XmToggleButtonSetState(list->or_mode_w, False, False);
    else
      XmToggleButtonSetState(list->or_mode_w, True, False);
  }
  else{
    if(cbs->set)
      XmToggleButtonSetState(list->and_mode_w, False, False);
    else
      XmToggleButtonSetState(list->and_mode_w, True, False);
  }
}


/******************************************************************
 *         static void arrow_callback()                           *
 * Description:                                                   *
 *      Shift arrow callback to represent how much bit will be    *
 *      shifted                                                   *
 *****************************************************************/
static void arrow_callback(w, list, cbs)
Widget w;
MaskWidgets *list;
XmArrowButtonCallbackStruct *cbs;
{
  int which,  bit_shift_value;
  Dimension   num_bit;
  Arg arg[10];
  int ac = 0;
  XmString str;
  char temp[40];
  int  mask_value;

  XtSetArg (arg[ac], XmNuserData, &which); ac++;
  XtGetValues(w, arg, ac);
  ac = 0;

  XtSetArg (arg[ac], XcodaNnumBit, &num_bit); ac++;
  XtGetValues(list->value_mask_w, arg, ac);
  ac = 0;

  XtSetArg (arg[ac], XmNuserData, &bit_shift_value); ac++;
  XtGetValues(list->right_shift_w, arg, ac);
  ac = 0;

  if(which == 0){ /* right arrow */
    if(bit_shift_value < num_bit - 1)
      bit_shift_value += 1;
  }
  else{
    if(bit_shift_value > 0)
      bit_shift_value -= 1;
  }

  sprintf(temp,"%d", bit_shift_value);
  str = XmStringCreateSimple(temp);
  XtSetArg (arg[ac], XmNlabelString, str); ac++;
  XtSetArg (arg[ac], XmNuserData, bit_shift_value); ac++;
  XtSetValues (list->right_shift_w, arg, ac);
  ac = 0;
  XmStringFree (str);
}

/**********************************************************************
 *       static void bit_select_callback()                            *
 * Description:                                                       *
 *     Bit selected callback to give mask pattern                     *
 *********************************************************************/
static void bit_select_callback(w, list, cbs)
Widget w;
MaskWidgets *list;
XmAnyCallbackStruct *cbs;
{
  char     temp[40];
  int      mask_value;

  mask_value = XcodaValueMaskGetValue(w);
  sprintf(temp,"0x%8x", mask_value);

  XmTextFieldSetString(list->text_w, temp);
}
  
/*********************************************************************
 *         static void apply_new_mask()                              *
 * Description:                                                      *
 *     Apply new integer value mask to this popup                    *
 ********************************************************************/
static void apply_new_mask(w, list, cbs)
Widget w;
MaskWidgets *list;
XmAnyCallbackStruct *cbs;
{
  int mask_value;
  int num_bit_shift;
  int mode;
  Arg arg[10];
  int ac = 0;
  char *buffer, *buffer1;

  mask_value = XcodaValueMaskGetValue(list->value_mask_w);
  XtSetArg (arg[ac], XmNuserData, &num_bit_shift); ac++;
  XtGetValues (list->right_shift_w, arg, ac);
  ac = 0;
  if(XmToggleButtonGetState(list->and_mode_w))
    mode = 0;
  else
    mode = 1;
  
  list->value_dialog->mask_value = mask_value;
  list->value_dialog->rshift_bit = num_bit_shift;
  list->value_dialog->mask_mode = mode;

  buffer = list->value_dialog->buffer;
  buffer1 = applyMaskToIntBuffer(buffer, mask_value, num_bit_shift, mode);
  XmTextSetString(list->value_dialog->text_w, buffer1);
  XtFree (buffer1);
}
  
/*********************************************************************
 *          char * applyMaskToIntBuffer()                            *
 * Descritpion:                                                      *
 *     Apply integer value mask and right shift to integer buffer    *
 ********************************************************************/
char *applyMaskToIntBuffer(buffer, mask_value, rbit_shift, mode)
char *buffer;
int  mask_value;
int  rbit_shift;
int  mode;   /* 0 and, 1 or */
{
  char *new_buffer, *p, *q;
  int  i, j, k; 
  int  temp, new_value;
  char out[12], in[12];

  new_buffer = (char *)malloc((strlen(buffer) + 1)*sizeof(char));
  if (new_buffer == NULL){
    fprintf(stderr,"Cannot allocate memory for new buffer in applying mask\n");
    exit(1);
  }
  i = strlen(buffer)/12;

  if ((strstr (buffer,"0x") != NULL) ||
      (strstr (buffer,"0X") != NULL)){  /* Hex Mode */
    p = buffer;
    q = new_buffer;
    while (*p != '\0'){
      sscanf(p, "%s", in);
      p = p + strlen(in);
      while (*p == ' ')
	p++;
      sscanf(in,"0x%x",&temp);
      /* convert to new value */
      if (mode == 0){
	new_value = temp & mask_value;
	if (rbit_shift)
	  new_value = new_value >> rbit_shift;
      }
      else{
	new_value = temp | mask_value;
	if (rbit_shift)
	  new_value = new_value >> rbit_shift;
      }	  
      if(hex_or_decimal == 1)
	sprintf(out, "0x%-10.8x",new_value);
      else
	sprintf(out, "%-12d", new_value);
      for(k=0;k<12;k++){
	*q = out[k];
	q++;
      }
    }
    *q = '\0';
  }
  else{ /* Decimal mode */
    p = buffer;
    q = new_buffer;
    while (*p != '\0'){
      sscanf(p, "%s", in);
      p = p + strlen(in);
      while (*p == ' ')
	p++;
      sscanf(in,"%d",&temp);
      /* change to new values */
      if (mode == 0){
	new_value = temp & mask_value;
	if (rbit_shift)
	  new_value = new_value >> rbit_shift;
      }
      else{
	new_value = temp | mask_value;
	if (rbit_shift)
	  new_value = new_value >> rbit_shift;
      }
      if(hex_or_decimal == 1)
	sprintf(out, "0x%-10.8x", new_value);
      else
	sprintf(out, "%-12d", new_value);
      for(k=0;k<12;k++){
	*q = out[k];
	q++;
      }
    }
    *q = '\0';
  }
  return new_buffer;
}
    
/********************************************************************
 *      static void unset_mask()                                    *
 * Description:                                                     *
 *     Unset mask for this window                                   *
 *******************************************************************/
static void unset_mask (w, list, cbs)
Widget w;
MaskWidgets *list;
XmAnyCallbackStruct *cbs;
{
  int mask_value = (int) 0xffffffff;
  Arg arg[10];
  int ac = 0;
  XmString t;
  
  XcodaValueMaskSetValue (list->value_mask_w, mask_value);
  list->value_dialog->mask_value = mask_value;
  XmTextFieldSetString(list->text_w, "0xffffffff");
  
  t = XmStringCreateSimple("0");
  XtSetArg (arg[ac], XmNuserData, 0); ac++;
  XtSetArg (arg[ac], XmNlabelString, t); ac++;  
  XtSetValues (list->right_shift_w, arg, ac);
  ac = 0;
  XmStringFree (t);

  apply_new_mask (w, list, cbs);
}
  
/******************************************************************
 *         static void ok_new_mask()                              *
 * Description:                                                   *
 *       Ok button callback                                       *
 *****************************************************************/
static void ok_new_mask(w, list, cbs)
Widget w;
MaskWidgets *list;
XmAnyCallbackStruct *cbs;
{
  apply_new_mask(w, list, cbs);
  XtPopdown (list->shell);
}

/*****************************************************************
 *         static void cancel_new_mask()                         *
 * Description:                                                  *
 *      Cancel Button Callback                                   *
 *****************************************************************/
static void cancel_new_mask(w, list, cbs)
Widget w;
MaskWidgets *list;
XmAnyCallbackStruct *cbs;
{
  XtPopdown (list->shell);
}

/****************************************************************
 *         void destroy_callback()                              *
 * Description:                                                 *
 *    Free memoty assocaited with mask dialog                   *
 *    Upon destruction of all widgets                           *
 ***************************************************************/
static void destroy_callback(w, list, cbs)
Widget w;
MaskWidgets *list;
XmAnyCallbackStruct *cbs;
{
  free (list);
}
