/*-----------------------------------------------------------------------------
 * COPYRIGHT (c) 1991,1992 Southeastern Universities Research Association,
 *                         Continuous Electron Beam Accelerator Facility
 *
 * This software was developed under a United States Government license
 * described in the NOTICE file included as part of this distribution.
 *
 * CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
 * Email: coda@cebaf.gov  Tel: (804) 249-7101  Fax: (804) 249-7363
 *-----------------------------------------------------------------------------
 * 
 * Description:
 *	CODA Value Mask Widget
 *	
 * Author:  Jie Chen, CEBAF Data Acquisition Group
 *
 * Revision History:
 *   $Log: ValueMask.c,v $
 *   Revision 1.2  1999/10/25 14:42:57  rwm
 *   Always include stdio.h
 *
 *   Revision 1.1.1.1  1996/09/19 18:26:25  chen
 *   original port to solaris
 *
 *	  Revision 1.1  1994/12/08  19:55:28  chen
 *	  Initial revision
 *
 *	  Revision 1.1  1994/04/06  14:35:09  chen
 *	  Initial revision
 *
 *	  Revision 1.3  1994/02/07  16:30:30  chen
 *	  set line attributes
 *
 *	  Revision 1.2  1994/02/07  16:12:34  chen
 *	  handle rectangle size right
 *
 *	  Revision 1.1  1994/02/07  15:24:43  chen
 *	  Initial revision
 *
 *	  
 */
#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "ValueMask.h"
#include "ValueMaskP.h"

#define Offset(field) XtOffsetOf(XcodaValueMaskRec, valueMask.field)

/********************************************************************
 *            Full Class Record constant                            *
 *******************************************************************/
/* private data */
static XtResource resources[] = {
  {XcodaNnumBit, XcodaCNumBit, XtRDimension, sizeof(Dimension),
     XtOffsetOf(XcodaValueMaskRec, valueMask.num_bit),
     XtRImmediate, (XtPointer)32},
  {XcodaNbitSelectCallback, XcodaCBitSelectCallback, XtRCallback, 
     sizeof(XtCallbackList), 
     XtOffsetOf(XcodaValueMaskRec, valueMask.callback),
     XtRCallback, (XtPointer)NULL},
  {XcodaNforeground, XcodaCForeground, XtRPixel, sizeof(Pixel), 
     XtOffsetOf(XcodaValueMaskRec, valueMask.foreground),
     XtRString, (XtPointer)XtDefaultForeground},
};

#undef Offset

static char defaultTranslations[]=
   "<Btn1Down>:            select()       \n\
    <Btn1Motion> :         high_light()";


static void Select();
static void High_light();

static XtActionsRec actions[]={
  {"select",     Select},
  {"high_light", High_light},
};

/* forward declaration */
static void  Initialize();
static void  Resize();
static void  Redisplay ();
static Boolean SetValues();
static void    Destroy();
static void  drawSetRectangle();
static void  drawUnsetRectangle();

/* class record declarations */
XcodaValueMaskClassRec xcodaValueMaskClassRec = {
  /* core class part */
  {
    /* super class                 */ (WidgetClass)&widgetClassRec,
    /* class_name                  */ "ValueMask",
    /* widget_size                 */ sizeof(XcodaValueMaskRec),
    /* class_initialize            */ NULL,
    /* class_part_initialize       */ NULL,
    /* class_initied               */ FALSE,
    /* initialize                  */ Initialize,
    /* initialize_hook             */ NULL,
    /* realize                     */ XtInheritRealize,
    /* actions                     */ actions,
    /* num_actions                 */ XtNumber(actions),
    /* resources                   */ resources,
    /* num_of_resources            */ XtNumber(resources),
    /* xrm_class                   */ NULLQUARK,
    /* compress_motion             */ TRUE,
    /* compress_exposure           */ TRUE,
    /* compress_enterleave         */ TRUE,
    /* visible_interet             */ FALSE,
    /* destroy                     */ Destroy,
    /* resize                      */ Resize,
    /* expose                      */ Redisplay,
    /* set_values                  */ SetValues,
    /* set_values_hook             */ NULL,
    /* set_values_almost           */ XtInheritSetValuesAlmost,
    /* get_values_hook             */ NULL,
    /* accept_focus                */ NULL,
    /* version                     */ XtVersion,
    /* callback offsets            */ NULL,
    /* tm_table                    */ defaultTranslations,
    /* query_geometry              */ NULL,
    /* display_accelerator         */ XtInheritDisplayAccelerator,
    /* extension                   */ NULL
  },
    /* value mask class part       */
  {
    /* ignore                      */ 0,
  }				    
};

WidgetClass xcodaValueMaskWidgetClass = (WidgetClass)&xcodaValueMaskClassRec;

static int power(n, power)
unsigned int n, power;
{
  int i;
  int      ret;

  ret = 1;

  for(i=0;i<power;i++)
    ret = ret*n;
  return ret;
}


static void GetNormalGC(vw)
     XcodaValueMaskWidget vw;
{
  XGCValues values;

  values.foreground = vw->core.background_pixel;
  values.background = vw->core.background_pixel;

  vw->valueMask.erase_gc = XtGetGC(
	 (Widget)vw,
	 (unsigned) GCForeground| GCBackground,
         &values);

  values.foreground = vw->valueMask.foreground;
  values.background = vw->core.background_pixel;

  vw->valueMask.gc = XtGetGC(
	 (Widget)vw,
	 (unsigned) GCForeground| GCBackground,
         &values);
}

static void SetDimensions(vw)
     XcodaValueMaskWidget vw;
{
  vw->core.width = (vw->valueMask.num_bit)*
    (vw->valueMask.bit_size + vw->valueMask.space);

  vw->core.height = (vw->valueMask.bit_size) +
    2*(vw->valueMask.space);
}

static void SetPositions(vw)
     XcodaValueMaskWidget vw;
{
  int i;
  Position xx, yy;
  Dimension real_space;

  real_space = vw->valueMask.space + vw->valueMask.bit_size;

  for (i=0; i < vw->valueMask.num_bit; i++){
    xx = vw->valueMask.space + (vw->valueMask.num_bit - 1 - i)*real_space;
    yy = (vw->core.height - vw->valueMask.bit_size)/2.0;
    if (yy < 0 )
      yy = 0;
    vw->valueMask.bit_x_pos[i] = xx;
    vw->valueMask.bit_y_pos[i] = yy;
  }
}

/****************************************************************************
 *          static void Initialize ()                                       *
 * Description:                                                             *
 *     Initialize all data structure elements                               *
 ***************************************************************************/
static void Initialize(request, new, args, num_args)
     Widget request, new;
     ArgList args;
     Cardinal *num_args;
{
  int i;

  XcodaValueMaskWidget vw = (XcodaValueMaskWidget) new;

  if (vw->valueMask.num_bit < 8){
    XtWarning("Too Few Bits");
    vw->valueMask.num_bit = 32;
  }
  vw->valueMask.bit_size = 8;
  vw->valueMask.space = 5;
  vw->valueMask.mask_value = -1;
  vw->valueMask.pre_x = 0;
  vw->valueMask.pre_y = 0;
  
  GetNormalGC (vw);

  SetDimensions (vw);
  
  SetPositions (vw);

  for(i=0;i < vw->valueMask.num_bit; i++)
    vw->valueMask.bit_set[i] = 1;
    
}

/*****************************************************************************
 *                   static void Redisplay ()                                *
 * Description:                                                              *
 *       Redraw everything for expose event                                  *
 ****************************************************************************/
static void Redisplay (w, event, region)
     Widget w;
     XEvent *event;
     Region region;
{
  int i;
  XcodaValueMaskWidget vw = (XcodaValueMaskWidget) w;
  GC gc;
  Dimension size;

  gc = vw->valueMask.gc;
  size = vw->valueMask.bit_size;

  for(i=0;i<vw->valueMask.num_bit;i++){
    if ((i+1) % 8 == 0 && i != 0 && i != vw->valueMask.num_bit - 1){
      XSetLineAttributes(XtDisplay(w), gc, 2, LineSolid, CapNotLast, JoinMiter);
      XDrawLine (XtDisplay(w), XtWindow(w), gc, 
		 vw->valueMask.bit_x_pos[i] - 2,
		 vw->valueMask.bit_y_pos[i] - 2, 
		 vw->valueMask.bit_x_pos[i] - 2,
		 vw->valueMask.bit_y_pos[i] + size + 2);
      XSetLineAttributes(XtDisplay(w), gc, 1, LineSolid, CapNotLast, JoinMiter);
    }
    if (vw->valueMask.bit_set[i]){
      drawSetRectangle(XtDisplay(w), XtWindow(w), gc, vw->valueMask.bit_x_pos[i],
		       vw->valueMask.bit_y_pos[i], size, size);
    }
    else{
      drawUnsetRectangle(XtDisplay(w), XtWindow(w), gc, vw->valueMask.bit_x_pos[i],
			 vw->valueMask.bit_y_pos[i], size, size);
    }
  }
}

/*******************************************************************************
 *             static Boolean SetValues()                                      *
 * Description:                                                                *
 *     Set Values from ValueMask Widget                                        *
 ******************************************************************************/
static Boolean SetValues(current, request, new, args, num_args)
     Widget current, request, new;
     ArgList args;
     Cardinal *num_args;
{
  XcodaValueMaskWidget oldvw = (XcodaValueMaskWidget)current;
  XcodaValueMaskWidget reqvw = (XcodaValueMaskWidget)request;
  XcodaValueMaskWidget newvw = (XcodaValueMaskWidget)new;
  Boolean redisplay = False;
  Boolean was_resized = False;

#define NE(field) (oldvw->field != newvw->field)

  if(NE(valueMask.num_bit)){
    SetDimensions (newvw);
    SetPositions (newvw);
    was_resized = True;
    redisplay = True;
  }
  if(NE(valueMask.foreground) || NE(core.background_pixel)){
    XtReleaseGC(new, oldvw->valueMask.gc);
    XtReleaseGC(new, oldvw->valueMask.erase_gc);
    GetNormalGC (newvw);
    redisplay = True;
  }
  return redisplay || was_resized;
#undef NE
}
  
/*****************************************************************************
 *          static void Destroy ()                                           *
 * Description:                                                              *
 *       Release all GC and memory                                           *
 ****************************************************************************/
static void Destroy(w)
     Widget w;
{
  XcodaValueMaskWidget vw = (XcodaValueMaskWidget)w;

  XtReleaseGC(w, vw->valueMask.gc);
  XtReleaseGC(w, vw->valueMask.erase_gc);
}

/****************************************************************************
 *         static void Resize()                                             *
 * Description:                                                             *
 *      Resize event handler                                                *
 ***************************************************************************/
static void Resize(w)
     Widget w;
{
  XcodaValueMaskWidget vw = (XcodaValueMaskWidget)w;
  Dimension real_space;
  
  real_space = (vw->core.width)/(vw->valueMask.num_bit);
  vw->valueMask.space = real_space - vw->valueMask.bit_size;
  SetPositions(vw);

  if(XtIsRealized(w)){
    XClearWindow(XtDisplay(w), XtWindow(w));
    (*(XtClass(w)->core_class.expose))(w,
       (XEvent *)NULL, (Region) NULL);
  }
}

static int WhichClicked(vw, x, y)
     XcodaValueMaskWidget vw;
     Position x, y;
{
  int i;
  int ret_num;

  for(i=0;i<vw->valueMask.num_bit; i++){
    if( (x >= vw->valueMask.bit_x_pos[i]) && 
        (x < vw->valueMask.bit_x_pos[i] + vw->valueMask.bit_size) &&
        (y >= vw->valueMask.bit_y_pos[i]) &&
        (y < vw->valueMask.bit_y_pos[i] + vw->valueMask.bit_size))
      break;
  }
  if (i >= vw->valueMask.num_bit)
    return -1;
  else
    return i;
}

static void Select(w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
{
  int pre_j, j;
  XcodaValueMaskWidget vw = (XcodaValueMaskWidget)w;
  Dimension size = vw->valueMask.bit_size;

  if(event != NULL && event->type == ButtonPress 
     && event->xany.window == XtWindow(w)){
    XButtonEvent *b = &event->xbutton;
    j = WhichClicked (vw, b->x, b->y);
    if (j >= 0){
      drawSetRectangle(XtDisplay(w), XtWindow(w), vw->valueMask.erase_gc ,
		       vw->valueMask.bit_x_pos[j],
		       vw->valueMask.bit_y_pos[j], size, size);
      if (vw->valueMask.bit_set[j]){
	drawUnsetRectangle(XtDisplay(w), XtWindow(w),vw->valueMask.gc, 
			   vw->valueMask.bit_x_pos[j],
			   vw->valueMask.bit_y_pos[j], size, size);
	vw->valueMask.bit_set[j] = 0;
      }
      else{
	drawSetRectangle(XtDisplay(w), XtWindow(w), vw->valueMask.gc ,
			 vw->valueMask.bit_x_pos[j],
			 vw->valueMask.bit_y_pos[j], size, size);
	vw->valueMask.bit_set[j] = 1;
      }
    }
    else
      return;

    XtCallCallbackList (vw, vw->valueMask.callback, NULL);
  }
}

static void High_light(w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
{
  int pre_j, j;
  XcodaValueMaskWidget vw = (XcodaValueMaskWidget)w;
  Dimension size = vw->valueMask.bit_size;

  if(event != NULL && event->xany.window == XtWindow(w)){
    XMotionEvent *b = &event->xmotion;
    pre_j = WhichClicked(vw, vw->valueMask.pre_x, vw->valueMask.pre_y);
    j = WhichClicked (vw, b->x, b->y);
    vw->valueMask.pre_x = b->x;
    vw->valueMask.pre_y = b->y;
    if (j >= 0 && pre_j != j) {
      drawSetRectangle(XtDisplay(w), XtWindow(w), vw->valueMask.erase_gc ,
		       vw->valueMask.bit_x_pos[j],
		       vw->valueMask.bit_y_pos[j], size, size);
      if (vw->valueMask.bit_set[j]){
	drawUnsetRectangle(XtDisplay(w), XtWindow(w),vw->valueMask.gc, 
			   vw->valueMask.bit_x_pos[j],
			   vw->valueMask.bit_y_pos[j], size, size);
	vw->valueMask.bit_set[j] = 0;
      }
      else{
	drawSetRectangle(XtDisplay(w), XtWindow(w), vw->valueMask.gc ,
			 vw->valueMask.bit_x_pos[j],
			 vw->valueMask.bit_y_pos[j], size, size);
	vw->valueMask.bit_set[j] = 1;
      }
    }
    else
      return;
    XtCallCallbackList (vw, vw->valueMask.callback, NULL);
  }
}

/* Supporting routines */
int XcodaValueMaskGetValue(w)
     Widget w;
{
  int i;
  int sum = 0;
  XcodaValueMaskWidget vw = (XcodaValueMaskWidget)w;

  for(i=0;i<vw->valueMask.num_bit; i++){
    if(vw->valueMask.bit_set[i]){
      sum = sum + power(2,i);
    }
  }
  return sum;
}

void XcodaValueMaskSetValue(w, value)
     Widget w;
     int    value;
{
  int i;
  XcodaValueMaskWidget vw = (XcodaValueMaskWidget)w;

  for(i=0;i<vw->valueMask.num_bit; i++){
    if((value & power(2, i)) != 0)
      vw->valueMask.bit_set[i] = 1;
    else
      vw->valueMask.bit_set[i] = 0;
  }
  if(XtIsRealized(w)){
    XClearWindow(XtDisplay(w), XtWindow(w));
    (*(XtClass(w)->core_class.expose))(w,
       (XEvent *)NULL, (Region) NULL);
  }
}  

/*********************Set dark rectangle********************/
static void drawSetRectangle(dpy, win, gc, x, y, wd, ht)
Display *dpy;
Drawable win;
GC       gc;
Position x, y;
Dimension wd, ht;
{
  XSetLineAttributes(dpy, gc, 1, LineSolid, CapNotLast, JoinMiter);
  XDrawRectangle(dpy, win, gc, x, y, wd, ht);
  XFillRectangle(dpy, win, gc, x, y, wd, ht);
}

static void drawUnsetRectangle(dpy, win, gc, x, y, wd, ht)
Display *dpy;
Drawable win;
GC       gc;
Position x, y;
Dimension wd, ht;
{
  XSetLineAttributes(dpy, gc, 1, LineSolid, CapNotLast, JoinMiter);
  XDrawRectangle(dpy, win, gc, x, y, wd, ht);  
}
