/*-----------------------------------------------------------------------------
 * Copyright (c) 1991,1992 Southeastern Universities Research Association,
 *                         Continuous Electron Beam Accelerator Facility
 *
 * This software was developed under a United States Government license
 * described in the NOTICE file included as part of this distribution.
 *
 * CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
 * Email: coda@cebaf.gov  Tel: (804) 249-7101  Fax: (804) 249-7363
 *-----------------------------------------------------------------------------
 * 
 * Description:
 *     construct a tree structure from a physical data
 *     event using most left child and right sibling 
 *     data structure
 *	
 * Author:  Jie Chen, CEBAF Data Acquisition Group
 *
 * Revision History:
 *   $Log: event_tree.c,v $
 *   Revision 1.1.1.1  1996/09/24 14:21:28  chen
 *   Initial import to coda_2.0
 *
 *	  Revision 1.2  1995/04/05  20:15:41  chen
 *	  fix a bug for a bad event
 *
 *	  Revision 1.1  1994/12/08  14:44:53  chen
 *	  Initial revision
 *
 *	  Revision 1.3  1994/05/16  15:38:44  chen
 *	  release memory for evStack
 *
 *	  Revision 1.2  1994/05/13  18:39:11  chen
 *	  memory management
 *
 *	  Revision 1.1  1994/04/12  16:20:11  chen
 *	  Initial revision
 *
 *	  Revision 1.1  1992/07/28  20:05:33  chen
 *	  Initial revision
 *
 */

#include <stdio.h>
#include "cef.h"

extern char *strsave();
extern void bufcpy(),bufcpy8(),bufcpy2();
extern void GetName();


/*************************************************************
 *         Stack Utilities                                   *
 * Description:                                              *
 *     This stack keeps information abot parent              *
 ************************************************************/
EvStack *init_EvStack()
{
  EvStack *head;
  head=(EvStack *)calloc(1,sizeof(EvStack));
  if(head == NULL){
    fprintf(stderr,"No memory for EvStack head.\n");
    exit(1);
  }
  head->length = 0;
  head->posi = 0;
  head->type = 0x0;
  head->tag = 0x0;
  head->num =0x0;
  head->next = NULL;
  return(head);
}

/*************************************************************
 *        EvStack *top (head)                                *
 * Description:                                              *
 *     return top of the Event stack                         *
 ************************************************************/
EvStack *top(head)
EvStack *head;
{
  EvStack *p;
   
  p=head;
  if( p->next == NULL)
    return(NULL);
  else
    return(p->next);
}

/*************************************************************
 *        void pop_off(head)                                 *
 * Description:                                              *
 *      pop off top of the stack                             *
 *      free the memory                                      *
 ************************************************************/
void pop_off(head)
EvStack *head;
{
  EvStack *p,*q;
   
  q=head;
  if(q->next == NULL){
    fprintf(stderr,"Empty stack. \n");
    return;
  }
  p=q->next;
  q->next = p->next;
  free (p);
}

/*****************************************************************
 *        void push_on (size, posi, type, tag, num, head)        *
 * Decription:                                                   *
 *     Push an item to the event stack                           *
 ****************************************************************/
void push_on(size,posi,type,tag,num,head)
int size,posi,type,tag,num;
EvStack *head;
{
  EvStack *p,*q;
   
  p=(EvStack *)calloc(1,sizeof(EvStack));
  if(p == NULL){
    fprintf(stderr,"No memory for stack items.\n");
    exit(1);
  }
  q=head;
  p->length=size;
  p->posi=posi;
  p->type=type;
  p->tag=tag;
  p->num=num;
  p->next = q->next;
  q->next = p;
}

/*******************************************************
 *       void free_ev_stack()                          *
 * Description:                                        *
 *     Free Event Statck Memory                        *
 ******************************************************/
static void free_ev_stack(head)
EvStack *head;
{
  EvStack *p, *q;

  p = head;
  while (p != NULL){
    q = p->next;
    free (p);
    p = q;
  }
}
      
/******************************************************
 * evTreeNode *EvTreeContruct()                       *
 * Description:                                       *
 *    Construct a Tree Structure which will be        *
 *    Displayed by  cefdmp                            *
 *****************************************************/
evTreeNode *EvTreeConstruct(ev,t_type,t_length)
Event ev[];
short t_type;
int   t_length;
{
  EvStack     *head;
  Look_Ahead  lk;
  evTreeNode  *root,*parent,*p,*q;
  int         i,j,k,l,depth,current_type;
  int         header1,header2,num_of_output;
  int         ev_size,ev_tag,ev_num,ev_type;
  int         bk_size,bk_tag,bk_type,bk_num;
  int         sg_size,sg_tag,sg_type;
  short       pk_size,pk_tag,pack;
  int         is_packet;
  char        title[80];
  char        temp[4],temp2[2],blank[80],temp8[8],ev_name[32],bk_name[32];
  char        sg_name[32],pk_name[32];

  head=init_EvStack();

  i=0;                 /* now i is a index pointing to 16 bit*/
  if(t_type == 0xC){   /* Notation for whole event */
    bufcpy(temp,ev);
    ev_size = *(int *)(temp);  /*ev_szie in unit 32 bit*/
    i = i + 2;     /*increase by 2 each time for 32 bit calculation*/
    bufcpy(temp,&ev[i*2]);
    header2= *(int *)(temp);
    ev_tag=(header2 >> 16) & (0x0000ffff);
    ev_type=(header2 >>8) & (0x000000ff);
    ev_num=(header2) &(0x000000ff);


    /* Allocate root for a tree*/
    root=(evTreeNode *)calloc(1,sizeof(evTreeNode));
    root->length=(ev_size+1)*2;
    root->start=0;
    root->tag_v=ev_tag;
    root->type= ev_type;
    root->parent_type = 0x10;
    root->num=ev_num;
    root->parent = NULL;
    root->mleft = NULL;
    root->rsibling = NULL;
    GetName(root,ev_name,title);
    root->name = strsave(ev_name);
    root->title= strsave(title);
    parent = root;
    
    i = i + 2;
/* set up stack keeping track of parents*/
    if(root->type >= 0x10 && root->type <= 0x30 ){      /* contains children */
      push_on((ev_size+1)*2,i-4,ev_type,ev_tag,ev_num,head);
      lk.head_pos = i;
      lk.type = root->type;   
      if(lk.type ==0x10)
	ev_size = ev_size + 1;
    }
    else{  
      root->parent_type = 0x10;
      free_ev_stack (head);
      return(root);
    }
  }
  else{
    lk.head_pos = i*2;  /*here i = 0 */
    lk.type = t_type;
    if(t_type == 0x10)
      ev_size = t_length + 1;
    else
      ev_size = t_length;
    root = (evTreeNode *)calloc(1,sizeof(evTreeNode));
    root->length = -1;
  }
 
    
  while(i < ev_size*2){
    if((top(head)) != NULL)
      while(i == (top(head))->length + (top(head))->posi){    /* make sure go back to root*/
	pop_off(head);
	head->length-=1;
	parent=parent->parent; 
      }
    if(i == lk.head_pos ){         /* dealing with head */
      if((top(head)) != NULL)
	lk.type = ((top(head))->type);
      switch(lk.type){
      case 0x10:
	bufcpy(temp,&ev[i*2]);
	header1= *(int *)(temp);
	bk_size = header1;
	i = i + 2;
	bufcpy(temp,&ev[i*2]);
	header2= *(int *)(temp);
	bk_tag=(header2 >>16) & (0x0000ffff);
	bk_type=(header2 >>8) & (0x000000ff);
	bk_num=(header2) &(0x000000ff);
	depth=head->length;
	
	if (bk_size <= 0){ /* sometimes bad things happen */
	  return (evTreeNode *)0;
	}
	if(bk_type >= 0x10){  /* contains children*/
	  push_on((bk_size+1)*2,i-2,bk_type,bk_tag,bk_num,head);
	  lk.head_pos=i+2;
	  head->length += 1;
	  
	  if(root->length == -1){
	    root->length =(bk_size+1)*2;
	    root->start = i-2;
	    root->tag_v=bk_tag;
	    root->num = bk_num;
	    root->type = bk_type;
	    root->parent_type = 0x10;
	    root->parent= NULL;
	    root->mleft = NULL;
	    root->rsibling = NULL;
	    GetName(root,bk_name,title);
	    root->name = strsave(bk_name);
	    root->title = strsave(title);
	    parent = root;
	  }
	  else{
	    p = (evTreeNode *)calloc(1,sizeof(evTreeNode));
	    p->length = (bk_size+1)*2;
	    p->start = i-2;
	    p->tag_v = bk_tag;
	    p->num = bk_num;
	    p->type = bk_type;
	    p->parent_type = 0x10;
	    /*updat parent and sibling */
	    if(parent->mleft == NULL){
	      p->parent = parent;
	      parent->mleft = p;
	      p->mleft = NULL;
	      p->rsibling = NULL;
	      GetName(p,bk_name,title);
	      p->name = strsave(bk_name);
	      p->title = strsave(title);
	      parent = p;
	    }
	    else{
	      for(q=parent->mleft;q->rsibling != NULL; q=q->rsibling)
		;
	      q->rsibling = p;
	      p->parent = q->parent;
	      p->mleft = NULL;
	      p->rsibling = NULL;
	      GetName(p,bk_name,title);
	      p->name = strsave(bk_name);
	      p->title = strsave(title);
	      parent = p;
	    }
	  }
	  i=i+2;
	}
	else{
	  current_type = bk_type;
	  lk.head_pos=i+bk_size*2;
	  
	  if(root->length == -1){
	    root->length =(bk_size+1)*2;
	    root->start = i-2;
	    root->tag_v=bk_tag;
	    root->num = bk_num;
	    root->type = bk_type;
	    root->parent_type = 0x10;
	    root->parent= NULL;
	    root->mleft = NULL;
	    root->rsibling = NULL;
	    GetName(root,bk_name,title);
	    root->name = strsave(bk_name);
	    root->title = strsave(title);
	    parent = root;
	  }
	  else{
	    p = (evTreeNode *)calloc(1,sizeof(evTreeNode));
	    p->length = (bk_size+1)*2;
	    p->start = i-2;
	    p->tag_v = bk_tag;
	    p->num = bk_num;
	    p->type = bk_type;
	    p->parent_type = 0x10;
	    /*updat parent and sibling */
	    if(parent->mleft == NULL){
	      p->parent = parent;
	      parent->mleft = p;
	      p->mleft = NULL;
	      p->rsibling = NULL;
	      GetName(p,bk_name,title);
	      p->name = strsave(bk_name);
	      p->title = strsave(title);
	    }
	    else{
	      for(q=parent->mleft;q->rsibling != NULL; q=q->rsibling)
		;
	      q->rsibling = p;
	      p->parent = q->parent;
	      p->mleft = NULL;
	      p->rsibling = NULL;
	      GetName(p,bk_name,title);
	      p->name = strsave(bk_name);
	      p->title = strsave(title);
	    }
	  }
	  
	  i=i+2;
	}		
	break;
      case 0x20:
	bufcpy(temp,&ev[i*2]);
	header2 = *(int *)(temp);
	sg_size=(header2)&(0x0000ffff); /* can be changed*/
	sg_size = sg_size + 1;
	sg_tag=(header2 >> 24) & (0x000000ff);
	sg_type=(header2 >> 16)&(0x000000ff);

	
	if(sg_type >= 0x20){      /* contains children */
	  push_on((sg_size)*2,i,sg_type,sg_tag,NULL,head);
	  lk.head_pos = i+2;
	  head->length +=1;

	  if(root->length == -1){
	    root->length =(sg_size)*2;
	    root->start = i;
	    root->tag_v=sg_tag;
	    root->num = -1;
	    root->type = sg_type;
	    root->parent_type = 0x20;
	    root->parent= NULL;
	    root->mleft = NULL;
	    root->rsibling = NULL;
	    GetName(root,sg_name,title);
	    root->name = strsave(sg_name);
	    root->title = strsave(title);
	    parent = root;
	  }
	  else{
	    p = (evTreeNode *)calloc(1,sizeof(evTreeNode));
	    p->length = (sg_size)*2;
	    p->start = i;
	    p->tag_v = sg_tag;
	    p->num = -1;
	    p->type = sg_type;
	    p->parent_type = lk.type;
	    /*updat parent and sibling */
	    if(parent->mleft == NULL){
	      p->parent = parent;
	      parent->mleft = p;
	      p->mleft = NULL;
	      p->rsibling = NULL;
	      GetName(p,sg_name,title);
	      p->name = strsave(sg_name);
	      p->title = strsave(title);
	      parent = p;
	    }
	    else{
	      for(q=parent->mleft;q->rsibling != NULL; q=q->rsibling)
		;
	      q->rsibling = p;
	      p->parent = q->parent;
	      p->mleft = NULL;
	      p->rsibling = NULL;
	      GetName(p,sg_name,title);
	      p->name = strsave(sg_name);
	      p->title = strsave(title);
	      parent = p;
	    }
	  }
	  i=i+2;
	}
	else{
	  current_type =sg_type;
	  lk.head_pos = i+sg_size*2;

	  if(root->length == -1){
	    root->length =sg_size*2;
	    root->start = i;
	    root->tag_v=sg_tag;
	    root->num = -1;
	    root->type = sg_type;
	    root->parent_type = 0x20;
	    root->parent= NULL;
	    root->mleft = NULL;
	    root->rsibling = NULL;
	    GetName(root,sg_name,title);
	    root->name = strsave(sg_name);
	    root->title = strsave(title);
	    parent = root;
	  }
	  else{
	    p = (evTreeNode *)calloc(1,sizeof(evTreeNode));
	    p->length = sg_size*2;
	    p->start = i;
	    p->tag_v = sg_tag;
	    p->num = -1;
	    p->type = sg_type;
	    p->parent_type = 0x20;
	    /*updat parent and sibling */
	    if(parent->mleft == NULL){
	      p->parent = parent;
	      parent->mleft = p;
	      p->mleft = NULL;
	      p->rsibling = NULL;
	      GetName(p,sg_name,title);
	      p->name = strsave(sg_name);
	      p->title = strsave(title);
	    }
	    else{
	      for(q=parent->mleft;q->rsibling != NULL; q=q->rsibling)
		;
	      q->rsibling = p;
	      p->parent = q->parent;
	      p->mleft = NULL;
	      p->rsibling = NULL;
	      GetName(p,sg_name,title);
	      p->name = strsave(sg_name);
	      p->title = strsave(title);
	    }
	  }
	  
	  i=i+2;
	}
	break;
      default:         /*packet type*/
	bufcpy2(temp2,&ev[i*2]);
	pack = *(short *)(temp2);
	if(pack == 0x0000){   /*empty packet increase by 1*/
	  lk.head_pos = i + 1;
	  i++;
	}
	else{
	  pk_tag=(pack >> 8) & (0x00ff);
	  pk_size=(pack)&(0x00ff);
	  
	  current_type = lk.type;
	  lk.head_pos=i+ pk_size + 1;

	  if(root->length == -1){
	    root->length = pk_size + 1;
	    root->start = i;
	    root->tag_v=pk_tag;
	    root->num = -1;
	    root->type = lk.type;
	    root->parent_type = lk.type;
	    root->parent= NULL;
	    root->mleft = NULL;
	    root->rsibling = NULL;
	    GetName(root,pk_name,title);
	    root->name = strsave(pk_name);
	    root->title = strsave(title);
	    parent = root;
	  }
	  else{
	    p = (evTreeNode *)calloc(1,sizeof(evTreeNode));
	    p->length = pk_size + 1;
	    p->start = i;
	    p->tag_v = pk_tag;
	    p->num = -1;
	    p->type = lk.type;
	    p->parent_type = lk.type;
	    /*updat parent and sibling */
	    if(parent->mleft == NULL){
	      p->parent = parent;
	      parent->mleft = p;
	      p->mleft = NULL;
	      p->rsibling = NULL;
	      GetName(p,pk_name,title);
	      p->name = strsave(pk_name);
	      p->title = strsave(title);
	    }
	    else{
	      for(q=parent->mleft;q->rsibling != NULL; q=q->rsibling)
		;
	      q->rsibling = p;
	      p->parent = q->parent;
	      p->mleft = NULL;
	      p->rsibling = NULL;
	      GetName(p,pk_name,title);
	      p->name = strsave(pk_name);
	      p->title = strsave(title);
	    }
	  }
	  /* still use i = i + 1 but handle with care when I dump it*/
	  i = i + 1;
	}
	break;
      }
    }
    else                  /* deal with real data */
      i = lk.head_pos;  /* skip throgh real data*/
  }
  free_ev_stack (head);
  return(root);
}                          

/*******************************************************
 *       void free_ev_tree()                           *
 * Description:                                        *
 *     Recusively free memory allocated by             *
 *     above tree construction routine                 *
 ******************************************************/
void free_ev_tree (ev_t)
evTreeNode *ev_t;
{
  evTreeNode *mleft, *rsibling;

  mleft = ev_t->mleft;
  rsibling = ev_t->rsibling;
  if (ev_t->name != NULL)
    free (ev_t->name);
  if (ev_t->title != NULL)
    free (ev_t->title);
  free (ev_t);
  if (mleft != NULL)
    free_ev_tree (mleft);
  if (rsibling != NULL)
    free_ev_tree (rsibling);
}

/*******************************************************
 *        int is_same_node (p, q)                      *
 * Description:                                        *
 *     Check to see whether two nodes p ?= q           *
 *     0: yes, 1: no                                   *
 ******************************************************/
int is_same_node(p,q)  /* == 0 means yes, 1 meas no*/
evTreeNode *p;
evTreeNode *q;
{
  if(p->tag_v != q->tag_v)
    return(1);
  if(p->start != q->start)
    return(1);
  if(p->length != q->length)
    return(1);
  if(p->type != q->type)
    return(1);

  return(0);
}

/*******************************************************
 *          void tree_print (root)                     *
 * Description:                                        *
 *      Print out evTreeNode structure                 *
 ******************************************************/
void tree_print(root)
evTreeNode *root;
{
  evTreeNode *p,*q;
   
  p=root;
  if(p!=NULL){
    if(p->parent != NULL)
      printf("node is %s and its parent is %s length is %d\n",
	     p->name,p->parent->name, p->length);
    else
      printf("node is %s and size is %d\n",p->name, p->length);
    tree_print(p->mleft);
    tree_print(p->rsibling);
  }
}

/********************************************************
 *         void copy_evnode_value(p, q)                 *
 * Description:                                         *
 *    Copy p to q, memory already allocated             *
 *******************************************************/
void copy_evnode_value(p,q)  /*copy p to q just values*/
evTreeNode *p,*q;
{
  q->start = p->start;
  q->length = p->length;
  q->tag_v = p->tag_v;
  q->name = strsave(p->name);
  q->title = strsave(p->title);
  q->num = p->num;
  q->parent_type = p->parent_type;
  q->type = p->type;
  q->parent = NULL;
  q->mleft = NULL;
  q->rsibling = NULL;
}
