/*
 * ns.c --
 *
 *  Introduction -- This file implements the extra Tcl/Tk commands
 *	needed by the name server.
 *
 * Copyright (c) 1995 Cornell University
 * All rights reserved.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for research and educational purposes, without fee, and
 * without written agreement is hereby granted, provided that the above
 * copyright notice and the following two paragraphs appear in all copies
 * of this software.
 *
 * IN NO EVENT SHALL CORNELL UNIVERSITY BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF CORNELL
 * UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * CORNELL UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND CORNELL UNIVERSITY HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <tcl.h>
#include <tclInt.h>
#include <tk.h>
#include <stdio.h>

#ifdef VXWORKS
 #include <time.h>
 #include <sys/times.h>
 extern int clock_gettime();
#else
 #include <time.h>
 #include <signal.h>
 #include <sys/param.h>
 #include <sys/types.h>
 #include <sys/time.h>
 #include <sys/stat.h>
 #include <sys/file.h>
 #include <sys/ioctl.h>

 extern int errno;
 #include <errno.h>
 
 #ifndef NO_X11
 #  define Widget int
 #  include "codaRegistry.h"
 #endif

 #ifdef HAVE_FCNTL_H
 #  include <fcntl.h>
 #endif
 
 #ifdef HAVE_SYS_STREAM_H
 #  include <sys/stream.h>
 #endif



/*----------------------------------------------------------------------
 * Function: Tns_forkCmd
 * Purpose:
 *     Implements the fork Tcl command. It returns TCL_ERROR on a failed
 *     fork, TCL_OK on a correct fork.  For the return value in TCL, 0
 *     is returned if we are in the child, the process # if we are in
 *     the parent.
 * Tcl Command Args:
 *     arg1: process group name
 *----------------------------------------------------------------------*/
int
Tns_forkCmd(ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
                          	/* Main window associated with
				 * interpreter. */
                       		/* Current interpreter. */
             			/* Number of arguments. */
                		/* Argument strings. */
{
    int pid;

    if (argc != 1) {
	Tcl_AppendResult (interp, "wrong # args: should be \"",
			  argv[0], 
			  (char *) NULL);
	return TCL_ERROR;
    }

    pid = fork();
    if (pid == -1) {
	Tcl_AppendResult (interp, argv[0], ": Unable to fork process",
			  (char *) NULL);
	return TCL_ERROR;
    }

#if 0
    if (pid == 0) {
	/* Close the stdin */
	close(1);
    }
#endif

    Tcl_ResetResult(interp);
    sprintf(interp->result, "%d", pid);
    return TCL_OK;
}

/*----------------------------------------------------------------------
 * Function: Tns_execvpCmd
 * Purpose:
 *     Implements the execvp Tcl command. It returns TCL_ERROR on a failed
 *     exec, TCL_OK on a correct exec.
 *     the parent.
 * Tcl Command Args:
 *     arg1: program name
 *     ard2..n: program args
 *----------------------------------------------------------------------*/
int
Tns_execvpCmd(ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
                          	/* Main window associated with
				 * interpreter. */
                       		/* Current interpreter. */
             			/* Number of arguments. */
                		/* Argument strings. */
{
    int ret;
    char **argp = argv + 1;
    if (argc == 1) {
	Tcl_AppendResult (interp, "wrong # args: should be \"",
			  argv[0], 
			  " <program> <arg1> .. <argn> ",
			  (char *) NULL);
	return TCL_ERROR;
    }

    ret = execvp(argv[1],argp);

    if (ret == -1) {
      Tcl_AppendResult (interp, argv[0], ": Unable to exec process, ",
			argv[0],
			" - ",
			strerror(errno),
			(char *) NULL);
      return TCL_ERROR;
    }
    
    /* In theory we never get here... */
    
    Tcl_ResetResult(interp);
    sprintf(interp->result, "%d", ret);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Tns_DupCmd --
 *
 *	Implements the "ns_dup" Tcl command.  This duplicates a
 *	file descriptors, specifying the value of the new file
 *	descriptor.
 *
 * Results:
 *	A standard Tcl result
 *
 * Side effects:
 *	From now on, there will be two file descriptors
 *
 *--------------------------------------------------------------
 */
int
Tns_DupCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Main window associated with
				 * interpreter. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    int i;
    int fd[2];
    
    /*
     * Initial value needed only to stop compiler warnings.
     */
    fd[0] = 0;
    fd[1] = 0;

    /*
     * Check syntax
     */
    if (argc != 3) {
	Tcl_AppendResult (interp, "wrong # args: should be \"",
			  argv[0], "fileid filed",
			  (char *) NULL);
	return TCL_ERROR;
    }

    for (i = 1; i <= 2; i++) {
	if ((argv[i][0] == 'f') && (argv[i][1] == 'i') && (argv[i][2] == 'l')
	    & (argv[i][3] == 'e')) {
	    char *end;

	    fd[i-1] = strtoul(argv[i]+4, &end, 10);
	    if ((end == argv[i]+4) || (*end != 0)) {
		goto badId;
	    }
	} else if ((argv[i][0] == 's') && (argv[i][1] == 't')
		   && (argv[i][2] == 'd')) {
	    if (strcmp(argv[i]+3, "in") == 0) {
		fd[i-1] = 0;
	    } else if (strcmp(argv[i]+3, "out") == 0) {
		fd[i-1] = 1;
	    } else if (strcmp(argv[i]+3, "err") == 0) {
		fd[i-1] = 2;
	    } else {
		goto badId;
	    }
	} else {
badId:
	    Tcl_AppendResult(interp, "dup bad file identifier \"", argv[i],
			     "\"", (char *) NULL);
	    return TCL_ERROR;
	}
    }

    if (dup2(fd[0], fd[1]) == -1) {
	Tcl_AppendResult(interp, "couldn't dup \"", argv[1], " onto ", argv[2],
			 "\": ", Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
    }

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Tns_strptimeCmd --
 *
 *	Implement the Tcl "ns_ptime" command.  This returns an
 *	integer (seconds since 00:00:00 UTC, January 1, 1970)
 *	that indicates the time of "timeval" which is, in turn,
 *	decoded using the "format" string. Be careful of daylight
 *	saving time or "dst".
 *
 *   -- A "isdst flag" positive value interprets "timeval" to be in
 *	  dst. This results in the subtraction of one hour if this
 *	  routine is called during main timezone time.
 *   -- A value of 0 forces main timezone time, and thus results in
 *	  the addition of one hour if called during dst.
 *   -- A negative value does NOT adjust the "timeval" time.
 *	  Ie., "timeval" is assumed to be the current dst status.
 *   -- With no given "isdst flag", a negative value is assumed.
 *
 *--------------------------------------------------------------
 */
Tns_strptimeCmd(clientData, interp, argc, argv)
ClientData clientData;         /* ignored */
Tcl_Interp *interp;            /* tcl interpreter */
int argc;                      /* Number of arguments */
char **argv;                   /* Arg list */
{
  struct tm tm;
  char *fmt;
  int dstFlag=-2;
  
  if ((argc == 1)||(argc > 4)) {
    Tcl_AppendResult(interp, argv[0], " <timeval> [<format>] [<isdst flag>]", NULL);
    return TCL_ERROR;
  }
  
  if (argc == 2) {
    fmt = "";
  } else {
    fmt = argv[2];
  }
  
  if (argc == 4) {
    dstFlag = atoi(argv[3]);
  }
    
  /* printf("ns_ptime: %s, format=%s, dstFlag=%d\n",argv[1],fmt,dstFlag); */
  strptime (argv[1], fmt, &tm);
  tm.tm_isdst = dstFlag;
  
  sprintf (interp->result, "%d", mktime(&tm));
  return TCL_OK;
}

#endif /* not VXWORKS */

/*
 *--------------------------------------------------------------
 *
 * Tns_SystimeCmd --
 *
 *	Implement the Tcl "ns_systime" command.  This returns a
 *	string that indicates the current value on the system clock
 *	as a double precision value.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
int
Tns_SystimeCmd (clientData, interp, argc, argv)
    ClientData clientData;         /* ignored */
    Tcl_Interp *interp;            /* tcl interpreter */
    int argc;                      /* Number of arguments */
    char **argv;                   /* Arg list */
{
#ifndef LINUX
    struct timespec tv;
#else 
    struct timeval tv;
#endif
    if (argc != 1) {
        Tcl_AppendResult(interp, argv[0], " doesn't take any args", NULL);
        return TCL_ERROR;
    }
#ifndef LINUX
    clock_gettime(0, &tv);
    sprintf (interp->result, "%d.%09d", tv.tv_sec, tv.tv_nsec);
#else 
    gettimeofday(&tv, NULL);
    sprintf (interp->result, "%d.%06d", tv.tv_sec, tv.tv_usec);
#endif

    return TCL_OK;
}

Tns_CtimeCmd (clientData, interp, argc, argv)
    ClientData clientData;         /* ignored */
    Tcl_Interp *interp;            /* tcl interpreter */
    int argc;                      /* Number of arguments */
    char **argv;                   /* Arg list */
{
  struct timeval tv;

  if (argc != 3) {
    Tcl_AppendResult(interp, argv[0], " <timeval> <format>", NULL);
    return TCL_ERROR;
  }

  sscanf (argv[1],"%d.%06d", &tv.tv_sec, &tv.tv_usec);
#ifdef VXWORKS68K51
  Tcl_AppendResult(interp, ctime (tv.tv_sec), NULL);
#else
  strftime (interp->result, 50, argv[2], localtime((time_t *) &tv.tv_sec));
#endif

  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Tns_SystimeCmd --
 *
 *	Implement the Tcl "ns_systime" command.  This returns a
 *	string that indicates the current value on the system clock
 *	as a double precision value.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
int
Tns_TrueSystimeCmd (clientData, interp, argc, argv)
ClientData clientData;         /* ignored */
Tcl_Interp *interp;            /* tcl interpreter */
int argc;                      /* Number of arguments */
char **argv;                   /* Arg list */
{
#ifndef LINUX
  struct timespec tv;
#else
  struct timeval tv;
#endif
  double true_time;
  if (argc != 1) {
    Tcl_AppendResult(interp, argv[0], " doesn't take any args", NULL);
    return TCL_ERROR;
  }
#ifndef LINUX
  (void) clock_gettime(0, &tv);
  true_time = tv.tv_sec + tv.tv_nsec/1000000000.0;  
#else
  gettimeofday(&tv,NULL);
  true_time = tv.tv_sec + tv.tv_usec/1000000.0;  
#endif
  sprintf (interp->result, "%f", true_time);
  
  return TCL_OK;
}

#ifndef VXWORKS

Tns_GidCmd (clientData, interp, argc, argv)
    ClientData clientData;         /* ignored */
    Tcl_Interp *interp;            /* tcl interpreter */
    int argc;                      /* Number of arguments */
    char **argv;                   /* Arg list */
{
  char tmp[20];
  sprintf(tmp,"%d",getgid());
  Tcl_AppendResult(interp, tmp, NULL);
  return TCL_OK;
}

Tns_UidCmd (clientData, interp, argc, argv)
    ClientData clientData;         /* ignored */
    Tcl_Interp *interp;            /* tcl interpreter */
    int argc;                      /* Number of arguments */
    char **argv;                   /* Arg list */
{
  char tmp[20];
  sprintf(tmp,"%d",getuid());
  Tcl_AppendResult(interp, tmp, NULL);
  return TCL_OK;
}
#ifndef NO_X11
Tns_codaSendCmd (clientData, interp, argc, argv)
    ClientData clientData;         /* ignored */
    Tcl_Interp *interp;            /* tcl interpreter */
    int argc;                      /* Number of arguments */
    char **argv;                   /* Arg list */
{
  return coda_send(Tk_Display(Tk_MainWindow(interp)),argv[1],argv[2]);
}

Tns_codaRCFrame (clientData, interp, argc, argv)
    ClientData clientData;         /* ignored */
    Tcl_Interp *interp;            /* tcl interpreter */
    int argc;                      /* Number of arguments */
    char **argv;                   /* Arg list */
{
  char tmp1[400];
  char tmp2[400];
  int wid;

  if (argc != 3) {
    Tcl_SetResult(interp,"usage : rcframe ?widget? ?name?",TCL_STATIC);
    return TCL_ERROR;
  }

  sprintf(tmp1,"%s_WINDOW",argv[2]);
  wid = CODAGetAppWindow(Tk_Display(Tk_MainWindow(interp)),tmp1);

  if (wid == 0) {
    int ix;

    sprintf(tmp1,"t:%d %s",getpid(),argv[2]);
    coda_send(Tk_Display(Tk_MainWindow(interp)),"RUNCONTROL",tmp1);

    for (ix = 0;ix<500;ix++) {
      sleep(1);
      sprintf(tmp1,"%s_WINDOW",argv[2]);
      wid = CODAGetAppWindow(Tk_Display(Tk_MainWindow(interp)),tmp1);
      if (wid) break;
    }
    if (wid == 0) {
      Tcl_SetResult(interp,"window not created",TCL_STATIC);
      return TCL_ERROR;
    }
  }
  sprintf(tmp2,"toplevel %s -use %d",argv[1],wid);
  
  return Tcl_Eval(interp,tmp2); 
}
#endif
#endif
/*
 *--------------------------------------------------------------
 *
 * Tns_Init --
 *
 *	Initialize the full Tcl-ISIS package.
 *
 * Results:
 *	None
 *
 * Side effects:
 *	ISIS related commands are bound to the interpreter.
 *
 *--------------------------------------------------------------
 */
int
Tns_Init (interp)
    Tcl_Interp *interp;
{
#ifndef VXWORKS
    Tcl_CreateCommand(interp, "ns_execvp", Tns_execvpCmd,
		      (ClientData) NULL, (void (*)()) NULL);
    Tcl_CreateCommand(interp, "ns_dup", Tns_DupCmd,
		      (ClientData) NULL, (void (*)()) NULL);
    Tcl_CreateCommand(interp, "ns_ptime", Tns_strptimeCmd,
		      (ClientData) NULL, (void (*)()) NULL);
    Tcl_CreateCommand(interp, "gid", Tns_GidCmd,
		      (ClientData) NULL, (void (*)()) NULL);
    Tcl_CreateCommand(interp, "uid", Tns_UidCmd,
		      (ClientData) NULL, (void (*)()) NULL);
# ifndef NO_X11

    Tcl_CreateCommand(interp, "coda_send", Tns_codaSendCmd,
		      (ClientData) NULL, (void (*)()) NULL);
    Tcl_CreateCommand(interp, "rcframe", Tns_codaRCFrame,
		      (ClientData) NULL, (void (*)()) NULL);
# endif
#endif
    Tcl_CreateCommand(interp, "ns_systime", Tns_SystimeCmd,
		      (ClientData) NULL, (void (*)()) NULL);
    Tcl_CreateCommand(interp, "ns_time", Tns_TrueSystimeCmd,
		      (ClientData) NULL, (void (*)()) NULL);
    Tcl_CreateCommand(interp, "ns_ctime", Tns_CtimeCmd,
		      (ClientData) NULL, (void (*)()) NULL);
    return TCL_OK;
}
