#@package: help help helpcd helppwd apropos

#==============================================================================
# help.tcl --
#     Tcl help command. (see Tcl shell manual)
#==============================================================================

#------------------------------------------------------------------------------
# Take a path name which might have . and .. elements and flatten them out.

proc help:flattenPath {pathName} {
    set newPath {}
    foreach element [split $pathName /] {
        if {"$element" == "."} {
           continue
        }
        if {"$element" == ".."} {
            if {[llength [join $newPath /]] == 0} {
                error "Help: name goes above subject directory root"}
            set newPath [lreplace $newPath [expr [llength $newPath]-1] [expr [llength $newPath]-1]]
            continue
        }
        lappend newPath $element
    }
    set newPath [join $newPath /]
    
    # Take care of the case where we started with something line "/" or "/."

    if {("$newPath" == "") && [string match "/*" $pathName]} {
        set newPath "/"}
        
    return $newPath
}

#------------------------------------------------------------------------------
# Take the help current directory and a path and evaluate it into a help root-
# based path name.

proc help:EvalPath {pathName} {
    global env

    if {"$pathName" == ""} {
	return $env(help:curDir)
    }
    set pathName $env(help:curDir)/$pathName
    set pathName [help:flattenPath $pathName]
    if {[string match "*/" $pathName] && ($pathName != ".")} {
        set pathName [string range $pathName 0 [expr [string length $pathName]-1]]}

    return $pathName    
}

#------------------------------------------------------------------------------
# Display a line of output, pausing waiting for input before displaying if the
# screen size has been reached.  Return 1 if output is to continue, return
# 0 if no more should be outputed, indicated by input other than return.
#

proc help:Display {line} {
    global env
    if {$env(help:lineCnt) >= 23} {
        set env(help:lineCnt) 0
        puts stdout ":" nonewline
        flush stdout
        gets stdin response
        if {$response != " " && $response != ""} {
            return 0}
    }
    puts stdout $line
    incr env(help:lineCnt)
}

#------------------------------------------------------------------------------
# Display a file.

proc help:DisplayFile {filepath} {

    set inFH [open $filepath r]
    while {[gets $inFH fileBuf] >= 0} {
        if {![help:Display $fileBuf]} {
            break}
    }
    close $inFH

}    

#------------------------------------------------------------------------------
# Procedure to return contents of a directory.  A list is returned, consisting
# of two lists.  The first list are all the directories (subjects) in the
# specified directory.  The second is all of the help files.  Eash sub-list
# is sorted in alphabetical order.
#

proc help:ListDir {dirPath} {
    set dirList {}
    set fileList {}
    if {[catch {set dirFiles [glob $dirPath/*]}] != 0} {
        error "No files in subject directory: $dirPath"}
    foreach fileName $dirFiles {
        if [file isdirectory $fileName] {
            lappend dirList "[file tail $fileName]/"
        } else {
            lappend fileList [file tail $fileName]
        }
    }
   return [list [lsort $dirList] [lsort $fileList]]
}

#------------------------------------------------------------------------------
# Display a list of file names in a column format. This use columns of 14 
# characters 3 blanks.

proc help:DisplayColumns {nameList} {
    set count 0
    set outLine ""
    foreach name $nameList {
        if {$count == 0} {
            append outLine "   "}
        append outLine $name
        if {[incr count] < 4} {
            set padLen [expr 17-[string length $name]]
            if {$padLen < 3} {
               set padLen 3}
            set outLine [format "$outLine%[expr $padLen]s" " "]
        } else {
           if {![help:Display $outLine]} {
               return}
           set outLine ""
           set count 0
        }
    }
    if {$count != 0} {
        help:Display $outLine}
    return
}


#------------------------------------------------------------------------------
# Help command main.

proc Help {{subject {}}} {
    global env

    set env(help:lineCnt) 0

    # Special case "help help", so we can get it at any level.

    if {($subject == "help") || ($subject == "?")} {
        help:DisplayFile "$env(help:root)/help"
        return
    }

    set request [help:EvalPath $subject]
    set requestPath "$request"

    if {![file exists $requestPath]} {
        error "Help:\"$request\" does not exist"}
    
    if [file isdirectory $requestPath] {
        set dirList [help:ListDir $requestPath]
        set subList  [lindex $dirList 0]
        set fileList [lindex $dirList 1]
        if {[llength $subList] != 0} {
            help:Display "\nSubjects available in $request:"
            help:DisplayColumns $subList
        }
        if {[llength $fileList] != 0} {
            help:Display "\nHelp files available in $request:"
            help:DisplayColumns $fileList
        }
    } else {
        help:DisplayFile $requestPath
    }
    return
}


#------------------------------------------------------------------------------
# Helpcd main.
#   
# The name of the new current directory is assembled from the current 
# directory and the argument.  The name will be flatten and any trailing
# "/" will be removed, unless the name is just "/".

proc helpcd {{dir /}} {
    global env

    set request [help:EvalPath $dir]
    set requestPath "$request"
    if {[string match "$request/*" $env(help:root)]} {
	puts "helpcd goes above help root, ignored"
	return
    }

    if {![file exists $requestPath]} {
        error "Helpcd: \"$request\" does not exist"}
    
    if {![file isdirectory $requestPath]} {
        error "Helpcd: \"$request\" is not a directory"}

    set env(help:curDir) $request
    puts "current directory now:  $request"
    return    
}

#------------------------------------------------------------------------------
# Helpcd main.

proc helppwd {} {
        global env
        puts "Current help subject directory: $env(help:curDir)"
}

#==============================================================================
#     Tcl apropos command.  (see Tcl shell manual)
#------------------------------------------------------------------------------

proc help {{manFile {}}} {
    global env
    if {$manFile != ""} {
	set fh [exec find $env(help:root) -name $manFile]
	if { $fh != ""} {
	    puts $fh
	    regsub $env(help:root) $fh {} manFile
	}
    }
    Help $manFile
}

#------------------------------------------------------------------------------
# One time initialization done when the file is sourced.
#
global env

#set env(help:root) [searchpath help]
set env(help:root) $env(CODA)/HP_UX/help
set env(help:curDir) $env(CODA)/HP_UX/help
set env(help:outBuf) {}
