#-----------------------------------------------------------------------------
#                                buildhelp.tcl
#-----------------------------------------------------------------------------
#
# Program to extract help files from TCL manual pages or TCL script files.
# The help directories are built as a hierarchical tree of subjects and help
# files.  
#
# For nroff man pages, the areas of text to extract are delimited with:
#
#     '@help: subjectdir/helpfile
#     '@endhelp
#
# start in column one. The text between these markers is extracted and stored
# in help/subjectdir/help.  The file must not exists, this is done to enforced 
# cleaning out the directories before help file generation is started, thus
# removing any stale files.  The extracted text is run through:
#
#     nroff -man|col -xb   {col -b on BSD derived systems}
#
# If there is other text to include in the helpfile, but not in the manual 
# page, the text, along with nroff formatting commands, may be included using:
#
#     '@:Other text to include in the help page.
#
# A entry in the brief file, used by apropos my be included by:
#
#     '@brief: Short, one line description
#
# These brief request must occur with in the bounds of a help section.
#
# If some header text, such as nroff macros, need to be preappended to the
# text streem before it is run through nroff, then that text can be bracketed
# with:
#
#     '@header
#     '@endheader
#
# If multiple header blocks are encountered, they will all be preappended.
#
# A similar construct is used for manual page name index generation:
#
#      ;@index: subject1 subjectN
#
# This is used by `installTcl' to generate the name index files.  There should
# be one per file, usuall right before the name line.  The subjects listed are
# all of the procedures or commands to link to the manual page, usually the
# same as on the .SH NAME line.
#
# For TCL script files, which are indentified because they end in ".tcl",
# the text to be extracted is delimited by:
#
#    #@help: subjectdir/helpfile
#    #@endhelp
#
# And brief lines are in the form:
#
#     #@brief: Short, one line description
#
# The only processing done on text extracted from .tcl files it to replace
# the # in column one with a space.
#
#
#-----------------------------------------------------------------------------
# 
# To run this program:
#
#   tcl buildhelp.tcl [-m mergeTree] [-i nameindex] helpDir file-1 file-2 ...
#
# o -m mergeTree is a tree of help code, plus a brief file to merge with the
#   help files that are to be extracted.  This will become part of the new
#   help tree.  Used to merge in the documentation from UCB Tcl.
# o -i nameindex is an name index file to create from the '@index markers in
#   the man files.
# o helpDir is the help tree root directory.  helpDir should  exists, but any
#   subdirectories that don't exists will be created.  helpDir should be
#   cleaned up before the start of manual page generation, as this program
#   will not overwrite existing files.
# o file-n are the nroff manual pages (.man) or .tcl or .tlib files to extract
#   the help files from.
#-----------------------------------------------------------------------------

#-----------------------------------------------------------------------------
# Truncate a file name of a help file if the system does not support long
# file names.  If the name starts with `Tcl_', then this prefix is removed.
# If the name is then over 14 characters, it is truncated to 14 charactes
#  
proc TruncFileName {pathName} {
    global G_truncFileNames

    if {!$G_truncFileNames} {
        return $pathName}
    set fileName [file tail $pathName]
    if {[regexp "^Tcl_" $fileName]} {
        set fileName [string range $fileName 4 end]}
    set fileName [string range $fileName 0 13]
    return "[file dirname $pathName]/$fileName"
}

#-----------------------------------------------------------------------------
# Proc to ensure that all directories for the specified file path exists,
# and if they don't create them.

proc EnsureDirs {filePath} {
    set dirPath [file dirname $filePath]
    if {![file exists $dirPath]} {
        exec mkdir -p $dirPath}
}


#-----------------------------------------------------------------------------
#
# Proc to extract nroff text to use as a header to all pass to nroff when
# processing a help file.
#    manPageFH - The file handle of the manual page.
#

proc ExtractNroffHeader {manPageFH} {
    global nroffHeader
    while {[gets $manPageFH manLine] >= 0} {
        if {[string first "'@endheader" $manLine] == 0} {
            break;
            }
        if {[string first "'@:" $manLine] == 0} {
            set manLine [string range $manLine 3 end]
            }
        append nroffHeader "$manLine\n"
        }
}

#-----------------------------------------------------------------------------
#
# Proc to extract a nroff help file when it is located in the text.
#    manPageFH - The file handle of the manual page.
#    manLine - The '@help: line starting the data to extract.
#

proc ExtractNroffHelp {rootName helpName manPageFH line } {
    global G_helpDir nroffHeader G_briefHelpFH G_colArgs foundBrief

    set helpFile [TruncFileName "$G_helpDir/$rootName/$helpName"]
    EnsureDirs $helpFile
    set helpFH [open "| nroff -man | col $G_colArgs > $helpFile" w]
    puts "    creating help file $helpName"

    # Nroff commands from .TH macro to get the formatting right.  The `\n'
    # are newline separators to output, the `\\n' become `\n' in the text.
        
    puts $helpFH ".ad b\n.PD\n.nrIN \\n()Mu\n.nr)R 0\n.nr)I \\n()Mu"
    puts $helpFH ".nr)R 0\n.\}E\n.DT\n.na\n.nh"
    puts $helpFH $nroffHeader
    puts $helpFH ".SH NAME"
    puts $helpFH $line    

    while {[gets $manPageFH manLine] >= 0} {
	puts $helpFH $manLine    
    }
    close $helpFH
    exec chmod a+r $helpFile
}

#-----------------------------------------------------------------------------
#
# Proc to extract a tcl script help file when it is located in the text.
#    ScriptPageFH - The file handle of the .tcl file.
#    ScriptLine - The #@help: line starting the data to extract.
#

proc ExtractScriptHelp {ScriptPageFH ScriptLine} {
    global G_helpDir G_briefHelpFH
    set helpName [string trim [string range $ScriptLine 7 end]]
    set helpFile "$G_helpDir/$helpName"
    if {[file exists $helpFile]} {
        error "Help file already exists: $helpFile"}
    EnsureDirs $helpFile
    set helpFH [open $helpFile w]
    puts "    creating help file $helpName"
    set foundBrief 0
    while {[gets $ScriptPageFH ScriptLine] >= 0} {
        if {[string first "#@endhelp" $ScriptLine] == 0} {
            break;
        }
        if {[string first ".SH SYNOPSYS" $ScriptLine] == 0} {
            if $foundBrief {
                error {Duplicate ".SH SYNOPSYS" entry"}
            }
            set foundBrief 1
	    puts $G_briefHelpFH "$helpName\t[string range $ScriptLine 8 end]"
            continue;
        }
        if {[string first "#@help" $ScriptLine] == 0} {
            error {"#@help" found within another help section"}
        }
        if {[clength $ScriptLine] > 1} {
            set ScriptLine " [string range $ScriptLine 1 end]"
        } else {
            set ScriptLine ""
        }
        puts $helpFH $ScriptLine
        }
    close $helpFH
    exec chmod a+r $helpFile
}

#-----------------------------------------------------------------------------
#
# Proc to scan a nroff manual file looking for the start of a help text
# sections and extracting those sections.
#    pathName - Full path name of file to extract documentation from.
#

proc ProcessNroffFile {rootName pathName} {
    global  nroffHeader G_briefHelpFH
    set helpName [file rootname [file tail $pathName]]
    set fileName [file tail $pathName]
    
    set nroffHeader {}
    set manPageFH [open $pathName r]
    puts "    scanning nroff file $pathName"
    set matchInfo(fileName) [file tail $pathName]

    while {[gets $manPageFH line] >= 0} {
	if {[regexp "^\.SH NAME$" $line]} {
	    gets $manPageFH line
	    ExtractNroffHelp $rootName $helpName $manPageFH $line
	}	
    }
    close $manPageFH
}

#-----------------------------------------------------------------------------
#
# Proc to scan a Tcl script file looking for the start of a
# help text sections and extracting those sections.
#    pathName - Full path name of file to extract documentation from.
#

proc ProcessTclScript {pathName} {
    global nroffHeader
    
    set scriptFH [open "$pathName" r]
    
    puts "    scanning tcl file $pathName"
    set matchInfo(fileName) [file tail $pathName]
        
    while {[gets $scriptFH line] >= 0} {
	if { [regexp "^#@help:" $line] } {
	     ExtractScriptHelp $scriptFH $line
	}
    }
    
    close $scriptFH
}

#-----------------------------------------------------------------------------
# Proc to copy the help merge tree, excluding the brief file.
# 

proc CopyMergeTree {helpDirPath mergeTree} {
    if {"[cindex $helpDirPath 0]" != "/"} {
        set helpDirPath "[pwd]/$helpDirPath"
    }
    set oldDir [pwd]
    cd $mergeTree

    set curHelpDir "."

    for_recursive_glob mergeFile {.} {
        if {"$mergeFile" == "./brief"} {
            continue}
            set helpFile "$helpDirPath/$mergeFile"
        if {[file isdirectory $mergeFile]} {
            if ![file exists $helpFile] {
                exec mkdir $helpFile
                exec chmod a+rx $helpFile
            }
        } else {
            if {[file exists $helpFile]} {
                error "Help file already exists: $helpFile"}
            set inFH [open $mergeFile r]
            set outFH [open $helpFile w]
            copyfile $inFH $outFH
            close $outFH
            close $inFH
            exec chmod a+r $helpFile
        }
    }
    cd $oldDir
}

proc ProcessFiles {f} {
    while {[gets $f  manFile] >=0 } {
	puts $manFile
        set manFile [glob $manFile]
	regsub \./ $manFile {} manFile
	regsub /.* $manFile {} rootName 
	set status [catch {ProcessNroffFile $rootName $manFile} msg]
        if {$status != 0} {
            puts "Error extracting help from: $manFile"
            puts $msg
            global errorInfo
        }
    }
}

proc RemoveFiles {f} {
    while {[gets $f  manFile] >=0 } {
	global G_helpDir
        set manFile [glob $manFile]
	regsub \./ $manFile {} manFile
	regsub /.* $manFile {} rootName 
	set helpName [file rootname [file tail $manFile]]
	set helpFile [TruncFileName "$G_helpDir/$rootName/$helpName"]
	puts $helpFile
	catch {exec rm -f $helpFile    }
    }
}

proc RemoveAll {helpDirPath} {
    global G_helpDir G_truncFileNames
    set G_truncFileNames 0
    set G_helpDir [glob $helpDirPath]

    set fh [open "| find . -name *.man" r]
    RemoveFiles $fh
    close $fh

    set fh [open "| find . -name *.3" r]
    RemoveFiles $fh
    close $fh

    set fh [open "| find . -name *.2" r]
    RemoveFiles $fh
    close $fh

    set fh [open "| find . -name *.1" r]
    RemoveFiles $fh
    close $fh

    set fh [open "| find . -name *.n" r]
    RemoveFiles $fh
    close $fh
}

#-----------------------------------------------------------------------------
# GenerateHelp: main procedure.  Generates help from specified files.
#    helpDirPath - Directory were the help files go.
#    mergeTree - Help file tree to merge with the extracted help files.
#    manIndexFile - Manual page index file to build, or {} to not build one.
#    sourceFiles - List of files to extract help files from.

proc GenerateHelp {helpDirPath} {
    global G_helpDir G_truncFileNames G_manIndexFH 
    global G_briefHelpFH G_colArgs

    puts ""
    puts "Begin building help tree"

    # Determine version of col command to use (no -x on BSD)
        set G_colArgs {-b}

    set G_helpDir $helpDirPath

    set status [catch {set tmpFH [open xxx $G_helpDir/AVeryVeryBigFileName w]}]
    if {$status != 0} {
        set G_truncFileNames 1
    } else {
        close $tmpFH
        unlink $G_helpDir/AVeryVeryBigFileName
        set G_truncFileNames 0
    }

    set G_manIndexFH {}

    set fh [open "| find . -name *.man" r]
    ProcessFiles $fh
    close $fh

    set fh [open "| find . -name *.3" r]
    ProcessFiles $fh
    close $fh

    set fh [open "| find . -name *.2" r]
    ProcessFiles $fh
    close $fh

    set fh [open "| find . -name *.1" r]
    ProcessFiles $fh
    close $fh

    set fh [open "| find . -name *.n" r]
    ProcessFiles $fh
    close $fh

    if {$G_manIndexFH != ""} {
        close $G_manIndexFH
    }
    puts "*** completed extraction of all help files"
}

#-----------------------------------------------------------------------------
# Print a usage message and exit the program
proc Usage {} {
    puts {Wrong args: [-m mergetree] [-i index] helpdir manfile1 [manfile2..]}
    exit 1
}

#-----------------------------------------------------------------------------
# Main program body, decides if help is interactive or batch.

if { $tcl_interactive } {
    puts "To extract help, use the command:"
    puts "  GenerateHelp helpDir"
} else {
    set mergeTree {}
    set manIndexFile {}
    
    GenerateHelp [lindex $argv 0] 
   exit
}
