
/*
 * fifo.c   :  This file contains all the fifo routines that operate 
 *          :  on individual fifos (int fifo_xxx(int which_fifo,...)
 * Author   :  C.Witzig
 * Date     :  Apr 1,  1992 (no joke!)
 * Mods     :  Dec 12, 1994: added prescale option to fifo (routines
 *                           get_next_fifo, fifo_make and
 *                           fifo_header_print)
 *          :  Feb 21, 1995: add buf into shmctl in order to
 *                           comile on HP
 *          :  Mar 30, 1995: replace conditions in get_next_fifo
 *                           with macros that are defined in 
 *                           $ONLINE_INCLUDE/dd_fifo_conditions.h
 *          :  Apr 3,  1995: add fifo_sys_header before the fifos in shm
 *          :  Sep 1,  1995: add (un)-lock_input_fifo to allow coordination
 *                           between processes
 *          :  Feb 15, 1996: add fifo_exist(char *fifoname)
 *
 *
 */

/* 
 * This file has the all the relevant fifo routines (way too  many!!!) 
 * and consists of the following parts:
 * 1. SYSTEM CALLS using standard unix (no IRIX extensions)
 *                 for - locking and unlocking the fifos using semaphores
 *                     - semaphore operations
 *                     - shared memory operations.
 * 2. USER CALLS   for writing/reading initialising fifos and getting 
 *                 informations about them.
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <time.h>

#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>
#include <sys/signal.h>

#include <db.h>
#include <fifo.h>

#include <dd_log.h>
#include <dd_filter.h>
#include <dd_fifo_conditions.h>

#define TRUE 1
#define FALSE 0

/*
 * Important variables for the fifos:
 * fifos[MAX_FIFO_NB]   : the fifos !!!
 * my_own_fifo          : the fifo to which the process is 
 *                      : attached to. This is the default 
 *                      : fifo from where you will read the 
 *                      : data.
 * fifo_shmid, *fifo_shm_start: Id and pointer to the shared
 *                        memory where the fifos are.
 * fifo_semid           : semaphores are used as counter for
 *                        the fifo contents (not to be con-
 *                        fused with p->cnt!)
 * fifo_lock_semid      : semaphore for locking the fifos.
 *                        (note the special use of the 
 *                        MAX_FIFO_NB+1 semaphore for locking
 *                        the creation of new fifos.
 */

static int nevoldin[MAX_FIFO_NB_MAX], nevoldout[MAX_FIFO_NB_MAX];

a_fifo fifos[MAX_FIFO_NB_MAX];
int my_own_fifo = -1;

int fifo_shmid;
a_fifo *fifo_shm_start;
fifo_sys_header *fifo_sys_start;

int fifo_semid, fifo_lock_semid;

/*
 * we have these shm variables in 
 * local memory in order to simply their access
 */
int fifo_depth;
int max_fifo_nb;

/* 
 * For initialisation purposes...
 */
static int silly_variable = 0x69696969;

static struct fifo_entry fev_init = {
  -1,-1,&silly_variable,-1,-1,-1,-1,-1};

static char *fifo_sys_status[] = {"UNUSED","INITIALISED","ERROR"};

static struct fifo_header fhdr_init = {
  -1,
  "------",
  "---",
  FSTAT_UNUSED, 
  FMODE_ALL, 
  -1, -1, -1, -1, 0, FWAIT_SLEEP, FIFO_UNPROTECTED, -1, -1, -1};

/* corresponds to the FSTAT_xxx, FMODE_xxx and FWAIT_xxx in fifo.h. 
   Be careful with fifo_waitmodes as this index does not agree with 
   the index for FSTAT_ASYNC (being IPC_NOWAIT). */
static char *fifo_states[3] = {
  "UNUSED", "  IDLE", "ACTIVE"};
static char *fifo_modes[4] = {
  "ALL  ", "COND ", "ONREQ", "NULL"};
static char *fifo_waitmodes[2] = {
  "SLEEP", "ASYNC"};
static char *fifo_protection[] = {
  "UNPROTECTED","PROTECTED  "};
static char *suser_mode[] = {
  "MULTI ", "SINGLE"};

#ifdef sun
union {
  int val;
  struct semid_ds *buf;
  ushort *array;
}fifo_arg;

union {
  int val;
  struct semid_ds *buf;
  ushort *array;
}fifo_larg;

#else 

union semun fifo_arg, fifo_larg;

#endif

static ushort array[MAX_FIFO_NB_MAX+1];

/*************************************************************************/
/*                                                                       */
/* ********************** SYSTEM CALLS ********************************* */
/*                                                                       */
/*************************************************************************/

/*
-------------------------------------------------------------------------
                 LOCKING AND UNLOCKING THE FIFOs
-------------------------------------------------------------------------
*/
  
static int fifo_lock(int which_fifo)
/*
 * Locking mechanism must ignore interrupting system call, in order to
 * guarantee the integrity of the fifo content. (If signal arrives while
 * waiting for unlocked fifo, this looks like a signal anywhere in 
 * non sleeping code.)
 */
 
/* Grab the lock sem associated with a particular fifo. Wait for it.
 * All fifo info is stored in shared mem at "fifo_shm_start", but
 * each Nth fifo's chunk (fifo_sm_start + N) is separately protected.
 * CTCTCT
 */
{
  int waiting;
  struct sembuf fifo_sembuf;

  fifo_sembuf.sem_num = which_fifo;
  fifo_sembuf.sem_op  = -1;
  fifo_sembuf.sem_flg = SEM_UNDO;

  waiting = 1;
  while ( waiting > 0 ){ 
    if ( (waiting = semop(fifo_lock_semid,&fifo_sembuf,1)) != 0 ){
      if ( errno == EINTR ){
	waiting = 1;
      }
      else{
	sprintf(dd_log_msg," fifo_lock: semop %s \n",strerror(errno));
	dd_log(DD_ERROR, dd_log_msg);
	return -1;
      }
    }
  }
  return 0;
}


static int fifo_unlock(int which_fifo)
{
  struct sembuf fifo_sembuf;

  fifo_sembuf.sem_num = which_fifo;
  fifo_sembuf.sem_op  = 1;
  fifo_sembuf.sem_flg = SEM_UNDO;

  if (semop(fifo_lock_semid,&fifo_sembuf,1) != 0){
    sprintf(dd_log_msg," fifo_unlock: semop %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  return 0;
}

/* 
 * Whenever a process wants to add/remove fifos, it must be
 * garanteed, that only one process does this at a time.
 * The following two calls lock/unlock the modification
 * semaphore, which is the last semaphore in the lock
 * semaphore set.
 */

int fifo_mod_lock()
{
  fifo_lock(max_fifo_nb);
  return 0;
}

int fifo_mod_unlock()
{
  fifo_unlock(max_fifo_nb);
  return 0;
}


/* 
-----------------------------------------------------------------------
                SEMAPHORE OPERATIONS FOR THE FIFOS 
-----------------------------------------------------------------------
*/


static int fifo_init_sem(char *filename, int nfifos)
/*
 * Initialises the fifo semaphores by
 * attaching to them. If this fails, it creates and
 * initialises them.
 * argument: system name
 */
{
  int i;
  int sflag;
  int creat_insem, creat_locksem;
  key_t key;

  sflag = 0666;
  creat_insem = 0;
  key = ftok(filename, FIFO_SEM_KEY);
  if ( key == -1 ) {
    perror(filename);
    return -1;
  }
  if ( (fifo_semid = semget(key,nfifos,sflag)) < 0 ){
    creat_insem = 1;
    sflag = 0666 | IPC_CREAT;
    if ( (fifo_semid = semget(key,nfifos,sflag)) < 0 ){
      sprintf(dd_log_msg," fifo_init_sem: semget creating input  %s \n",strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
      return(-1);
    }
  }

  i = nfifos + 1;
  sflag = 0666;
  creat_locksem = 0;
  key = ftok(filename, FIFO_SEM_KEY_LOCK);
  if ( (fifo_lock_semid = semget(key,i,sflag)) < 0 ){
    creat_locksem = 1;
    sflag = 0666 | IPC_CREAT;
    if ( (fifo_lock_semid = semget(key,i,sflag)) < 0 ){
      sprintf(dd_log_msg," fifo_init_sem: semget creating input lock %s \n",strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
      return(-1);
    }
  }

  if (creat_insem) {
    fifo_arg.array = array;

    fifo_arg.array[0] = 0;
    for (i=1;i<nfifos;i++)
      fifo_arg.array[i] = 0;
    if (semctl(fifo_semid,0,SETALL,fifo_arg) < 0){
      sprintf(dd_log_msg," fifo_init_sem: semctl(SETALL) %s \n",strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
      return(-1);
    }
  }
  if (creat_locksem){
    fifo_larg.array = array;

    for (i=0;i<nfifos+1;i++)
      fifo_larg.array[i] = 1;
    if (semctl(fifo_lock_semid,0,SETALL,fifo_larg) < 0){
      sprintf(dd_log_msg," fifo_init_sem: lock semctl(SETALL) %s \n",strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
      return(-1);
    }
  }

  return 0;
}

static int fifo_close_sem(int idestroy)
{
  if (idestroy){
    if (semctl(fifo_semid,0,IPC_RMID,0) < 0){
      sprintf(dd_log_msg," fifo_close_sem: fifo_semid %s \n",strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
     return -1;
    }
    if (semctl(fifo_lock_semid,0,IPC_RMID,0) < 0){
      sprintf(dd_log_msg," fifo_close_sem fifo_lock_semid %s \n",strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
  }
  return 0;
}

/* give up sem CTCTCT */
static int fifo_incr_semcnt(int which_fifo)
{
  struct sembuf fifo_sembuf;

  fifo_sembuf.sem_num = which_fifo;
  fifo_sembuf.sem_op  = 1;
  fifo_sembuf.sem_flg = 0;
  if (semop(fifo_semid,&fifo_sembuf,1) != 0){
    sprintf(dd_log_msg," fifo_incr_semcnt: semop %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  return 0;
}

/* take sem CTCTCT */
static int fifo_decr_semcnt(int which_fifo, int sem_flg)
{
  struct sembuf fifo_sembuf;

  fifo_sembuf.sem_num = which_fifo;
  fifo_sembuf.sem_op  = -1;
  fifo_sembuf.sem_flg = sem_flg;
  if (semop(fifo_semid,&fifo_sembuf,1) != 0){
	 if ( (errno == EINTR) || (errno == EAGAIN) )
		return errno;
	 else{
		sprintf(dd_log_msg," fifo_decr_semcnt: semop %s \n",strerror(errno));
		dd_log(DD_ERROR, dd_log_msg);
		return -1;
	 }
  }
  return 0;
}

static int fifo_get_semcnt(int *semcnt)
{
  int i,len;
  struct semid_ds fifo_semid_ds;

  fifo_arg.buf = &fifo_semid_ds;
  if (semctl(fifo_semid,0,IPC_STAT,fifo_arg) != 0){
    sprintf(dd_log_msg," fifo_get_semcnt: semctl(IPC_STAT) %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  len = fifo_arg.buf->sem_nsems;
  fifo_arg.array = array;
  if (semctl(fifo_semid,0,GETALL,fifo_arg) != 0){
    sprintf(dd_log_msg," fifo_get_semcnt: semctl(GETALL) %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  for (i=0;i<len;i++)
    semcnt[i] = fifo_arg.array[i];
  return 0;
}

static int fifo_get_lsemcnt(int *semcnt)
{
  int i,len;
  struct semid_ds fifo_semid_ds;

  fifo_larg.buf = &fifo_semid_ds;
  if (semctl(fifo_lock_semid,0,IPC_STAT,fifo_larg) != 0){
    sprintf(dd_log_msg," fifo_get_lsemcnt: semctl(IPC_STAT) %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  len = fifo_larg.buf->sem_nsems;
  fifo_larg.array = array;
  if (semctl(fifo_lock_semid,0,GETALL,fifo_larg) != 0){
    sprintf(dd_log_msg," fifo_get_lsemcnt: semctl(GETALL) %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  for (i=0;i<len;i++)
    semcnt[i] = fifo_larg.array[i];
  return 0;
}

static int fifo_set_semcnt(int which_sem, int sem_cnt)
{
  fifo_arg.val = sem_cnt;
  if (semctl(fifo_semid,which_sem,SETVAL,fifo_arg) != 0){
    sprintf(dd_log_msg," fifo_set_semcnt: semctl(SETVAL) %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  return 0;
}

int fifo_get_asemcnt(int which_sem, int *sem_cnt)
{
  int i;

  if ( (i = semctl(fifo_semid,which_sem,GETVAL,0)) < 0 ) {
    sprintf(dd_log_msg," fifo_get_asemcnt: semctl(GETVAL) %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  *sem_cnt = i;
  return 0;
}


/* 
-----------------------------------------------------------------------
            SHARED MEMORY OPERATIONS FOR THE FIFOS 
-----------------------------------------------------------------------
*/

static int fifo_init_shm(char *filename, int nfifos, int fdepth)
/*
 * Attaches to the fifo shared memory. If it doesn't exist, it
 * creates them. 
 * argument: filename == fifo system name
 */
{
  int size;
  int i;
  int ishmflag;
  key_t key;
  static int first = 1;
  int created = 0;

  if ( first ) {
    if ( nfifos > MAX_FIFO_NB_MAX ) {
      sprintf(dd_log_msg,"fifo_init_shm: nfifos = %d > MAX_FIFO_NB_MAX = %d - set to MAX_FIFO_NB_MAX \n",nfifos,MAX_FIFO_NB_MAX);
      dd_log(DD_ERROR, dd_log_msg);
     nfifos = MAX_FIFO_NB_MAX;
    }
    if ( fdepth > FIFO_DEPTH_MAX ) {
      sprintf(dd_log_msg,"fifo_init_shm: fdepth = %d > FIFO_DEPTH_MAX = %d  - set to FIFO_DEPTH_MAX \n",fdepth,MAX_FIFO_NB_MAX);
      dd_log(DD_ERROR, dd_log_msg);
      fdepth = FIFO_DEPTH_MAX;
    }

    size = sizeof(struct a_fifo)*nfifos + sizeof(fifo_sys_header);

    ishmflag = 0666;
    key = ftok(filename,FIFO_SHM_KEY);
    if ( key == -1 ) {
      perror(filename);
      return -1;
    }

    if((fifo_shmid = shmget(key,size,ishmflag)) < 0) {
      created = 1;
      ishmflag = 0666 | IPC_CREAT;
      if((fifo_shmid = shmget(key,size,ishmflag)) < 0) {
	sprintf(dd_log_msg,"fifo_init_shm: shmget createing %s \n",strerror(errno));
	dd_log(DD_ERROR, dd_log_msg);
	return -1;
      }
    }
    
    if ( (fifo_sys_start = (fifo_sys_header*) shmat(fifo_shmid,0,0) ) == NULL) {
      sprintf(dd_log_msg,"fifo_init_shm: shmat %s \n",strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }

    fifo_shm_start = (a_fifo*) (fifo_sys_start + 1);

    if (created){ 
      strcpy(fifo_sys_start->sys_name,filename);
      fifo_sys_start->pid_creat = getpid();
      fifo_sys_start->time_creat = time(NULL);
      fifo_sys_start->status = FIFO_SYS_INITIALISED;
      fifo_sys_start->max_fifo_nb = nfifos;
      fifo_sys_start->fifo_depth = fdepth;
      for (i=0;i<fifo_sys_start->max_fifo_nb;i++)
	fifo_init(i);
    }
    else{
      if ( ( nfifos > 0 ) && ( nfifos != fifo_sys_start->max_fifo_nb ) ) {
	  sprintf(dd_log_msg,"fifo_init_shm: fifo_init_shm: number of fifos mismatch: %d %d - attach to old value \n", 
		  nfifos, fifo_sys_start->max_fifo_nb);
	  dd_log(DD_ERROR, dd_log_msg);
	}
      if ( (fdepth > 0) &&  (fdepth != fifo_sys_start->fifo_depth) ) {
	sprintf(dd_log_msg,"fifo_init_shm: fifo depth mismatch: %d %d attach to old value \n", 
	       fdepth, fifo_sys_start->fifo_depth);
	dd_log(DD_ERROR, dd_log_msg);
      }
    }
    /* first = 0; GHGH*/
    fifo_depth = fifo_sys_start->fifo_depth;
    max_fifo_nb = fifo_sys_start->max_fifo_nb;
  }
  return 0;
}

static int fifo_close_shm(int idestroy)
{
  struct shmid_ds *buf;

  if (idestroy)
    if (shmctl(fifo_shmid,IPC_RMID, buf) < 0){
      sprintf(dd_log_msg,"fifo_close_shm: shmctl %s \n",strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
     return -1;
    }
  else
    if (shmdt(fifo_sys_start) < 0){
      sprintf(dd_log_msg,"fifo_close_shm: shmdt %s \n",strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
  return 0;
}


/*************************************************************************/
/*                                                                       */
/* **********************  USER  CALLS  ******************************** */
/*                                                                       */
/*************************************************************************/


int fifo_write(int which_fifo, struct fifo_entry fevent)
{
  struct a_fifo *pfifo;
  
  fifo_lock(which_fifo); 
 
  pfifo = fifo_shm_start + which_fifo;
  if (pfifo->cnt ==  fifo_depth) {
    sprintf(dd_log_msg," fifo_write: VERY SERIOUS FATAL FIFO ERROR fifo %i\n",
				which_fifo);
    dd_log(DD_ERROR, dd_log_msg);
    sprintf(dd_log_msg," fifo_write: REINITIALISE DD  \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  pfifo->fentry[pfifo->wrptr] = fevent;
  pfifo->wrptr = (++pfifo->wrptr)%fifo_depth;
  pfifo->cnt++;
  pfifo->nopin++;

  if (fifo_incr_semcnt(which_fifo) != 0)
    return -1;

  if (fifo_unlock(which_fifo) !=0)
    return -1;

  return 0;

}


int fifo_read(int which_fifo, struct fifo_entry *p2fevent, int mode)
/*
 * Reads the next fifo_entry from the fifo which_fifo.
 * Return code -1: some very serious error
 *             >0: no available pointer in the fifo or interrupted system
 *                 call while waiting for the next event. (the variable
 *                 errno is returned:
 *                 errno = EAGAIN: IPC_NOWAIT flag is set 
 *                                 (i.e. fifo in async mode)
 *                 errno = EINTR : interrupted system call 
 *                                 (i.e. signal arrived while waiting).
 */
{
  struct fifo_entry *ptr,fevent;
  int status, sem_flg;
  struct a_fifo *pfifo;

  ptr = NULL;
  pfifo = fifo_shm_start + which_fifo;
  ( mode >= 0 ) ? (sem_flg = mode) : (sem_flg = pfifo->fhdr.fwait_mode);

  if ( ( status = fifo_decr_semcnt(which_fifo, sem_flg) ) != 0) {
    if ( status < 0 ) {
      sprintf(dd_log_msg," fifo_read: error in fifo_decr_semcnt %s \n",strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
    }
    return status;
  }

  fifo_lock(which_fifo); 

  if (pfifo->cnt != 0){
    ptr = &pfifo->fentry[pfifo->rdptr];
    fevent = *ptr;
    /* rdptr is next event to be read CTCTCT */
    pfifo->rdptr = (++pfifo->rdptr)%fifo_depth; 
    /* cnt is # events left to be read CTCTCT */
    pfifo->cnt--;
    pfifo->nopout++;
  }
  else{
    sprintf(dd_log_msg," proc %i: fifo_read: attempt to read empty fifo %i\n",
	    getpid(),which_fifo);
    dd_log(DD_ERROR, dd_log_msg);
    if (fifo_unlock(which_fifo) != 0 )
      return -1;
    *p2fevent = fev_init;
    return 1;
  }

  if (fifo_unlock(which_fifo) !=0 )
    return -1;

  *p2fevent = fevent;

  return 0;
}

int fifo_attach(char *filename, int nfifos, int fdepth)
/*
 * Attaches the fifo system to the process that calls it. 
 * If the sem and shm don't exists, creates them 
 * Argument: system name, nb of fifos and fifo depth.
 * We have to initialize  first sem and then shm, but
 * the check whether the arguments are the same as in 
 * shm buffer is done later.
 */
{
  static int first = TRUE;

  if ( ! first )
    return 0;

  if ( fifo_init_sem(filename, nfifos) < 0 ) 
    return -1 ;

  if ( fifo_init_shm(filename, nfifos, fdepth) < 0 ) 
    return -1;

  /*first = FALSE; GHGH*/
  return 0;
}


int fifo_init(int which_fifo)
/*
 * Initialises the fifos (header and body) and deletes any connection 
 * to previous assignments.
 */
{
  int i,ii;
  int newsemcnt;
  struct a_fifo *p;
  char *pp;

  fifo_lock(which_fifo);

  nevoldin[which_fifo] = nevoldout[which_fifo] = 0;

  newsemcnt = 0;
  fifo_set_semcnt(which_fifo,newsemcnt);

  p = fifo_shm_start + which_fifo;
  p->fhdr = fhdr_init;
  p->cnt = NULL;
  p->wrptr = NULL;
  p->rdptr = NULL;
  p->noptry = NULL;
  p->nopin = NULL;
  p->nopout = NULL;
  for (ii = 0;ii<fifo_depth;ii++)
    p->fentry[ii] = fev_init;

  fifo_unlock(which_fifo);

  return 0;
}


int fifo_make(char *fname, char *ftype, struct fifo_mode fmode)
/*
 * Makes a fifo for a process. The return value is either the fifo number,
 * to which the process must refer to in subsequent calls (my_own_fifo) 
 * or - when < 0 - signals an error code.
 * The character ftype is either "STA" or "DYN". If it is "STA" and ends with a 
 * number, then this number is interpreted as the requested fifo number.
 * If the requested fifo is already in use  then an error code is returned.
 * If successful, the fifo is initialised and set into IDLE mode.
 * (Note: We must loop over all fifos.)
 */
{
  int i, icnt, free_fifo,this_fifo;
  char fifo_type[4];
  struct a_fifo *p;

  /* fifo_mod_lock() = fifo_lock(max_fifo_nb) CTCTCT */
  /* Locking mechanism must ignore interrupting system call, in order to
   * guarantee the integrity of the fifo content. (If signal arrives while
   * waiting for unlocked fifo, this looks like a signal anywhere in 
   * non sleeping code.) CTCTCT 
   */
  fifo_mod_lock();

  this_fifo = -1;
  free_fifo = -1;
  
  /* if fifo is static ...  CTCTCT */
  if ( strstr(ftype,"STA") != NULL ){
    sscanf(ftype,"%3s%i",fifo_type,&free_fifo);
    p = fifo_shm_start + free_fifo;
    /* if fifo name at shm != init name ... */
    if ( strcmp(p->fhdr.fname,fhdr_init.fname) != 0 ){
      sprintf(dd_log_msg,"fifo_make: requested fifo already in use\n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
  }
  /* if fifo is regular ...  CTCTCT */
  else{
    strcpy(fifo_type, ftype );
    for (i=0;i<max_fifo_nb;i++){
      p = fifo_shm_start + i;
     /* if fifo name = requested name ...  CTCTCT */
     if (strcmp(p->fhdr.fname,fname) == 0){
	if ( (p->fhdr.fsuser == FMODE_SINGLE_USER) && (p->fhdr.fnproc_att > 0) ) {
	  sprintf(dd_log_msg,"fifo_make: requested fifo already in use in SINGLE USER mode\n");
	  dd_log(DD_ERROR, dd_log_msg);
	  return -1;
	}
	this_fifo = i;
	p->fhdr.fnproc_att =  p->fhdr.fnproc_att++;
      }
      /* finds the first free fifo */
      if (( free_fifo == -1) && (p->fhdr.fstatus == FSTAT_UNUSED))
	free_fifo = i;
    }
  }
  
  /* if no existing multi-user fifo with that name is found ... */
  if (this_fifo == -1){
    /* if free fifo found */
    if ( free_fifo != -1 ){
      this_fifo = free_fifo;
      /* get sem count */
      if ( fifo_get_asemcnt(this_fifo,&icnt) < 0){
	sprintf(dd_log_msg,"fifo_make: error in fifo_get_asemcnt \n");
	dd_log(DD_ERROR, dd_log_msg);
	free_fifo = -1;
      }
      if ( icnt != 0){
	sprintf(dd_log_msg,"fifo_make: creating new fifo which is nonempty! \n");
	dd_log(DD_ERROR, dd_log_msg);
	free_fifo = -1;
      }
      else{
	p = fifo_shm_start + this_fifo;
	p->fhdr.fnumber = this_fifo;
	strcpy(p->fhdr.fname,fname);
	strcpy(p->fhdr.ftype,fifo_type); 
	p->fhdr.fstatus = FSTAT_IDLE;
	p->fhdr.fnproc_att = 1;
	p->fhdr.fpidcreat = getpid();
	if ( fmode.mode == FMODE_ONREQ )
	  p->fhdr.freqcnt = 1;
	else
	  p->fhdr.freqcnt = -1;
	p->fhdr.fmode = fmode.mode;
	p->fhdr.fwait_mode = fmode.wait;
	p->fhdr.fprescale = fmode.prescale;
	p->fhdr.fsuser = fmode.suser;
	p->fhdr.fctlw1 = *fmode.p2ctl++;
	p->fhdr.fctlb1 = *fmode.p2ctl++;
	p->fhdr.fctlw2 = *fmode.p2ctl++;
	p->fhdr.fctlb2 = *fmode.p2ctl;
      }
    }
  }
  if ( free_fifo == -1 ){
    sprintf(dd_log_msg,"fifo_make: Error: no free fifo\n");
    dd_log(DD_ERROR, dd_log_msg);
    this_fifo = -1;
  }

  fifo_mod_unlock();
  
  return this_fifo;
}

int fifo_perm2empty(int which_fifo, int *perm)
/*
 * Check whether the process has the right to remove the fifo. This is the 
 * case, if:
 * 1. The fifo is not the INPUT_FIFO
 * 2. There are no other processes attached to the fifo.
 * 3. If there is a permission to remove the fifo, then the fifo change 
 *    semaphore will be set!!!
 */
{
  struct a_fifo *p2f;

  *perm = 0;

  p2f = fifo_shm_start + which_fifo;
  
  fifo_mod_lock();

  if (which_fifo == INPUT_FIFO)
    return 0;

  if ( ( strcmp(p2f->fhdr.ftype,"STA") == 0) && ( p2f->fhdr.fnproc_att > 2))
    return 0;

  if ( (strcmp(p2f->fhdr.ftype,"DYN") == 0) && ( p2f->fhdr.fnproc_att > 1))
    return 0;
  
  *perm = 1;
  
  return 0;
}


int fifo_remove(int which_fifo)
/* 
 * Is called when a process wants to deattach from the dd and must therefore
 * reset its fifo, which is completely initialised and free to be
 * allocated to another process in fifo_make.
 * NOTE: It is assumed that the routine fifo_perm2empty is called beforehand
 *       and gave permission to remove (and correspondingly the change lock
 *       semaphore is set!!!).
 */
{
  struct a_fifo *p;
  int pid,i;
  int this_fifo;

  p = fifo_shm_start + which_fifo;
  p->fhdr.fnproc_att--;

  if (which_fifo == INPUT_FIFO){
    fifo_mod_unlock();
    return 0;
  }

  if ( ( strcmp(p->fhdr.ftype,"STA") == 0) && ( p->fhdr.fnproc_att > 1) ) {
    fifo_mod_unlock();
    return 0;
  }

  if ( ( strcmp(p->fhdr.ftype,"DYN") == 0) && ( p->fhdr.fnproc_att > 0) ) {
    fifo_mod_unlock();
    return 0;
  }

  p->cnt = NULL;
  p->wrptr = NULL;
  p->rdptr = NULL;
  p->nopin = NULL;
  p->nopout = NULL;

  p->fhdr.freqcnt = -1;

  if (strcmp(p->fhdr.ftype,"STA") == 0)
    p->fhdr.fstatus = FSTAT_IDLE;
  else{
    p->fhdr.fnumber = -1;
    p->fhdr.fpidcreat = -1;
    p->fhdr.fnproc_att = -1;
    p->fhdr.fmode = -1;
    p->fhdr.fwait_mode = -1;
    p->fhdr.freqcnt = -1;
    p->fhdr.fctlw1 = -1;
    p->fhdr.fctlb1 = -1;
    p->fhdr.fctlw2 = -1;
    p->fhdr.fctlb2 = -1;
    strcpy(p->fhdr.fname,"------");
    strcpy(p->fhdr.ftype,"--- ");
    p->fhdr.fstatus = FSTAT_UNUSED;
  }

  fifo_mod_unlock();
  return 0;
}

int fifo_set_status(int which_fifo, int status)
{
  struct a_fifo *p;

  if (which_fifo > max_fifo_nb)
    return -1;
  p = fifo_shm_start + which_fifo;
  p->fhdr.fstatus = status;
  return 0;
}

int fifo_set_protection(int which_fifo, int protection)
{
  struct a_fifo *p;

  if (which_fifo > max_fifo_nb)
    return -1;
  p = fifo_shm_start + which_fifo;
  p->fhdr.fprotection = protection;
  return 0;
}


int fifo_set_header(int which_fifo, struct fifo_header fhdr)
{
  struct a_fifo *p;

  if (which_fifo > max_fifo_nb)
    return -1;

  fifo_mod_lock();

  p = fifo_shm_start + which_fifo;
  p->fhdr  = fhdr;

  fifo_mod_unlock();

  return 0;
}

int fifo_set_reqcnt(int which_fifo, int cnt)
{
  struct a_fifo *p;

  if (which_fifo > max_fifo_nb)
    return -1;

  fifo_mod_lock();

  p = fifo_shm_start + which_fifo;
  p->fhdr.freqcnt = cnt;

  fifo_mod_unlock();

  return 0;
}


int fifo_get_header(int which_fifo, struct fifo_header *fhdr)
{
  struct a_fifo *p;

  if (which_fifo > max_fifo_nb)
    return -1;

  p = fifo_shm_start + which_fifo;
  *fhdr = p->fhdr;

  return 0;
}

/* returns fifo position in shm (0-N) or -1 if non-existant */
int fifo_exist(char *fifoname)
{
  int i;
  struct a_fifo *p;

  for (i=0;i<max_fifo_nb;i++){
    p = fifo_shm_start + i;
    if (strcmp(p->fhdr.fname,fifoname) == 0)
      return i;
  }

  return -1;
}

int fifo_close(int idestroy)
/*
 * Deattaches the process from the fifo shared memory if !idestroy.
 * If idestroy, then it removes the shared memory and semaphore.
 */
{
  /* removes shm for all fifos if idestroy>0,
   * detach from shm if idestroy<=0 CTCTCT */
  if ( fifo_close_shm(idestroy) != 0)
    return -1;
    
  /* removes sems for all fifos if idestroy>0 CTCTCT */
  if ( fifo_close_sem(idestroy) != 0)
    return -1;

  my_own_fifo = -1;

  return 0;
}

int fifo_print_sys()
{
  int i;
  if ( fifo_sys_start == NULL ){
    printf(" no such system\n");
    return -1;
  }

  printf("\n\n ************* FIFO SYSTEM: %s ************* \n",fifo_sys_start->sys_name);
  printf("SYSTEM INFO: created by %d at %s",(int)fifo_sys_start->pid_creat,ctime(&fifo_sys_start->time_creat));
  printf("           : status : %s\n",fifo_sys_status[fifo_sys_start->status]);
  printf("           : number of fifos %d of depth %d\n",fifo_sys_start->max_fifo_nb,fifo_sys_start->fifo_depth);
  printf("\n\n");
  return 0;
}



int fifo_print_header(int which_fifo)
{
  char cio[1];
  char *cp;
  struct a_fifo *p;
  int imode, semcnt[MAX_FIFO_NB_MAX];

  if (fifo_get_semcnt(semcnt) != 0)
    return -1;

  p = fifo_shm_start + which_fifo;
  if ( p->fhdr.fstatus == FSTAT_UNUSED ) 
    return 0;

  if ( p->fhdr.fwait_mode == FWAIT_SLEEP )
	 imode = 0;
  else if ( p->fhdr.fwait_mode == FWAIT_ASYNC )
	 imode = 1;

  printf(" \n FIFO %i %s                        status: %s \n", 
	 p->fhdr.fnumber,p->fhdr.fname, fifo_states[p->fhdr.fstatus]);

  printf("******************************************************\n");
  printf(" type %s  protection: %s    created by pid %i \n", 
	 p->fhdr.ftype, fifo_protection[p->fhdr.fprotection], p->fhdr.fpidcreat);
  printf(" mode %s wait_mode %s suser %s nproc_att %3i\n",
	 fifo_modes[p->fhdr.fmode], fifo_waitmodes[imode], suser_mode[p->fhdr.fsuser],
	 p->fhdr.fnproc_att);

  if ( p->fhdr.fprescale > 1 ) 
    printf(" fifo prescale %i: tries %i \n",p->fhdr.fprescale, p->noptry);

  printf(" ctl2 words: ctlw1 %5i ctlb1 %x ctlw2 %5i ctlb2 %x \n", 
	 p->fhdr.fctlw1,p->fhdr.fctlb1,p->fhdr.fctlw2,p->fhdr.fctlb2);
  printf(" nopin nopout reqcnt %i %i %i \n", 
	 p->nopin,p->nopout,p->fhdr.freqcnt);  
  printf(" fifo semaphore  = %i \n",semcnt[which_fifo]);
  printf(" fifo count      = %i \n",p->cnt);
  printf(" fifo wr/rd ptr  = %i %i \n",p->wrptr,p->rdptr);

  return 0;
}
  
int fifo_print(int which_fifo, int first, int last)
/* 
 * Dumps the contents of fifo which_fifo from location first to last.
 */
{
  int i,ii;
  char cio[1];
  struct a_fifo *p;

  p = fifo_shm_start + which_fifo;

  fifo_print_header(which_fifo);

  /* should be ||(last <= 0) CTCTCT */
  if ( (last > fifo_depth) || (last == 0)) 
    last = fifo_depth;
  for (ii = first;ii<last;ii++){
    cio[0] = ' ';
    cio[1] = ' ';
    /* why is read labeled 'O' and write labeled 'I' ? CTCTCT */
    if (ii == p->wrptr) cio[0] = 'I';
    if (ii == p->rdptr) cio[1] = 'O';
    printf(" %c %c %3i %10i %6x %10p %6i %6i %6x %6i %6x \n", 
	   cio[0],cio[1],ii,p->fentry[ii].shmid, p->fentry[ii].dboff, 
	   p->fentry[ii].p2da, p->fentry[ii].len, 
	   p->fentry[ii].ctlw1,p->fentry[ii].ctlb1,
	   p->fentry[ii].ctlw2,p->fentry[ii].ctlb2);
  }
  return 0;
}

int fifo_rate(int *fstatus, int *fcnt, int *dnevin, int *dnevout)
/*
 * returns arrays with the status, fifo count and difference in
 * of events that came in/went out of the fifo since the last
 * time that it was called
 */
{
  int semcnt[MAX_FIFO_NB_MAX];
  struct a_fifo *pfifo;
  int i;

  if (fifo_get_semcnt(semcnt) != 0)
    return -1;

  for (i=0;i<max_fifo_nb;i++){
    pfifo = fifo_shm_start + i;
    *fstatus++ = pfifo->fhdr.fstatus;
    *fcnt++ = semcnt[i];
    if (nevoldin[i]) {
      *dnevin++ = pfifo->nopin - nevoldin[i];
    } else {
      *dnevin++ =0;
    }
    /* store ->nopin in static array CTCTCT */
    nevoldin[i] =  pfifo->nopin;
    
    if (nevoldout[i]) {
      *dnevout++ = pfifo->nopout - nevoldout[i];
    } else {
      *dnevout++ = 0;
    }
    /* store ->nopout in static array CTCTCT */
    nevoldout[i] =  pfifo->nopout;
  }
  return 0;
}

int fifo_sem_status()
/*
 * gives an overview of the fifo count semaphore data structure
 */
{
  int i, ival, ival1;
  struct semid_ds fifo_sem_ds,fifo_lsem_ds;
  int semcnt[MAX_FIFO_NB_MAX];
  int semlcnt[MAX_FIFO_NB_MAX+1];

  if (fifo_get_semcnt(semcnt) != 0)
    return -1;
  if (fifo_get_lsemcnt(semlcnt) != 0)
    return -1;

  printf(" *************************************************** \n");
  printf("\n");
  printf("                  SEMAPHORE STATUS \n");
  printf("                ==================== \n");

  ival = semctl(fifo_semid,0,GETPID,0);
  printf(" last operation by ............. %i ",ival);
  ival = semctl(fifo_lock_semid,0,GETPID,0);
  printf("  %i \n",ival);

  if (semctl(fifo_semid,0,IPC_STAT,&fifo_sem_ds) != 0){
    sprintf(dd_log_msg," fifo_status: semctl(IPC_STAT) %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  if (semctl(fifo_lock_semid,0,IPC_STAT,&fifo_lsem_ds) != 0){
    sprintf(dd_log_msg," fifo_status: semctl(IPC_STAT) %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
   return -1;
  }
  printf(" User ID  ...................... %i  %i \n", 
	 fifo_sem_ds.sem_perm.uid,fifo_lsem_ds.sem_perm.uid);
  printf(" Group ID ...................... %i  %i \n", 
	 fifo_sem_ds.sem_perm.gid,fifo_lsem_ds.sem_perm.gid);
  printf(" Operation permissions ......... O%o O%o\n", 
	 fifo_sem_ds.sem_perm.mode,fifo_lsem_ds.sem_perm.mode);
  printf(" Number of semaphores in set ... %i  %i \n", 
	 fifo_sem_ds.sem_nsems,fifo_lsem_ds.sem_nsems);
  printf(" Time of last operation ........ %d  %i \n", 
	 fifo_sem_ds.sem_otime,fifo_lsem_ds.sem_otime);
  printf(" Time of last change ........... %d  %i \n", 
	 fifo_sem_ds.sem_ctime,fifo_lsem_ds.sem_ctime);
  printf(" \n");
  
  printf(" FIFOs:                                  fifo sem  | fifo lock sem \n");
  for (i=0;i<max_fifo_nb;i++){
    printf(" FIFO %3i : ",i);
    ival = semctl(fifo_semid,i,GETNCNT,0);
    ival1 = semctl(fifo_lock_semid,i,GETNCNT,0);
    printf(" sem count / n proc waiting %4i %4i  | %4i %4i \n", 
	   semcnt[i],ival,semlcnt[i],ival1);
  }

  ival1 = semctl(fifo_lock_semid,max_fifo_nb,GETNCNT,0);
  printf("\n \n ...... nb proc waiting for change  %i \n",ival1);

  printf(" *************************************************** \n");
  return 0;
}

int fifo_unblock_proc(int which_fifo)
/* This function seems to simply give up semaphores until there
 * are no more processes waiting to use the protected resource.
 * It's only used in "fifos_unblock" which is only used in "dds_wakeup"
 * which wakes up all processes that sleep on a signal.
 * CTCTCT 
 */
{
  struct a_fifo *p2f;
  int nloop;

  p2f = fifo_shm_start+which_fifo;

  nloop = 0;
  
 /* while (number of processes waiting for sem to increase != 0) and
  *       (looping thru number of attached processes + 1) ...
  *       CTCTCT
  */
  while ( (semctl(fifo_semid,which_fifo,GETNCNT,0)) && 
	 (nloop <= p2f->fhdr.fnproc_att) ){
    /* give up sem CTCTCT */
    if (fifo_incr_semcnt(which_fifo)){
      sprintf(dd_log_msg," fifo_unblock: fifo_incr_semcnt \n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
    nloop++;
  }

  /* # of processes waiting for sem to increase still != 0 */
  if (semctl(fifo_semid,which_fifo,GETNCNT,0)) {
    sprintf(dd_log_msg," fifo_unblock: Still hanging processes %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
  }

  return 0;
}


int lock_input_fifo()
{
  return fifo_lock(INPUT_FIFO);
}

int unlock_input_fifo()
{
  return fifo_unlock(INPUT_FIFO);
}

int get_next_fifo(int which_fifo, struct fifo_entry fev)
/*
 * Called when a process got an event from its fifo and wants to put it
 * back into the dd system.
 * Starting from fifo number my_own_fifo, it loops through the remaining
 * fifos and checks whether the event fits into another fifo. If not, then
 * it is put back into the INPUT_FIFO.
 * Note that the NULL fifo (in mode FMODE_NULL) will never receive an event.
 */
{
  struct a_fifo *p2f;
  int this_fifo;
  int i;

  i = which_fifo+1;
  p2f = fifo_shm_start+i;
  this_fifo = -1;

  while ( (i<max_fifo_nb) && (this_fifo == -1) ){
    if (p2f->fhdr.fstatus == FSTAT_ACTIVE){
      if (p2f->fhdr.fmode == FMODE_ALL)
        this_fifo = i;
      else if (p2f->fhdr.fmode == FMODE_COND) {
	if ( (p2f->fhdr.fctlw1 != -1) && FIFO_COND1 )
	  this_fifo = i;
	if ( (p2f->fhdr.fctlb1 != -1) && FIFO_COND2 )
	  this_fifo = i;
	if ( (p2f->fhdr.fctlw2 != -1) && FIFO_COND3 )
	  this_fifo = i;
	if ( (p2f->fhdr.fctlb2 != -1) && FIFO_COND4 )
	  this_fifo = i;
      }
      else if (p2f->fhdr.fmode == FMODE_ONREQ){
        if (p2f->cnt < p2f->fhdr.freqcnt)
          this_fifo = i;
      }
      
#if defined(__sun) || defined(__linux)
      /*
       * add another condition - FMODE_USER - which allows a user
       * function - ddu_filter_fev - to determine whether an event
       * is of interest to a fifo CTCTCT
       */
      else if (p2f->fhdr.fmode >= FMODE_USER) {
	if ((*ddu_filter) (p2f, fev)) {
	  this_fifo = i;
        }
      }
#endif
      
      if ( this_fifo != -1 ) 
	if ( (p2f->noptry++%p2f->fhdr.fprescale) ) 
	  this_fifo = -1;
    }
    i++;
    p2f++;
  }

  if (this_fifo == -1)
    this_fifo = INPUT_FIFO;

  return this_fifo;
}
