/*
 * dd_sys.c :  General calls for the dd system (start with dds_)
 *          :  These are the routines that should be called from the
 *          :  dd control program (or likewise) the run control.
 * Author   :  C.Witzig
 * Date     :  Apr 1, 1992 (no joke!)
 * Mods     : 
 *          : Mar 29/30, 1995: modify the philosophy of who
 *                   starts up the system and when (now 
 *                   have several masters, which can be consumers
 *                   as well. Thel ast one turns the lights on
 *                   should normally be the dd_garbage_d).
 *          : Mar 30, 1995: add dds_read_config which
 *                   allows to determine the configuration 
 *                   at run time.
 *          : Aug 31, 1995: 
 *                   add dds_flush (basically the old reset)
 *
 *          : Jan 10, 1996:
 *                   add dds calls over the network
 *
 */

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>

#include <dd_config.h>
#include <dd_signal_def.h>
#include <dd_log.h>
#include <dd_user.h>
#include <dd_sys.h>

#define TRUE 1
#define FALSE 0

#define MAX_SHUTDOWN_WAIT 50


/*
 * default name for the ftok file in /tmp
 */
static char *dd_default_name = DEFAULT_DD_NAME;
static char dd_filename[1024] = "";

static char dd_name_env[1024];
static char dd_conf_env[1024];

/*
 *====================================================
 * static routines
 * - dds_check_configuration 
 * - dds_check_dd_configuration
 * - dds_read_configuration
 * - dds_get_nproc_att
 * 
 * These routines are NOT called over the network
 *====================================================
 *
 */


static int dds_check_configuration()
{
/*
 * tests upper and lower limit of the user configuration
 */

  if ( dd_dcom->dd_config.max_fifo_nb <= 0  ) {
    sprintf(dd_log_msg,"dds_check_configuration: max_fifo_nb negative \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( dd_dcom->dd_config.fifo_depth <= 0 ) {
    sprintf(dd_log_msg,"dds_check_configuration: fifo depth negative\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( dd_dcom->dd_config.db_maxnbev <= 0 ) {
    sprintf(dd_log_msg,"dds_check_configuration: static buffer count negative \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( dd_dcom->dd_config.db_maxevsize <= 0 ) {
    sprintf(dd_log_msg,"dds_check_configuration: nb of static buffers negative \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( dd_dcom->dd_config.db_max_buf_vol <= 0 ) {
    sprintf(dd_log_msg,"dds_check_configuration: nb of volatile buffers negative \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  if ( dd_dcom->dd_config.max_fifo_nb > MAX_FIFO_NB_MAX ) {
    sprintf(dd_log_msg,"dds_check_configuration: too many fifos: %d %d \n",
	    dd_dcom->dd_config.max_fifo_nb, MAX_FIFO_NB_MAX);
    dd_log(DD_ERROR, dd_log_msg); 
    return -1;
  }

  if ( dd_dcom->dd_config.fifo_depth > FIFO_DEPTH_MAX ) {
    sprintf(dd_log_msg,"dds_check_configuration: fifos too deep: %d %d \n",
	    dd_dcom->dd_config.fifo_depth, FIFO_DEPTH_MAX);
    dd_log(DD_ERROR, dd_log_msg); 
    return -1;
  }

  if ( dd_dcom->dd_config.db_maxnbev > DB_MAXNBEV_MAX ) {
    sprintf(dd_log_msg,"dds_check_configuration: too many static buffers: %d %d  \n",
	    dd_dcom->dd_config.db_maxnbev, DB_MAXNBEV_MAX );
    dd_log(DD_ERROR, dd_log_msg); 
    return -1;
  }

  if ( dd_dcom->dd_config.db_max_buf_vol > DB_MAX_BUF_VOL_MAX ) {
    sprintf(dd_log_msg,"dds_check_configuration: too many volatile buffers: %d %d  \n",
	    dd_dcom->dd_config.db_max_buf_vol, DB_MAX_BUF_VOL_MAX);
    dd_log(DD_ERROR, dd_log_msg); 
    return -1;
  }

  if ( dd_dcom->dd_config.fifo_depth <
	  dd_dcom->dd_config.db_maxnbev + dd_dcom->dd_config.db_max_buf_vol +1 ) {
    sprintf(dd_log_msg,"dds_check_configuration: fifos too shallow  \n");
    dd_log(DD_ERROR, dd_log_msg);
     return -1;
  }

  return 0;
}

static int dds_check_dd_configuration()
{
  FILE *fp;
  int i, n, val[DD_NCONF], rval[DD_NCONF];
  char *p2c, name[100];

  for (i=0;i<DD_NCONF;i++)
    val[DD_NCONF] = 0;

  if ( ((p2c = getenv("DD_CONFIGURATION")) != NULL ) ) {
    if ( (fp = fopen(p2c,"r")) != NULL ) {
      sprintf(dd_log_msg,"dds_check_new_configuration: attaching to existant DD system\n");
      dd_log(DD_INFO, dd_log_msg);
      sprintf(dd_log_msg,"dds_check_new_configuration: configuration file %s defined \n",p2c);
      dd_log(DD_INFO, dd_log_msg);
      while ( (fscanf(fp,"%s %d",name,&n)) != EOF ){
	for (i=0; i<DD_NCONF; i++) {
	  if ( !(strcasecmp(name, dd_conf_name[i]) ) ){
	    val[i] = n;
	    break;
	  }
	}
      }
      fclose(fp);

      rval[0] = dd_dcom->dd_config.max_fifo_nb;
      rval[1] = dd_dcom->dd_config.fifo_depth;
      rval[2] = dd_dcom->dd_config.db_maxnbev;
      rval[3] = dd_dcom->dd_config.db_maxevsize;
      rval[4] = dd_dcom->dd_config.db_max_buf_vol;
      
      for (i=0; i<DD_NCONF; i++){
	if ( rval[i] !=  val[i] ) 
	  sprintf(dd_log_msg,"dds_check_new_configuration: values differ: %s old: %d new: %d\n",
	       dd_conf_name[i],rval[i], val[i]);
	dd_log(DD_ERROR, dd_log_msg);
      }
    }
  }
  return 0;
}

static void dds_read_configuration()
{
  FILE *fp;
  int i,n,val[DD_NCONF];
  char *p2c, name[100];

  for (i=0;i<DD_NCONF;i++)
    val[DD_NCONF] = 0;

  if ( ((p2c = getenv("DD_CONFIGURATION")) != NULL ) ) {
    if ( (fp = fopen(p2c,"r")) != NULL ) {
      while ( (fscanf(fp,"%s %d",name,&n)) != EOF ){
	for (i=0; i<DD_NCONF; i++) {
	  if ( !(strcasecmp(name, dd_conf_name[i]) ) ){
	    val[i] = n;
	    sprintf(dd_log_msg,"  >>> %s %d\n",dd_conf_name[i],val[i]);
	    dd_log(DD_INFO, dd_log_msg);
	    break;
	  }
	}
      }
      fclose(fp);
      sprintf(dd_log_msg,"\n");
      dd_log(DD_INFO, dd_log_msg);
     
      dd_dcom->dd_config.max_fifo_nb = val[0];
      dd_dcom->dd_config.fifo_depth = val[1];
      dd_dcom->dd_config.db_maxnbev = val[2];
      dd_dcom->dd_config.db_maxevsize = val[3];
      dd_dcom->dd_config.db_max_buf_vol = val[4];
      
      if ( dds_check_configuration() == 0 ) 
	return;
      else {
	sprintf(dd_log_msg,"dds_read_configuration: bad configuation values \n");
	dd_log(DD_ERROR, dd_log_msg);
      }
    }
  }

  dd_dcom->dd_config.max_fifo_nb = MAX_FIFO_NB_MAX;
  dd_dcom->dd_config.fifo_depth = FIFO_DEPTH_MAX;
  dd_dcom->dd_config.db_maxnbev = DB_MAXNBEV_MAX;
  dd_dcom->dd_config.db_maxevsize = DB_MAXEVSIZE_MAX;
  dd_dcom->dd_config.db_max_buf_vol = DB_MAX_BUF_VOL_MAX;
}

static int dds_get_nproc_att()
{
/*
 * returns the number of processes attached
 */
  int i, nproc;
  nproc = 0;
  for (i=0; i<DD_MAX_NPROC_ATT; i++) {
    if ( dd_dcom->dd_proc[i].pid > 0 ){
      nproc++;
    }
  }
  return nproc;
}

int dds_find_dd_system(char *p2file, char *dd_system, char *hostname, char *conf_file)
{
/* 
 * looks in the file p2file to see whether the DD system *dd_system is defined and 
 * and returns *hostname and conf_file
 *
 * Returns 0 if found, -1 if not found
 * NOTE: hostname and conf_file are the pointers where we copy the info
 */
  FILE *fp;
  int not_found = TRUE;

  if ( (fp = fopen(p2file,"r")) == NULL ) {
    sprintf(dd_log_msg,"dds_find_dd_systemc: DDD_CONFIGURATION_FILE %s NOT found\n",p2file);
    dd_log(DD_ERROR, dd_log_msg);
   }
  else{
    char line[100], dd_name[100];
    
    while ( fgets(line, sizeof(line), fp) ) {
      if ( line[0] != '#') {
	sscanf(line, "%s %s %s",dd_name, hostname, conf_file);
	if ( strcmp(dd_name, dd_system) == 0 ) {
	  not_found = FALSE;
	  break;
	}
      }
    }
  }
  return not_found;
}
  


void dds_get_filename(char *p2filename)
{
/*
 * read the environment variable DD_NAME
 * if $DDD_CONFIGURATION_FILE is defined, then 
 * scan the file to get the DD parameters
 * Find out whether we are on the same host
 * p2filename points to the DD_NAME environment
 * variable
 */
  char *p2c, *p2file;

  if ( (p2c = getenv("DD_NAME")) == NULL )
    p2c = dd_default_name;

  if ( (p2file = getenv("DDD_CONFIGURATION_FILE")) != NULL ){
    FILE *fp;
    char dd_name[1024], hostname[1024], conf_file[1024];
    char *p2cc;

    if ( (p2cc = strrchr(p2c,'@')) != NULL ) 
      *p2cc = '\0';

   /* 
    * look in the file p2file to see whether the DD system *dd_system is defined 
    * and return *hostname and conf_file
    */
    if ( dds_find_dd_system(p2file, p2c, hostname, conf_file) != 0 ) {
      sprintf(dd_log_msg,"DD system %s NOT found in DDD_CONFIGURATION_FILE %s\n",
	     p2c, p2file);
      dd_log(DD_ERROR, dd_log_msg); 
    }
    else {
      int len1, len2;
      char my_hostname[HOSTNAME_LEN];

      if ( gethostname(my_hostname,HOSTNAME_LEN-1) < 0)
	perror("ddn_set_hostname: error in gethostname");

      sprintf(dd_name_env,"DD_NAME=%s@%s", p2c, my_hostname);

      len1 = strlen(my_hostname);
      len2 = strlen(hostname);
      if (len1 > len2)
	len1 = len2;
      if ( strncmp(hostname, my_hostname, len1) == 0 ) {
	char *p2c = strrchr(dd_name_env, '@');
	*p2c = '\0';
      }
      if ( putenv(dd_name_env) ) 
	printf("dds_get_filename: error in putenv for dd_name %s\n",dd_name_env);
      
      if ( strcmp(conf_file,"NULL") != 0 ) {
	sprintf(dd_conf_env,"DD_CONFIGURATION=%s", conf_file);
	if ( putenv(dd_conf_env) ) {
	  sprintf(dd_log_msg, "dds_get_filename: error in putenv for dd_conf_file %s\n",dd_conf_env);
	  dd_log(DD_ERROR, dd_log_msg);
	}
      }
/*    now that we redefined DD_NAME we have to make sure we point to the right value */
      p2c = getenv("DD_NAME");
    }
  }
  sprintf(p2filename,"/tmp/%s",p2c);
  return;
}



/*
 *====================================================
 *
 * system routines
 * - dds_create
 * - dds_init
 * - dds_wait4proc
 * - dds_wait4sleep
 * - dds_wakeup
 * - dds_prepare_shutdown
 * - dds_flush
 * - dds_close
 *
 * These routines are called over the network
 *
 *====================================================
 *
 */

int dds_create()
/*
 * create the dd system, that is:
 * 1. Check whether there is room for another dd master
 * 2. make the fifos
 * 3. create the common data buffer db
 * 4. and connect them
 * 5. dd_dcom will be fully initialised if one of the 
 *    other systems (fifo,db,dcom) has been initialised.
 */
{
  int first;
  char *p2c;
  FILE *fp;

  first = 0;
  
 /* CTCTCT
  * Read the environment variable DD_NAME.
  * If $DDD_CONFIGURATION_FILE is defined, then 
  * scan the file to get the DD parameters.
  * Find out whether we are on the same host and
  * change DD_NAME to point to local host.
  * dd_filename points to "/tmp/DD_NAME" 
  */
  dds_get_filename(dd_filename);

 /*
  * Call for setting up the host names and find out
  * whether we are connected over TCP/IP.
  * Argument can be either $DD_NAME or $FIFONAME
  * with @hostname for specifing the host. We 
  * must therefore check that there is no mismatch
  * in case both names are defined.
  * CTCTCT
  * Check to see if dd_hostname & arg are same host. If not, quit.
  * Set dd_hostname to arg's hostname. Compare new dd_hostname to
  * my_hostname. If same, dd2tcp = 0, else dd2tcp = 1.
  * If i_am_dd2tcp_server, get peername and put in dd_peername.
  */
  if (ddn_set_hostnames(dd_filename) < 0){
    sprintf(dd_log_msg,"dds_create: error in ddn_set_hostnames\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( strlen(dd_hostname) > 0 ) {
    sprintf(dd_log_msg, "Creating DD system %s on host %s from host %s\n",dd_filename, dd_hostname, my_hostname);
    dd_log(DD_INFO, dd_log_msg);
  }
  else {
    sprintf(dd_log_msg, "Creating DD system %s locally on host %s\n",dd_filename, my_hostname);
    dd_log(DD_INFO, dd_log_msg);
  } 
  if ( dd2tcp ) {
    if ( (ddn_server_running()) == FALSE  ){
      if ( ddn_client_startup(dd_hostname) )
	return -1;
      atexit(dd_cleanup);
    }
    return ddsnc_create();
  }

  if ( (fp = fopen(dd_filename,"r")) == NULL ) {
    time_t tt;
    if ( (fp = fopen(dd_filename,"w")) == NULL ) {
      sprintf(dd_log_msg,"dds_create: cannot open file %s\n",dd_filename);
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
    time(&tt);
    fprintf(fp,"DD SYSTEM: %s created by proc %d at %s",dd_filename, getpid(), ctime(&tt) );
    fclose(fp);
  }
  
  /* get pointer to shared memory of dd_dcom structure
   * create & initialize (blocking with sem #0) if necessary CTCTCT */
  dd_dcom = dd_dcom_attach(dd_filename);

  if ( dd_dcom_lock() ){
    sprintf(dd_log_msg,"dds_create: error in dd_dcom_lock\n");
    dd_log(DD_ERROR, dd_log_msg);
    dd_dcom_unlock();
    dds_close(1);
    return -1;
  }

  if (dd_dcom->dd_nmstr == 0) {
    first = 1;
    /* read configuration in DD_CONFIGURATION CTCTCT */
    dds_read_configuration(); 
    sprintf(dd_log_msg,"dds_create: system not in use: I will create it!\n");
    dd_log(DD_ERROR, dd_log_msg);
    sprintf(dd_log_msg,"dds_create:fifos %d, depth %d, events %d, volatile %d, size %d\n",
	    dd_dcom->dd_config.max_fifo_nb,
	    dd_dcom->dd_config.fifo_depth,
	    dd_dcom->dd_config.db_maxnbev,
	    dd_dcom->dd_config.db_max_buf_vol,
	    dd_dcom->dd_config.db_maxevsize
	    );
    dd_log(DD_ERROR, dd_log_msg);
  }
  else {
    sprintf(dd_log_msg,"dds_create: system already in use: I will attach to it \n");
    dd_log(DD_INFO, dd_log_msg);
    /* check values in dd_com structure against DD_CONFIGURATION CTCTCT */
    dds_check_dd_configuration();
  }

  /* add master to list, find num of sem (dd_master[i].dd_sem) & ival ("".dd_semval) CTCTCT */
  if ( dd_dcom_addmaster() < 0){
    sprintf(dd_log_msg,"dds_create: cannot add one more master\n");
    dd_log(DD_ERROR, dd_log_msg);
    dd_dcom_unlock();
    dds_close(1);
   return -1;
  }
  
  /* init (set to 0) or find sems & shm for each fifo */
  /* Sems: one for each fifo for event counting,      */
  /*       one for each fifo + 1 for locking fifo shm */
  /* Shm:  mem for all fifo structures */
  /* Someone's been playing with fifo size */
  /* dbfi.c, fifo.c CTCTCT */
  if ( fifos_attach(dd_filename) ){
    sprintf(dd_log_msg," dds_create: fifo_attach failed \n");
    dd_log(DD_ERROR, dd_log_msg);
    dd_dcom_unlock();
    
    dds_close(1);
    return -1;
  }
  
  /* allocated shm for all events (db.c) CTCTCT */
  if ( db_init(dd_filename, dd_dcom->dd_config.db_maxnbev, dd_dcom->dd_config.db_maxevsize) ) {
    sprintf(dd_log_msg," dds_create: db_init failed \n");
    dd_log(DD_ERROR, dd_log_msg);
    dd_dcom_unlock();
    dds_close(1);
    return -1;
  }

  if (first){
    /* Initializes dd_dcom structure (grabs sem. above) CTCTCT   */
    /* Some important-looking code is commented out. (dd_dcom.c) */
    if ( dd_dcom_init() ){
      sprintf(dd_log_msg," dds_create: error in dd_dcom_init\n");
      dd_log(DD_ERROR, dd_log_msg);
      dd_dcom_unlock();
      dds_close(1);
      return -1;
    }
    
   /* CTCTCT
    * Connect the db and the fifo together, that is:
    * 1. get the db and fill the pointers to the slots into the INPUT fifo.
    * 2. initialise INPUT fifo
    * 3. set INPUT fifo active
    * 4. make static TAPE fifo
    * 5. make static BROADCAST fifo
    * 6. make static GARBAGE fifo
    *
    * Fills the INPUT FIFO with the pointers to the slots in the db.
    * In addition it saves the offset within db for later crosscheck.
    */
    if (dbfi_connect() ){
      sprintf(dd_log_msg," dds_create: cannot connect db and fifos \n");
      dd_log(DD_ERROR, dd_log_msg);
      dd_dcom_unlock();
      dds_close(1);
      return -1;
    }

    if ( dbfi_fork_garbage_d() ){
      sprintf(dd_log_msg,"dds_create: error in dbfi_fork_garbage \n");
      dd_log(DD_ERROR, dd_log_msg);
      dds_close(1);
      return -1;
    }

  }
  if ( dd_dcom_unlock() ){
    sprintf(dd_log_msg," dds_create: error in dd_dcom_unlock\n");
    dd_log(DD_ERROR, dd_log_msg);
    dds_close(1);
    return -1;
  }

  dd_dcom->dd_status = DD_STATUS_OK;

  return 0;
}


int dds_init()
/*
 * reinitializes the fifos and db, that is:
 * 1. empty all the fifos, throwing away all the volatile buffers
 * 2. initialise the fifos
 * 3. connect fifo and db again (dbfi_connect), i.e. fill the INPUT fifo
 *    again with valid pointers.
 * NOTE: The connection between the processes and the fifos will be lost
 *       (i.e. the fifo headers are initialised).
 */
{
  if ( ! dd_attached() ) {
    sprintf(dd_log_msg,"dds_init:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

 /*
  * Sends the necessary data to the server:
  * 1. environment variable DD_NAME
  * 2. length of the fifo name (incl. terminating null character!)
  * 3. fifoname
  * 4. fifo mode, fifo wait and control words.
  * 5. returns the status as obtained from ddu_init on the server side.
  */
  if ( dd2tcp )
    return ddsnc_init();

  if ( dd_dcom_lock() ){
    sprintf(dd_log_msg," dds_init: error in dd_dcom_unlock\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
 /*
  * Resets the db and fifos, that is:
  * empties fifos if they're not yet empty and check with the saved
  * set in dcom to make sure that no pointer got lost (this is only
  * a last crosscheck that the dd system was working ok in the last run).
  * NOTE: It does not touch the fifo headers ..... just makes sure that ALL
  *       the fifos are empty (including INPUT fifo). 
  */
 /* Deletes shared mem from sys ?! once for each fifo entry ?!! CTCTCT
  * dbfi_reset (in dbfi.c) calls dbfi_free_fev(fev) for each value
  * where fev = fifo entry of every fifo. dbfi_free_fev (in dbfi.c)
  * calls db_freemem(fev.shmid) (in db.c) which calls
  * shmctl(fev.shmid,IPC_RMID,shmbuf)
  */
  if (dbfi_reset() ){
    sprintf(dd_log_msg," dds_init: fifo_reset failed \n");
    dd_log(DD_ERROR, dd_log_msg);
    if ( dd_dcom_unlock() ){
      sprintf(dd_log_msg," dds_init: error in dd_dcom_unlock\n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
    return -1;
  }

  /* reset dd_dcom shared memory data */
  dd_dcom_init();

 /*
  * Reinitialises the fifos completely (all the connections to 
  * processes are lost) and connects db and fifo. (dbfi.c)
  * Calls fifo_init (fifo.c) which initializes the fifo sems
  * by attaching to them. If this fails it creates and initializes
  * them. (fifo_semid, nfifos) (fifo_lock_semid, nfifos+1)
  */
  if ( fifos_init() ){
    sprintf(dd_log_msg," dds_init: fifos_init failed \n");
    dd_log(DD_ERROR, dd_log_msg);
    if ( dd_dcom_unlock() ){
      sprintf(dd_log_msg," dds_init: error in dd_dcom_unlock\n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
    dd_error_inv();
    return -1;
  }
  
 /* (dbfi.c) 
  * Connect the db and the fifo together, that is:
  * 1. get the db and fill the pointers to the slots into the INPUT fifo.
  * 2. initialise INPUT fifo
  * 3. set INPUT fifo active
  */
  if ( dbfi_connect() ){
    sprintf(dd_log_msg," dds_init: cannot connect db and fifos \n");
    dd_log(DD_ERROR, dd_log_msg);
    if ( dd_dcom_unlock() ){
      sprintf(dd_log_msg," dds_init: error in dd_dcom_unlock\n");
      dd_log(DD_ERROR, dd_log_msg);
       return -1;
    }
    dd_error_inv();
    return -1;
  }

  if ( dd_dcom_unlock() ){
    sprintf(dd_log_msg," dds_init: error in dd_dcom_unlock\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  dd_dcom->dd_status = DD_STATUS_OK;

  return 0;
}

int dds_wait4proc(int ntime_ms)
/*
 * wait for all processes to have have no 
 * events left in their fifo.
 * times out after wait_wait ( milli seconds)
 */
{
  int fifo_nonempty, i, max_wait, nwait;
  int fstatus[MAX_FIFO_NB_MAX],fcount[MAX_FIFO_NB_MAX];
  int nev1[MAX_FIFO_NB_MAX],nev2[MAX_FIFO_NB_MAX];

  if ( ! dd_attached() ) {
    sprintf(dd_log_msg,"dds_wait4proc:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( dd2tcp )
    return ddsnc_wait4proc(ntime_ms);

  fifo_nonempty = FALSE;
  max_wait = ntime_ms/20 + 1;
  nwait = 0;

  dd_dcom->dd_status  = DD_STATUS_WAIT;

  do {
    fifo_nonempty = FALSE;

    if ( fifo_rate(fstatus,fcount,nev1,nev2) ){
      sprintf(dd_log_msg,"dds_wait4proc: cannot get fifo counts \n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }

    dd_dcom_lock();
    for (i=0;i<DD_MAX_NPROC_ATT;i++){
      if ( ( dd_dcom->dd_proc[i].pid > 0) && 
	  ( dd_dcom->dd_proc[i].fifo != INPUT_FIFO ) && 
	  ( dd_dcom->dd_proc[i].status != DD_PROC_STATE_IDLE) ){
	if ( (fstatus[dd_dcom->dd_proc[i].fifo] == FSTAT_ACTIVE ) &&
	    (fcount[dd_dcom->dd_proc[i].fifo] > 0 ) ){
	  fifo_nonempty = TRUE;
	  break;
	}
      }
    }
    dd_dcom_unlock();

    if ( fifo_nonempty ){ 
      dd_nap(0.05); 
      nwait++;
    }
  } while ( (fifo_nonempty == TRUE) && (nwait < max_wait) );

  dd_dcom->dd_status  = DD_STATUS_OK;

  sprintf(dd_log_msg,"dd_wait4proc: nwaits: %d nproc waiting %d\n",nwait,fifo_nonempty);
  dd_log(DD_INFO, dd_log_msg);
 
  return ( nwait >= max_wait );
}


int dds_wakeup()
/*
 * wakes up all the processes that sleep on a signal 
 */
{
  if ( ! dd_attached() ) {
    sprintf(dd_log_msg,"dds_wakeup:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( dd2tcp )
    return ddsnc_wakeup();

  if ( dd_dcom_lock() ){
    sprintf(dd_log_msg," dds_wakeup: error in dd_dcom_unlock\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  if ( fifos_unblock() ) {
    sprintf(dd_log_msg," dds_wakeup: error in fifos_unblock\n");
    dd_log(DD_ERROR, dd_log_msg);
    if ( dd_dcom_unlock() ){
      sprintf(dd_log_msg," dds_wakeup: error in dd_dcom_unlock\n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
    dd_error_inv();
    return -1;
  }
  
  if ( dd_dcom_unlock() ){
    sprintf(dd_log_msg," dds_wakeup: error in dd_dcom_unlock\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  return 0;
}

int dds_prepare_shutdown()
{
/*
 * Sends a signal to every process that is still attached to the
 * DD system. This signal will then force the process to detach.
 * Is called before the DD system is removed
 */
  int i, nsleep, nproc, proc_list[DD_MAX_NPROC_ATT];

  if ( ! dd_attached() ) {
    sprintf(dd_log_msg,"dds_prepare_shutdown:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( dd2tcp )
    return ddsnc_prepare_shutdown();

  if ( dd_dcom_lock() ){
    sprintf(dd_log_msg," dds_prepare_shutdown: error in dd_dcom_lock\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  nproc = 0;
  for (i=0; i<DD_MAX_NPROC_ATT; i++) {
    if ( (dd_dcom->dd_proc[i].pid > 0) && ( dd_dcom->dd_proc[i].pid != getpid()) ){
      proc_list[nproc++] = dd_dcom->dd_proc[i].pid;
    }
  }
 
  for (i=0; i<nproc; i++) {
    int status; 
    if ( (status = kill(proc_list[i], DD_SHUTDOWN_SIGNAL)) ){
      sprintf(dd_log_msg, "dds_prepare_shutdown: sending signal to %d: %s\n",
	    proc_list[i],strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
    }
  }

  if ( dd_dcom_unlock() ){
    sprintf(dd_log_msg," dds_prepare_shutdown: error in dd_dcom_unlock\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  nsleep = 0;
  while ( (nproc = dds_get_nproc_att()) != 1 ){
    dd_nap(0.2);
    if ( (++nsleep) >= MAX_SHUTDOWN_WAIT ){
      dds_close(1);
      break;
    }
  }

  return 0;
}



int dds_reset()
/*
 * reinitializes the fifos and db, that is:
 * 1. empty all the fifos, the events are thrown away
 * 2. connect fifo and db again (dbfi_connect), i.e. fill the INPUT fifo
 *    again with valid pointers.
 * NOTE: The connection between the processes and the fifos is not lost.
 *    because dd_dcom_lock is used, there cannot be a broadcast while
 *    we are flushing, so we don't have to worry about this.
 */
{
  if ( ! dd_attached() ) {
    sprintf(dd_log_msg,"dds_reset:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( dd2tcp )
    return ddsnc_reset();

  if ( dd_dcom_lock() ){
    sprintf(dd_log_msg," dds_reset: error in dd_dcom_lock\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if (dbfi_reset() ){
    sprintf(dd_log_msg," dds_reset: fifo_reset failed \n");
    dd_log(DD_ERROR, dd_log_msg);
    dd_dcom_unlock();
    dd_error_inv();
    return -1;
  }

  if ( dbfi_fill_input_fifo() ){
    sprintf(dd_log_msg," dds_reset: error in dbfi_fill_input_fifo \n");
    dd_log(DD_ERROR, dd_log_msg);
    dd_error_inv();
    dd_dcom_unlock();
    return -1;
  }

  if ( dd_dcom_unlock() ){
    sprintf(dd_log_msg," dds_reset: error in dd_dcom_unlock\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  return 0;
}


int dds_flush(int brc_flush)
/*
 * flushes the dd system. 
 * if brc_flush != 0 makes a brccmd to attempt flushing attached DD systems
 */
{
  if ( ! dd_attached() ) {
    sprintf(dd_log_msg,"dds_flush:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( dd2tcp )
    return ddsnc_flush(brc_flush);

  return dbfi_flush(brc_flush);
}


int dds_close(int enforce)
/*
 * closes the dd system completely, that is:
 * - removes all the semaphores and  shared memory segments for db and fifos.
 * - deattaches dd_dcom.
 */
{
  char *p2c, command[1024];

  if ( ! dd_attached() ) {
    sprintf(dd_log_msg,"dds_close:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( dd2tcp )
    return ddsnc_close(enforce);

  if ( dd_dcom == NULL ) 
    return 0;

  if ( ! enforce ) {
    if ( dd_dcom_lock() ){
      sprintf(dd_log_msg," dds_close: error in dd_dcom_lock\n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
  }

  dd_dcom_remmaster();

  if ( (dd_dcom->dd_nmstr == 0) || (enforce) ){

    dds_get_filename(dd_filename);
    sprintf(command,"/bin/rm %s",dd_filename);
    system(command);

    if ( db_close(1) ) {
      sprintf(dd_log_msg," dds_close: db_close failed \n");
      dd_log(DD_ERROR, dd_log_msg);
       dd_error_inv();
      return -1;
    }
    if ( fifo_close(1) ){
      sprintf(dd_log_msg," dds_close: fifo_close failed \n");
      dd_log(DD_ERROR, dd_log_msg);
       dd_error_inv();
      return -1;
    }
    dd_dcom->dd_status = DD_STATUS_UNUSED;
    dd_dcom = dd_dcom_delete();
    
  }
  if ( dd_dcom ) {
    if ( dd_dcom_unlock() ){
      sprintf(dd_log_msg," dds_close: error in dd_dcom_unlock\n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
    dd_dcom->dd_status = DD_STATUS_UNUSED;
    dd_dcom = dd_dcom_detach();
  }

  return 0;
}  




