/*
 * dd_network_util.c :  Utility routines for the network connection
 *          :  (can be called by either client or server)
 *          :  The server specific part is in dd_network_server.c,
 *          :  the client specific part in dd_network_client.c
 *      
 * Author   :  C.Witzig
 * Date     :  Jan 6, 1996 (based on old version with inetd)
 * Mods     :  Apr 15, 1996: network byte ordering introduced for Alpha
 *
 */



/* 
 * An Important Note on Network Byte Ordering:
 *
 * We transfer the data to/from the network in the network byte ordering.
 * The hosts in another byte ordering must find out who they are and 
 * swap accordingly.
 * Consequences: A transfer from e.g. Alpha to Alpha will swap the bytes
 * twice, but we mostly have SGIs...
 * Routines:
 * wrsocket/resocket     : do the actual read and write
 * writesocket/readsocket: do the byte swapping part. 
 *                         In order to avoid overhead, use a flag byte_swapping.
 * This way of handling the byte swapping is sort of ugly, but it saves
 * resources compared to using a more elegant solution such as xdr. This
 * makes sense if one has predominantly machines that have the network  
 * byte ordering but is clearly an limitation (and ugly implementation)
 * for the byte swapping problem.
 *
 * In certain applications the user data is not supposed to be swapped at
 * all (i.e. if the user application has a built in mechanism that
 * decodes the byte ordering itself.) 
 * The global variable ddn_trf_udata sets the byte swapping flag (char or
 * long variable) for every transfer in writesocket(..., ddn_trf_udata).
 * This variable can be set using the routin ddu_set_netswap(int swap_flag)
 */


#include <stdio.h>
#include <unistd.h>
#include <syslog.h>
#include <string.h>
#include <errno.h>
#include <signal.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>

#include <malloc.h>

#include <dd_log.h>
#include <dd_user.h>

#define TRUE 1
#define FALSE 0


/*-----------------------------------------------------------------------*/

/* 
 * Global variables that are declared as external in dd_network.h
 * and can be accessed from code outside this file.
 */

int dd2tcp = 0;
int i_am_dd2tcp_producer = 0;
int i_am_dd2tcp_consumer = 0;
int i_am_dd2tcp_server = 0;
int ddn_trf_usrdata = DDN_TRF_DATA;

int dd_sockfd;
FILE *dd_fp;

char dd_hostname[HOSTNAME_LEN];
char dd_peername[HOSTNAME_LEN];
char my_hostname[HOSTNAME_LEN];

/* 
 * Test data block to be sent back and forth for testing
*/
int test_data_send[TEST_DATA_LENGTH], test_data_rec[TEST_DATA_LENGTH];



/*-----------------------------------------------------------------------*/
 
/* 
 * Network Byte Ordering:
 * In order to avoid overhead, use a flag byte_swapping.
 */
int byte_swapping;


static int set_byte_swapping()
{
  unsigned int i = 0xabcd1234;
  int res = ( htonl(i) == i ) ? FALSE : TRUE;
  /*printf("byte swap flag == %d\n",res);*/
  return res;
}

int ddn_set_byte_swapping()
{
  byte_swapping = set_byte_swapping();

  return byte_swapping;
}

int ddn_set_dataswap(int flag)
/*
 * Sets ddn_trf_usrdata to enable/disable swapping of user data 
 * when transferred over the network.
 * flag = DDN_TRF_SWAP or DDN_TRF_NOSWAP
 */
{
  ddn_trf_usrdata = flag;

  return 0;
}

int ddn_set_dd_fp()
{
  if ( (dd_fp = fdopen(dd_sockfd,"r")) == NULL ) {
    sprintf(dd_log_msg,"ddn_set_dd_fp: error in fdopen: %s\n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  return 0;
}


void ddn_set_cmd(int cmd, int len, struct ddn_cmd *ddn_cmd)
{
  ddn_cmd->header = DDN_HEADER;
  ddn_cmd->cmd = cmd;
  ddn_cmd->len = len;
  return;
}


int ddn_set_hostnames(char *fname)
/*
 * Call for setting up the host names and find out
 * whether we are connected over TCP/IP
 * Argument can be either $DD_NAME  or $FIFONAME
 * with @hostname for specifing the host. We 
 * must therefore check that there is no mismatch
 * in case both names are defined (tricky as 
 * we might be using not the full IP address
 * (eg only bnlku7.phy instead of the full name).
 */ 
{
  int min, len1, len2;
  char *p2c, *p2v;
  struct sockaddr_in peername;
  int peernamelen = sizeof(peername);
  struct hostent *hp;

  if ( gethostname(my_hostname,HOSTNAME_LEN-1) < 0)
         perror("ddn_set_hostname: error in gethostname");

  byte_swapping = ddn_set_byte_swapping();

  if ( strlen(dd_hostname) > 0 ) {
    int diff;
    char *p2c = strrchr(fname,'@');
    if ( p2c ) {
      len1 = strlen(dd_hostname);
      len2 = strlen(++p2c);
      if ( len1 > len2 ) 
        diff = strncasecmp(dd_hostname, p2c, len2);
      else
        diff = strncasecmp(p2c, dd_hostname, len1);
      if ( diff != 0 ) {
        sprintf(dd_log_msg, "ddn_set_hostnames: mismatch in dd_hostname: *%s* *%s* \n",
                dd_hostname, p2c);
        dd_log(DD_ERROR, dd_log_msg);
        return -1;
      }
    }
  }

  if ( (p2c = strrchr(fname,'@')) != NULL )
    strcpy(dd_hostname, ++p2c);

  len1 = strlen(dd_hostname);
  len2 = strlen(my_hostname);
  if ( len1 > len2)
    len1 = len2;
  
  if ( (p2c == NULL) || (strncasecmp(my_hostname, dd_hostname, len1)  == 0) )
    dd2tcp = 0;
  else
    dd2tcp = 1;
  
  if ( i_am_dd2tcp_server ){
    if (getpeername(dd_sockfd, (struct sockaddr *)&peername, &peernamelen) < 0){
      sprintf(dd_log_msg,"ddn_set_hostnames: error in getpeername\n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
    
    p2v = (void*) &peername.sin_addr;
    hp = gethostbyaddr(p2v,sizeof(struct in_addr),AF_INET);
    if (hp == NULL){
      sprintf(dd_log_msg,"ddn_set_hostnames: error in gethostbyaddr\n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
    strncpy(dd_peername,hp->h_name,HOSTNAME_LEN-1);
  }
  else
    strcpy(dd_peername,"");
  
return 0;
}



/*
 * Calls for sending/receiving data over the network
 * --------------------------------------------------
 * See top of file for info on how the byte swapping is done.
 */

static int wrsocket(int fdes, void *p2source, int nbytes)
{
  int status;

  if ( (status = write(fdes, p2source, nbytes )) < 0){
    sprintf(dd_log_msg,"wrsocket: error in writing to socket - status = %d \n",status, strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  return 0;
}

static int resocket(int fdes, void *p2dest, int nbytes)
{
  int iread = 0;
  int status;

  while ( iread != nbytes ) {
    if ( (status = read(fdes, (char*)p2dest+iread, nbytes-iread)) < 0){
      sprintf(dd_log_msg," readsock: error in read - status = %d \n",status, strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
    else if ( status == 0 ) {
      sprintf(dd_log_msg," readsocket: read sees EOF -  status = %d \n",status, strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
      break;
    }
    iread+= status;
  }
  
  return 0;
}
    

int readsocket(int fdes, void *p2dest, int nbytes)
{
  int data_type;

  if ( resocket(fdes, (void*) &data_type, sizeof(data_type)) )
    return -1;

  if ( resocket(fdes, p2dest, nbytes) ) 
    return -1;

  if ( byte_swapping ) {
    data_type = ntohl(data_type);
    if ( data_type == DDN_TRF_LONG ) {
      int i, *p = (int*) p2dest;
      for (i=0; i++ <nbytes/4; p++)
        *p = ntohl(*p);
    }
  }

  return 0;
}


int writesocket(int fdes, void *p2source, int nbytes, int data_type)
{
  int i, status, byte_marker;
  int tmp, *p2data = p2source;

  tmp = htonl(data_type);
  if ( wrsocket(fdes, &tmp, sizeof(tmp)) < 0 )
    return -1;

  if ( byte_swapping ) {
    printf("swap\n");
    if ( data_type == DDN_TRF_LONG ) {
      int i, tmp, *p = (int*) p2source;
      for (i=0; i++<nbytes/4; p++) {
        tmp = htonl(*p);
        if ( wrsocket(fdes, &tmp, sizeof(tmp)) < 0 ) 
          return -1;
      }
      return 0;
    }
  }

  if ( wrsocket(fdes, p2data, nbytes) < 0 )
    return -1;

  return 0;
}




/*
 * Calls for sending/receiving an fev over the network
 * --------------------------------------------------
 * See top of file for info on how the byte swapping is done.
 * The fev structure has pointers which may be of different size
 * on different platforms. We therefore transfer it piece by piece
 */

int ddn_send_fev(int fdes, struct fifo_entry fev)
{
  if (writesocket(dd_sockfd,(void*)&fev.len, sizeof(fev.len), DDN_TRF_LONG ) < 0){
    sprintf(dd_log_msg,"ddn_send_fev: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if (writesocket(dd_sockfd,(void*)&fev.ctlw1, sizeof(fev.ctlw1), DDN_TRF_LONG ) < 0){
    sprintf(dd_log_msg,"ddn_send_fev: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if (writesocket(dd_sockfd,(void*)&fev.ctlb1, sizeof(fev.ctlb1), DDN_TRF_LONG ) < 0){
    sprintf(dd_log_msg,"ddn_send_fev: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if (writesocket(dd_sockfd,(void*)&fev.ctlw2, sizeof(fev.ctlw2), DDN_TRF_LONG ) < 0){
    sprintf(dd_log_msg,"ddn_send_fev: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if (writesocket(dd_sockfd,(void*)&fev.ctlb2, sizeof(fev.ctlb2), DDN_TRF_LONG ) < 0){
    sprintf(dd_log_msg,"ddn_send_fev: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  return 0;
}

int ddn_receive_fev(int fdes, struct fifo_entry *fev)
{
  if ( readsocket(dd_sockfd, (void*)&(fev->len), sizeof(fev->len) ) < 0){
    sprintf(dd_log_msg,"ddn_receive_fev: error in readsocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( readsocket(dd_sockfd, (void*) &(fev->ctlw1), sizeof(fev->ctlw1) ) < 0){
    sprintf(dd_log_msg,"ddn_receive_fev: error in readsocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  if ( readsocket(dd_sockfd, (void*) &(fev->ctlb1), sizeof(fev->ctlb1) ) < 0){
    sprintf(dd_log_msg,"ddn_receive_fev: error in readsocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  if ( readsocket(dd_sockfd, (void*) &(fev->ctlw2), sizeof(fev->ctlw2) ) < 0){
    sprintf(dd_log_msg,"ddn_receive_fev: error in readsocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  if ( readsocket(dd_sockfd, (void*) &(fev->ctlb2), sizeof(fev->ctlb2) ) < 0){
    sprintf(dd_log_msg,"ddn_receive_fev: error in readsocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  return 0;
}
