/*
 * dd_network_client.c : Code for the client part of the
 *          :  DD network connection (starts with ddnc_xxzx)
 * Author   :  C.Witzig
 * Date     :  Jan 6, 1996 (based on what was there from 92)
 * Mods     :  
 *          :  Feb 15, 1996: added ddnc_fifo_exist(char *fifoname)
 *          :  Apr 16, 1996: added ddnc_fifo_active(char *fifoname)
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <string.h>
#include <errno.h>
#include <signal.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>

#include <malloc.h>

#include <dd_log.h>
#include <dd_signal_def.h>
#include <dd_user.h>

#define TRUE 1
#define FALSE 0


static void *base_camp;
static size_t base_camp_size = 0;

static int server_running = FALSE;

static int ddnc_user_int_rec = FALSE;
static int ddnc_swap = TRUE;

/* For initialisation purposes */
static struct fifo_entry fev_init = {
  -1,-1,NULL,-1,-1,-1,-1,-1};

void ddnc_broken_pipe(int sig)
{
  sprintf(dd_log_msg,"ddnc_broken_pipe: pipe to server broken !!! \n");
  dd_log(DD_ERROR, dd_log_msg);
  sprintf(dd_log_msg,"ddnc_broken_pipe: REINITIALISE \n");
  dd_log(DD_ERROR, dd_log_msg);
  dd_sockfd = -1;
  dd_fp = NULL;
  server_running = FALSE;
  
  return;
}

void ddnc_user_int(int sig)
{
  sprintf(dd_log_msg, "user interrupt attached to signal %d seen\n",DD_USER_INTERRUPT);
  dd_log(DD_INFO, dd_log_msg);

  if ( DD_USER_INTERRUPT > 0 )
    signal(DD_USER_INTERRUPT, ddnc_user_int);

  ddnc_user_int_rec = TRUE;

  ddnc_exit();

  return;
}



int ddn_client_startup(char *dd_hostname)
{
  struct hostent *p_host;
  struct sockaddr_in server_addr;
  char host[100];

  strncpy(host, dd_hostname, sizeof(host));
  if ( (p_host = gethostbyname(dd_hostname)) == NULL ) {
   sprintf(dd_log_msg,"ddn_client_startup: error in gethostbyname: %s \n",strerror(errno));
   dd_log(DD_ERROR, dd_log_msg);
   return -1;
  }

  bzero( (char*) &server_addr, sizeof(server_addr) );
  server_addr.sin_family = AF_INET;
  bcopy(p_host->h_addr, &(server_addr.sin_addr.s_addr), p_host->h_length);
  server_addr.sin_port = htons(DD_SERVER_TCP_PORT);

  if ( (dd_sockfd = socket(AF_INET, SOCK_STREAM, 0) ) < 0 ) {
   sprintf(dd_log_msg,"ddn_client_startup: error in socket: %s \n",strerror(errno));
    return -1;
  }

  if ( connect(dd_sockfd, (struct sockaddr *)&server_addr, sizeof(server_addr)) < 0 ) {
   sprintf(dd_log_msg,"dd_client: error in connect: %s \n",strerror(errno));
   dd_log(DD_ERROR, dd_log_msg);
   return -1;
  }

  if ( ddn_set_dd_fp() ) {
    sprintf(dd_log_msg,"ddn_client_startup: error in ddn_set_dd_fp\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( DD_USER_INTERRUPT > 0 )
    signal(DD_USER_INTERRUPT, ddnc_user_int);

  signal(SIGPIPE, ddnc_broken_pipe);
  server_running = TRUE;
  return 0;
}



/*
 * Routine that tells us whether the client is still attached
 * over the network
 * -----------------------------------------------------------
 */

int ddn_server_running()
{
  return server_running;
}



/*
 *******************************************************
 * Calls executed by the client ddnc_xyz for ddu_calls *
 *******************************************************
 */

int ddnc_exists()
{
  struct ddn_cmd ddn_cmd;
  int status;
  
  ddn_set_cmd(DDN_CMD_EXISTS, 0, &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_stop: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);
  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}


int ddnc_attach()
{
  struct ddn_cmd ddn_cmd;
  int status;
  
  ddn_set_cmd(DDN_CMD_ATTACH, 0, &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_stop: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);
  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}


int ddnc_init(char *fname, struct fifo_mode fmode)
/*
 * Sends the necessary data to the server:
 * 1. environment variable DD_NAME
 * 2. length of the fifo name (incl. terminating null character!)
 * 3. fifoname
 * 4. fifo mode, fifo wait and control words.
 * 5. returns the status as obtained from ddu_init on the server side.
 */
{
  int len, status, nbytes;
  char *p2c;
  struct ddn_cmd ddn_cmd;

  if ( strstr(fname,"INPUT") != NULL )
    i_am_dd2tcp_producer = 1;
  else
    i_am_dd2tcp_consumer = 1;

  ddn_set_cmd(DDN_CMD_INIT, 0, &ddn_cmd);
  writesocket(dd_sockfd, (void*)&ddn_cmd, sizeof(ddn_cmd), DDN_TRF_LONG);
  fflush(dd_fp);

  if ( (p2c = getenv("DD_NAME")) == NULL ) {
    len = 0;
    writesocket(dd_sockfd, &len, sizeof(len), DDN_TRF_LONG);
  }
  else {
    len = strlen(p2c)+1;
    writesocket(dd_sockfd, &len, sizeof(len), DDN_TRF_LONG);
    fflush(dd_fp);
    writesocket(dd_sockfd, p2c, len, DDN_TRF_CHAR);
  }
  fflush(dd_fp);

  len = strlen(fname)+1;
  writesocket(dd_sockfd, &len, sizeof(len), DDN_TRF_LONG);
  fflush(dd_fp);
  writesocket(dd_sockfd, fname, len, DDN_TRF_CHAR);
  fflush(dd_fp);

/* transfer fmode individually - has int* member in it! */
  writesocket(dd_sockfd, (void*)&fmode.mode, sizeof(fmode.mode), DDN_TRF_LONG );
  writesocket(dd_sockfd, (void*)&fmode.wait, sizeof(fmode.wait), DDN_TRF_LONG );
  writesocket(dd_sockfd, (void*)&fmode.prescale, sizeof(fmode.prescale), DDN_TRF_LONG );
  writesocket(dd_sockfd, (void*)&fmode.suser, sizeof(fmode.suser), DDN_TRF_LONG );

  if ( writesocket(dd_sockfd, fmode.p2ctl, 4*FIFO_HDR_CTL_LEN, DDN_TRF_LONG) < 0 ) {
    sprintf(dd_log_msg,"ddnc_init: error in writesocket for fmode.p2ctl \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
    
  fflush(dd_fp);

  readsocket(dd_sockfd, &status, sizeof(status));

  atexit(ddnc_exit);
  return status;
}

int ddnc_start()
{
  struct ddn_cmd ddn_cmd;
  int status;
  
  ddn_set_cmd(DDN_CMD_START, 0, &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_start: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);
  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}

int ddnc_stop()
{
  struct ddn_cmd ddn_cmd;
  int status;
  
  ddn_set_cmd(DDN_CMD_STOP, 0, &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
         sprintf(dd_log_msg,"ddnc_stop: error in writesocket \n");
         dd_log(DD_ERROR, dd_log_msg);
         return -1;
  }
  fflush(dd_fp);
  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}

int ddnc_req_fev(size_t ev_size, struct fifo_entry *fev)
{
  struct fifo_entry fev1;

  if ( ev_size > base_camp_size/4 ){
    if ( base_camp_size > 0 ) {
      free(base_camp);
      base_camp_size = 0;
    }
    base_camp_size = 4*ev_size;
    base_camp = malloc(base_camp_size);
    if (base_camp == NULL){
      sprintf(dd_log_msg,"ddnc_req_fev: malloc: %i !!!\n",base_camp_size);
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
  }

  fev1 = fev_init;
  fev1.len  = ev_size;
  fev1.p2da = (int*)base_camp;

  *fev = fev1;

  return 0;
}

int  ddnc_put_fev(struct fifo_entry fev)
{
  int status;
  struct ddn_cmd ddn_cmd;

  ddn_set_cmd(DDN_CMD_PUT_FEV, fev.len, &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_put_fev: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  if ( ddn_send_fev(dd_sockfd, fev) < 0 ) {
    sprintf(dd_log_msg,"ddnc_put_fev: error in ddn_send_fev \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if (writesocket(dd_sockfd,(void*)fev.p2da, 4*fev.len, ddn_trf_usrdata) < 0){
         sprintf(dd_log_msg,"ddnc_put_fev: error in writesocket \n");
         dd_log(DD_ERROR, dd_log_msg);
         return -1;
  }
  fflush(dd_fp);

  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}


int  ddnc_ins_fev(struct fifo_entry fev)
{
  int status;
  struct ddn_cmd ddn_cmd;

  ddn_set_cmd(DDN_CMD_INS_FEV, fev.len, &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_ins_fev: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  if ( ddn_send_fev(dd_sockfd, fev ) < 0){
    sprintf(dd_log_msg,"ddn_ins_fev: error in ddn_send_fev \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  if (writesocket(dd_sockfd,(void*)fev.p2da, 4*fev.len, ddn_trf_usrdata ) < 0){
    sprintf(dd_log_msg,"ddnc_ins_fev: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  fflush(dd_fp);

  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}

int  ddnc_brc_fev(struct fifo_entry fev, int wait_flag)
{
  int status;
  struct ddn_cmd ddn_cmd;

  ddn_set_cmd(DDN_CMD_BRC_FEV, fev.len, &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_brc_fev: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  if (writesocket(dd_sockfd,&wait_flag,sizeof(wait_flag), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_brc_fev: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  if ( ddn_send_fev(dd_sockfd, fev) < 0 ) {
    sprintf(dd_log_msg,"ddnc_brc_fev: error in ddn_send_fev \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if (writesocket(dd_sockfd,(void*)fev.p2da, 4*fev.len, ddn_trf_usrdata ) < 0){
    sprintf(dd_log_msg,"ddnc_brc_fev: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}

int ddnc_set_brc(int brc_flag)
{
  int status;
  struct ddn_cmd ddn_cmd;

  ddn_set_cmd(DDN_CMD_SET_BRC, sizeof(brc_flag), &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_set_brc: error in writesocket cmd\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  if (writesocket(dd_sockfd,&brc_flag,sizeof(brc_flag), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_set_brc: error in writesocket brc_flag\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);
  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}

int ddnc_set_protection(int protection)
{
  int status;
  struct ddn_cmd ddn_cmd;

  ddn_set_cmd(DDN_CMD_SET_PROTECTION, sizeof(protection), &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_set_protection: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);
  
  if (writesocket(dd_sockfd,&protection,sizeof(protection), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_set_protection: error in writesocket protection\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}

int ddnc_set_dd_wait(int dd_wait)
{
  int status;
  struct ddn_cmd ddn_cmd;

  ddn_set_cmd(DDN_CMD_SET_DD_WAIT, sizeof(dd_wait), &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_set_protection: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  if (writesocket(dd_sockfd,&dd_wait,sizeof(dd_wait), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_set_dd_wait: error in writesocket dd_wait\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}


int ddnc_set_netswap(int swap)
{
  int status;
  struct ddn_cmd ddn_cmd;

  ddn_set_cmd(DDN_CMD_SET_NETSWAP, sizeof(swap), &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_set_protection: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  if (writesocket(dd_sockfd,&swap,sizeof(swap), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_set_netswap: error in writesocket swap\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}



int ddnc_set_reqcnt(int cnt)
{
  int status;
  struct ddn_cmd ddn_cmd;

  ddn_set_cmd(DDN_CMD_SET_REQCNT, sizeof(cnt), &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_set_reqcnt: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);
  if (writesocket(dd_sockfd,&cnt,sizeof(cnt), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_set_reqcnt: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}


int ddnc_get_free_buffers()
{
  int status;
  struct ddn_cmd ddn_cmd;

  ddn_set_cmd(DDN_CMD_GET_FREE_BUFFERS, 0, &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_set_reqcnt: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}

int ddnc_fifo_exist(char *fifoname)
{
  int status, nbytes;
  struct ddn_cmd ddn_cmd;

  nbytes = strlen(fifoname) + 1;
  ddn_set_cmd(DDN_CMD_FIFO_EXIST, nbytes , &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_fifo_exist: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);
  
  if (writesocket(dd_sockfd, fifoname, nbytes, DDN_TRF_CHAR) < 0 ){
    sprintf(dd_log_msg,"ddnc_fifo_exist: error in writesocket protection\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}

int ddnc_fifo_active(char *fifoname)
{
  int status, nbytes;
  struct ddn_cmd ddn_cmd;

  nbytes = strlen(fifoname) + 1;
  ddn_set_cmd(DDN_CMD_FIFO_ACTIVE, nbytes , &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_fifo_exist: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);
  
  if (writesocket(dd_sockfd, fifoname, nbytes, DDN_TRF_CHAR) < 0 ){
    sprintf(dd_log_msg,"ddnc_fifo_exist: error in writesocket protection\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}



int ddnc_get_fev(struct fifo_entry *fev)
{
  struct fifo_entry fev1;
  int status;
  struct ddn_cmd ddn_cmd;

  ddn_set_cmd(DDN_CMD_GET_FEV, 0, &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_put_fev: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  if ( ddn_receive_fev(dd_sockfd, &fev1) < 0 ) {
    sprintf(dd_log_msg,"ddnc_get_fev: error in ddn_receive_fev \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( fev1.len > 0 ){
    if ( fev1.len > base_camp_size/4 ){
      if ( base_camp_size > 0 ) {
        free(base_camp);
        base_camp_size = 0;
      }
      base_camp_size = 4*fev1.len;
      base_camp = malloc(base_camp_size);
      if (base_camp == NULL){
        sprintf(dd_log_msg,"ddnc_get_fev: malloc: could not make base camp !!!\n");
        dd_log(DD_ERROR, dd_log_msg);
        return -1;
      }
    }
    
    fev1.p2da = (int*) base_camp;
    if ( readsocket(dd_sockfd, (void*)base_camp, 4*fev1.len) < 0){
      sprintf(dd_log_msg,"ddnc_get_fev: error in readsocket \n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
  }

  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  if ( !(status) && (fev1.len == 0) )
    status = 1;
  *fev = fev1;

  return status;
}



int ddnc_close()
{
  int status;
  struct ddn_cmd ddn_cmd;

  ddn_set_cmd(DDN_CMD_CLOSE, 0, &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_close: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}

void ddnc_exit()
{
  struct ddn_cmd ddn_cmd;

  if ( dd_sockfd >= 0 ) {
    ddn_set_cmd(DDN_CMD_EXIT, 0, &ddn_cmd);
    if ( writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
      sprintf(dd_log_msg,"ddnc_exit: error in writesocket \n");
      dd_log(DD_ERROR, dd_log_msg);
    }
    fflush(dd_fp);

    dd_nap(0.1);
    close(dd_sockfd);
    dd_sockfd = -1;
    dd_fp = NULL;
    server_running = FALSE;
  }

  return;
}




/*
 *******************************************************
 * Calls executed by the client ddnc_xyz for dds calls *
 *******************************************************
 */

/*
 * the structure of the system commands is always the
 * same, in contrast to the user commands
 */

static int ddsnc_sys_cmd(int cmd, int add_data)
{
  struct ddn_cmd ddn_cmd;
  int status;
  
  ddn_set_cmd(cmd, add_data, &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
         sprintf(dd_log_msg,"ddsnc_sys_cmd: error in writesocket \n");
         dd_log(DD_ERROR, dd_log_msg);
         return -1;
  }
  fflush(dd_fp);
  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}


int ddsnc_create()
{
  struct ddn_cmd ddn_cmd;
  int status, len;
  char *p2c;

  ddn_set_cmd(DDSN_CMD_CREATE, 0, &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddsnc_create: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  len = 0;
  if ( (p2c = getenv("DD_NAME")) != NULL ) {
    len = strlen(p2c);
    len++;
  }
  writesocket(dd_sockfd,&len,sizeof(len), DDN_TRF_LONG);
  fflush(dd_fp);
  if ( len > 0 ) 
    writesocket(dd_sockfd, p2c, len, DDN_TRF_CHAR);
  fflush(dd_fp);
  len = 0;
  if ( (p2c = getenv("DD_CONFIGURATION")) != NULL )
    len = strlen(p2c);
  if ( len > 0 )
    len++;
  writesocket(dd_sockfd,&len,sizeof(len), DDN_TRF_LONG);
  fflush(dd_fp);
  if ( len > 0 ) 
    writesocket(dd_sockfd, p2c, len, DDN_TRF_CHAR);
  fflush(dd_fp);

  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  return status;
}

int ddsnc_init()
{
  return ddsnc_sys_cmd(DDSN_CMD_INIT, 0);
}

int ddsnc_wait4proc(int ntime_ms)
{
  return ddsnc_sys_cmd(DDSN_CMD_WAIT4PROC, ntime_ms);
}

int ddsnc_wakeup()
{
  return ddsnc_sys_cmd(DDSN_CMD_WAKEUP, 0);
}

int ddsnc_prepare_shutdown()
{
  return ddsnc_sys_cmd(DDSN_CMD_PREPARE_SHUTDOWN, 0);
}

int ddsnc_reset()
{
  return ddsnc_sys_cmd(DDSN_CMD_RESET, 0);
}

int ddsnc_flush(int brc_flg)
{
  return ddsnc_sys_cmd(DDSN_CMD_FLUSH, brc_flg);
}

int ddsnc_close(int enforce)
{
  return ddsnc_sys_cmd(DDSN_CMD_CLOSE, enforce);
}



/*
 *******************************************************
 * Other calls                                         *
 *******************************************************
 */

int ddnc_test_link()
{
  static int first = 0;
  int i,len;
  int status,error,err_trf;
  struct ddn_cmd ddn_cmd;

  ddn_set_cmd(DDN_CMD_TEST_TRF, sizeof(test_data_send), &ddn_cmd);

  if ( first == 0){
    first = 1;
    for (i=0;i<sizeof(test_data_send)/4;i++)
      test_data_send[i] = i;
  }

  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_test_link: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  if (writesocket(dd_sockfd,test_data_send,sizeof(test_data_send), DDN_TRF_LONG) < 0 ){
    sprintf(dd_log_msg,"ddnc_test_link: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fflush(dd_fp);

  if (readsocket(dd_sockfd,test_data_rec,sizeof(test_data_send)) < 0 ){
    sprintf(dd_log_msg,"ddnc_test_link: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if (readsocket(dd_sockfd, &status, sizeof(status)) < 0 ){
    sprintf(dd_log_msg,"ddnc_test_link: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

/*  
  for (i=0; i<sizeof(test_data_send)/4; i++)
   sprintf(dd_log_msg,"test data %d %d %d \n",i,test_data_send[i], test_data_rec[i]);
   dd_log(DD_ERROR, dd_log_msg);
*/

  err_trf = 0;
  for (i=0;i<sizeof(test_data_send)/4;i++){
    if (test_data_send[i] != test_data_rec[i])
      err_trf++;
  }
  if (err_trf != 0){
    sprintf(dd_log_msg,"ddn_network_send_test: %i transmission errors occured!!! \n",err_trf);
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  return 0;
}
