/*
 * dd_link_util.c : Utility routines for the dd_link
 *
 * Author   :  C.Witzig
 * Date     :  Jan 10, 1996
 *
 * Mods     :  Oct 7, 1996: fixed bug in create_dd_link (transfer
 *                    mode words individually (discovered by M.Leitch)
 *
 */


#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/times.h>
#include <sys/stat.h>
#include <signal.h>

#include <dd_sys.h>
#include <dd_user.h>
#include <dd_usr_brccmd.h>
#include <dd_link_util.h>

#define TRUE 1
#define FALSE 0

void dd_link_flush(int w1, int b1)
/*
 * does the flush over the network to the remote
 * DD system
 */
{
  if ( ddsnc_flush() ) 
    printf("dd_link: error in dd_link_flush\n");

  return;
}  


void dd_link_define_brccmd()
{
  ddu_set_brccmd(DD_BRCCMD_FLUSH, &dd_link_flush);
}
  

int create_dd_link(char *to_dd, char *other_host)
{
/*
 * start the server on the other end and give him
 * the command to attach to the DD system
 * We also let him become a DD master (and let him
 * thus create the system if it is not in use yet).
 *
 * NOTE: we are basically redoing the work that ddu
 * is supposed to do. But as we have only one DD per
 * process at this time, we have to duplicate the work.
 * This is clearly a current limitation.
 */

/* 
 * fifo setup parameters 
 */
  char *fifo = "INPUT";
  int fctl[4] = {-1, -1, -1, -1};
  struct fifo_mode fmode = {FMODE_ALL, FWAIT_SLEEP, FMODE_NOPRESCALE, FMODE_MULTI_USER, NULL};

  struct ddn_cmd ddn_cmd;
  int status, len;

  if ( ddn_client_startup(other_host) ) {
    printf("dd_link: error in ddn_client_startup\n");
    return -1;
  }

  ddn_set_cmd(DDSN_CMD_CREATE, 0, &ddn_cmd);
  if (writesocket(dd_sockfd,&ddn_cmd,sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ){
    fprintf(stderr,"ddsnc_create: error in writesocket \n");
    return -1;
  }
  fflush(dd_fp);

  len = strlen(to_dd);
  writesocket(dd_sockfd,&len,sizeof(len), DDN_TRF_LONG);
  fflush(dd_fp);
  if ( len > 0 ) 
    writesocket(dd_sockfd, to_dd, len, DDN_TRF_CHAR);
  fflush(dd_fp);

/* DD configuration is read from the ddd conf file */
  len = 0;
  writesocket(dd_sockfd,&len,sizeof(len), DDN_TRF_LONG);
  fflush(dd_fp);

  readsocket(dd_sockfd,(void*)&status,sizeof(status));
  if ( status != 0 ) {
    printf("dds_create for DD %s failed on host %s\n",
           to_dd, other_host);
    return -1;
  }

/* now duplicate ddu_init for attaching to INPUT fifo */
  ddn_set_cmd(DDN_CMD_INIT, 0, &ddn_cmd);
  writesocket(dd_sockfd, (void*)&ddn_cmd, sizeof(ddn_cmd), DDN_TRF_LONG);
  fflush(dd_fp);
  
  len = strlen(to_dd)+1;
  writesocket(dd_sockfd,&len,sizeof(len), DDN_TRF_LONG);
  fflush(dd_fp);
  if ( len > 0 ) 
    writesocket(dd_sockfd, to_dd, len, DDN_TRF_CHAR);
  fflush(dd_fp);

  len = strlen(fifo)+1;
  writesocket(dd_sockfd, &len, sizeof(len), DDN_TRF_LONG);
  fflush(dd_fp);
  writesocket(dd_sockfd, fifo, len, DDN_TRF_CHAR);
  fflush(dd_fp);

  fmode.p2ctl = fctl;

/* 
 * transfer the words individually 
 */
  writesocket(dd_sockfd, (void*)&fmode.mode, sizeof(fmode.mode), DDN_TRF_LONG);
  writesocket(dd_sockfd, (void*)&fmode.wait, sizeof(fmode.wait), DDN_TRF_LONG);
  writesocket(dd_sockfd, (void*)&fmode.prescale, sizeof(fmode.prescale), DDN_TRF_LONG);
  writesocket(dd_sockfd, (void*)&fmode.suser, sizeof(fmode.suser), DDN_TRF_LONG);

  writesocket(dd_sockfd, fctl, sizeof(fctl), DDN_TRF_LONG );
  fflush(dd_fp);

  readsocket(dd_sockfd, &status, sizeof(status));
  if ( status != 0 ) {
    printf("ddu_init for DD %s failed on host %s\n",
           to_dd, other_host);
    return -1;
  }

  atexit(ddnc_exit);

  return 0;
}


int delete_dd_link()
{
/*
 * close down the link and detach from the local DD
 * system
 */
  ddnc_exit();
  
  ddu_stop();
  ddu_set_brc(DD_BRC_OFF);

  ddu_close();

  return 0;
}

int do_dd_trf()
{
  int status;
  struct fifo_entry fev;

  if ( (status = ddu_get_fev(&fev)) != 0 )
    return status;

  if ( ddnc_put_fev(fev) < 0 ) {
    ddu_put_fev(fev);
    return -1;
  }

  if ( ddu_put_fev(fev) < 0 )
    return -1;

  return 0;
}



int do_dd_mtrf()
/*
 * example how multiple events can be transfered over the 
 * DD link. We first get the desired number of events, then
 * transfer them and read the status of the server at the other
 * end. Make sure you free up all the events before you exit.
 * Note: The status is read after the fev events are put back
 * into the local DD system to allow the server to retrieve the data
 * out of the DD system.
 */
{
  struct fifo_entry fev[DD_LINK_MAXEV];
  struct ddn_cmd ddn_cmd;
  int status, nfev, ifev = 0;
  int n_lw = 0;
  int trf_header = DDN_TRF_HEADER;
  int error = FALSE;

#ifdef UNAME_SGI
  struct tms stime, etime;
  time_t t1, t2;
  double rate, delta_t1, delta_t2;
#endif

#ifdef UNAME_SGI
  times(&stime);
  time(&t1);
#endif

  while ( (status = ddu_get_fev(&fev[ifev]) ) == 0 ) {
    n_lw += fev[ifev].len;
    if ( n_lw > DD_LINK_MAXLONGWORDS )
      break;
    if ( ++ifev >= DD_LINK_MAXEV )
      break;
  }

  if ( status < 0 ) {
    sprintf(dd_log_msg, "dd_link: ddu_get_fev returns %d\n",status);
    dd_log(DD_ERROR, dd_log_msg);
    error = TRUE;
  }

  ddn_set_cmd(DDN_CMD_PUT_NFEV, ifev, &ddn_cmd);
  if ( writesocket(dd_sockfd, &ddn_cmd, sizeof(ddn_cmd), DDN_TRF_LONG) < 0 ) {
    sprintf(dd_log_msg,"ddnc_put_fev: error in writesocket \n");
    dd_log(DD_ERROR, dd_log_msg);
    error = TRUE;
  }
  fflush(dd_fp);
  
  nfev = ifev;  
  for (ifev = 0; (ifev < nfev) && !error ; ifev++) {
    if (writesocket(dd_sockfd,(void*)&trf_header, sizeof(trf_header), DDN_TRF_LONG ) < 0){
      sprintf(dd_log_msg,"ddnc_put_fev: error in writesocket \n");
      dd_log(DD_ERROR, dd_log_msg);
      error = TRUE;
    }
    if (writesocket(dd_sockfd, (void*)&fev[ifev], sizeof(struct fifo_entry), DDN_TRF_LONG ) < 0){
      sprintf(dd_log_msg,"ddnc_put_fev: error in writesocket \n");
      dd_log(DD_ERROR, dd_log_msg);
      error = TRUE;
    }
    
    if (writesocket(dd_sockfd,(void*)fev[ifev].p2da, 4*fev[ifev].len , DDN_TRF_DATA) < 0){
      sprintf(dd_log_msg,"ddnc_put_fev: error in writesocket \n");
      dd_log(DD_ERROR, dd_log_msg);
      error = TRUE;
    }
    fflush(dd_fp);
  }

  for (ifev = 0; ifev < nfev; ifev++) {
    if ( ddu_put_fev(fev[ifev]) < 0 ) {
      sprintf(dd_log_msg, "dd_link: ddu_put_fev returns %d\n",status);
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
  }
  
  readsocket(dd_sockfd,(void*)&status,sizeof(status));

  if ( nfev == 0 )
    dd_nap(0.1);

#ifdef UNAME_SGI
  if ( nfev > 0 ) {
    times(&etime);
    time(&t2);
    delta_t1 =  (etime.tms_utime + etime.tms_stime) - (stime.tms_utime + stime.tms_stime);
    delta_t1 = delta_t1 / ((double)CLK_TCK);
    delta_t2 = t2 - t1;
    rate = n_lw*4.0 / (delta_t2 * 1024.0);
    printf("dd_link: do_dd_mtrf: #%d events : rate: %6.2f CPU frac: %6.2f delta_t: %6.2f time: %s ", 
           nfev, rate, (delta_t1/delta_t2), delta_t2, ctime(&t2));
  }
#endif  

/*
 * if an error occured, we might have to overwrite any good status
 * of another command
 */
  if ( error ) 
    status = -1;

  return status;
}
