/*
 * dd_link.c : Main link between two DD systems to ship 
 *           : data from one DD system to another one
 *           : in a very general way
 *
 * Author   :  C.Witzig
 * Date     :  Jan 10, 1996
 *
 * Mods     :  Oct  8, 1996: added REQCNT in conf.file
 *
 */


/*
 * Usage:   
 *-------
 * dd_link from to file
 * where: from : DD system to retrieve the data from
 *        to   : DD system where the data has to be sent to
 *        file : optional - filename where to read the setup 
 *               parameters. By default the dd_link will work
 *               in sleep mode and ship events ONREQ 
 *
 */

#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <signal.h>
#include <dd_sys.h>
#include <dd_user.h>
#include <dd_link_util.h>

#define TRUE 1
#define FALSE 0

int go_on;

void
sig_handler ()
{
  go_on = 0;
  return;
}

main(int argc,char **argv)
{
/* arguments */
  char from_dd[100], to_dd[100], filename[100];
/* 
 * fifo setup parameters 
 */
  int fctl[4] = {-1, -1, -1, -1};
  struct fifo_mode fmode;
  int freq_cnt = 30;

/* DD at the other end */
  char other_host[100];

/* trf counter */
  int n_trf_done = 0;

/* other parameters */
  int status;
  int len1, len2;
  char *p2file;
  char hostname[100], conf_file[100], my_hostname[100];
  char fname[20];

/* environment variables */
  static char dd_env_name[100];

/*
 * check arguments and set DD parameters 
 */
  if (argc < 3 || argc > 4 ) {
    printf("DD link: wrong syntax: \n");
    printf("         correct: dd_link from_dd to_dd file\n");
    return;
  }

  strcpy(from_dd, argv[1]);
  strcpy(to_dd, argv[2]);
  sprintf(fname, "%s->%s",from_dd,to_dd);

  fmode.mode = FMODE_ONREQ;
  fmode.wait = FWAIT_ASYNC;
  fmode.prescale = 1;
  fmode.suser = FMODE_MULTI_USER;
  fmode.p2ctl = fctl;


/*
 * if the DD option file is defined read it
 * and override the default values of fmode
 */
  if ( argc == 4 ) {
    FILE *fp;
    char line[100], crap[100];
    strcpy(filename, argv[3]);

    if ( (fp = fopen(filename,"r")) == NULL ) {
      printf("dd_link: cannot open file %s\n",filename);
      exit(1);
    }

    while ( fgets(line, sizeof(line), fp) ) {
      if ( line[0] != '#') {
        if ( strncasecmp(line, "MODE", strlen("MODE")) == 0 )
          sscanf(line, "%s %d", crap, &fmode.mode);
        if ( strncasecmp(line, "SEL", strlen("SEL")) == 0 ) 
          sscanf(line, "%s %d %d %d %d",crap, &fctl[0], &fctl[1], &fctl[2], &fctl[3]);
        else if ( strncasecmp(line, "PS_FACTOR", strlen("PS_FACTOR")) == 0 )
          sscanf(line, "%s %d", crap, &fmode.prescale);
        else if ( strncasecmp(line, "WAIT", strlen("WAIT")) == 0 )
          sscanf(line, "%s %d", crap, &fmode.wait);
        else if ( strncasecmp(line, "REQCNT", strlen("REQCNT")) == 0 )
          sscanf(line, "%s %d", crap, &freq_cnt);
      }
    }
  }
    


/*
 * find the DD systems in the configuration file and check the
 * hosts for both ends
 */
  if ( (p2file = getenv("DDD_CONFIGURATION_FILE")) == NULL ) {
    printf("Variable DDD_CONFIGURATION_FILE not defined\n");
    exit(1);
  }

  if ( dds_find_dd_system(p2file, from_dd, hostname, conf_file) != 0 ) {
    printf("DD system %s not found in DDD_CONFIGURATION_FILE %s\n",
           from_dd, p2file);
    exit(1);
  }

  gethostname(my_hostname, sizeof(hostname));
  len1 = strlen(my_hostname);
  len2 = strlen(hostname);
  if (len1 > len2)
    len1 = len2;

  if ( strncmp(my_hostname, hostname, len1) != 0 ) {
    printf("dd_link MUST be started on host %s and not %s\n",
           hostname, my_hostname);
    exit(1);
  }

  if ( dds_find_dd_system(p2file, to_dd, other_host, conf_file) != 0 ) {
    printf("DD system %s not found in DDD_CONFIGURATION_FILE %s\n",
           to_dd, p2file);
    exit(1);
  }
  
  sprintf(dd_env_name,"DD_NAME=%s",from_dd);
  if ( putenv(dd_env_name) != 0 ) {
    printf("Error in putenv %s\n",dd_env_name);
    exit(1);
  }

/*
 * now the actual work - create the link and do the trf
 */
  if ( ddu_attach() ) {
    printf("dd_link: ddu_attach failed \n");
    return;
  }

  if ( (status = ddu_init(fname, fmode) ) != 0 ) {
    printf("dd_link: ddu_init: status returned %d\n",status);
    return;
  }

  if ( (status = ddu_set_reqcnt(freq_cnt)) != 0 ) {
    printf("dd_link: error in ddu_set_reqcnt\n");
    ddu_close();
    return;
  }

  if ( (status = ddu_set_brc(DD_BRC_USMAIL) ) != 0 ) {
    printf("dd_link: error in ddu_set_brc\n");
    ddu_close();
    return;
  }

  dd_link_define_brccmd();

  if ( create_dd_link(to_dd, other_host) ) {
    printf("dd_link: error in creating DD link\n");
    return;
  }

  if ( ddu_start() ) {
    printf("dd_link: ddu_init: status returned %d\n",status);
    delete_dd_link();
    return;
  }
    
  printf("dd_link: %d started\n", getpid());

  while ( ( status = do_dd_trf() ) >= 0 ) {
    n_trf_done++;
    if ( (n_trf_done%100) == 0 )
      printf("now I am at trf #%d\n",n_trf_done);
    if ( status > 0 ) {
      printf("dd_link: %d got no events \n",getpid());
      dd_nap(0.5);
    }
  }
 
  if ( status ) 
    printf("dd_link: pid=%d returns status %d\n", getpid(), status);

  if ( delete_dd_link() ) {
    printf("dd_link: error in deleting DD link\n");
    return;
  }

  return;
}
       

