/*
 *  DD_DCOMLIB:
 *
 *  The processes attached to the DD system share a shm segment
 *  where they register, keep their statistics and 
 *  and they syncronize themself through the locking mechanism 
 *  dd_dcom_lock
 * 
 *  The code has been pirated from S.Adlers E787 dcom code
 *  but extended and adapted to a standalone 
 *  DD system.
 * 
 *  Apr 5, 1995 CW
 *
 *  Mods: 
 * 
 */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <sys/msg.h>

#include <dd_dcom.h>
#include <dd_log.h>

#define TRUE 1
#define FALSE 0

#define DD_DCOM_INIT_VALUE 0

static struct dd_dcom_str *dd_dcom_addr;
static int first = TRUE;

static int dd_dcom_semid = -1;
static int dd_dcom_shmid = -1;
static int dd_dcom_nsem = -1;

#ifdef sun
union {
  int val;
  struct semid_ds *buf;
  ushort *array;
} dd_dcom_arg;
#else 
union semun dd_dcom_arg;
#endif

static ushort array[MAX_DD_SEM];



/*
 * The next four routines are used to lock parts of the dcom area.
 * ---------------------------------------------------------------
 */

int dd_dcom_lock()
{
/*
 * locks dd_dcom
 * protect against interrupted system calls
 */
  int waiting;
  struct sembuf dcom_sembuf;

  dcom_sembuf.sem_num = 0;
  dcom_sembuf.sem_op  = -1;
  dcom_sembuf.sem_flg = SEM_UNDO;

  waiting = 1;
  while ( waiting > 0 ){
    if ( (waiting = semop(dd_dcom_semid,&dcom_sembuf,1)) != 0){
      if (errno = EINTR)
	waiting = 1;
      else{
	sprintf(dd_log_msg,"dcom_lock: error in semop: %s\n",strerror(errno));
	dd_log(DD_ERROR, dd_log_msg);
	return -1;
      }
    }
  }
  return 0;
}

int dd_dcom_unlock()
{
  struct sembuf dcom_sembuf;

  dcom_sembuf.sem_num = 0;
  dcom_sembuf.sem_op  = 1;
  dcom_sembuf.sem_flg = SEM_UNDO;

  if (semop(dd_dcom_semid,&dcom_sembuf,1) != 0){
    sprintf(dd_log_msg,"dcom_lock: error in semop: %s\n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  return 0;
}

int dd_dcom_set_procsem(int isem, int semval)
{
  struct sembuf dcom_sembuf;

  dcom_sembuf.sem_num = isem;
  dcom_sembuf.sem_op  = semval;
  dcom_sembuf.sem_flg = SEM_UNDO;

  if (semop(dd_dcom_semid,&dcom_sembuf,1) != 0){
    sprintf(dd_log_msg,"dd_dcom_set_procsem: error in semop: %s\n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  return 0;
}

int dd_dcom_get_semcnt(int *semcnt)
{
  int i,len;
  struct semid_ds dd_dcom_semid_ds;

  dd_dcom_arg.buf = &dd_dcom_semid_ds;
  if (semctl(dd_dcom_semid,0,IPC_STAT,dd_dcom_arg) != 0){
    sprintf(dd_log_msg," dd_dcom_get_semcnt: semctl %s \n ",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  len = dd_dcom_arg.buf->sem_nsems;
  dd_dcom_arg.array = array;
  if (semctl(dd_dcom_semid,0,GETALL,dd_dcom_arg) != 0){
    sprintf(dd_log_msg," dd_dcom_get_semcnt: semctl %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  for (i=0;i<len;i++) 
    semcnt[i] = dd_dcom_arg.array[i];

  return 0;
}



/*
 * dd_dcomlib routines
 */


int dd_dcom_exists(char *filename)
{
  int flag;
  key_t key;
  
  flag = 0666;
  if ( (key = ftok(filename,DD_DCOM_SHM_KEY)) < 0 ) {
    return FALSE;
  }

  dd_dcom_shmid = shmget(key,sizeof(struct dd_dcom_str),flag);

  return ( (dd_dcom_shmid < 0) ? FALSE : TRUE);
}

struct dd_dcom_str *dd_dcom_attach(char *filename)
{
  int nsem;
  int init, flag;
  struct shmid_ds shm_buff;
  key_t key;

  init = FALSE;
  if ( (key = ftok(filename,DD_DCOM_SHM_KEY)) < 0 ) {
    sprintf(dd_log_msg,"dd_dcom_attach: error in ftok: %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return NULL;
  }
  flag = 0666;
  dd_dcom_semid = semget(key, MAX_DD_SEM, flag);
  if ( dd_dcom_semid == -1 ) {
    flag = 0666 | IPC_CREAT;
    if ( (dd_dcom_semid = semget(key, MAX_DD_SEM, flag)) < 0 ){
      sprintf(dd_log_msg,"dd_dcom_attach: error in semget: %s\n",strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
      return NULL;
    }
    dd_dcom_arg.array = array;
    
      dd_dcom_arg.array[0] = 1;
      if (semctl(dd_dcom_semid,0,SETALL,dd_dcom_arg) < 0){
	sprintf(dd_log_msg," dd_dcom_attach: error in semctl %s ", strerror(errno));
	dd_log(DD_ERROR, dd_log_msg);
	return NULL;
      }
  }
  
  dd_dcom_lock();
  
  flag = 0666;
  dd_dcom_shmid = shmget(key,sizeof(struct dd_dcom_str),flag);
  if (dd_dcom_shmid == -1) {
    flag = 0666 | IPC_CREAT;
    dd_dcom_shmid = shmget(key,sizeof(struct dd_dcom_str),flag);
    if ( dd_dcom_shmid < 0 ) {
      sprintf(dd_log_msg,"dd_dcom_attach: error in shmget: %s\n",strerror(errno));
      dd_log(DD_ERROR, dd_log_msg);
      dd_dcom_unlock();
      return NULL;
    }
    else 
      init = TRUE;
  }
  
  if ( (dd_dcom_addr = (struct dd_dcom_str*) shmat(dd_dcom_shmid,NULL, 0)) == NULL ){
    sprintf(dd_log_msg,"dd_dcom_attach: error in shmat: %s\n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    dd_dcom_unlock();
    return NULL;
  }
  
  if ( init ) 
    memset(dd_dcom_addr, DD_DCOM_INIT_VALUE, sizeof(struct dd_dcom_str) );
  
  /*first = FALSE;GHGHGH*/
  dd_dcom_unlock();
  
  return dd_dcom_addr;
}


struct dd_dcom_str *dd_dcom_detach()
{
  if (shmdt(dd_dcom_addr) == -1) {
    sprintf(dd_log_msg,"dd_dcom_detach: error in shmdt: %s\n",strerror(errno));
    first = TRUE;
    dd_log(DD_ERROR, dd_log_msg);
  }
  else{
    first = TRUE;
    dd_dcom_addr = NULL;
  }
  dd_dcom_semid = -1;
  dd_dcom_shmid = -1;
  dd_dcom_nsem = -1;
  my_own_ddpid = -1;
  return dd_dcom_addr;
}

struct dd_dcom_str *dd_dcom_delete()
{
  struct shmid_ds *buf;
  
  if ( ( shmctl(dd_dcom_shmid,IPC_RMID, buf) ) < 0){ 
    sprintf(dd_log_msg,"dcom_delete_close: shmctl %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
  }
  
  if ( ( semctl(dd_dcom_semid, 0, IPC_RMID, 0) ) < 0){
    sprintf(dd_log_msg,"dcom_delete: semctl %s \n",strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
  }

  dd_dcom_addr = NULL;

  return dd_dcom_addr;
}

