/*----------------------------------------------------------------------------*
 *  Copyright (c) 1991, 1992  Southeastern Universities Research Association, *
 *                            Continuous Electron Beam Accelerator Facility   *
 *                                                                            *
 *    This software was developed under a United States Government license    *
 *    described in the NOTICE file included as part of this distribution.     *
 *                                                                            *
 * CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606 *
 *      heyes@cebaf.gov   Tel: (804) 249-7030    Fax: (804) 249-7363          *
 *----------------------------------------------------------------------------*
 * Module: camacLib.c
 *
 * Description: VC2117 Camac Controller VxWorks
 *              Camac Function Library. 
 *
 * Author:
 *	Dave Barker
 *	CEBAF Data Acquisition Group
 *
 * Revision History:
 *	  Revision 1.0  1993/05/25  17:38:00  barker
 *
 *   NOTES:                                        
 *   This library of Camac Functions is specific to
 *   the following configuration:                  
 *   VC2117 Camac Controller With MC68030 CPU.     
 *   VxWorks 5.0.2b Real time Executive.           
 *                                                 
 */



/* Include Files */


/* Header file for Zilog 8536 register definitions */
#include "z8536.h"


/* Global Definitions  */
#define BIT_24_MASK   0xffffff
#define Q_BIT_MASK    0x80000000
#define X_BIT_MASK    0x40000000
#define STATION_MASK  0x7c0
#define ADDRESS_MASK  0x3c

#define ERROR -1
#define OK     0

#define DEBUG 1

#define UTINY char
#define ZERO  0

/* Zilog 8536 Base addresses */
#define  Z8536_1_BASE_ADDR    0xe00100
#define  Z8536_2_BASE_ADDR    0xe00200

/* LAM interrupt vectors */
#define INT_VEC_LAM1  0x48
#define INT_VEC_LAM2  0x48
#define INT_VEC_LAM3  0x48

#define INUM_TO_IVEC(intNum)    ((VOIDFUNCPTR *) ((intNum) << 2))


/* Type definitions */
typedef void (*VOIDFUNCPTR)(); /* Remove when VxWorks includes are added. */


/* Structure Definitions */

/* 8536 registers are located in the top 8 bits of the */
/* longword addresses.                                 */
struct zcio
    {   char port_c;    char ca[3];
        char port_b;    char cb[3];
        char port_a;    char cc[3];
        char control;   char cd[3];
    };


/* external definitions */
int  cdreg (int *ext, int b, int c, int n, int a);
void cfsa (int f, int ext, int *data, int *q);
void cc_cfsa (int f, int ext, int *data, int *q);
void cccz (int ext);
void cccc (int ext);
void ccci (int ext, int l);
void cccd (int ext, int l);                                      /* Dummy */
void ctgl (int ext, int *l);                                     /* Dummy */
void cdlam (int *lam, int b, int c, int n, int m, int *inta);
void cclm (int lam, int l);
void cclc (int lam);
void ctlm (int lam, int *l);
void cclnk (int lam, VOIDFUNCPTR label);                         /* Dummy */
void ccgl (int lam, int l);
void cssa (int f, int ext, short *data, int *q);
void ccrgl (int lam);
void ctci (int ext, int *l);                                     /* Dummy */
void ctcd (int ext, int *l);                                     /* dummy */
void cclwt (int lam);
void ccinit (int b); 
void cfga (int fa[], int exta[], int intc[], int qa[], int cb[]);
void csga (int fa[], int exta[], short intc[], int qa[], int cb[]);
void cfmad (int f, int extb[], int intc[], int cb[]);
void csmad (int f, int extb[], short intc[], int cb[]);
void cfubc (int f, int ext, int intc[], int cb[]);
void csubc (int f, int ext, short intc[], int cb[]);
void cfubl (int f, int ext, int intc[], int cb[]);
void csubl (int f, int ext, short intc[], int cb[]);
void cfubr (int f, int ext, int inta[], int cb[]);
void csubr (int f, int ext, short inta[], int cb[]);
void cglam (int lam, int *b, int *c, int *n, int *a, int inta[]);
void cgreg (int ext, int *b, int *c, int *n, int *a);
void ctstat (int *k);

/* internal definitions */
void wait();
void cio_reset();
void z1write();
void z1read();
void z1or();
void z1and();
void z2write();
void z2read();
void z2or();
void z2and();
void cio_state_zero();




/******************************************************/
/*   External Function : cdreg                        */
/*                                                    */
/*   Function : Declare CAMAC Register.               */
/*                                                    */
/*   Parameters :  int *ext   Returned hardware       */
/*                           specific external address*/
/*                 int b     CAMAC branch number      */
/*                 int c     CAMAC crate number.      */
/*                 int n     CAMAC slot number.       */
/*                 int a     CAMAC address number.    */
/*                                                    */
/*   returns -1 if error, 0 if ok.                    */
/*                                                    */
/******************************************************/

int cdreg (int *ext, int b, int c, int n, int a)
{
    /* VCC2117 NAF encoding is as follows: */
    /* bits 0-1   not used.                */
    /* bits 2-5   CAMAC address a          */
    /* bits 6-10  CAMAC slot number n      */
    /* bits 11-15 CAMAC function code f    */
    /* bit 16     No Wait Flag             */
    /* bit 17     not used. (not set)      */
    /* bit 18     Internal register access */
    /*            flag. internal cycle if  */
    /*            set, otherwise CAMAC     */
    /*            cycle.                   */
    /* bits 20-21 not used (not set)       */
    /* bits 22-23 Set for base address     */
    /*            0xc00000.                */
    /* bits 24-31 not used.                */

    /* parameters b and c are not used     */

    /* initialise ext to zero */
    *ext = 0;
   
    if ( (a >= 16) || (a < 0))
      {
      if(DEBUG)printf("VC2117 CAMAC LIB: CDREG: address 'a' out of range\n");
      return(ERROR);
      }

    if ( (n >= 32) || (n < 0))
      {
      if(DEBUG)printf("VC2117 CAMAC LIB: CDREG: slot number 'n' out of range\n");
      return(ERROR);
      }

    *ext = 0xc00000 | (a<<2) | (n<<6);
    return(OK);

}



/******************************************************/
/*   External Function : cc_cfsa                      */
/*                                                    */
/*   Function : Perform Single CAMAC Action Internal  */
/*              to the camac controller.              */
/*                                                    */
/*   Parameters :  int f     CAMAC function code.     */
/*                 int ext   CAMAC external address   */
/*                 int *data 24bit data for read/write*/
/*                 int *q    returned status.         */
/*                                                    */
/******************************************************/

void cc_cfsa (int f, int ext, int *data, int *q)
{
long *camac_ptr;

    *q = 1;

    if ( (f >= 32) || (f < 0))
      {
      if(DEBUG)printf("VC2117 CAMAC LIB: CFSA: function 'f' out of range\n");
      return;
      }

    camac_ptr = (long *)(ext | (f<<11)|0x40000);

    if (f & 0x8)           /* No Read or Write operations */
      {
      *camac_ptr = 0;

      /* Get status from slot 30 address 0 */
      /* bit 31 is the Q status.           */

      camac_ptr = (long *)0xc40780;
      *q = ((*camac_ptr&Q_BIT_MASK)!=0);
      }
    else
      {
      if ( f & 0x10 )      /* Write Operation */
        {
        *camac_ptr = *data & BIT_24_MASK;
      
        /* Get status from slot 30 address 0 */
        /* bit 31 is the Q status.           */

        camac_ptr = (long *)0xc40780;
        *q = ((*camac_ptr&Q_BIT_MASK)!=0);
        }
      else                 /* Read Operation */
        {
        *data = *camac_ptr & BIT_24_MASK;

        *q = ((*camac_ptr&Q_BIT_MASK)!=0);
        }

      }
}
        
      

/******************************************************/
/*   External Function : cfsa                         */
/*                                                    */
/*   Function : Perform Single CAMAC Action.          */
/*                                                    */
/*   Parameters :  int f     CAMAC function code.     */
/*                 int ext   CAMAC external address   */
/*                 int *data 24bit data for read/write*/
/*                 int *q    returned status.         */
/*                                                    */
/******************************************************/

void cfsa (int f, int ext, int *data, int *q)
{
long *camac_ptr;

    *q = 1;

    if ( (f >= 32) || (f < 0))
      {
      if(DEBUG)printf("VC2117 CAMAC LIB: CFSA: function 'f' out of range\n");
      return;
      }

    camac_ptr = (long *)(ext | (f<<11));

    if (f & 0x8)           /* No Read or Write operations */
      {
      *camac_ptr = 0;

      /* Get status from slot 30 address 0 */
      /* bit 31 is the Q status.           */

      camac_ptr = (long *)0xc40780;
      *q = ((*camac_ptr&Q_BIT_MASK)!=0);
      }
    else
      {
      if ( f & 0x10 )      /* Write Operation */
        {
        *camac_ptr = *data & BIT_24_MASK;
      
        /* Get status from slot 30 address 0 */
        /* bit 31 is the Q status.           */

        camac_ptr = (long *)0xc40780;
        *q = ((*camac_ptr&Q_BIT_MASK)!=0);
        }
      else                 /* Read Operation */
        {
        *data = *camac_ptr & BIT_24_MASK;

        *q = ((*camac_ptr&Q_BIT_MASK)!=0);
        }

      }
}
        

/******************************************************/
/*   External Function : cccz                         */
/*                                                    */
/*   Function : Generate Dataway Initialise.          */
/*                                                    */
/*   Parameters :  int ext   CAMAC external address   */
/*                                                    */
/******************************************************/

void cccz (int ext)
{
long *camac_ptr;

    /* Generates a Z cycle on the CAMAC bus */

    /* Execute Function 26, address 8, slot 28 */
    camac_ptr = (long *)0xc0d720;

    *camac_ptr = 0;
}



/******************************************************/
/*   External Function : cccc                         */
/*                                                    */
/*   Function : Generate Crate Clear.                 */
/*                                                    */
/*   Parameters :  int ext   CAMAC external address   */
/*                                                    */
/******************************************************/

void cccc (int ext)
{
long *camac_ptr;

    /* Generates a C cycle on the CAMAC bus */

    /* Execute Function 26, address 9, slot 28 */
    camac_ptr = (long *)0xc0d724;

    *camac_ptr = 0;
}



/******************************************************/
/*   External Function : ccci                         */
/*                                                    */
/*   Function : Set or Clear Dataway Inhibit          */
/*                                                    */
/*   Parameters :  int ext   CAMAC external address   */
/*                 int l     logical truth value      */
/*                                                    */
/******************************************************/

void ccci (int ext, int l)
{
long *camac_ptr;

    if (l)                  /* Set Dataway Inhibit by invoking */
      {                     /* function 26, address 9, slot 30 */
      camac_ptr = (long *) 0xc0d7a4;
      }
    else                    /* Reset Dataway Inhibit using     */
      {                     /* function 24, address 9, slot 30 */
      camac_ptr = (long *) 0xc0c7a4;
      }

    *camac_ptr = 0;
}





/******************************************************/
/*   External Function : cccd                         */
/*                                                    */
/*   Function : Enable or Disable Crate Demand.       */
/*                                                    */
/*   Parameters :  int ext   CAMAC external address   */
/*                 int l     logical truth value      */
/*                                                    */
/******************************************************/

void cccd (int ext, int l)
{

    /* Crate Demand Enable is not specified in the VCC2117 */
    /* manual. Therefore this is a dummy stub, included    */
    /* for completeness.                                   */

}




/******************************************************/
/*   External Function : ctgl                         */
/*                                                    */
/*   Function : Test Crate Demand Present.            */
/*                                                    */
/*   Parameters :  int ext   CAMAC external address   */
/*                 int *l    returned logical value   */
/*                                                    */
/******************************************************/

void ctgl (int ext, int *l)
{

    /* Test Crate Demand Present is not specified in the */
    /* VCC2117 manual. Therefore this is a dummy stub,   */
    /* included  for completeness.                       */

    *l = 0;

}




/******************************************************/
/*   External Function : cdlam                        */
/*                                                    */
/*   Function : Declare LAM.                          */
/*                                                    */
/*   Parameters :  int *lam  returned LAM address     */
/*                 int b     CAMAC branch number      */
/*                 int c     CAMAC crate number.      */
/*                 int n     CAMAC slot number.       */
/*                 int m     LAM address specifier    */
/*                 int *inta integer array            */
/*                                                    */
/******************************************************/

void cdlam (int *lam, int b, int c, int n, int m, int *inta)
{

    /* cdlam not specified by VCC2117 manual. Implement */
    /* same code as cdreg (see cdreg for full description) */

    cdreg ((int *)lam, b, c, n, m);

    /* initialise ext to zero */
    *lam = 0;
   
    if ( (m >= 16) || (m < 0))
      {
      if(DEBUG)printf("VC2117 CAMAC LIB: CDLAM: address 'm' out of range\n");
      return;
      }

    if ( (n >= 32) || (n < 0))
      {
      if(DEBUG)printf("VC2117 CAMAC LIB: CDLAM: slot number 'n' out of range\n");
      return;
      }

    *lam = 0xc00000 | (m<<2) | (n<<6);
    return;

}




/******************************************************/
/*   External Function : cclm                         */
/*                                                    */
/*   Function : Enable or Disable LAM                 */
/*                                                    */
/*   Parameters :  int lam   LAM identifier.          */
/*                 int l     logical truth value      */
/*                                                    */
/******************************************************/

void cclm (int lam, int l)
{
long *camac_ptr;

    /* Function 26 Enables the LAM. */
    /* Function 24 Disables the LAM. */

    if (l)
      {
      /* Enable graded lam, function 26 */
      camac_ptr = (long *)(lam | ((0x1a)<<11));
      }
    else
      {
      /* Disable graded lam, function 24 */
      camac_ptr = (long *)(lam | ((0x18)<<11));
      }

    *camac_ptr = 0;
}





/******************************************************/
/*   External Function : ccgl                         */
/*                                                    */
/*   Function : Enable or Disable Graded LAM.         */
/*                                                    */
/*   Parameters :  int lam   LAM identifier.          */
/*                 int l     logical truth value      */
/*                                                    */
/******************************************************/

void ccgl (int lam, int l)
{
long *camac_ptr;

    /* Enable or disable graded lam */
       
    if(l)
      {
      /* Enable Port A and B  */
      z1or(ZCIO_MCC, ZCIO_MCC_PAE);
      z1or(ZCIO_MCC, ZCIO_MCC_PBE);

      z2or(ZCIO_MCC, ZCIO_MCC_PAE);
      z2or(ZCIO_MCC, ZCIO_MCC_PBE);

      z1write(ZCIO_PACS, ZCIO_CS_SIE);
      z1write(ZCIO_PBCS, ZCIO_CS_SIE);
      z2write(ZCIO_PACS, ZCIO_CS_SIE);

      z1or(ZCIO_MIC, ZCIO_MIC_MIE); 
      z2or(ZCIO_MIC, ZCIO_MIC_MIE); 

      }
    else
      {

      z1write(ZCIO_PACS, (char)ZCIO_CS_CLIE);
      z1write(ZCIO_PBCS, (char)ZCIO_CS_CLIE);
      z2write(ZCIO_PACS, (char)ZCIO_CS_CLIE);

      }

}


/******************************************************/
/*   External Function : cclc                         */
/*                                                    */
/*   Function : Clear LAM                             */
/*                                                    */
/*   Parameters :  int lam   LAM identifier.          */
/*                                                    */
/******************************************************/

void cclc (int lam)
{
long *camac_ptr;

    /* LAM is cleared by function 10 */

    camac_ptr = (long *)(lam | ((0x0a)<<11));

    *camac_ptr = 0;

}



/******************************************************/
/*   External Function : ctlm                         */
/*                                                    */
/*   Function : Test LAM.                             */
/*                                                    */
/*   Parameters :  int lam   LAM identifier.          */
/*                 int *l    returned logical value   */
/*                                                    */
/******************************************************/

void ctlm (int lam, int *l)
{
int data;

    /* LAM is tested using Function 8, returned value */
    /* is q.                                          */

    cfsa (8, lam, &data, l);

}


/******************************************************/
/*   External Function : cclnk                        */
/*                                                    */
/*   Function : Link LAM to Service Procedure         */
/*                                                    */
/*   Parameters :  int lam   LAM identifier.          */
/*                 VOIDFUNCPTR label Function to call */
/*                                                    */
/******************************************************/

void cclnk (int lam, VOIDFUNCPTR label)
{
int result;
int lam_id;

    /* Find lam Id from lam identifier. */
    lam_id = (lam&STATION_MASK)>>7;

    if (lam_id<=8)
      {
      /* LAM in range 1 to 8, connect to    */
      /* interrupt vector port A chip #1    */
      result=intConnect(INUM_TO_IVEC(INT_VEC_LAM1),
               (VOIDFUNCPTR)label,
               0);
      if(result!=0)printf("Error: intConnect\n");
      if(DEBUG)printf("intConnect result %d\n",result);
      }
    else if ((lam_id > 8) && (lam_id <= 16))
      {
      /* LAM in range 9 to 16, connect to    */
      /* interrupt vector port B chip #1    */
      result=intConnect(INUM_TO_IVEC(INT_VEC_LAM2),
               (VOIDFUNCPTR)label,
               0);
      if(result!=0)printf("Error: intConnect\n");
      if(DEBUG)printf("intConnect result %d\n",result);
      }
    else
      {
      /* LAM in range 17 to 24, connect to    */
      /* interrupt vector port A chip #2    */
      result=intConnect(INUM_TO_IVEC(INT_VEC_LAM3),
               (VOIDFUNCPTR)label,
               0);
      if(result!=0)printf("Error: intConnect\n");
      if(DEBUG)printf("intConnect result %d\n",result);
      }

}


/******************************************************/
/*   External Function : cssa                         */
/*                                                    */
/*   Function : Perform Single CAMAC Action (Short    */
/*              Word).                                */
/*                                                    */
/*   Parameters :  int f     CAMAC function code.     */
/*                 int ext   CAMAC external address   */
/*               short *data 16bit data for read/write*/
/*                 int *q    returned status.         */
/*                                                    */
/******************************************************/

void cssa (int f, int ext, short *data, int *q)
{
long *camac_ptr;

    *q = 1;

    if ( (f >= 32) || (f < 0))
      {
      if(DEBUG)printf("VC2117 CAMAC LIB: CSSA: function 'f' out of range\n");
      return;
      }

    camac_ptr = (long *)(ext | (f<<11));

    if (f & 0x8)           /* No Read or Write operations */
      {
      *camac_ptr = 0;

      /* Get status from slot 30 address 0 */
      /* bit 31 is the Q status.           */

      camac_ptr = (long *)0xc40780;
      *q = ((*camac_ptr&Q_BIT_MASK)!=0);
      }
    else
      {
      if ( f & 0x10 )      /* Write Operation */
        {
        *camac_ptr = (int)*data;
      
        /* Get status from slot 30 address 0 */
        /* bit 31 is the Q status.           */

        camac_ptr = (long *)0xc40780;
        *q = ((*camac_ptr&Q_BIT_MASK)!=0);
        }
      else                 /* Read Operation */
        {
        *data = (short)*camac_ptr;

        *q = ((*camac_ptr&Q_BIT_MASK)!=0);
        }

      }
}
        

/******************************************************/
/*   External Function : ccrgl                        */
/*                                                    */
/*   Function : Re-enable GLAM.                       */
/*                                                    */
/*   Parameters :  int lam   lam identifier.          */
/*                                                    */
/******************************************************/

void ccrgl (int lam)
{


    /* Unsure of function, will perform an */
    /* enable of  lam interrupts           */
    /* on the controller Zilog CIO chips,  */
       
    /* DUMMY */


}



/******************************************************/
/*   External Function : ctci                         */
/*                                                    */
/*   Function : Test Dataway Inhibit.                 */
/*                                                    */
/*   Parameters :  int ext   External identifier.     */
/*                 int *l    Logical value.           */
/*                                                    */
/******************************************************/

void ctci (int ext, int *l)
{
long *camac_ptr;


    /* Test Dataway Inhibit by invoking */
    /* function 27, address 9, slot 30 */
    camac_ptr = (long *) 0xc0dfa4;
    *camac_ptr = 0;

    /* get status information. */
    camac_ptr = (long *)0xc40780;
    *l = ((*camac_ptr&Q_BIT_MASK)!=0);

}




/******************************************************/
/*   External Function : ctcd                         */
/*                                                    */
/*   Function : Test Crate Demand Enabled.            */
/*                                                    */
/*   Parameters :  int ext   External identifier.     */
/*                 int *l    Logical value.           */
/*                                                    */
/******************************************************/

void ctcd (int ext, int *l)
{

       /* This function is not specified.  */
       /* Dummy Stub included for completeness */

       *l = 0;
}


/******************************************************/
/*   External Function : cclwt                        */
/*                                                    */
/*   Function : LAM Wait                              */
/*                                                    */
/*   Parameters :  int lam   LAM identifier.          */
/*                                                    */
/******************************************************/

void cclwt (int lam)
{

       /* Following the example code in cds002, directory */
       /* /usr/local/coda/1.2/source/camac, file camacLib.c */
       /* this function is implemented as a vxWorks task */
       /* suspension.       */

       taskSuspend (0);
}

/******************************************************/
/*   External Function : ccinit                       */
/*                                                    */
/*   Function : Branch Initialise.                    */
/*                                                    */
/*   Parameters :  int b   Branch identifier.         */
/*                                                    */
/******************************************************/

void ccinit (int b)
{
int ext;


     /* This function initialises the crate. It is   */
     /* called by the prestart routine, prior to     */
     /* data acquisition.                            */
     /* This routine is mainly concerned with the    */
     /* reset and initialisation of the VCC2117      */
     /* LAM interface. The crate LAMS are connected  */
     /* to the input ports of the Zilog 8536 chips.  */
     /* This routine resets both the Zilog CIO chips,*/
     /* and then programs them for correct LAM       */
     /* processing.                                  */

     cdreg(&ext,0,0,0,0);

     /* Generate dataway initialise */
     cccz (ext);

     /* Clear dataway inhibit */
     ccci (ext, 0);


     /* Disable ports on Chip #1, but ensure Counter Timer */
     /* is unaffected.   */
     z1write((char)ZCIO_MCC,(char)ZCIO_MCC_CT1E);
     z2write((char)ZCIO_MCC,(char)ZERO);

     
     /* Initialise Zilog 8536 Chip #1.              */
     /* (handles LAMS 1 to 16)                      */

     /* Following CES VCC2117 Manual, page 109. */


     /* Set port A Mode Specification into bit port, 'OR' mode. */
     z1write((char)ZCIO_PAMS, (char)ZERO);
     z1or(ZCIO_PAMS, ZCIO_PMS_PMS_OR);

     /* Set port B Mode Specification into bit port, 'OR' mode. */
     z1write((char)ZCIO_PBMS, (char)ZERO);
     z1or(ZCIO_PBMS, ZCIO_PMS_PMS_OR);


     /* Set port A Handshake Specification Register */
     z1write((char)ZCIO_PAHS, (char)ZERO);

     /* Set port B Handshake Specification Register */
     z1write((char)ZCIO_PBHS, (char)ZERO);


     /* Set port A Command and Status Register */
     if(!DEBUG)z1write((char)ZCIO_PACS, (char)ZERO);

     /* Set port B Command and Status Register */
     z1write((char)ZCIO_PBCS, (char)ZERO);


     /* Set port A Data Path Polarity Register */
     /* Set to INVERTING (as in CES manual). */
     z1write((char)ZCIO_PADPP, (char)ZCIO_DPP_INVERT);

     /* Set port B Data Path Polarity Register */
     z1write((char)ZCIO_PBDPP, (char)ZCIO_DPP_INVERT);

     /* Set port C Data Path Polarity Register */
     /* z1write((char)ZCIO_PCDPP, (char)ZERO); */


     /* Set port A Data Direction Register to input. */
     z1write((char)ZCIO_PADD,(char)ZCIO_DD_IN);

     /* Set port B Data Direction Register to input. */
     z1write((char)ZCIO_PBDD,(char)ZCIO_DD_IN);

     /* Set port C Data Direction Register to input. */
     /* z1write((char)ZCIO_PCDD,(char)0x0f); */

     /* Set port A Special I/O Control Register */
     z1write((char)ZCIO_PASIOC, (char)ZCIO_SIO_NORMAL);

     /* Set port B Special I/O Control Register */
     z1write((char)ZCIO_PBSIOC, (char)ZCIO_SIO_NORMAL);

     /* Set port C Special I/O Control Register */
     /* z1write((char)ZCIO_PCSIOC, (char)ZCIO_SIO_NORMAL); */

     /* Clear port A data register */
     z1write((char)ZCIO_PAD, (char)ZERO);

     /* Clear port B data register */
     z1write((char)ZCIO_PBD, (char)ZERO);

     /* Clear port C data register */
     /* z1write((char)ZCIO_PCD, (char)ZERO); */


     /* Set port A Pattern Definition Registers.         */
     /* Set the Pattern Definition Registers to raise an */
     /* interrupt on a zero to one transistion.          */

     z1write((char)ZCIO_PAPP, (char)ZCIO_PP_ONE);
     z1write((char)ZCIO_PAPT, (char)ZCIO_PT_ONE);
     z1write((char)ZCIO_PAPM, (char)ZCIO_PM_ONE);

     /* Set port B Pattern Definition Registers.         */
     /* Set the Pattern Definition Registers to raise an */
     /* interrupt on a zero to one transistion.          */

     z1write((char)ZCIO_PBPP, (char)ZCIO_PP_ONE);
     z1write((char)ZCIO_PBPT, (char)ZCIO_PT_ONE);
     z1write((char)ZCIO_PBPM, (char)ZCIO_PM_ONE);



     /* Set port A Interrupt Vector */
     z1write ((char)ZCIO_PAIV, (char)INT_VEC_LAM1);

     /* Set port B Interrupt Vector */
     z1write ((char)ZCIO_PBIV, (char)INT_VEC_LAM2);

     /* Enable Port A and B */
     z1or(ZCIO_MCC, ZCIO_MCC_PAE); 
     z1or(ZCIO_MCC, ZCIO_MCC_PBE);



     /* Initialise Zilog 8536 Chip #2.              */
     /* (handles LAMS 17 to 24)                     */


     /* Set port A Mode Specification into bit port, 'OR' mode. */

     z2write((char)ZCIO_PAMS, (char)ZERO);
     z2or(ZCIO_PAMS, ZCIO_PMS_PMS_OR);

     /* Set port B Mode Specification into bit port mode */
     z2write((char)ZCIO_PBMS, (char)ZERO);


     /* Set port A Handshake Specification Register */
     z2write((char)ZCIO_PAHS, (char)ZERO);

     /* Set port B Handshake Specification Register */
     z2write((char)ZCIO_PBHS, (char)ZERO);


     /* Set port A Command and Status Register */
     z2write((char)ZCIO_PACS, (char)ZERO);

     /* Set port B Command and Status Register */
     z2write((char)ZCIO_PBCS, (char)ZERO);


     /* Set port A Data Path Polarity Register */
     /* Set to INVERTING (as in CES manual). */
     z2write((char)ZCIO_PADPP, (char)ZCIO_DPP_INVERT);

     /* Set port B Data Path Polarity Register */
     /* A bit set means inverting.             */
     /* bit 0 invert                           */
     /* bit 1 invert                           */
     /* bit 2 non-invert                       */
     /* bit 3 non-invert                       */
     /* bit 4 non-invert                       */
     /* bit 5 non-invert                       */
     /* bit 6 not applicable.                  */
     /* bit 7 non-invert                       */

     z2write((char)ZCIO_PBDPP, (char)0x03);

     /* Set port C Data Path Polarity Register */
     /* z2write((char)ZCIO_PCDPP, (char)0x0f); */


     /* Set port A Data Direction Register to input. */
     z2write((char)ZCIO_PADD,(char)ZCIO_DD_IN);

     /* Set port B Data Direction Register according */
     /* to CES 2117page 26 */
     /* Setting the bit means it is an input bit.    */
     /* bit 0 output     */
     /* bit 1 input      */
     /* bit 2 input      */
     /* bit 3 input      */
     /* bit 4 input      */
     /* bit 5 output     */
     /* bit 6 not appl.  */
     /* bit 7 input      */

     z2write((char)ZCIO_PBDD,(char)0x9e);

     /* Set port C Data Direction Register to input. */
     /* z2write((char)ZCIO_PCDD,(char)ZERO); */


     /* Set port A Special I/O Control Register */
     z2write((char)ZCIO_PASIOC, (char)ZCIO_SIO_NORMAL);

     /* Set port B Special I/O Control Register */
     z2write((char)ZCIO_PBSIOC, (char)ZCIO_SIO_NORMAL);

     /* Set port C Special I/O Control Register */
     /* z2write((char)ZCIO_PCSIOC, (char)ZCIO_SIO_NORMAL); */

     /* Init port A data Register */
     z2write((char)ZCIO_PAD, (char)0x00);

     /* Init port B data Register */
     z2write((char)ZCIO_PBD, (char)0x01);

     /* Init port C data Register */
     /* z2write((char)ZCIO_PCD, (char)0x00); */


     /* Set port A Pattern Definition Registers.         */
     /* Set the Pattern Definition Registers to raise an */
     /* interrupt on a zero to one transistion.          */

     z2write((char)ZCIO_PAPP, (char)ZCIO_PP_ONE);
     z2write((char)ZCIO_PAPT, (char)ZCIO_PT_ONE);
     z2write((char)ZCIO_PAPM, (char)ZCIO_PM_ONE);

     /* Set port B Pattern Definition Registers.         */
     z2write((char)ZCIO_PBPP, (char)ZERO);
     z2write((char)ZCIO_PBPT, (char)ZERO);
     z2write((char)ZCIO_PBPM, (char)ZERO);


     /* Set port A Interrupt Vector */
     z2write ((char)ZCIO_PAIV, (char)INT_VEC_LAM3);

     /* Enable Port A and B */

     z2or(ZCIO_MCC, ZCIO_MCC_PAE); 
     z2or(ZCIO_MCC, ZCIO_MCC_PBE);

     /* Enable interrupts */

}



/******************************************************/
/*   External Function : cfga                         */
/*                                                    */
/*   Function : General Multiple Action.              */
/*                                                    */
/*   Parameters :  fa    Function array.              */
/*                 exta  External address array.      */
/*                 intc  Camac data array.            */
/*                 qa    Q response array.            */
/*                 cb    Control block.               */
/*                                                    */
/******************************************************/

void cfga (int fa[], int exta[], int intc[], int qa[], int cb[])
{
int result;
int i;

     /* Control block composition;   */
     /* cb[0] Repeat Count.          */
     /* cb[1] Tally                  */
     /* cb[2] LAM id.                */
     /* cb[3] Channel identifier     */

     if (cb[2])
       {
       /* poll for lam identified by cb[2] */
       result=0;
       while(result == 0)
         ctlm(cb[2],&result);
       }

     /* perform all functions */
     cb[1]=1;
     for (i=0;i<cb[0];i++)
       {
       cfsa(fa[i],exta[i],&intc[i],&qa[i]);
       cb[1]++;
       }

}




/******************************************************/
/*   External Function : csga                         */
/*                                                    */
/*   Function : General Multiple Action.(short word)  */
/*                                                    */
/*   Parameters :  fa    Function array.              */
/*                 exta  External address array.      */
/*                 intc  Camac data array.            */
/*                 qa    Q response array.            */
/*                 cb    Control block.               */
/*                                                    */
/******************************************************/

void csga (int fa[], int exta[], short intc[], int qa[], int cb[])
{
int result;
int i;

     /* Control block composition;   */
     /* cb[0] Repeat Count.          */
     /* cb[1] Tally                  */
     /* cb[2] LAM id.                */
     /* cb[3] Channel identifier     */

     if (cb[2])
       {
       /* poll for lam identified by cb[2] */
       result=0;
       while(result == 0)
         ctlm(cb[2],&result);
       }

     /* perform all functions */
     cb[1]=1;
     for (i=0;i<cb[0];i++)
       {
       cssa(fa[i],exta[i],&intc[i],&qa[i]);
       cb[1]++;
       }

}



/******************************************************/
/*   External Function : cfmad                        */
/*                                                    */
/*   Function : Address Scan.                         */
/*                                                    */
/*   Parameters :  f     Function                     */
/*                 extb  External addresses.          */
/*                 intc  Camac data array.            */
/*                 cb    Control block.               */
/*                                                    */
/******************************************************/

void cfmad (int f, int extb[], int intc[], int cb[])
{
int result;
int branch;
int crate;
int station;
int addr;
int ext;
int q;


     /* Control block composition;   */
     /* cb[0] Repeat Count.          */
     /* cb[1] Tally                  */
     /* cb[2] LAM id.                */
     /* cb[3] Channel identifier     */

     if (cb[2])
       {
       /* poll for lam identified by cb[2] */
       result=0;
       while(result == 0)
         ctlm(cb[2],&result);
       }

     /* decompose the starting address given by */
     /* extb[0] into its component branch, crate*/
     /* station and address numbers.            */

     cgreg( extb[0], &branch, &crate, &station, &addr);

     ext = extb[0];
     cb[1]=1;

     while ( (ext<extb[1]) && (cb[1]<cb[0]) )
       {
       cdreg( &ext, branch, crate, station, addr);

       cfsa (f, ext, &intc[cb[1]-1], &q);

       if (q)
         {
         addr++;
         if (addr>15)
           {
           station++;
           addr=0;
           }
         cb[1]++;
         }
       else
         {
         station++;
         addr=0;
         }
       }
}






/******************************************************/
/*   External Function : csmad                        */
/*                                                    */
/*   Function : Address Scan. (short word)            */
/*                                                    */
/*   Parameters :  f     Function                     */
/*                 extb  External addresses.          */
/*                 intc  Camac data array.            */
/*                 cb    Control block.               */
/*                                                    */
/******************************************************/

void csmad (int f, int extb[], short intc[], int cb[])
{
int result;
int branch;
int crate;
int station;
int addr;
int ext;
int q;


     /* Control block composition;   */
     /* cb[0] Repeat Count.          */
     /* cb[1] Tally                  */
     /* cb[2] LAM id.                */
     /* cb[3] Channel identifier     */

     if (cb[2])
       {
       /* poll for lam identified by cb[2] */
       result=0;
       while(result == 0)
         ctlm(cb[2],&result);
       }

     /* decompose the starting address given by */
     /* extb[0] into its component branch, crate*/
     /* station and address numbers.            */

     cgreg( extb[0], &branch, &crate, &station, &addr);

     ext = extb[0];
     cb[1]=1;

     while ( (ext<extb[1]) && (cb[1]<cb[0]) )
       {
       cdreg( &ext, branch, crate, station, addr);

       cssa (f, ext, &intc[cb[1]-1], &q);


       if (q)
         {
         addr++;
         if (addr>15)
           {
           station++;
           addr=0;
           }
         cb[1]++;
         }
       else
         {
         station++;
         addr=0;
         }
       }
}






/******************************************************/
/*   External Function : cfubc                        */
/*                                                    */
/*   Function : Controller-Synchronised Block Transfer*/
/*                                                    */
/*   Parameters :  f     Function                     */
/*                 ext   External address.            */
/*                 intc  Camac data array.            */
/*                 cb    Control block.               */
/*                                                    */
/******************************************************/

void cfubc (int f, int ext, int intc[], int cb[])
{
int result;
int q;


     /* Control block composition;   */
     /* cb[0] Repeat Count.          */
     /* cb[1] Tally                  */
     /* cb[2] LAM id.                */
     /* cb[3] Channel identifier     */

     if (cb[2])
       {
       /* poll for lam identified by cb[2] */
       result=0;
       while(result == 0)
         ctlm(cb[2],&result);
       }

     /* repeatedly peform function f at address ext */
     /* until the required number of iterations     */
     /* has been completed, or the q response is    */
     /* zero.                                       */

     q=1;
     cb[1]=0;

     while ( (cb[1]<cb[0]) && (q==1) )
       {
       cb[1]++;
       cfsa ( f, ext, &intc[cb[1]-1], &q);
       }
}




/******************************************************/
/*   External Function : csubc                        */
/*                                                    */
/*   Function : Controller-Synchronised Block Transfer*/
/*              (short word.)                         */
/*                                                    */
/*   Parameters :  f     Function                     */
/*                 ext   External address.            */
/*                 intc  Camac data array.            */
/*                 cb    Control block.               */
/*                                                    */
/******************************************************/

void csubc (int f, int ext, short intc[], int cb[])
{
int result;
int q;


     /* Control block composition;   */
     /* cb[0] Repeat Count.          */
     /* cb[1] Tally                  */
     /* cb[2] LAM id.                */
     /* cb[3] Channel identifier     */

     if (cb[2])
       {
       /* poll for lam identified by cb[2] */
       result=0;
       while(result == 0)
         ctlm(cb[2],&result);
       }

     /* repeatedly peform function f at address ext */
     /* until the required number of iterations     */
     /* has been completed, or the q response is    */
     /* zero.                                       */

     q=1;
     cb[1]=0;

     while ( (cb[1]<cb[0]) && (q==1) )
       {
       cb[1]++;
       cssa ( f, ext, &intc[cb[1]-1], &q);
       }
}



/******************************************************/
/*   External Function : cfubl                        */
/*                                                    */
/*   Function : LAM-Synchronised Block Transfer.      */
/*                                                    */
/*   Parameters :  f     Function                     */
/*                 ext   External address.            */
/*                 intc  Camac data array.            */
/*                 cb    Control block.               */
/*                                                    */
/******************************************************/

void cfubl (int f, int ext, int intc[], int cb[])
{
int result;
int q;

     /* Control block composition;   */
     /* cb[0] Repeat Count.          */
     /* cb[1] Tally                  */
     /* cb[2] LAM id.                */
     /* cb[3] Channel identifier     */


     q=1;
     cb[1]=0;

     /* repeatedly peform function f at address ext */
     /* until the required number of iterations     */
     /* has been completed, or the q response is    */
     /* zero.                                       */
     while ( (cb[1]<cb[0]) && (q==1) )
       {
       cb[1]++;

       /* poll for lam identified by cb[2] */
       result=0;
       while(result == 0)
         ctlm(cb[2],&result);

       cfsa ( f, ext, &intc[cb[1]-1], &q);
       }
}




/******************************************************/
/*   External Function : csubl                        */
/*                                                    */
/*   Function : LAM-Synchronised Block Transfer.      */
/*              (short word).                         */
/*                                                    */
/*   Parameters :  f     Function                     */
/*                 ext   External address.            */
/*                 intc  Camac data array.            */
/*                 cb    Control block.               */
/*                                                    */
/******************************************************/

void csubl (int f, int ext, short intc[], int cb[])
{
int result;
int q;

     /* Control block composition;   */
     /* cb[0] Repeat Count.          */
     /* cb[1] Tally                  */
     /* cb[2] LAM id.                */
     /* cb[3] Channel identifier     */


     q=1;
     cb[1]=0;

     /* repeatedly peform function f at address ext */
     /* until the required number of iterations     */
     /* has been completed, or the q response is    */
     /* zero.                                       */
     while ( (cb[1]<cb[0]) && (q==1) )
       {
       cb[1]++;

       /* poll for lam identified by cb[2] */
       result=0;
       while(result == 0)
         ctlm(cb[2],&result);

       cssa ( f, ext, &intc[cb[1]-1], &q);
       }
}



/******************************************************/
/*   External Function : cfubr                        */
/*                                                    */
/*   Function : Repeat Mode Block Transfer.           */
/*                                                    */
/*   Parameters :  f     Function                     */
/*                 ext   External address.            */
/*                 inta  Camac data array.            */
/*                 cb    Control block.               */
/*                                                    */
/******************************************************/

void cfubr (int f, int ext, int inta[], int cb[])
{
int result;
int q;

     /* Control block composition;   */
     /* cb[0] Repeat Count.          */
     /* cb[1] Tally                  */
     /* cb[2] LAM id.                */
     /* cb[3] Channel identifier     */

     if (cb[2])
       {
       /* poll for lam identified by cb[2] */
       result=0;
       while(result == 0)
         ctlm(cb[2],&result);
       }

     cb[1]=0;

     /* repeatedly peform function f at address ext */
     /* until the required number of iterations     */
     /* has been completed, or the q response is    */
     /* zero.                                       */
     while ( cb[1]<cb[0] )
       {
       cb[1]++;

       q=0;
       while (q==0)
         cfsa ( f, ext, &inta[cb[1]-1], &q);
       }
}



/******************************************************/
/*   External Function : csubr                        */
/*                                                    */
/*   Function : Repeat Mode Block Transfer.(short wrd)*/
/*                                                    */
/*   Parameters :  f     Function                     */
/*                 ext   External address.            */
/*                 inta  Camac data array.            */
/*                 cb    Control block.               */
/*                                                    */
/******************************************************/

void csubr (int f, int ext, short inta[], int cb[])
{
int result;
int q;

     /* Control block composition;   */
     /* cb[0] Repeat Count.          */
     /* cb[1] Tally                  */
     /* cb[2] LAM id.                */
     /* cb[3] Channel identifier     */

     if (cb[2])
       {
       /* poll for lam identified by cb[2] */
       result=0;
       while(result == 0)
         ctlm(cb[2],&result);
       }

     cb[1]=0;

     /* repeatedly peform function f at address ext */
     /* until the required number of iterations     */
     /* has been completed, or the q response is    */
     /* zero.                                       */
     while ( cb[1]<cb[0] )
       {
       cb[1]++;

       q=0;
       while (q==0)
         cssa ( f, ext, &inta[cb[1]-1], &q);
       }
}



/******************************************************/
/*   External Function : cglam                        */
/*                                                    */
/*   Function : Decompose a camac external address    */
/*              into its component branch, crate,     */
/*              station and address components.       */
/*                                                    */
/******************************************************/

void cglam (int lam, int *b, int *c, int *n, int *a, int inta[])
{
    /* VCC2117 NAF encoding is as follows: */
    /* bits 0-1   not used.                */
    /* bits 2-5   CAMAC address a          */
    /* bits 6-10  CAMAC slot number n      */
    /* bits 11-15 CAMAC function code f    */
    /* bit 16     No Wait Flag             */
    /* bit 17     not used. (not set)      */
    /* bit 18     Internal register access */
    /*            flag. internal cycle if  */
    /*            set, otherwise CAMAC     */
    /*            cycle.                   */
    /* bits 20-21 not used (not set)       */
    /* bits 22-23 Set for base address     */
    /*            0xc00000.                */
    /* bits 24-31 not used.                */

    /* parameters b and c are not used     */

    *b = 0;
    *c = 0;

    *n = lam & STATION_MASK;

    *a = lam & ADDRESS_MASK;


}



/******************************************************/
/*   External Function : cgreg                        */
/*                                                    */
/*   Function : Decompose a camac external address    */
/*              into its component branch, crate,     */
/*              station and address components.       */
/*                                                    */
/******************************************************/

void cgreg (int ext, int *b, int *c, int *n, int *a)
{
    /* VCC2117 NAF encoding is as follows: */
    /* bits 0-1   not used.                */
    /* bits 2-5   CAMAC address a          */
    /* bits 6-10  CAMAC slot number n      */
    /* bits 11-15 CAMAC function code f    */
    /* bit 16     No Wait Flag             */
    /* bit 17     not used. (not set)      */
    /* bit 18     Internal register access */
    /*            flag. internal cycle if  */
    /*            set, otherwise CAMAC     */
    /*            cycle.                   */
    /* bits 20-21 not used (not set)       */
    /* bits 22-23 Set for base address     */
    /*            0xc00000.                */
    /* bits 24-31 not used.                */

    /* parameters b and c are not used     */

    *b = 0;
    *c = 0;

    *n = ext & STATION_MASK;

    *a = ext & ADDRESS_MASK;


}




/******************************************************/
/*   External Function : ctstat                       */
/*                                                    */
/*   Function : Return Q and X status codes.          */
/*                                                    */
/*   Parameters :  int *k returned result.            */
/*                                                    */
/******************************************************/

void ctstat (int *k)
{
long *camac_ptr;

    /* The returned value of k is defined as 4e+d */
    /* (pg 17 IEEE CAMAC Standards). In this      */
    /* implementation, e will always be zero.     */
    /* Therefore, k is defined as;                */
    /*                                            */
    /* k            Q  X                          */
    /* 0            1  1                          */
    /* 1            0  1                          */
    /* 2            1  0                          */
    /* 3            0  0                          */

    /* get status information. */
    camac_ptr = (long *)0xc40780;

    *k= 0;

    if ((*camac_ptr&Q_BIT_MASK)==0)
      *k=*k|0x1;
    if ((*camac_ptr&X_BIT_MASK)==0)
      *k=*k|0x2;

}





/******************************************************/
/*   External Function : cdcrt                        */
/*                                                    */
/*   Function : Define Crate Identifier.              */
/*                                                    */
/*   Parameters :  int c crate number.                */
/*                 int intb integer array.            */
/*                                                    */
/******************************************************/

void cdcrt ( int c, int intb[] )
{

    /* Not defined. */
}



/*****************************************************/
/*                                                   */
/*      INTERNAL FUNCTIONS.                          */
/*                                                   */
/*****************************************************/




/******************************************************/
/*   Internal Function : wait                         */
/*                                                    */
/*   Function : Perform a wait to allow the 8536      */
/*              Zilog chips recovery time.            */
/*                                                    */
/*   Parameters :                                     */
/*                                                    */
/******************************************************/

void wait()
{
      __asm__("nop;nop;nop;nop;nop;nop;nop");
}



/******************************************************/
/*   Internal Function : cio_reset                    */
/*                                                    */
/*   Function : Reset both Zilog 8536 CIO chips.      */
/*                                                    */
/*   Parameters :                                     */
/*                                                    */
/******************************************************/

void cio_reset()
{
char dummy;
char *ptr;
char *p;



     /* Zilog 8536 Chip #1 */

     ptr = (char *)Z8536_1_BASE_ADDR;
     p = &((struct zcio*)ptr)-> control;

     /* Put chip into state 0 */
     /* (See page 108, VCC2117 Manual) */
     /* Perform read, write, read. This */
     /* will leave the chip in state 0 */
     /* no matter what its original  */
     /* state was. */

     dummy = *p;
     wait();

     *p = (char)0x0;
     wait();

     dummy= *p;
     wait();

     /* Write address of master interrupt control register. */
     *p = (char) ZCIO_MIC;
     wait();

     /* Set the reset bit */
     *p = (char)ZCIO_MIC_RESET;
     wait();

     /* Clear the reset bit */
     *p = (char) ZCIO_MIC;
     wait();
     *p = (char)ZERO;
     wait();

     /* Dummy read to ensure state 0 */
     dummy = *p;
     wait();

     /* Zilog 8536 Chip #2 */

     ptr = (char *)Z8536_2_BASE_ADDR;
     p = &((struct zcio*)ptr)-> control;

     /* Put chip into state 0 */
     /* (See page 108, VCC2117 Manual) */
     /* Perform read, write, read. This */
     /* will leave the chip in state 0 */
     /* no matter what its original  */
     /* state was. */

     dummy = *p;
     wait();

     *p = (char)0x0;
     wait();

     dummy= *p;
     wait();

     /* Write address of master interrupt control register. */
     *p = (char) ZCIO_MIC;
     wait();

     /* Set the reset bit */
     *p = (char)ZCIO_MIC_RESET;
     wait();
     /* Clear the reset bit */
     *p = (char) ZCIO_MIC;
     wait();
     *p = (char)ZERO;
     wait();

     /* Dummy read to ensure state 0 */
     dummy = *p;
     wait();


}

/******************************************************/
/*   Internal Function : z1write                      */
/*                                                    */
/*   Function : Perform a write of two pieces of      */
/*              information to the Zilog 8536 #1      */
/*              control register. The information     */
/*              comprises an address, and the data to */
/*              be written to that address.           */
/*              Each write is preceded by a dummy     */
/*              read to ensure that the chip is in    */
/*              mode 0. All operations are bounded by */
/*              a CPU wait to allow for the 1us       */
/*              recovery time.                        */
/*                                                    */
/*   Parameters : char addr;   The destination        */
/*                             register's address.    */
/*                char data;   The data to be written */
/*                                                    */
/*                                                    */
/******************************************************/

void z1write(addr, data)
char addr;
char data;
{
char *p;
char *ptr;
char dummy;

     ptr = (char *)Z8536_1_BASE_ADDR;
     p = &((struct zcio*)ptr)-> control;

     /* Dummy read */
     dummy = *p;
     wait();

     /* Perform the write */
     *p = addr;
     wait();
     *p = data;
     wait();
}



/******************************************************/
/*   Internal Function : z2write                      */
/*                                                    */
/*   Function : Perform a write of two pieces of      */
/*              information to the Zilog 8536 #2      */
/*              control register. The information     */
/*              comprises an address, and the data to */
/*              be written to that address.           */
/*              Each write is preceded by a dummy     */
/*              read to ensure that the chip is in    */
/*              mode 0. All operations are bounded by */
/*              a CPU wait to allow for the 1us       */
/*              recovery time.                        */
/*                                                    */
/*   Parameters : char addr;   The destination        */
/*                             register's address.    */
/*                char data;   The data to be written */
/*                                                    */
/*                                                    */
/******************************************************/

void z2write(addr, data)
char addr;
char data;
{
char *p;
char *ptr;
char dummy;

     ptr = (char *)Z8536_2_BASE_ADDR;
     p = &((struct zcio*)ptr)-> control;

     /* Dummy read */
     dummy = *p;
     wait();

     /* Perform the write */
     *p = addr;
     wait();
     *p = data;
     wait();
}



/******************************************************/
/*   Internal Function : z1read                       */
/*                                                    */
/*   Function : Perform a read operation from the     */
/*              Zilog 8536 #1 Control Register. A     */
/*              read operation comprises of writting  */
/*              the target register address into the  */
/*              control register, followed by reading */
/*              the data from this same register.     */
/*              Each read  is preceded by a dummy     */
/*              read to ensure that the chip is in    */
/*              mode 0. All operations are bounded by */
/*              a CPU wait to allow for the 1us       */
/*              recovery time.                        */
/*                                                    */
/*   Parameters : char addr;   The destination        */
/*                             register's address.    */
/*                char *data;  The read data.         */
/*                                                    */
/*                                                    */
/******************************************************/

void z1read(addr, data)
char addr;
char *data;
{
char *p;
char *ptr;
char dummy;

     ptr = (char *)Z8536_1_BASE_ADDR;
     p = &((struct zcio*)ptr)-> control;

     /* Dummy read */
     dummy = *p;
     wait();

     *p = addr;
     wait();
     *data = *p;
     wait();
}



/******************************************************/
/*   Internal Function : z2read                       */
/*                                                    */
/*   Function : Perform a read operation from the     */
/*              Zilog 8536 #2 Control Register. A     */
/*              read operation comprises of writing   */
/*              the target register address into the  */
/*              control register, followed by reading */
/*              the data from this same register.     */
/*              Each read  is preceded by a dummy     */
/*              read to ensure that the chip is in    */
/*              mode 0. All operations are bounded by */
/*              a CPU wait to allow for the 1us       */
/*              recovery time.                        */
/*                                                    */
/*   Parameters : char addr;   The destination        */
/*                             register's address.    */
/*                char *data;  The read data.         */
/*                                                    */
/*                                                    */
/******************************************************/

void z2read(addr, data)
char addr;
char *data;
{
char *p;
char *ptr;
char dummy;

     ptr = (char *)Z8536_2_BASE_ADDR;
     p = &((struct zcio*)ptr)-> control;

     /* Dummy read */
     dummy = *p;
     wait();

     *p = addr;
     wait();
     *data = *p;
     wait();
}


/******************************************************/
/*   Internal Function : Z1or                         */
/*                                                    */
/*   Function : Perform an OR operation on the        */
/*              contents of a register on the Zilog   */
/*              8536 #1 Chip. The operation comprises */
/*              of Reading the current contents of    */
/*              the register, and then writing back   */
/*              this value OR'ed with the data.       */
/*              Each write is preceded by a dummy     */
/*              read to ensure that the chip is in    */
/*              mode 0. All operations are bounded by */
/*              a CPU wait to allow for the 1us       */
/*              recovery time.                        */
/*                                                    */
/*   Parameters : char addr;   Address of register.   */
/*                char data;   Data to be OR'ed       */
/*                                                    */
/*                                                    */
/******************************************************/

void z1or(addr, data)
char addr;
char data;
{
char *p;
char *ptr;
char v;
char dummy;

     ptr = (char *)Z8536_1_BASE_ADDR;
     p = &((struct zcio*)ptr)-> control;

     /* Dummy read */
     dummy = *p;
     wait();

     *p = addr;
     wait();
     v = *p;
     wait();
     *p = addr;
     wait();
     *p = (v|data);
     wait();
}



/******************************************************/
/*   Internal Function : Z1and                        */
/*                                                    */
/*   Function : Perform an AND operation on the       */
/*              contents of a register on the Zilog   */
/*              8536 #1 Chip.                         */
/*              Each write is preceded by a dummy     */
/*              read to ensure that the chip is in    */
/*              mode 0. All operations are bounded by */
/*              a CPU wait to allow for the 1us       */
/*              recovery time.                        */
/*                                                    */
/*   Parameters : char addr;   Address of register.   */
/*                char data;   Data to be AND'ed      */
/*                                                    */
/*                                                    */
/******************************************************/

void z1and(addr, data)
char addr;
char data;
{
char *p;
char *ptr;
char v;
char dummy;

     ptr = (char *)Z8536_1_BASE_ADDR;
     p = &((struct zcio*)ptr)-> control;

     /* Dummy read */
     dummy = *p;
     wait();

     *p = addr;
     wait();
     v = *p;
     wait();
     *p = addr;
     wait();
     *p = (v&data);
     wait();
}
   

/******************************************************/
/*   Internal Function : Z2or                         */
/*                                                    */
/*   Function : Perform an OR operation on the        */
/*              contents of a register on the Zilog   */
/*              8536 #2 Chip. The operation comprises */
/*              of Reading the current contents of    */
/*              the register, and then writing back   */
/*              this value OR'ed with the data.       */
/*              Each write is preceded by a dummy     */
/*              read to ensure that the chip is in    */
/*              mode 0. All operations are bounded by */
/*              a CPU wait to allow for the 1us       */
/*              recovery time.                        */
/*                                                    */
/*   Parameters : char addr;   Address of register.   */
/*                char data;   Data to be OR'ed       */
/*                                                    */
/*                                                    */
/******************************************************/

void z2or(addr, data)
char addr;
char data;
{
char *p;
char *ptr;
char v;
char dummy;

     ptr = (char *)Z8536_2_BASE_ADDR;
     p = &((struct zcio*)ptr)-> control;

     /* Dummy read */
     dummy = *p;
     wait();

     *p = addr;
     wait();
     v = *p;
     wait();
     *p = addr;
     wait();
     *p = (v|data);
     wait();
}


/******************************************************/
/*   Internal Function : Z2and                        */
/*                                                    */
/*   Function : Perform an AND operation on the       */
/*              contents of a register on the Zilog   */
/*              8536 #2 Chip.                         */
/*              Each write is preceded by a dummy     */
/*              read to ensure that the chip is in    */
/*              mode 0. All operations are bounded by */
/*              a CPU wait to allow for the 1us       */
/*              recovery time.                        */
/*                                                    */
/*   Parameters : char addr;   Address of register.   */
/*                char data;   Data to be AND'ed      */
/*                                                    */
/*                                                    */
/******************************************************/

void z2and(addr, data)
char addr;
char data;
{
char *p;
char *ptr;
char v;
char dummy;

     ptr = (char *)Z8536_2_BASE_ADDR;
     p = &((struct zcio*)ptr)-> control;

     /* Dummy read */
     dummy = *p;
     wait();

     *p = addr;
     wait();
     v = *p;
     wait();
     *p = addr;
     wait();
     *p = (v&data);
     wait();
}


/******************************************************/
/*   Internal Function : cio_state_zero.              */
/*                                                    */
/*   Function : Put both Zilog CIO chips into state   */
/*              zero, ready to read the address of the*/
/*              first operation.                      */
/*                                                    */
/******************************************************/

void cio_state_zero()
{
char dummy;
char *ptr;
char *p;



     /* Zilog 8536 Chip #1 */

     ptr = (char *)Z8536_1_BASE_ADDR;
     p = &((struct zcio*)ptr)-> control;

     /* Put chip into state 0 */
     /* (See page 108, VCC2117 Manual) */
     /* Perform read, write, read. This */
     /* will leave the chip in state 0 */
     /* no matter what its original  */
     /* state was. */

     dummy = *p;
     wait();

     *p = (char)0x0;
     wait();

     dummy= *p;
     wait();



     /* Zilog 8536 Chip #2 */

     ptr = (char *)Z8536_2_BASE_ADDR;
     p = &((struct zcio*)ptr)-> control;

     /* Put chip into state 0 */
     /* (See page 108, VCC2117 Manual) */
     /* Perform read, write, read. This */
     /* will leave the chip in state 0 */
     /* no matter what its original  */
     /* state was. */

     dummy = *p;
     wait();

     *p = (char)0x0;
     wait();

     dummy= *p;
     wait();

}


