/* 
 * tkTest.c --
 *
 *	This file contains C command procedures for a bunch of additional
 *	Tcl commands that are used for testing out Tcl's C interfaces.
 *	These commands are not normally included in Tcl applications;
 *	they're only used for testing.
 *
 * Copyright (c) 1993-1994 The Regents of the University of California.
 * Copyright (c) 1994-1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tkInt.h"
#include "tkPort.h"	
#include "tix.h"	

/*
 * The table below describes events and is used by the "testevent"
 * command.
 */

typedef struct {
    char *name;			/* Name of event. */
    int type;			/* Event type for X, such as
				 * ButtonPress. */
} EventInfo;

static EventInfo eventArray[] = {
    {"Motion",		MotionNotify},
    {"Button",		ButtonPress},
    {"ButtonPress",	ButtonPress},
    {"ButtonRelease",	ButtonRelease},
    {"Colormap",	ColormapNotify},
    {"Enter",		EnterNotify},
    {"Leave",		LeaveNotify},
    {"Expose",		Expose},
    {"FocusIn",		FocusIn},
    {"FocusOut",	FocusOut},
    {"Keymap",		KeymapNotify},
    {"Key",		KeyPress},
    {"KeyPress",	KeyPress},
    {"KeyRelease",	KeyRelease},
    {"Property",	PropertyNotify},
    {"ResizeRequest",	ResizeRequest},
    {"Circulate",	CirculateNotify},
    {"Configure",	ConfigureNotify},
    {"Destroy",		DestroyNotify},
    {"Gravity",		GravityNotify},
    {"Map",		MapNotify},
    {"Reparent",	ReparentNotify},
    {"Unmap",		UnmapNotify},
    {"Visibility",	VisibilityNotify},
    {"CirculateRequest",CirculateRequest},
    {"ConfigureRequest",ConfigureRequest},
    {"MapRequest",	MapRequest},
    {(char *) NULL,	0}
};

/*
 * The defines and table below are used to classify events into
 * various groups.  The reason for this is that logically identical
 * fields (e.g. "state") appear at different places in different
 * types of events.  The classification masks can be used to figure
 * out quickly where to extract information from events.
 */

#define KEY_BUTTON_MOTION	0x1
#define CROSSING		0x2
#define FOCUS			0x4
#define EXPOSE			0x8
#define VISIBILITY		0x10
#define CREATE			0x20
#define MAP			0x40
#define REPARENT		0x80
#define CONFIG			0x100
#define CONFIG_REQ		0x200
#define RESIZE_REQ		0x400
#define GRAVITY			0x800
#define PROP			0x1000
#define SEL_CLEAR		0x2000
#define SEL_REQ			0x4000
#define SEL_NOTIFY		0x8000
#define COLORMAP		0x10000
#define MAPPING			0x20000

static int flagArray[LASTEvent] = {
   /* Not used */		0,
   /* Not used */		0,
   /* KeyPress */		KEY_BUTTON_MOTION,
   /* KeyRelease */		KEY_BUTTON_MOTION,
   /* ButtonPress */		KEY_BUTTON_MOTION,
   /* ButtonRelease */		KEY_BUTTON_MOTION,
   /* MotionNotify */		KEY_BUTTON_MOTION,
   /* EnterNotify */		CROSSING,
   /* LeaveNotify */		CROSSING,
   /* FocusIn */		FOCUS,
   /* FocusOut */		FOCUS,
   /* KeymapNotify */		0,
   /* Expose */			EXPOSE,
   /* GraphicsExpose */		EXPOSE,
   /* NoExpose */		0,
   /* VisibilityNotify */	VISIBILITY,
   /* CreateNotify */		CREATE,
   /* DestroyNotify */		0,
   /* UnmapNotify */		0,
   /* MapNotify */		MAP,
   /* MapRequest */		0,
   /* ReparentNotify */		REPARENT,
   /* ConfigureNotify */	CONFIG,
   /* ConfigureRequest */	CONFIG_REQ,
   /* GravityNotify */		0,
   /* ResizeRequest */		RESIZE_REQ,
   /* CirculateNotify */	0,
   /* CirculateRequest */	0,
   /* PropertyNotify */		PROP,
   /* SelectionClear */		SEL_CLEAR,
   /* SelectionRequest */	SEL_REQ,
   /* SelectionNotify */	SEL_NOTIFY,
   /* ColormapNotify */		COLORMAP,
   /* ClientMessage */		0,
   /* MappingNotify */		MAPPING
};


/*
 *----------------------------------------------------------------------
 *
 * TestEventCmd --
 *
 *	This procedure implements the "testevent" command.  It allows
 *	events to be generated on the fly, for testing event-handling.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Creates and handles events.
 *
 *----------------------------------------------------------------------
 */

int
Tix_TestEventCmd(clientData, interp, argc, argv)
    ClientData clientData;		/* Main window for application. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    Tk_Window main = (Tk_Window) clientData;
    Tk_Window tkwin, tkwin2;
    XEvent event;
    EventInfo *eiPtr;
    char *field, *value;
    int i, number, flags;
    KeySym keysym;

    if ((argc < 3) || !(argc & 1)) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" window type ?field value field value ...?\"",
		(char *) NULL);
	return TCL_ERROR;
    }
    tkwin = Tk_NameToWindow(interp, argv[1], main);
    if (tkwin == NULL) {
	return TCL_ERROR;
    }

    /*
     * Get the type of the event.
     */

    memset((VOID *) &event, 0, sizeof(event));
    for (eiPtr = eventArray; ; eiPtr++) {
	if (eiPtr->name == NULL) {
	    Tcl_AppendResult(interp, "bad event type \"", argv[2],
		    "\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (strcmp(eiPtr->name, argv[2]) == 0) {
	    event.xany.type = eiPtr->type;
	    break;
	}
    }

    /*
     * Fill in fields that are common to all events.
     */

    event.xany.serial = NextRequest(Tk_Display(tkwin));
    event.xany.send_event = False;
    event.xany.window = Tk_WindowId(tkwin);
    event.xany.display = Tk_Display(tkwin);

    /*
     * Process the remaining arguments to fill in additional fields
     * of the event.
     */

    flags = flagArray[event.xany.type];
    for (i = 3; i < argc; i += 2) {
	field = argv[i];
	value = argv[i+1];
	if (strcmp(field, "-above") == 0) {
	    tkwin2 = Tk_NameToWindow(interp, value, main);
	    if (tkwin2 == NULL) {
		return TCL_ERROR;
	    }
	    event.xconfigure.above = Tk_WindowId(tkwin2);
	} else if (strcmp(field, "-borderwidth") == 0) {
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    event.xcreatewindow.border_width = number;
	} else if (strcmp(field, "-button") == 0) {
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    event.xbutton.button = number;
	} else if (strcmp(field, "-count") == 0) {
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (flags & EXPOSE) {
		event.xexpose.count = number;
	    } else if (flags & MAPPING) {
		event.xmapping.count = number;
	    }
	} else if (strcmp(field, "-detail") == 0) {
	    if (flags & (CROSSING|FOCUS)) {
		if (strcmp(value, "NotifyAncestor") == 0) {
		    number = NotifyAncestor;
		} else if (strcmp(value, "NotifyVirtual") == 0) {
		    number = NotifyVirtual;
		} else if (strcmp(value, "NotifyInferior") == 0) {
		    number = NotifyInferior;
		} else if (strcmp(value, "NotifyNonlinear") == 0) {
		    number = NotifyNonlinear;
		} else if (strcmp(value, "NotifyNonlinearVirtual") == 0) {
		    number = NotifyNonlinearVirtual;
		} else if (strcmp(value, "NotifyPointer") == 0) {
		    number = NotifyPointer;
		} else if (strcmp(value, "NotifyPointerRoot") == 0) {
		    number = NotifyPointerRoot;
		} else if (strcmp(value, "NotifyDetailNone") == 0) {
		    number = NotifyDetailNone;
		} else {
		    Tcl_AppendResult(interp, "bad detail \"", value, "\"",
			    (char *) NULL);
		    return TCL_ERROR;
		}
		if (flags & FOCUS) {
		    event.xfocus.detail = number;
		} else {
		    event.xcrossing.detail = number;
		}
	    } else if (flags & CONFIG_REQ) {
		if (strcmp(value, "Above") == 0) {
		    number = Above;
		} else if (strcmp(value, "Below") == 0) {
		    number = Below;
		} else if (strcmp(value, "TopIf") == 0) {
		    number = TopIf;
		} else if (strcmp(value, "BottomIf") == 0) {
		    number = BottomIf;
		} else if (strcmp(value, "Opposite") == 0) {
		    number = Opposite;
		} else {
		    Tcl_AppendResult(interp, "bad detail \"", value, "\"",
			    (char *) NULL);
		    return TCL_ERROR;
		}
		event.xconfigurerequest.detail = number;
	    }
	} else if (strcmp(field, "-focus") == 0) {
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    event.xcrossing.focus = number;
	} else if (strcmp(field, "-height") == 0) {
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (flags & EXPOSE) {
		 event.xexpose.height = number;
	    } else if (flags & (CONFIG|CONFIG_REQ)) {
		event.xconfigure.height = number;
	    } else if (flags & RESIZE_REQ) {
		event.xresizerequest.height = number;
	    }
	} else if (strcmp(field, "-keycode") == 0) {
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    event.xkey.keycode = number;
	} else if (strcmp(field, "-keysym") == 0) {
	    keysym = TkStringToKeysym(value);
	    if (keysym == NoSymbol) {
		Tcl_AppendResult(interp, "unknown keysym \"", value,
			"\"", (char *) NULL);
		return TCL_ERROR;
	    }
	    number = XKeysymToKeycode(event.xany.display, keysym);
	    if (number == 0) {
		Tcl_AppendResult(interp, "no keycode for keysym \"", value,
			"\"", (char *) NULL);
		return TCL_ERROR;
	    }
	    event.xkey.keycode = number;
	} else if (strcmp(field, "-mode") == 0) {
	    if (strcmp(value, "NotifyNormal") == 0) {
		number = NotifyNormal;
	    } else if (strcmp(value, "NotifyGrab") == 0) {
		number = NotifyGrab;
	    } else if (strcmp(value, "NotifyUngrab") == 0) {
		number = NotifyUngrab;
	    } else if (strcmp(value, "NotifyWhileGrabbed") == 0) {
		number = NotifyWhileGrabbed;
	    } else {
		Tcl_AppendResult(interp, "bad mode \"", value, "\"",
			(char *) NULL);
		return TCL_ERROR;
	    }
	    if (flags & CROSSING) {
		event.xcrossing.mode = number;
	    } else if (flags & FOCUS) {
		event.xfocus.mode = number;
	    }
	} else if (strcmp(field, "-override") == 0) {
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (flags & CREATE) {
		event.xcreatewindow.override_redirect = number;
	    } else if (flags & MAP) {
		event.xmap.override_redirect = number;
	    } else if (flags & REPARENT) {
		event.xreparent.override_redirect = number;
	    } else if (flags & CONFIG) {
		event.xconfigure.override_redirect = number;
	    }
	} else if (strcmp(field, "-place") == 0) {
	    if (strcmp(value, "PlaceOnTop") == 0) {
		event.xcirculate.place = PlaceOnTop;
	    } else if (strcmp(value, "PlaceOnBottom") == 0) {
		event.xcirculate.place = PlaceOnBottom;
	    } else if (strcmp(value, "bogus") == 0) {
		event.xcirculate.place = 147;
	    } else {
		Tcl_AppendResult(interp, "bad place \"", value, "\"",
			(char *) NULL);
		return TCL_ERROR;
	    }
	} else if (strcmp(field, "-root") == 0) {
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    event.xkey.root = number;
	} else if (strcmp(field, "-rootx") == 0) {
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    event.xkey.x_root = number;
	} else if (strcmp(field, "-rooty") == 0) {
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    event.xkey.y_root = number;
	} else if (strcmp(field, "-sendevent") == 0) {
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    event.xany.send_event = number;
	} else if (strcmp(field, "-serial") == 0) {
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    event.xany.serial = number;
	} else if (strcmp(field, "-state") == 0) {
	    if (flags & KEY_BUTTON_MOTION) {
		if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		    return TCL_ERROR;
		}
		event.xkey.state = number;
	    } else if (flags & CROSSING) {
		if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		    return TCL_ERROR;
		}
		event.xcrossing.state = number;
	    } else if (flags & VISIBILITY) {
		if (strcmp(value, "VisibilityUnobscured") == 0) {
		    number = VisibilityUnobscured;
		} else if (strcmp(value, "VisibilityPartiallyObscured") == 0) {
		    number = VisibilityPartiallyObscured;
		} else if (strcmp(value, "VisibilityFullyObscured") == 0) {
		    number = VisibilityFullyObscured;
		} else {
		    Tcl_AppendResult(interp, "bad state \"", value, "\"",
			    (char *) NULL);
		    return TCL_ERROR;
		}
		event.xvisibility.state = number;
	    }
	} else if (strcmp(field, "-subwindow") == 0) {
	    tkwin2 = Tk_NameToWindow(interp, value, main);
	    if (tkwin2 == NULL) {
		return TCL_ERROR;
	    }
	    event.xkey.subwindow = Tk_WindowId(tkwin2);
	} else if (strcmp(field, "-time") == 0) {
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (flags & (KEY_BUTTON_MOTION|PROP|SEL_CLEAR)) {
		event.xkey.time = (Time) number;
	    } else if (flags & SEL_REQ) {
		event.xselectionrequest.time = (Time) number;
	    } else if (flags & SEL_NOTIFY) {
		event.xselection.time = (Time) number;
	    }
	} else if (strcmp(field, "-valueMask") == 0) {
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    event.xconfigurerequest.value_mask = number;
	} else if (strcmp(field, "-width") == 0) {
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (flags & EXPOSE) {
		event.xexpose.width = number;
	    } else if (flags & (CONFIG|CONFIG_REQ)) {
		event.xconfigure.width = number;
	    } else if (flags & RESIZE_REQ) {
		event.xresizerequest.width = number;
	    }
	} else if (strcmp(field, "-window") == 0) {
	    tkwin2 = Tk_NameToWindow(interp, value, main);
	    if (tkwin2 == NULL) {
		return TCL_ERROR;
	    }
	    event.xmap.window = Tk_WindowId(tkwin2);
	} else if (strcmp(field, "-x") == 0) {
	    int rootX, rootY;
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    Tk_GetRootCoords(tkwin, &rootX, &rootY);
	    rootX += number;
	    if (flags & KEY_BUTTON_MOTION) {
		event.xkey.x = number;
		event.xkey.x_root = rootX;
	    } else if (flags & EXPOSE) {
		event.xexpose.x = number;
	    } else if (flags & (CREATE|CONFIG|GRAVITY|CONFIG_REQ)) {
		event.xcreatewindow.x = number;
	    } else if (flags & REPARENT) {
		event.xreparent.x = number;
	    } else if (flags & CROSSING) {
		event.xcrossing.x = number;
		event.xcrossing.x_root = rootY;
	    }
	} else if (strcmp(field, "-y") == 0) {
	    int rootX, rootY;
	    if (Tcl_GetInt(interp, value, &number) != TCL_OK) {
		return TCL_ERROR;
	    }
	    Tk_GetRootCoords(tkwin, &rootX, &rootY);
	    rootY += number;
	    if (flags & KEY_BUTTON_MOTION) {
		event.xkey.y = number;
		event.xkey.y_root = rootY;
	    } else if (flags & EXPOSE) {
		event.xexpose.y = number;
	    } else if (flags & (CREATE|CONFIG|GRAVITY|CONFIG_REQ)) {
		event.xcreatewindow.y = number;
	    } else if (flags & REPARENT) {
		event.xreparent.y = number;
	    } else if (flags & CROSSING) {
		event.xcrossing.y = number;
		event.xcrossing.y_root = rootY;
	    }
	} else {
	    Tcl_AppendResult(interp, "bad option \"", field, "\"",
		    (char *) NULL);
	    return TCL_ERROR;
	}
    }
    Tk_HandleEvent(&event);
    return TCL_OK;
}
